/**
 * jQuery json-viewer
 * @author: Alexandre Bodelot <alexandre.bodelot@gmail.com>
 */

  /**
   * Check if arg is either an array with at least 1 element, or a dict with at least 1 key
   * @return boolean
   */
  function isCollapsable(arg) {
    return arg instanceof Object && Object.keys(arg).length > 0;
  }

  /**
   * Check if a string represents a valid url
   * @return boolean
   */
  function isUrl(string) {
     var regexp = /^(ftp|http|https):\/\/(\w+:{0,1}\w*@)?(\S+)(:[0-9]+)?(\/|\/([\w#!:.?+=&%@!\-\/]))?/;
     return regexp.test(string);
  }

  /**
   * Transform a json object into html representation
   * @return string
   */
  function json2html(baseDom, json, options) {
    var html = '';
    if (typeof json === 'string') {
      /* Escape tags */
      json = json.replace(/&/g, '&amp;').replace(/</g, '&lt;').replace(/>/g, '&gt;');
      if (isUrl(json))
        html += '<a href="' + json + '" class="json-string">' + json + '</a>';
      else
        html += '<span class="json-string">"' + json + '"</span>';
    }
    else if (typeof json === 'number') {
      html += '<span class="json-literal">' + json + '</span>';
    }
    else if (typeof json === 'boolean') {
      html += '<span class="json-literal">' + json + '</span>';
    }
    else if (json === null) {
      html += '<span class="json-literal">null</span>';
    }
    else if (json instanceof Array) {
      if (json.length > 0) {
        html += '[<ol class="json-array">';
        
        var indexHtml;
        for (var i = 0; i < json.length; ++i) {
          html += '<li>';

          // 2018-7-19 falcon add index for array items
          indexHtml ='<label class="dw-array-index">' + i + ':</label>';
          
          /* Add toggle button if item is collapsable */
          if (isCollapsable(json[i])) {
            //html += '<a a class="json-toggle" onclick="jsonToggleClicked(this, &quot;' + baseDom.id + '&quot;)"></a>';
        	  html += '<a a class="json-toggle" onclick="jsonToggleClicked(this, &quot;' + baseDom.id + '&quot;)">' + indexHtml + '</a>';
          }
          else {
        	  // 2018-7-19 falcon add index for array items
        	  html += indexHtml;
          }
          
          html += json2html(baseDom, json[i], options);
          /* Add comma if item is not last */
          if (i < json.length - 1) {
            html += ',';
          }
          
          html += '</li>';
        }
        html += '</ol>]';
      }
      else {
        html += '[]';
      }
    }
    else if (typeof json === 'object') {
      var key_count = Object.keys(json).length;
      if (key_count > 0) {
        html += '{<ul class="json-dict">';
        for (var key in json) {
          if (json.hasOwnProperty(key)) {
            html += '<li class="dw-json-property">';
            
            // 2018-7-6 falcon 加上 <a class="dw-json-property-name">
            var keyRepr = options.withQuotes ?
              '<span class="json-string">"' + '<a class="dw-json-property-name">' + key + "</a>" + '"</span>' : '<a class="dw-json-property-name">' + key + '</a>';
            /* Add toggle button if item is collapsable */
            if (isCollapsable(json[key])) {
              html += '<a a class="json-toggle" onclick="jsonToggleClicked(this, &quot;' + baseDom.id + '&quot;)">' + keyRepr + '</a>';
            }
            else {
              html += keyRepr;
            }
            html += ': ' + json2html(baseDom, json[key], options);
            /* Add comma if item is not last */
            if (--key_count > 0)
              html += ',';
            html += '</li>';
          }
        }
        html += '</ul>}';
      }
      else {
        html += '{}';
      }
    }
    return html;
  }

  /**
   * jQuery plugin method
   * @param json: a javascript object
   * @param options: an optional options hash
   */
  function render(baseDom, json, options) {
    options = options || {};

    $(baseDom).attr("options", 1);
    
      /* Transform to HTML */
      var html = json2html(baseDom, json, options);
      if (isCollapsable(json))
        html = '<a a class="json-toggle" onclick="jsonToggleClicked(this, &quot;' + baseDom.id + '&quot;)"></a>' + html;

      /* Insert HTML in target DOM element */
      $(baseDom).html(html);

      /* Bind click on toggle buttons */
      $(baseDom).off('click');
//      $(baseDom).on('click', 'a.json-toggle', jsonToggleClicked);
//
//      /* Simulate click on toggle button when placeholder is clicked */
//      $(baseDom).on('click', 'a.json-placeholder', jsonPlaceholderClicked);

      if (options.collapsed == true) {
        /* Trigger click to collapse all nodes */
        $(baseDom).find('a.json-toggle').click();
        
        // 2018-7-12 falcon 加入 document 前, toggle 會無效, 所以此處直接抓對象出來設定 
        $(baseDom).find('ul.json-dict, ol.json-array').css("display", "none");
      }

      onJsonViewerDisplayPropertyCountChanged(options, baseDom);
  };
  
  function onJsonViewerDisplayPropertyCountChanged(options, baseDom) {
	  
	  var list = $(baseDom).find("a.dw-object-placeholder");
	  
	  if (options.showPropertySeq) {
			
		  list.css("display", "");
	  }
	  else {
			
		  list.css("display", "none");
	  }
  }
  
  function jsonToggleClicked(sourceTarget, baseDomId) {
      var target = $(sourceTarget).toggleClass('collapsed').siblings('ul.json-dict, ol.json-array');
      target.toggle();
      
      // 2018-7-18 falcon 在加入 document 之前, :visible 判斷並不準確
      var isOpen = !$(sourceTarget).attr('class').split(/\s+/).includes("collapsed");
      
      if (target.is(':visible') || isOpen) {
        target.siblings('.json-placeholder').remove();
      }
      else {
        var count = target.children('li').length;
        var placeholder;
        var isArray = target.length > 0 ? target.attr('class').split(/\s+/).includes("json-array") : false;
        if (isArray) {
        	placeholder = "Array " + count + (count > 1 ? ' items' : ' item');
        }
        else {
        	// placeholder = count + (count > 1 ? ' items' : ' item');
        	placeholder = ".." + count;
        }
        
        var aclassValue = isArray ? "json-placeholder dw-array-placeholder" : "json-placeholder dw-object-placeholder";
        
        var placeholderObject = target.after('<a a class="' + aclassValue + '" onclick="jsonPlaceholderClicked()">' + placeholder + '</a>');
        
        // 2018-7-16 當 dom 還未加入 document 前是獲取不到 angular scope 的
        var angularScope = angular.element(document.getElementById(baseDomId)).scope();
        if (angularScope != undefined) {
        	
        	if (angularScope.tempData && angularScope.tempData.jsonviewer && !angularScope.tempData.jsonviewer.showPropertySeq) {
        		
        		placeholderObject.siblings("a.dw-object-placeholder").css("display", "none");
        	}
        }
      }
      return false;
  }
  
  function jsonPlaceholderClicked() {
      $(event.target).siblings('a.json-toggle').click();
      return false;
   }
