﻿
/**
 * 2016-11-22 falcon
 * Digiwin Unit Test Project Object 
 * @param name 專案名稱
 * @param tokenServiceInfo 2016-12-3 金鑰服務信息
 */
function DWUnitTestProject(name, tokenServiceInfo) {
	
	// project name
	this.name = name;
	
	// project description
	this.description = name;
	
	// test case token getter service
	this.tokenInfo = {}; // 金鑰取得信息
	if (tokenServiceInfo) {
		// 設定金鑰服務信息
		this.tokenInfo.serviceInfo = tokenServiceInfo;
	}
	
	// token service
	this.tokenInfo.serviceUrl = null; // 金鑰服務 URL (字串)
	this.tokenInfo.serviceParameters = null; // 金鑰服務參數 (字串)
	this.tokenInfo.fixedToken = null; // 固定金鑰值
	
	
	// test case list
	this.testCases = [];
}

/**
 * 設定金鑰服務信息
 * @param tokenServiceInfo 金鑰服務信息
 */
DWUnitTestProject.prototype.setTokenServiceInfo = function(tokenServiceInfo) {

	this.tokenInfo.serviceInfo = tokenServiceInfo;
}

/**
 * 加入新的測試案例
 */
DWUnitTestProject.prototype.addNewTestCase = function() {
	
	var newTestCase = new DWTestCase();
	newTestCase.id = this.generateGuid();
	this.testCases[this.testCases.length] = newTestCase;
	
	newTestCase.tokenInfo = { "pass": true };
	
	var uuid = DWUnitTestProject.prototype.generateGuid();
	newTestCase.idempotencyInfo = { "pass": true, "value": uuid };
	
	newTestCase.assertInfo = { "check" : true };
	
	return newTestCase;
}

/**
 * 產生 Guid
 * reference : http://stackoverflow.com/questions/105034/create-guid-uuid-in-javascript
 */
DWUnitTestProject.prototype.generateGuid = function() {
	function s4() {
		return Math.floor((1 + Math.random()) * 0x10000)
				.toString(16)
				.substring(1);
		  }
	return s4() + s4() + '-' + s4() + '-' + s4() + '-' +
			s4() + '-' + s4() + s4() + s4();
}

/**
 * 是否有選取的測試案例
 */
DWUnitTestProject.prototype.hasSelectedTestCases = function() {

	for (itemIdx in this.testCases) {
		
		if (this.testCases[itemIdx].selected) {
			return true;
		}
	}
	
	return false;
}

/**
 * 刪除選取的測試案例
 */
DWUnitTestProject.prototype.deleteSelectedTestCases = function() {
	
	var deletedItems = [];
	var lastIndex = this.testCases.length - 1;
	for (var i = lastIndex; i >= 0; i--) {
		
		if (this.testCases[i].selected) {
			deletedItems[deletedItems.length] = this.testCases[i];
			this.testCases.splice(i, 1);
		}
	}
	
	return deletedItems;
}

/**
 * 取得標準格式的測試報告
 * @param onlySelectedTestCase 只取選取的測試案例測試結果
 * @param description 描述
 * @return 標準格式的測試報告
 */
DWUnitTestProject.prototype.getStandardFormatTestingReport = function(onlySelectedTestCase, description) {
	
	var testingReport = {};
	
	if (typeof description == 'undefined') {
		
		testingReport.description = this.description;
	}
	else {
	
		testingReport.description = description;
	}

	var testResults = this.getTestingResults(onlySelectedTestCase);
	testingReport._results = testResults;
	
	return testingReport;
}

/**
 * 轉為 Digiwin 單元測試報告提交服務信息
 * @param token 使用者金鑰
 * @param standardFormatTestingReport 標準格式測試報告
 */
DWUnitTestProject.prototype.toDWUnitTestrReportSumbitServiceInfo = function(token, standardFormatTestingReport) {

	var dwUnitTestSubmitServiceInfo = {};
	var parameterList = [];

	parameterList[0] = { "name":"userId", "value": "00320" };
	parameterList[1] = { "name":"password", "value": "00320" };
	parameterList[2] = { "name":"description", "value": standardFormatTestingReport.description };
	
	// 2016-11-25 falcon 值先用 JSON 轉為字串, 否則到後面 GSON 轉不了
	parameterList[3] = { "name":"report", "value": JSON.stringify(standardFormatTestingReport) }; 
	
	dwUnitTestSubmitServiceInfo.parameters = parameterList;
	
	// 2017-3-17 falcon 設定 token
	dwUnitTestSubmitServiceInfo.headers = [];
	dwUnitTestSubmitServiceInfo.headers[0] = { "name":"token", "value": token };
	
	return dwUnitTestSubmitServiceInfo;
}

/**
 * 取得測試結果清單
 * @param onlySelectedTestCase 只取選取的測試案例測試結果
 * @return 測試結果清單
 */
DWUnitTestProject.prototype.getTestingResults = function(onlySelectedTestCase) {
	
	var testResults = [];
	
	var testCase;
	for (var idx in this.testCases) {
		
		testCase = this.testCases[idx];
		
		if (testCase.testingResult && ! testCase.testingResult.exception) {
			
			testResults[testResults.length] = testCase.testingResult;
		}
	}
	
	return testResults;
}

/**
 * 拷貝
 * @param objectTypeInstance 物件型別的實例
 */
DWUnitTestProject.prototype.copy = function(objectTypeInstance) {

	$.extend(true, this, objectTypeInstance);
	
	var testCase;
	var objectTypeTestCase;
	for (var idx in this.testCases) {
		
		objectTypeTestCase = this.testCases[idx];
		testCase = new DWTestCase();
		
		testCase.copy(objectTypeTestCase);
		
		// 覆蓋為還原的測試案例
		this.testCases[idx] = testCase;
	}
}

//===================================================================================

/**
 * 2016-11-22 falcon
 * Digiwin Unit Test Project Object 
 */
function DWTestCase() {
	
	this.serviceCommand = {};
	this.serviceCommand.moduleName = null;
	this.serviceCommand.serviceName = null;
	this.serviceCommand.methodName = null;
	this.serviceCommand.parameters = null;
	
	// 金鑰
	this.token = null
	
	// 測試結果
	this.testingResult = { "state": null }
	
	// 測試報告
	this.testingReport = null;
	
	// temp data property
	this.tempData = {};

	// view property
	this.selected = false;
	this.showReport = false;
}

/**
 * 轉為服務信息
 */
DWTestCase.prototype.toServiceInfo = function() {
	
	var serviceInfo = {};
	
	var moduleName = this.serviceCommand.moduleName;
	var serviceName = this.serviceCommand.serviceName;
	var methodName = this.serviceCommand.methodName;
	serviceInfo.url = getBasicUrlList()[0] + "/" +
		moduleName + "/" + serviceName + "/" + methodName;

	serviceInfo.parameters = this.parameters;
	
	return serviceInfo;
}

/**
 * 轉為 Digiwin 單元測試服務信息
 * @return Digiwin 單元測試服務信息
 */
DWTestCase.prototype.toDWUnitTestServiceInfo = function() {

	var dwUnitTestServiceInfo = {};
	
	var parameterList = [];
	parameterList[0] = { "name":"description", "value":"Unit Test" };

	var clonedServiceCommand = $.extend(true, {}, this.serviceCommand);

	// 處理 冪等性編號
	if (this.idempotencyInfo && this.idempotencyInfo.pass) {
		
		var id = this.idempotencyInfo.value ? this.idempotencyInfo.value : "";

		if (clonedServiceCommand.parameters) {
			
			try {
				
				var jsonObject = JSON.parse(clonedServiceCommand.parameters);
				
				if ($.isArray(jsonObject)) {
					
					clonedServiceCommand.idempotencyIdMergeFailedReason = "JSON String is Array!";
				}
				else {
					
					jsonObject["$id"] = id;
				}
			}
			catch (e) {
				
				// 冪等性編號合併失敗
				clonedServiceCommand.idempotencyIdMergeFailedReason = "JSON String parse failed!";
			}
		}
		else {
			
			clonedServiceCommand.parameters = { "$id": id }
		}
	}
	
	parameterList[1] = { "name":"serviceCommands", "value": [ clonedServiceCommand ] };
	
	// copy array properties
	dwUnitTestServiceInfo.parameters = parameterList;

	// 2016-1-4 falcon 處理 upload files
	if (this.uploadFiles) {
		
		var clonedUploadFiles = [];
		
		var uploadFile;
		var uploadFileParameterNameWithId;
		var uploadFileParameterNamePrefix = "uploadFiles$";
		for (var idx in this.uploadFiles) {
			
			uploadFile = $.extend(true, {}, this.uploadFiles[idx]);
			uploadFileParameterNameWithId = uploadFileParameterNamePrefix;
			if (uploadFile.name) {
				
				uploadFileParameterNameWithId += uploadFile.name;
			}
			
			uploadFile.name = uploadFileParameterNameWithId;
			clonedUploadFiles.push(uploadFile);
		}
	}
	
	dwUnitTestServiceInfo.uploadFiles = clonedUploadFiles;
	
	return dwUnitTestServiceInfo;
}

/**
 * 當本身無金鑰時將設定金鑰
 * @param token 金鑰
 */
DWTestCase.prototype.setTokenIfEmpty = function(token) {
	
	if (!this.token) {
		this.token = token;
	}
}

/**
 * 設定結果
 * @param latestTestingResult 最後的測試結果
 */
DWTestCase.prototype.setResult = function(latestTestingResult) {
	
	this.testingResult = latestTestingResult;
	
	this.createReport();
}

/**
 * 清除結果
 */
DWTestCase.prototype.clearResult = function() {
	
	this.testingResult = { "state": null }
}

/**
 * 更新報表狀態
 */
DWTestCase.prototype.updateReportState = function() {
	
	if (this.testingReport) {
	
		this.testingReport.state = this.getStateTooltip(); // 狀態說明
	}
}

/**
 * 建立測試報表
 */
DWTestCase.prototype.createReport = function() {
	
	this.testingReport = {};
	
	this.testingReport.token = this.token; // 使用的金鑰

	this.updateReportState();
	
	this.testingReport.startTime = "-"; // 開始測試時間
	this.testingReport.responseTime = "-"; // 回應時間(花費時間)
	if (this.testingResult) {
	
		// 開始使間
		if (this.testingResult.testingTime) {
			
			this.testingReport.startTime = this.testingResult.testingTime;
		}
		// 耗時
		if (this.testingResult.testingResponseTime || this.testingResult.testingResponseTime === 0) {
			
			var diffs = this.testingResult.testingResponseTime;
			
			// 格式化 long 時間
	        // var second = (diffs / 1000) % 60;
	        //2016-12-26 falcon issue -> 值 =62191, (diffs / 1000) % 60 => 2.1910000000000025???
			var second = ((diffs / 1000) % 60).toFixed(3); // 2016-12-26 falcon 強制取 3 位
			
	        var minute = Math.floor((diffs / (1000 * 60)) % 60)
	        var hour = Math.floor((diffs / (1000 * 60 * 60)) % 24);
	        //var millSeconds = (diffs % 1000) / 1000;
	        
	        var format = hour + " 時 " + minute + " 分 " + second + "秒" 
	        
	        this.testingReport.responseTime = format;
		}
		// 結果
		if (this.testingResult.exception) {
			// 這是在 dwUnitTester.afterRunTestCase 方法中處理的屬性
			this.testingReport.result = this.testingResult.exception;
		}
		else if (this.testingResult.testingResult) {
			this.testingReport.result = this.testingResult.testingResult;
		}
	}
}

/**
 * 取得狀態(0-pending, 1-success, 2-failure, 3-斷言失敗)
 */
DWTestCase.prototype.getState = function() {
	
	if (this.testingResult && this.testingResult.state) {
		return this.testingResult.state;
	}
	
	return 0;
}
/**
 * 設定狀態
 * @param state 狀態
 */
DWTestCase.prototype.setState = function(state) {
	
	if (this.testingResult) {
		
		this.testingResult.state = state;
		this.updateReportState();
	}
}

/**
 * 狀態(0-pending, 1-success, 2-failure)
 */
DWTestCase.prototype.getStateTooltip = function() {
	
	var state = this.getState();
	
	if (state == 0) {
		return "準備中";
	}
	else if (state == 1) {
		return "測試通過";
	}
	else if (state == 2) {
		return "測試失敗";
	}
	else if (state == 3) {
		return "斷言失敗";
	}
	
	return "未知的狀態=" + state;
}

/**
 * 拷貝
 * @param objectTypeInstance 物件型別的實例
 */
DWTestCase.prototype.copy = function(objectTypeInstance) {
	
	$.extend(true, this, objectTypeInstance);
	
	// 2017-1-23 falcon 還原 DWAssert
	if (objectTypeInstance && objectTypeInstance.assertInfo && objectTypeInstance.assertInfo.items) {

		var assert;
		var reloadType;
		var initScript;
		var objectTypeAssert;
		for (var idx in objectTypeInstance.assertInfo.items) {

			objectTypeAssert = objectTypeInstance.assertInfo.items[idx];
			reloadType = objectTypeAssert.reloadType;
			if (!reloadType) {
				
				throw new Error("idx=" + idx + " assert.reloadType is missing");
			}
			
			initScript = "new " + reloadType + "()";
			
			assert = eval(initScript);
			assert.copy(objectTypeAssert);
			
			// 覆蓋為還原的斷言
			this.assertInfo.items[idx] = assert;
		}
	}
}

//===================================================================================

/**
 * 2017-1-18 falcon
 * Digiwin 測試結果 Wrapper
 * @param unitTestServiceResult 單元測試服務結果
 */
function DWTestingResultWrapper(unitTestServiceResult) {
	
	if (!unitTestServiceResult) return;
	
	var testReport = unitTestServiceResult.getResult();
	if (testReport && testReport._results) {
		
		this.rawData = testReport._results[0];
	}
	
	this.initialize();
}

/**
 * 初始化
 */
DWTestingResultWrapper.prototype.initialize = function() {
	
	// do nothing
}

/**
 * 2017-1-18 falcon
 * 取得執行時間(毫秒)
 * @return 執行時間(毫秒)
 */
DWTestingResultWrapper.prototype.getDuration = function() {
	
	if (!this.rawData) return 0;
	
	return this.rawData.testingResponseTime;
}
/**
 * 2017-1-18 falcon
 * 取得結果
 * @return 結果
 */
DWTestingResultWrapper.prototype.getResult = function() {
	
	if (!this.rawData) return "";
	
	return this.rawData.testingResult;
}

//===================================================================================

/**
 * 2016-11-23 falcon
 */
function DWUnitTestExecutor() {
}

/**
 * 執行
 * @param unitTestProject 單元測試計畫
 * @param controller 控制者
 */
DWUnitTestExecutor.prototype.execute = function(controller) {
	
	if (!controller.currentProject) {
		
		return;
	}
	
	var token = this.getToken(controller.currentProject);
}

/**
 * 取得金鑰
 * @param unitTestProject 單元測試專案
 */
DWUnitTestExecutor.prototype.getToken = function(unitTestProject) {
	
	if (unitTestProject.fixedToken) {
		
		return unitTestProject.fixedToken;
	}
	
	// 由默認的 service info 取得
	if (unitTestProject.tokenInfo.serviceInfo) {
	}
	
	// 由自訂的 service url 取得
}

//===================================================================================

/**
 * 2016-11-30 falcon Digiwin 服務結果 Wrapper
 * @param digiwinServiceReturnResponse Digiwin 服務返回的回應物件
 * @param DWMessage Digiwin Message Service
 */
function DWServiceResultWrapper(digiwinServiceReturnResponse, DWMessage) {

	var target = digiwinServiceReturnResponse.currentTarget;
	if (target && target.constructor == XMLHttpRequest) {
	
		var virtualResponse = {};
		virtualResponse.status = target.status;
		
		try {
			
			if (typeof target.response === "string") {
				
				virtualResponse.data = JSON.parse(target.response);
			}
			else {
				
				virtualResponse.data = target.response;
			}
		}
		catch (e) {
			
			DWMessage.showWarning("嘗試解析結果為 JSON 物件時失敗!");
			virtualResponse.data = target.response;
		}
		
		this.response = virtualResponse;
	} 
	else {
		this.response = digiwinServiceReturnResponse;
	}
}

/**
 * 是否連線成功
 */
DWServiceResultWrapper.prototype.isSuccess = function() {
	
	// TODO the code below is only for chrome ?
	return this.getResponse().status == 200;
}

/**
 * 是否為連線問題
 */
DWServiceResultWrapper.prototype.isConnectionError = function() {
	
	// TODO the code below is only for chrome ?
	return this.getResponse().status == -1;
}

/**
 * 是否找不到微服務
 */
DWServiceResultWrapper.prototype.isServiceNotFound = function() {
	
	// TODO the code below is only for chrome ?
	return this.getResponse().status == 400;
}

/**
 * 取得原始結果 (標準格式)
 * @return 回應物件
 */
DWServiceResultWrapper.prototype.getResponse = function() {
	
	return this.response;
}

/**
 * 2017-1-15 falcon
 * 取得回應失敗的原因
 * @return 失敗的原因
 */
DWServiceResultWrapper.prototype.getFailedReason = function() {
	
	if (this.isConnectionError()) {
		
		return "無法連接 DWGateway 服務器!";
	}
	else if (!this.isSuccess()) {
		
		systemInfo = this.getSystemInfo();
		if (systemInfo && systemInfo.debugInfo && systemInfo.debugInfo.stackTrace) {
			
			return systemInfo.debugInfo.stackTrace;
		}
	}
	
	return "Unknown";
}

/**
 * 2017-1-15 falcon
 * 取得狀態的描述
 * @param 取得狀態的描述
 */
DWServiceResultWrapper.prototype.getStatus = function() {
	
	if (this.isConnectionError()) {
		
		return "無法連接 DWGateway 服務器!";
	}
	else {
		
		systemInfo = this.getSystemInfo();
		if (systemInfo && systemInfo.status) {
			
			var status = systemInfo.status;
			
			if (systemInfo.statusDescription) {
				
				status += " " + systemInfo.statusDescription;
			}
			
			return status;
		}
		else {
			
			return this.getResponse().status;
		}
	}
}

/**
 * 2016-11-30 falcon
 * 取得結果
 * @return 調用服務的結果
 */
DWServiceResultWrapper.prototype.getResult = function() {

	if (this.isSuccess()) {
		
		// 標準格式
		var result = this.response.data.response;
		
		// 2018-7-6 falcon 其他的調用並沒有 .response 字段
		if (result == undefined) return this.response.data;
		
		return result;
	}
	else if (!this.isConnectionError()) {

		// 2018-7-24 falcon 錯誤時要顯示錯誤 (ex. 如 iam)
		// dap 錯誤時還是有 response 只是是空的, 所以此處不用  ! 判斷
		if (this.response.data && this.response.data.response == undefined) {
			
			return this.response.data;
		}
		
		// TODO 改完 Exception 後, 此處代碼會和成功區塊相同
		// 標準格式
		return this.response.data.response;
	}
	
	return null;
}

/**
 * 2016-12-22 falcon
 * 取得系統信息
 * @return 調用服務的系統信息
 */
DWServiceResultWrapper.prototype.getSystemInfo = function() {
	
	var systemInfo = {};
	if (this.response.data) {
		
		// 2018-7-5 falcon 其他非標準格式沒有 system info
		if (this.response.data.response == undefined) return {};

		if (typeof(this.response.data) == "string") { // 2017-2-16 falcon 特殊處理
			
			systemInfo["unknownResponseData"] = this.response.data;
			
			return systemInfo;
		}
		
		for (var property in this.response.data) {

			if (property != "response") {

				// 2016-12-22 falcon 將 response 以外的屬性抄寫至 systemInfo 中
				systemInfo[property] = this.response.data[property];
			}
		}
	}
	
	this.updateExtraSystemInfo(systemInfo);
	
	return systemInfo;
}

/**
 * 2016-12-22 falcon
 * 更新額外的系統信息
 * @param systemInfo 系統信息
 * @return 更新額外系統信息的對象
 */
DWServiceResultWrapper.prototype.updateExtraSystemInfo = function(systemInfo) {
	
	if (systemInfo && systemInfo.duration) {
		
		var diffs = systemInfo.duration;
	
		// 格式化 long 時間
		var second = (diffs / 1000) % 60;
		var minute = Math.floor((diffs / (1000 * 60)) % 60)
		var hour = Math.floor((diffs / (1000 * 60 * 60)) % 24);
		//var millSeconds = (diffs % 1000) / 1000;
	
		var format = hour + " 時 " + minute + " 分 " + second + "秒"
		
		systemInfo["duration-format"] = format;
	}
}

//====== RAP Mock Data Service Result 以下

/**
 * 2017-1-23 falcon
 * RAP Mock Data 服務 回應物件
 */
function RapMockDataServiceResultWrapper(rapMockDataServiceResponse) {
	
	this.response = rapMockDataServiceResponse;
}

/**
 * 取得結果
 * @return 調用服務的結果 
 */
RapMockDataServiceResultWrapper.prototype.getResult = function() {
	
	var mockReturnObject;
	var rawObject = this.getResponse().data;
	if (rawObject && rawObject.result) {
		
		mockReturnObject = rawObject.result;
	}
	else { // has error
		
		mockReturnObject = rawObject;
	}
	
	if (mockReturnObject && mockReturnObject.result) {
		
		// 規範 (在回應參數定義一個鍵為  "result" 的參數)
		mockReturnObject = mockReturnObject.result;
	}
	
	return mockReturnObject;
}

/**
 * 取得原始結果 (標準格式)
 * @return 回應物件
 */
RapMockDataServiceResultWrapper.prototype.getResponse = function() {
	
	return this.response;
}

RapMockDataServiceResultWrapper.prototype.isSuccess = function() {
	
	// TODO the code below is only for chrome ?
	return this.getResponse().status == 200;
}

RapMockDataServiceResultWrapper.prototype.isNotFound = function() {
	
	var result = this.getResult();
	return  (result && result.errMsg == "no matched action"); 
}

/**
 * 是否為連線問題
 */
RapMockDataServiceResultWrapper.prototype.isConnectionError = function() {
	
	// TODO the code below is only for chrome ?
	return this.getResponse().status == -1;
}

//====== RAP Mock Data Service Result 以上


//===================================================================================

/**
 * 2016-11-29 falcon
 * Digiwin Rap Model Wrapper
 * @param rapQueryAPIReturnResponse RAP Query API 返回的回應物件
 */
function DWRapModelWrapper(rapQueryAPIReturnResponse) {
	
	var rapResponseData = rapQueryAPIReturnResponse.data;
	var modelJSONString = rapResponseData.modelJSON;
	
	this.rawModel = JSON.parse(modelJSONString);
	
	this.dwModuleInfoList = this.convertToModuleInfo();
}

/**
 * 轉換為 Digiwin 模組信息
 */
DWRapModelWrapper.prototype.convertToModuleInfo = function() {
	
	var dwModuleInfoList = [];
	var dwModuleInfo;
	var rapModuleInfo;
	for (var moduleIndex in this.rawModel.moduleList) {
		
		rapModuleInfo = this.rawModel.moduleList[moduleIndex];
		
		dwModuleInfo = {};
		dwModuleInfo.name = rapModuleInfo.name;
		
		dwModuleInfo.serviceList = this.convertToServiceList(rapModuleInfo);
		
		dwModuleInfoList.push(dwModuleInfo);
	}
	
	return dwModuleInfoList;
}

/**
 * 轉換為 Digiwin 服務清單
 * @param rapModuleInfo RAP 模組信息
 * @return Digiwin 服務清單
 */
DWRapModelWrapper.prototype.convertToServiceList = function(rapModuleInfo) {
	
	var serviceList = [];
	
	var dwServiceInfo;
	var rapServiceInfo;
	for (var serviceIndex in rapModuleInfo.pageList) {
		
		rapServiceInfo = rapModuleInfo.pageList[serviceIndex];
		
		dwServiceInfo = {};
		dwServiceInfo.name = rapServiceInfo.name;
		
		dwServiceInfo.methodList = this.convertToMethodList(rapServiceInfo);
		
		serviceList.push(dwServiceInfo);
	}
	
	return serviceList;
}

/**
 * 轉換為 Digiwin 方法清單
 * @param rapServiceInfo RAP 服務信息
 * @return Digiwin 方法清單
 */
DWRapModelWrapper.prototype.convertToMethodList = function(rapServiceInfo) {
	
	var methodList = [];
	
	var dwMethodInfo;
	var rapMethodInfo;
	var rapParameterInfo;
	var parameterName;
	for (var methodIndex in rapServiceInfo.actionList) {
		
		rapMethodInfo = rapServiceInfo.actionList[methodIndex];
		
		dwMethodInfo = {};
		dwMethodInfo.name = rapMethodInfo.name;
		dwMethodInfo.method = dwMethodInfo.name; // 間容舊格式;
		
		dwMethodInfo.allParameters = [];
		dwMethodInfo.parameters = [];
		dwMethodInfo.files = [];
		
		for (var parameterIndex in rapMethodInfo.requestParameterList) {
			
			rapParameterInfo = rapMethodInfo.requestParameterList[parameterIndex];
			parameterName = rapParameterInfo.identifier;
			
			dwMethodInfo.allParameters.push(parameterName);
			dwMethodInfo.parameters.push(parameterName);
		}
		
		var token = "";
		var signature = "(" + dwMethodInfo.allParameters.join(", ") + ")";
		
		dwMethodInfo.signature = signature + token;
		dwMethodInfo.displayName = dwMethodInfo.name + dwMethodInfo.signature;
		
		methodList.push(dwMethodInfo);
	}
	
	return methodList;
}

//===================================================================================

/**
 * Digiwin 使用者數據模型 Wrapper
 * @param response 回應
 */
function DWUserDataModelWrapper(response) {
	
	this.response = response;
}

/**
 * 2016-11-30 falcon
 * 取得結果
 * @return 調用服務的結果
 */
DWUserDataModelWrapper.prototype.getResult = function() {
	
	return this.getResponse().data;
}

/**
 * 取得原始結果 (標準格式)
 * @return 回應物件
 */
DWUserDataModelWrapper.prototype.getResponse = function() {
	
	return this.response;
}

DWUserDataModelWrapper.prototype.isSuccess = function() {
	
	// TODO the code below is only for chrome ?
	return this.getResponse().status == 200;
}

/**
 * 是否為連線問題
 */
DWUserDataModelWrapper.prototype.isConnectionError = function() {
	
	// TODO the code below is only for chrome ?
	return this.getResponse().status == -1;
}

/**
 * 是否為找不到
 */
DWUserDataModelWrapper.prototype.isNotFound = function() {
	
	// TODO the code below is only for chrome ?
	return this.getResponse().status == 404;
}

/**
 * 是否為檔案已經存在
 */
DWUserDataModelWrapper.prototype.isConflict = function() {
	
	// TODO the code below is only for chrome ?
	return this.getResponse().status == 409;
}

//===================================================================================

/**
 * Dwigiwin Assert 
 */
function DWAssert(inheritance) {
}

/**
 * 檢查
 * @param serviceResult 服務執行結果
 * @return 是否成功
 */
DWAssert.prototype.check = function(serviceResult) {
	
	return true;
}

/**
 * 拷貝
 * @param objectTypeInstance 物件型別的實例
 */
DWAssert.prototype.copy = function(objectTypeInstance) {

	$.extend(true, this, objectTypeInstance);
}

/**
 * 類型名稱
 */
DWAssert.prototype.typeName = "斷言";
/**
 * 是否可以編輯
 */
DWAssert.prototype.editable = false;

//====== RAP 默認輸出斷言
/**
 * Digiwin RAP 默認的輸出斷言
 */
function DWRapDefaultOutputAssert() {
	
	/**
	 * call base constructor
	 */
	DWAssert.call(this, null);
	
	this.hasInitialized = false;
	this.lockOutputObject = false;
	
	// 2017-1-23 falcon 載入時還原時使用
	this.reloadType = "DWRapDefaultOutputAssert";
}

/**
 * Javascript class inheritance.
 */
DWRapDefaultOutputAssert.prototype = new DWAssert(null);

DWRapDefaultOutputAssert.prototype.typeName = "RAP輸出";

/**
 * 初始化輸出物件
 * @param assertContext 斷言上下文
 */
DWRapDefaultOutputAssert.prototype.initializeOutputObject = function(assertContext) {
	
	// TODO 取  rap mock data
	if (assertContext.scope.rapMockDataService) {

		var assert = this;
		var serviceCommand = assertContext.testCase.serviceCommand;
		this.onRetrievingOutputObject = true;
		assertContext.scope.rapMockDataService.retrieveRapMockData(serviceCommand)
			.then(
				function(response) { // success

					assert.outputObject = response.getResult();
					assert.onRetrievingOutputObject = false;
				},
				function(response) { // failure

					var errMsg;
					if (response.isNotFound()) {
						
						errMsg = "沒有對應的 Mock 數據!";
					}
					else if (response.isConnectionError()) {
						
						errMsg = "無法連接 RAP 服務器!";
					}
					else {
						
						errMsg = "Unknown Error!";
					}
					
					assert.outputObject = new Error(errMsg);
					assert.onRetrievingOutputObject = false;
				}
			);
	}
}

/**
 * 檢查 (override)
 * @param resultInfo 結果信息
 * @param assertContext 斷言上下文
 * @return 是否成功
 */
DWRapDefaultOutputAssert.prototype.check = function(testingInfo, assertContext) {

	if (!this.lockOutputObject || !this.hasInitialized) {
	
		this.initializeOutputObject(assertContext);
	
		// TODO 改為  DWOutputAssert <- 新增時就下載??
//		while (this.onRetrievingOutputObject) {
//			
//			// 等待
//		} 
		
		var hasError = typeof(this.outputObject) == "object" && this.outputObject.constructor.name == "Error";
		if (hasError) {
			
			var error = this.outputObject;
			this.outputObject = null;
			throw error;
		}
		
		this.hasInitialized = this.lockOutputObject;
	}

	var result = testingInfo.getResult();
	var assertResult = this.outputObject;
	
	var isObjectType = typeof(assertResult) == "object";
	
	// TODO 比對結構
	if (isObjectType) {
		
	}
	else {
		
	}
	
	return true;
}


//====== DWOutputAssert 斷言

function DWOutputAssert(output) {
	
	/**
	 * call base constructor
	 */
	DWAssert.call(this, null);
	
	this.output = JSON.stringify(output, null, "\t");
	
	// 2017-1-23 falcon 載入時還原時使用
	this.reloadType = "DWOutputAssert";
	this.editInfo = {};
	this.editInfo.targetProperty = "output";
}

/**
 * Javascript class inheritance.
 */
DWOutputAssert.prototype = new DWAssert(null);

/**
 * 檢查 (override)
 * @param resultInfo 結果信息
 * @param assertContext 斷言上下文
 * @return 是否成功
 */
DWOutputAssert.prototype.check = function(testingInfo, assertContext) {
	
	var assertResult = null;
	
	try {
		
		if (this.output) {

			assertResult = JSON.parse(this.output);
		}
		else {
			
			assertResult = this.output;
		}
	}
	catch (e) {
		
		var message = "Assert 中的輸出格式無法轉為 JSON 物件!";
		if (e && e.message) {
			
			message += "\r\n";
			message += e.message;
		}
		
		throw new Error(message);
	}

	var result = testingInfo.getResult();
	
	return compareObject(assertResult, result);
}

DWOutputAssert.prototype.typeName = "輸出";
DWOutputAssert.prototype.editable = true;

//====== DWOutputAssert 斷言

//====== Javascript 斷言
/**
 * 構造函數
 * @param script 腳本
 */
function DWScriptAssert(script) {
	
	/**
	 * call base constructor
	 */
	DWAssert.call(this, null);
	
	this.script = script;
	
	// 2017-1-23 falcon 載入時還原時使用
	this.reloadType = "DWScriptAssert";
	this.editInfo = {};
	this.editInfo.targetProperty = "script";
}

/**
 * Javascript class inheritance.
 */
DWScriptAssert.prototype = new DWAssert(null);

/**
 * 檢查 (override)
 * @param resultInfo 結果信息
 * @param assertContext 斷言上下文
 * @return 是否成功
 */
DWScriptAssert.prototype.check = function(testingInfo, assertContext) {
	
	if (!this.script) throw new Error("script is empty or null!");

	var object = eval(this.script);
	
	return object ? true : false;
}

DWScriptAssert.prototype.typeName = "腳本";
DWScriptAssert.prototype.editable = true;

//====== Javascript 斷言

//====== 比對物件

/**
 * 拋出比對失敗的錯誤
 * @param propertyPath 屬性路徑
 * @param failedReason 失敗的原因
 */
function throwCompareFailedError(propertyPath, failedReason) {
	
	var errorMsg = "服務返回的";
	if (propertyPath) {
		
		errorMsg += "路徑 '" + propertyPath + "' 的值  ";
	}
	else {
		
		errorMsg += "服務返回的值  ";
	}
	
	var error = new Error(errorMsg + failedReason);
	error.checkedException = true;
	
	throw error;
}

/**
 * 比對物件
 * @param sourceObject 來源物件
 * @param targetObject 目標物件
 * @param propertyPath 屬性路徑
 * @return 是否成功
 */
function compareObject(sourceObject, targetObject, propertyPath) {
	
	var isArray = $.isArray(sourceObject);
	var isObjectType = typeof(sourceObject) == "object";
	
	if (!propertyPath) {
		
		propertyPath = "";
	}
	var currentPath = "";
	
	// TODO 比對結構
	if (sourceObject == null) {
		
		if (targetObject != undefined && targetObject != null) {
			
			throwCompareFailedError(propertyPath, "不是  null!");
		}
		
		return true;
	}
	else if (isArray) {
		
		if (!$.isArray(targetObject)) {
			
			throwCompareFailedError(propertyPath, "不是陣列!");
		}
		
		var sourceLength = sourceObject.length;
		var targetLength = targetObject.length;
		if (sourceLength != targetLength) {
			
			throwCompareFailedError(propertyPath, "陣列長度不一致!");
		}
		
		return true;
	}
	else if (sourceObject == "") { // 注意!  [] == "" -> true, 所以此邏輯要放在  isArray 後
		
		if (targetObject != undefined && targetObject != "") {
			
			throwCompareFailedError(propertyPath, "不是  空字串!");
		}
		
		return true;
	}
	else if (isObjectType) {
		
		if (typeof(targetObject) != "object") {
			
			throwCompareFailedError(propertyPath, "不是物件!");
		}
	
		for (var propertyName in sourceObject) {
			
			// 設定當前路徑
			currentPath = propertyPath ? propertyPath + "." + propertyName : propertyName;
			if (targetObject.hasOwnProperty(propertyName)) {

				// recursive check
				compareObject(sourceObject[propertyName], targetObject[propertyName], currentPath);
			}
			else {
				
				throwCompareFailedError(propertyPath, "沒有預期的屬性 -> " + propertyName);
			}
		}
		
		return true;
	}
	
	if (sourceObject != targetObject) {
		
		throwCompareFailedError(propertyPath, " -> " + targetObject + " 非預期!");
	}
	else {
		
		return true;
	}
}

//====== 比對物件