angular.module('app')
.service('settingService', ["$q", 'DWSaveService', 'DWMessage', function($q, DWSaveService, DWMessage) {
	
	this.setting = {};
	this.$q = $q;
	
	// ui 字段
	this.setting.dwgatewayUrls = [];
	this.setting.standardRestfulGatewayUrls = [];
	this.setting.eaiGatewayUrls = [];
	
	this.initSettingName = "setting";
	this.defaultSettingName = this.initSettingName;
	
	// ui 字段
	this.tempData = {};
	this.tempData.unitTest = {};
	
	/**
	 * 是否讀取完成
	 */
	this.isLoaded = false;
	/**
	 * 是否讀取中
	 */
	this.isLoading = false;

	/**
	 * 載入默認值
	 * @param targetSetting 目標設定
	 */
	this.loadDefault = function(targetSetting) {
		
		if (!targetSetting) return;
		
		// 默認日誌分析參數
		if (!targetSetting.logAnalyzer) targetSetting.logAnalyzer = {};
		if (!targetSetting.logAnalyzer.dwGateway) {
			
			var dwGateway = {};
			dwGateway.logSuffix = "gw";
			dwGateway.serverStartKey = "Root WebApplicationContext: initialization started";
			dwGateway.threadStartKey = ">invoke";
			dwGateway.threadInputKey = ">invoke";
			dwGateway.threadOutputKey = ">result";
			dwGateway.threadExceptionKey = ">Exception";
			dwGateway.threadInvokeOtherServiceKey = "Not Implement!!!";
			dwGateway.threadStartAndInputInTheSameLine = true; 
			
			targetSetting.logAnalyzer.dwGateway = dwGateway;
		}
		if (!targetSetting.logAnalyzer.dwDubbo) {
			
			var dwDubbo = {};
			dwDubbo.logSuffix = "";
			dwDubbo.serverStartKey = "Server starting...";
			dwDubbo.threadStartKey = ">>>Inject Profile Filter>";
			dwDubbo.threadInputKey = "io>invoke";
			dwDubbo.threadOutputKey = "io>result";
			dwDubbo.threadExceptionKey = "io>exception";
			dwDubbo.threadInvokeOtherServiceKey = ">invoke";
			dwDubbo.threadStartAndInputInTheSameLine = false; 
			
			targetSetting.logAnalyzer.dwDubbo = dwDubbo;
		}
		if (!targetSetting.logAnalyzer.public) {
			
			targetSetting.logAnalyzer.public = {};
			targetSetting.logAnalyzer.public.alertInvBaseCount = 200;
			targetSetting.logAnalyzer.public.exportAnalyzerResultSheet = false;
		}
	}
	
	/**
	 * 設定 設定物件
	 * @param setting 設定
	 */
	this.setSetting = function(setting) {
		
		if(!setting) {
			return;
		}
		
		this.setting = setting;
		this.loadDefault(this.setting);
		
		// 下列順序不可改變
		// 1.刷新 DWGateway URL 清單
		this.refreshDwgatewayUrls();
	
		if (!$.isArray(this.setting.dwgatewayIpPorts)) {
			
			this.setting.dwgatewayIpPorts = [];
			DWMessage.showWarning("設定檔中的  dwgatewayIpPorts 不是陣列型別!");
		}
		
		// 2.確認默認值是否合法
		if (!$.isNumeric(this.setting.defaultIpPortIndex) || this.setting.defaultIpPortIndex >= this.setting.dwgatewayIpPorts.length) {
			
			this.setting.defaultIpPortIndex = 0;
		}
		
		// 2.1.產生 defaultIpPort (2016-12-26 falcon request 頁面的  file link 會使用此值)
		if (this.setting.dwgatewayIpPorts.length > 0) {
			
			this.defaultIpPort = this.setting.dwgatewayIpPorts[0];
		}
		else {
			
			this.defaultIpPort = "";
		}

		// 3.產生 defaultDwgatewayUrl
		this.defaultDwgatewayUrl = "";
		this.defaultStandardRestfulGatewayUrl = "";
		this.defaultEaiGatewayUrl = "";
		if (this.setting.dwgatewayUrls.length > 0) {
			this.defaultDwgatewayUrl = this.setting.dwgatewayUrls[this.setting.defaultIpPortIndex];
			
			this.defaultStandardRestfulGatewayUrl = this.setting.standardRestfulGatewayUrls[this.setting.defaultIpPortIndex];
			this.defaultEaiGatewayUrl = this.setting.eaiGatewayUrls[this.setting.defaultIpPortIndex];
		}
		
		// 4.刷新金鑰服務信息
		this.refreshTokenServiceInfo();
		
		// 5.刷新單元測試服務信息
		this.refreshUnitTestServiceInfo();
	}
	
	/**
	 * 讀取後事件的監聽者清單
	 */
	this.afterLoadedEventListeners = [];
	
	/**
	 * 註冊讀取後的監聽者
	 * @param listener 監聽者
	 */
	this.registerOnAfterLoadedListener = function(listener) {
		
		if (listener == null) return;
		if (this.afterLoadedEventListeners.indexOf(listener) != -1) return;
		
		this.afterLoadedEventListeners.push(listener);
		
		if (this.isLoaded) {
			
			// 註冊時, 如果已經加載過立即調用 listener
			listener();
		}
	}
	
	/**
	 * 觸發讀取後事件
	 */
	this.fireAfterLoadedEvent = function() {
		
		for (var idx in this.afterLoadedEventListeners) {
			
			this.afterLoadedEventListeners[idx]();
		}
	}
	
	/**
	 * 取消註冊讀取後的監聽者
	 * @param listener 監聽者
	 */
	this.unregisterOnAfterLoadedListener = function(listener) {
		
		for (var idx in this.afterLoadedEventListeners) {

			if (listener == this.afterLoadedEventListeners[idx]) {
				
				this.afterLoadedEventListeners.splice(idx, 1);
				break;
			}
		}
	}
	
	/**
	 * 加載設定
	 * @param callback 回調方法
	 * @param refresh 是否強制重讀
	 */
	this.loadSetting = function(callback, refresh) {

		if (this.isLoaded && !refresh) {
			
			// 使用緩存
			
			if (callback) {
				callback();
			}
			
			return;
		}
		
		this.isLoading = true;
		
		var service = this;
		
		this.loadMetadata().then ( function(dsn) { // 2018-11-6 falcon 取出 defaultSettingName 之後再讀設定檔
			
			// 2018-11-6 falcon 使用  defaultSettingName
			var settingName = service.defaultSettingName;
			DWSaveService.getData('setting', settingName)
//			DWSaveService.getData('setting', 'setting')
			.then(function successCallback(userDataModel) {

				service.isLoaded = true;
				service.isLoading = false;
				
				var setting = userDataModel.getResult();
				service.setSetting(setting);
				
				if (callback) {
					callback();
				}
				
				service.fireAfterLoadedEvent();
			}, function errorCallback(userDataModel) {

				service.isLoaded = true;
				service.isLoading = false;
				
				if (userDataModel.isNotFound()) {

					var isInitName = service.initSettingName == settingName.toLowerCase();
					
					service.setSetting({});
					
					if (!isInitName) {
					
						DWMessage.showError("找不到設定檔 [ " + settingName + " ]!");
					}
				}
				else if (userDataModel.isConnectionError()) {
					
					DWMessage.showError("無法連接  DWRequester 服務器, 取得設定檔失敗!");
				}
				else {

					service.setSetting(service.setting);
					DWMessage.showError("讀取設定檔失敗, 還原至前次設定值!");
				}
				
				if (callback) {
					callback();
				}
				
				service.fireAfterLoadedEvent();
			});

		});
    }
	
	/**
	 * 刷新 DW Gateway URL 清單
	 * @return DW Gateway URL 清單
	 */
	this.refreshDwgatewayUrls = function() {
		
		var urls = [];
		var url = "";
		var basicUrl = "";
		
		var standardRestfulUrls = [];
		var eaiUrls = [];
		if (this.setting) {
			
			if (this.setting.basicUrl) {
				
				basicUrl = this.setting.basicUrl;
			}

			if (this.setting.dwgatewayIpPorts) {
				
				for (var itemIdx in this.setting.dwgatewayIpPorts) {
					
					url = this.setting.dwgatewayIpPorts[itemIdx];
					
					// 2021-5-27 falcon
					standardRestfulUrls.push(url + "restful/standard/");
					eaiUrls.push(url);
					
					if (basicUrl) {
						
						url += basicUrl;
					}
					
					urls.push(url);
				}
				
			}
			else if (basicUrl) {
				
				// 沒設定  ip & port 加一個默認的
				urls.push(basicUrl);
			}

		}
		
		this.setting.dwgatewayUrls = urls;
		this.setting.standardRestfulGatewayUrls = standardRestfulUrls;
		this.setting.eaiGatewayUrls = eaiUrls;
	}
	
	/**
	 * 2018-9-28 falcon 產生金鑰
	 */
	this.generateToken = function(dwServiceClient) {
		
		var def = this.$q.defer();

		var controller = this;
		dwServiceClient.invoke(this.tokenServiceInfo).then(
			function(tokenServiceResult) { // success

				try {
					var gettingTokenScript = controller.setting.tokenServiceGettingTokenScript;				
					if (!gettingTokenScript) {
						
						gettingTokenScript = "tokenServiceResult.getResult().token";
					}
	
					var token = eval(gettingTokenScript);
					if (!token) {
						
						var errorMessage = tokenServiceResult.response.data;
						def.reject(errorMessage)
					}
					else {
					
						def.resolve(token);
					}
				}
				catch (e) {
					
					var errorMessage = "取金鑰腳本 錯誤 - " + e.message;
					
					def.reject(errorMessage);

				}
			},
			function(tokenServiceResult) { // failure

				try {
					
					var gettingFailedScript = controller.setting.tokenServiceGettingFailedScript;
					if (!gettingFailedScript) {
						
						gettingFailedScript = "tokenServiceResult.getFailedReason()";
					}
	
					var errorMessage = eval(gettingFailedScript);
					
					def.reject(errorMessage);
				}
				catch (e) {
					
					var errorMessage = "取異常信息腳本 錯誤 - " + e.message;
					
					def.reject(errorMessage);
				}
			}
		);
		
		return def.promise;
	}
	
	/**
	 * 刷新金鑰服務信息
	 */
	this.refreshTokenServiceInfo = function() {

		var tokenServiceCompleteUrl = "";
		
		if (this.setting.tokenServiceUrl) {

			if (this.setting.tokenServiceUrl.toLowerCase().startsWith("http://") ||
					this.setting.tokenServiceUrl.toLowerCase().startsWith("https://")) {

				tokenServiceCompleteUrl = this.setting.tokenServiceUrl;
			}
			else {
				if (this.defaultDwgatewayUrl) {
					
					tokenServiceCompleteUrl = this.defaultDwgatewayUrl;
				}
				
				tokenServiceCompleteUrl += this.setting.tokenServiceUrl;
			}
		}
		
		var tokenServiceInfo = {};
		tokenServiceInfo.url = tokenServiceCompleteUrl;
		tokenServiceInfo.headers = [];
		tokenServiceInfo.parameters = [];
		tokenServiceInfo.httpMethod = "post";
		
		if (this.setting.tokenServiceHeaders) {
			
			try {
				
				var parameterObj = JSON.parse(this.setting.tokenServiceHeaders);
				var parameterValue;
				for (tokenParameterName in parameterObj) {
				
					parameterValue = parameterObj[tokenParameterName];
					var newParameter = {};
					
					newParameter.name = tokenParameterName;
					newParameter.value = parameterValue;
					
					tokenServiceInfo.headers.push(newParameter);
				}
			}
			catch(e) {

				DWMessage.showError("設定的Header參數解析失敗!");
			}
		}
		
		
		if (this.setting.tokenServiceParameter) {
			
			try {
				
				var parameterObj = JSON.parse(this.setting.tokenServiceParameter);
				var parameterValue;
				for (tokenParameterName in parameterObj) {
				
					parameterValue = parameterObj[tokenParameterName];
					var newParameter = {};
					
					newParameter.name = tokenParameterName;
					newParameter.value = parameterValue;
					
					tokenServiceInfo.parameters.push(newParameter);
				}
			}
			catch(e) {

				DWMessage.showError("設定的Body參數解析失敗!");
			}
		}
		
		this.tokenServiceInfo = tokenServiceInfo;
	}
	
	/**
	 * 刷新單元測試服務信息
	 */
	this.refreshUnitTestServiceInfo = function() {

		var unitTestRunServicePartialUrl = "UnitTester/DWUnitTestingService/run";
		var unitTestSubmitServicePartialUrl = "UnitTester/DWUnitTestingService/submitResult";
		var unitTestLoginServicePartialUrl = "UnitTester/DWUnitTestingService/getUserInfo";
		
		// 單元測試執行服務信息
		this.tempData.unitTest.unitTestRunServiceUrl = this.defaultDwgatewayUrl + unitTestRunServicePartialUrl;
		// 單元測試提交報表服務信息
		this.tempData.unitTest.unitTestSubmitResultServiceUrl = this.defaultDwgatewayUrl + unitTestSubmitServicePartialUrl;
		// 單元測試登錄服務信息
		this.tempData.unitTest.unitTestLoginServiceUrl = this.defaultDwgatewayUrl + unitTestLoginServicePartialUrl;
	}
	
	/**
	 * 2018-11-6 falcon 加載元數據
	 */
	this.loadMetadata = function() {
		
		var def = $q.defer();
		
		var service = this;
		DWSaveService.getData('setting', 'default')
			.then(function(userDataModel) { // success
				
				var result = userDataModel.getResult();
				
				// 會有換行符號
				result = result.replace(/(\r\n\t|\n|\r\t)/gm,"").trim();
				if (result.length > 0) {
					
					service.defaultSettingName = result;
				}
				
				def.resolve(service.defaultSettingName);
			},
			function() { // failure
				
				def.resolve(service.defaultSettingName);
			})
			
		return def.promise;
	}
	/**
	 * 2018-11-6 falcon 加載設定清單
	 */
	this.loadSettingList = function() {
		
	}
	
	this.loadMetadata();
	this.loadSettingList();
	
	// 讀取 設定檔
	this.loadSetting();

}]);