/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.ogm.autoindex;

import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Optional;
import org.neo4j.ogm.annotation.CompositeIndex;
import org.neo4j.ogm.autoindex.AutoIndex;
import org.neo4j.ogm.autoindex.IndexType;
import org.neo4j.ogm.autoindex.MissingIndexException;
import org.neo4j.ogm.config.Configuration;
import org.neo4j.ogm.metadata.ClassInfo;
import org.neo4j.ogm.metadata.FieldInfo;
import org.neo4j.ogm.metadata.MetaData;
import org.neo4j.ogm.model.RowModel;
import org.neo4j.ogm.request.DefaultRequest;
import org.neo4j.ogm.request.Statement;
import org.neo4j.ogm.response.Response;
import org.neo4j.ogm.session.Neo4jSession;
import org.neo4j.ogm.session.request.RowDataStatement;
import org.neo4j.ogm.transaction.Transaction;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class AutoIndexManager {
    private static final Logger LOGGER = LoggerFactory.getLogger(ClassInfo.class);
    private final List<AutoIndex> indexes;
    private Neo4jSession session;
    private final Configuration configuration;

    public AutoIndexManager(MetaData metaData, Configuration configuration, Neo4jSession session) {
        this.configuration = configuration;
        this.indexes = this.initialiseIndexMetadata(metaData);
        this.session = session;
    }

    private List<AutoIndex> initialiseIndexMetadata(MetaData metaData) {
        LOGGER.debug("Building Index Metadata.");
        ArrayList<AutoIndex> indexMetadata = new ArrayList<AutoIndex>();
        for (ClassInfo classInfo : metaData.persistentEntities()) {
            AutoIndex autoIndex;
            IndexType type;
            if (classInfo.containsIndexes()) {
                for (FieldInfo fieldInfo : classInfo.getIndexFields()) {
                    type = fieldInfo.isConstraint() ? IndexType.UNIQUE_CONSTRAINT : IndexType.SINGLE_INDEX;
                    autoIndex = new AutoIndex(type, classInfo.neo4jName(), new String[]{fieldInfo.property()});
                    LOGGER.debug("Adding Index [description={}]", (Object)autoIndex);
                    indexMetadata.add(autoIndex);
                }
                for (CompositeIndex index : classInfo.getCompositeIndexes()) {
                    type = index.unique() ? IndexType.NODE_KEY_CONSTRAINT : IndexType.COMPOSITE_INDEX;
                    String[] properties = index.value().length > 0 ? index.value() : index.properties();
                    AutoIndex autoIndex2 = new AutoIndex(type, classInfo.neo4jName(), properties);
                    LOGGER.debug("Adding composite index [description={}]", (Object)autoIndex2);
                    indexMetadata.add(autoIndex2);
                }
            }
            if (!classInfo.hasRequiredFields()) continue;
            for (FieldInfo requiredField : classInfo.requiredFields()) {
                type = classInfo.isRelationshipEntity() ? IndexType.REL_PROP_EXISTENCE_CONSTRAINT : IndexType.NODE_PROP_EXISTENCE_CONSTRAINT;
                autoIndex = new AutoIndex(type, classInfo.neo4jName(), new String[]{requiredField.property()});
                LOGGER.debug("Adding required constraint [description={}]", (Object)autoIndex);
                indexMetadata.add(autoIndex);
            }
        }
        return indexMetadata;
    }

    List<AutoIndex> getIndexes() {
        return this.indexes;
    }

    public void build() {
        switch (this.configuration.getAutoIndex()) {
            case ASSERT: {
                this.assertIndexes();
                break;
            }
            case UPDATE: {
                this.updateIndexes();
                break;
            }
            case VALIDATE: {
                this.validateIndexes();
                break;
            }
            case DUMP: {
                this.dumpIndexes();
            }
        }
    }

    private void dumpIndexes() {
        String newLine = System.lineSeparator();
        StringBuilder sb = new StringBuilder();
        for (AutoIndex index : this.indexes) {
            sb.append(index.getCreateStatement().getStatement()).append(newLine);
        }
        File file = new File(this.configuration.getDumpDir(), this.configuration.getDumpFilename());
        FileWriter writer = null;
        LOGGER.debug("Dumping Indexes to: [{}]", (Object)file.toString());
        try {
            writer = new FileWriter(file);
            writer.write(sb.toString());
        }
        catch (IOException e) {
            throw new RuntimeException("Could not write file to " + file.getAbsolutePath(), e);
        }
        finally {
            if (writer != null) {
                try {
                    writer.close();
                }
                catch (IOException iOException) {}
            }
        }
    }

    private void validateIndexes() {
        LOGGER.debug("Validating indexes and constraints");
        ArrayList<AutoIndex> copyOfIndexes = new ArrayList<AutoIndex>(this.indexes);
        List<AutoIndex> dbIndexes = this.loadIndexesFromDB();
        copyOfIndexes.removeAll(dbIndexes);
        if (!copyOfIndexes.isEmpty()) {
            String missingIndexes = "[";
            for (AutoIndex s : copyOfIndexes) {
                missingIndexes = missingIndexes + s.getDescription() + ", ";
            }
            missingIndexes = missingIndexes + "]";
            throw new MissingIndexException("Validation of Constraints and Indexes failed. Could not find the following : " + missingIndexes);
        }
    }

    private void assertIndexes() {
        LOGGER.debug("Asserting indexes and constraints");
        ArrayList<Statement> dropStatements = new ArrayList<Statement>();
        List<AutoIndex> dbIndexes = this.loadIndexesFromDB();
        for (AutoIndex dbIndex : dbIndexes) {
            LOGGER.debug("[{}] added to drop statements.", (Object)dbIndex.getDescription());
            dropStatements.add(dbIndex.getDropStatement());
        }
        org.neo4j.ogm.session.request.DefaultRequest dropIndexesRequest = new org.neo4j.ogm.session.request.DefaultRequest();
        dropIndexesRequest.setStatements(dropStatements);
        LOGGER.debug("Dropping all indexes and constraints");
        this.session.doInTransaction(() -> this.session.requestHandler().execute((DefaultRequest)dropIndexesRequest), Transaction.Type.READ_WRITE);
        this.create();
    }

    private List<AutoIndex> loadIndexesFromDB() {
        org.neo4j.ogm.session.request.DefaultRequest indexRequests = this.buildProcedures();
        ArrayList<AutoIndex> dbIndexes = new ArrayList<AutoIndex>();
        this.session.doInTransaction(() -> {
            try (Response response = this.session.requestHandler().execute((DefaultRequest)indexRequests);){
                RowModel rowModel;
                while ((rowModel = (RowModel)response.next()) != null) {
                    if (rowModel.getValues().length == 3 && rowModel.getValues()[2].equals("node_unique_property") || rowModel.getValues().length == 6 && rowModel.getValues()[4].equals("node_unique_property")) continue;
                    Optional<AutoIndex> dbIndex = AutoIndex.parse((String)rowModel.getValues()[0]);
                    dbIndex.ifPresent(dbIndexes::add);
                }
            }
        }, Transaction.Type.READ_WRITE);
        return dbIndexes;
    }

    private void updateIndexes() {
        LOGGER.info("Updating indexes and constraints");
        ArrayList<Statement> dropStatements = new ArrayList<Statement>();
        List<AutoIndex> dbIndexes = this.loadIndexesFromDB();
        for (AutoIndex dbIndex : dbIndexes) {
            if (!dbIndex.hasOpposite() || !this.indexes.contains(dbIndex.createOppositeIndex())) continue;
            dropStatements.add(dbIndex.getDropStatement());
        }
        this.executeStatements(dropStatements);
        ArrayList<Statement> createStatements = new ArrayList<Statement>();
        for (AutoIndex index : this.indexes) {
            if (dbIndexes.contains(index)) continue;
            createStatements.add(index.getCreateStatement());
        }
        this.executeStatements(createStatements);
    }

    private void executeStatements(List<Statement> statements) {
        org.neo4j.ogm.session.request.DefaultRequest request = new org.neo4j.ogm.session.request.DefaultRequest();
        request.setStatements(statements);
        this.session.doInTransaction(() -> {
            Response response = this.session.requestHandler().execute((DefaultRequest)request);
            Throwable throwable = null;
            if (response != null) {
                if (throwable != null) {
                    try {
                        response.close();
                    }
                    catch (Throwable throwable2) {
                        throwable.addSuppressed(throwable2);
                    }
                } else {
                    response.close();
                }
            }
        }, Transaction.Type.READ_WRITE);
    }

    private org.neo4j.ogm.session.request.DefaultRequest buildProcedures() {
        ArrayList<Statement> procedures = new ArrayList<Statement>();
        procedures.add(new RowDataStatement("CALL db.constraints()", Collections.emptyMap()));
        procedures.add(new RowDataStatement("CALL db.indexes()", Collections.emptyMap()));
        org.neo4j.ogm.session.request.DefaultRequest getIndexesRequest = new org.neo4j.ogm.session.request.DefaultRequest();
        getIndexesRequest.setStatements(procedures);
        return getIndexesRequest;
    }

    private void create() {
        ArrayList<Statement> statements = new ArrayList<Statement>();
        for (AutoIndex index : this.indexes) {
            Statement createStatement = index.getCreateStatement();
            LOGGER.debug("[{}] added to create statements.", (Object)createStatement);
            statements.add(createStatement);
        }
        org.neo4j.ogm.session.request.DefaultRequest request = new org.neo4j.ogm.session.request.DefaultRequest();
        request.setStatements(statements);
        LOGGER.debug("Creating indexes and constraints.");
        this.session.doInTransaction(() -> {
            Response response = this.session.requestHandler().execute((DefaultRequest)request);
            Throwable throwable = null;
            if (response != null) {
                if (throwable != null) {
                    try {
                        response.close();
                    }
                    catch (Throwable throwable2) {
                        throwable.addSuppressed(throwable2);
                    }
                } else {
                    response.close();
                }
            }
        }, Transaction.Type.READ_WRITE);
    }
}

