package com.digiwin.athena.adt.agileReport.service.impl.ania;

import com.digiwin.app.log.JsonUtil;
import com.digiwin.athena.adt.agileReport.constant.AgileDataEnum;
import com.digiwin.athena.adt.agileReport.service.AniaAssistantService;
import com.digiwin.athena.adt.domain.dto.ania.*;
import com.digiwin.athena.adt.domain.dto.km.KMCommandIntentionsDTO;
import com.digiwin.athena.adt.domain.dto.km.KMDatasetCommandIntentionsResDTO;
import com.digiwin.athena.adt.domain.dto.mongo.AgileDataCustomConfig;
import com.digiwin.athena.adt.domain.knowledge.KmService;
import com.digiwin.athena.appcore.auth.domain.AuthoredUser;
import com.digiwin.athena.appcore.util.JsonUtils;
import lombok.extern.log4j.Log4j2;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.data.mongodb.core.MongoTemplate;
import org.springframework.data.mongodb.core.query.Criteria;
import org.springframework.data.mongodb.core.query.Query;
import org.springframework.data.mongodb.core.query.Update;
import org.springframework.stereotype.Service;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.stream.Collectors;

import static com.digiwin.athena.adt.agileReport.constant.BusinessConstants.VERSION_1;
import static com.digiwin.athena.adt.agileReport.constant.BusinessConstants.VERSION_2;

/**
 * @Author: SunHong
 * @Date: 2025/5/13 9:50
 * @Description: 2.0 助理应用信息定义
 */
@Service
@Log4j2
public class AniaAssistantServiceImpl implements AniaAssistantService {

    @Autowired
    private KmService kmService;

    @Autowired
    @Qualifier("mongoAgileDataLogTemplate")
    private MongoTemplate mongoTemplate;

    @Value("${adt.url:}")
    private String adtUrl;


    /**
     * 根据助理id查询 敏数2.0 助理信息
     * @param req 助理ids
     * @param user 用户信息
     * @return 助理信息
     */
    @Override
    public AniaAssistantResDTO getPublishedInfo(AniaAssistantReqDTO req,AuthoredUser user) {
        // 请求助理集合
        List<AniaAssistantReqDTO.Assistant> assistantReq =  req.getAssistant();
        AniaAssistantResDTO res = new AniaAssistantResDTO();
        List<AniaAssistantResDTO.Assistant> assistantList = new ArrayList<>();
        for(AniaAssistantReqDTO.Assistant dto : assistantReq){
            String appCode = dto.getAssistantId();
            String version = dto.getVersion();
            String appSystem = "";
            List<Map<String, Object>> commandIntentions = new ArrayList<>();
            List<KMCommandIntentionsDTO> kmCommandIntentions = new ArrayList<>();
            AgileDataCustomConfig config = new AgileDataCustomConfig();
            // km获取指令集和开场白
            KMDatasetCommandIntentionsResDTO cIData = kmService.dataSetCommandIntentions(user,appCode,version);

            int appType = 6;
            if(Objects.nonNull(cIData) && Objects.nonNull(cIData.getAppType())){
                appType = cIData.getAppType();
                appSystem = cIData.getAppSystem();
            }
            AgileDataCustomConfig configReq = new AgileDataCustomConfig();
            configReq.setVersion(appType == 12 ? "V2" : "V1");
            // 获取动态前端组件配置
            List<AgileDataCustomConfig> customConfigs = this.getMongoCustomConfig(configReq,user);
            if(CollectionUtils.isNotEmpty(customConfigs)){
                config = customConfigs.get(0);
            }
            // 指令集 1.0 / 2.0
            if(6 == appType && AgileDataEnum.DATASET.getCode().equals(appSystem)){
                kmCommandIntentions = kmService.queryCommandIntentions(user);
            }else if (Objects.nonNull(cIData) && CollectionUtils.isNotEmpty(cIData.getCommandIntentions())){
                commandIntentions = cIData.getCommandIntentions();
            }
            // 组装应用助理信息
            AniaAssistantResDTO.Assistant assistant =  AniaAssistantResDTO.Assistant
                    .builderAssistant(version,appCode,cIData,config,adtUrl,commandIntentions,kmCommandIntentions);
            assistantList.add(assistant);
        }
        res.setAssistant(assistantList);
        log.info("娜娜回调查询发布出参:{}", JsonUtils.objectToString(res));
        //1.0
        return res;
    }

    /**
     * 助理应用前端组件信息定义
     * @param agileDataCustomConfig 入参
     * @param user token
     */
    @Override
    public void saveCustomConfig(AgileDataCustomConfig agileDataCustomConfig, AuthoredUser user) {
        String version = agileDataCustomConfig.getVersion();
        Query query = Query.query(Criteria.where("version").is(version));
        List<AgileDataCustomConfig> results = mongoTemplate.find(query, AgileDataCustomConfig.class);
        if(CollectionUtils.isNotEmpty(results)){
            Update update = new Update();
            update.set("pluginCustoms",agileDataCustomConfig.getPluginCustoms());
            mongoTemplate.updateFirst(query,update,AgileDataCustomConfig.class);
        }else{
            mongoTemplate.insert(agileDataCustomConfig);
        }
    }

    /**
     * 查询助理前端组件信息
     * @param agileDataCustomConfig bean
     * @param user token
     * @return 助理前端组件信息
     */
    @Override
    public List<AgileDataCustomConfig> getMongoCustomConfig(AgileDataCustomConfig agileDataCustomConfig, AuthoredUser user) {
        String version = agileDataCustomConfig.getVersion();
        Query query = Query.query(Criteria.where("version").is(version));
        return mongoTemplate.find(query, AgileDataCustomConfig.class);
    }

    @Override
    public AniaVersionResDTO getVersion(AniaVersionReqDTO req, AuthoredUser user) {
        AniaVersionResDTO res = new AniaVersionResDTO();
        List<AniaAssistantResDTO.Assistant> assistant = new ArrayList<>();
        String version = kmService.getVersion(user);
        if(Objects.nonNull(version)){
            AniaAssistantResDTO.Assistant versionBean = new AniaAssistantResDTO.Assistant();
            versionBean.setAssistantId(req.getAssistantId());
            versionBean.setVersion(version);
            assistant.add(versionBean);
        }
        res.setAssistant(assistant);
        return res;
    }

    @Override
    public AniaAssistantResDTO getInspirationPhrases(AniaAssistantReqDTO req,
                                                              AuthoredUser user) {

        List<AniaAssistantReqDTO.Assistant> assistantReq =  req.getAssistant();
        AniaAssistantResDTO result = new AniaAssistantResDTO();
        List<AniaAssistantResDTO.Assistant> assistantList = new ArrayList<>();
        for(AniaAssistantReqDTO.Assistant dto : assistantReq){
            AniaAssistantResDTO.Assistant assistant = new AniaAssistantResDTO.Assistant();
            String appCode = dto.getAssistantId();
            // km获取指令集和开场白
            KMDatasetCommandIntentionsResDTO cIData = kmService.dataSetCommandIntentions(user,appCode,"");
            List<Map<String, Object>> commandIntentions = new ArrayList<>();
            List<AniaAssistantResDTO.Assistant.InspirationPhrases> phrases = new ArrayList<>();
            if(Objects.nonNull(cIData) && StringUtils.isNotEmpty(cIData.getName())){
                int appType = cIData.getAppType();
                String appSystem = cIData.getAppSystem();
                // 指令集 1.0 / 2.0
                if(6 == appType && AgileDataEnum.DATASET.getCode().equals(appSystem)){
                    List<KMCommandIntentionsDTO> kmCommandIntentions = kmService.queryCommandIntentions(user);

                    Map<String, List<KMCommandIntentionsDTO>> kmCommandIntentionMap = kmCommandIntentions.stream()
                            // 过滤条件：applicationCode等于目标appCode
                            .filter(km -> km.getApplicationCode() != null && km.getApplicationCode().equals(appCode))
                            // 按指定字段分组（这里以applicationCode为例，你可以根据实际需求修改）
                            .collect(Collectors.groupingBy(KMCommandIntentionsDTO::getApplicationCode));

                    if(CollectionUtils.isNotEmpty(kmCommandIntentions)){
                        List<KMCommandIntentionsDTO> Intentions = kmCommandIntentionMap.get(appCode);
                        // 1.0 数据集指令集
                        phrases =  AniaAssistantResDTO.Assistant.InspirationPhrases
                                .builderInspirationPhrasesV1(Intentions);
                    }
                }else{
                    commandIntentions = cIData.getCommandIntentions();
                    phrases = AniaAssistantResDTO.Assistant.InspirationPhrases
                            .builderInspirationPhrases(commandIntentions);
                }
            }
            assistant.setInspirationPhrases(phrases);
            assistant.setAssistantId(appCode);
            assistantList.add(assistant);
        }
        result.setAssistant(assistantList);
        return result;
    }

    @Override
    public void deleteMongoCustomConfig(AgileDataCustomConfig agileDataCustomConfig, AuthoredUser user) {
        String version = agileDataCustomConfig.getVersion();
        Query query = Query.query(Criteria.where("version").is(version));
        mongoTemplate.remove(query,"agile_data_custom_config");
    }
}
