package com.digiwin.athena.adt.agileReport.service.impl.process.agileData;

import com.digiwin.athena.adt.agileReport.constant.AgileDataEnum;
import com.digiwin.athena.adt.agileReport.constant.BusinessConstants;
import com.digiwin.athena.adt.agileReport.constant.SchemaConstants;
import com.digiwin.athena.adt.agileReport.constant.SchemaDataEnum;
import com.digiwin.athena.adt.agileReport.controller.dto.AthenaMessageDTO;
import com.digiwin.athena.adt.agileReport.dao.AgileDataLogMapper;
import com.digiwin.athena.adt.agileReport.eventbus.AgileDataLogEvent;
import com.digiwin.athena.adt.agileReport.eventbus.AthenaMessageEvent;
import com.digiwin.athena.adt.agileReport.service.AgileDataClassificationsService;
import com.digiwin.athena.adt.agileReport.service.AgileDataLogService;
import com.digiwin.athena.adt.app.env.AdtEnvProperties;
import com.digiwin.athena.adt.domain.ade.ADEService;
import com.digiwin.athena.adt.domain.dto.ade.ADEScencDTO;
import com.digiwin.athena.adt.domain.dto.ade.AdeDatasetReqDTO;
import com.digiwin.athena.adt.domain.dto.ade.AdeMetricReqDTO;
import com.digiwin.athena.adt.domain.dto.agileReport.SnapShotDTO;
import com.digiwin.athena.adt.domain.dto.aiBoard.AIBoardDTO;
import com.digiwin.athena.adt.domain.dto.echo.AgileDataFileDTO;
import com.digiwin.athena.adt.domain.dto.echo.EchoSubmitReq;
import com.digiwin.athena.adt.domain.dto.km.KMPurchaseModelResDTO;
import com.digiwin.athena.adt.domain.dto.schema.QuerySchemaDatasetDTO;
import com.digiwin.athena.adt.domain.dto.schema.QuerySchemaMetricDTO;
import com.digiwin.athena.adt.domain.dto.schema.QuerySchemaResDTO;
import com.digiwin.athena.adt.domain.echo.EchoService;
import com.digiwin.athena.adt.domain.knowledge.KmService;
import com.digiwin.athena.adt.domain.po.AgileDataClassifications;
import com.digiwin.athena.adt.domain.po.AgileDataLog;
import com.digiwin.athena.adt.domain.report.impl.AbsAgileEventProcess;
import com.digiwin.athena.adt.domain.semc.SemcService;
import com.digiwin.athena.adt.sse.domain.EventData;
import com.digiwin.athena.adt.sse.domain.SseAniaEventEnum;
import com.digiwin.athena.adt.sse.utils.SseEmitterUtils;
import com.digiwin.athena.adt.util.CommonUtil;
import com.digiwin.athena.adt.util.DateUtils;
import com.digiwin.athena.adt.util.MessageUtil;
import com.digiwin.athena.appcore.auth.domain.AuthoredUser;
import com.digiwin.athena.appcore.util.JsonUtils;
import com.digiwin.athena.appcore.util.TimeUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.compress.utils.Lists;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.i18n.LocaleContextHolder;
import org.springframework.web.servlet.mvc.method.annotation.SseEmitter;

import javax.annotation.Resource;
import java.time.LocalDateTime;
import java.util.*;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.atomic.AtomicBoolean;

import static com.digiwin.athena.adt.agileReport.constant.BusinessConstants.VERSION_1;

/**
 * @Author: SunHong
 * @Date: 2024/6/3 13:51
 * @Description: 执行段 抽象公公共类
 */
@Slf4j
public abstract class AbsAgileDataProcess extends AbsAgileEventProcess {

    @Autowired
    private AgileDataLogService agileDataLogService;

    @Autowired
    private MessageUtil messageUtil;

    @Autowired
    private SemcService semcService;

    @Autowired
    private AgileDataClassificationsService agileDataClassificationsService;

    @Autowired
    private AdtEnvProperties adtEnvProperties;

    @Resource
    private ADEService adeService;

    @Autowired
    private KmService kmService;

    @Autowired
    private AgileDataLogMapper agileDataLogMapper;

    @Autowired
    private EchoService echoService;

    public ADEScencDTO buildCommonAdeScene(AthenaMessageEvent event,Map<String, Object> data){
        ADEScencDTO adeScencDTO = new ADEScencDTO();
        List<Map<String, Object>> eocMaps = new ArrayList<>();
        if(Objects.nonNull(data.get("eocMaps"))){
            eocMaps = (List<Map<String, Object>>) data.get("eocMaps");
        }
        // 2.0 订阅消息 写入 特定的type
        if(event.isSubscribe()){
            Map<String,Object> param = new HashMap<>();
            param.put("type","2");
            param.put("undeletable",event.isUndeletable());
            param.put("asaCode",event.getAsaCode());
            param.put("ruleId",event.getMsgExt().get("ruleId"));
            adeScencDTO.setParam(param);
        }
        adeScencDTO.setTenantId(event.getUser().getTenantId());
        adeScencDTO.setRequestor(event.getUser().getUserId());
        adeScencDTO.setRequestPersonName(event.getUser().getUserName());
        adeScencDTO.setLocale(LocaleContextHolder.getLocale().toString());
        adeScencDTO.setRequestTime(TimeUtils.format(LocalDateTime.now(), "yyyy-MM-dd HH:mm:ss"));
        adeScencDTO.setScene(event.getSceneDTO());
        adeScencDTO.setTemplateCode(MapUtils.getString(data, "templateCode"));
        adeScencDTO.setTarget(MapUtils.getString(data, "target"));
        adeScencDTO.setDimension(data.containsKey("dimension") ? MapUtils.getObject(data, "dimension") : Lists.newArrayList());
        adeScencDTO.setDimensionCnt(MapUtils.getInteger(data, "dimensionCnt",null));
        adeScencDTO.setEocMaps(eocMaps);
        // 存快照问句
        adeScencDTO.setQuestion(event.getQuestion());
        adeScencDTO.setMessageId(String.valueOf(event.getGenerateSerialNo()));
        adeScencDTO.setMethod(SchemaConstants.METHOD_DATA_FLOW);
        adeScencDTO.setProductLineInfo(event.getProductLineInfo());
        adeScencDTO.setProductVersion(event.getProductVersion());
        adeScencDTO.setSaveTransResult(event.getSaveTransResult());
        return adeScencDTO;
    }


    /**
     * 记录异常日志 501
     * @param sentences sentences
     * @param event event
     */
    public void saveQuerySchemaErrorByCode(AthenaMessageEvent event,List<String> sentences) {
        Map<String, Object> msgBody = new HashMap<>();
        String sysError = messageUtil.getMessageByLangNameWithFormat(
                "message.system.schema.error",event.getLang());
        msgBody.put("prompt", sysError);
        //记录异常日志
        semcService.sendMessageToGpt(event,msgBody);
        this.saveAbnormalLog(event,sysError,1,0);
    }

    /**
     * 记录adt日志
     * @param event 上下文
     * @param errorMessage 异常消息
     * @param type type
     * @param success 0 1 2
     */

    public void saveAbnormalLog(AthenaMessageEvent event,
                                String errorMessage,
                                Integer type,
                                Integer success){
        AgileDataLogEvent agileDataLogEvent = new AgileDataLogEvent();
        String combinationQuestion = event.getCombinationQuestion();
        String question = event.getQuestion();
        Long generateSerialNo = event.getGenerateSerialNo();
        //尽量用组合问句记录日志
        if(StringUtils.isNotEmpty(combinationQuestion)){
            question = combinationQuestion;
        }
        agileDataLogEvent.setQuestion(question);
        agileDataLogEvent.setGenerateSerialNo(generateSerialNo);
        agileDataLogEvent.setErrorAnswer(errorMessage);
        agileDataLogEvent.setAuthoredUser(event.getUser());
        agileDataLogEvent.setTerminal(event.getTerminal());
        // 1 多目标 3 多场景 5 多应用 10 多sheet 导正记录为正常语句
        if("1".equals(event.getMessageType()) || "3".equals(event.getMessageType())
                || "5".equals(event.getMessageType()) || "10".equals(event.getMessageType())){
            agileDataLogEvent.setSuccess(1);
        }else {
            agileDataLogEvent.setSuccess(success);
        }
        agileDataLogEvent.setType(type);
        agileDataLogEvent.setAskTime(event.getAskTime());
        agileDataLogEvent.setAnswerTime(LocalDateTime.now());
        agileDataLogEvent.setAdeScencDTO(event.getAdeScencDTO());
        agileDataLogEvent.setPtxId(event.getPtxId());
        agileDataLogEvent.setSourceCode(event.getSourceCode());
        agileDataLogEvent.setSourceName(event.getSourceName());
        agileDataLogEvent.setApplicationCode(event.getAppCode());
        agileDataLogEvent.setApplicationName(event.getAppName());
        agileDataLogEvent.setLogSaveSuccess("N");
        agileDataLogEvent.setReason("");
        agileDataLogEvent.setProductVersion(event.getProductVersion());
        agileDataLogService.saveAgileDataLog(agileDataLogEvent);
    }

    public String getKmCodeByEventOrSchemaData(AthenaMessageEvent event, Map<String, Object> data) {
        if (CollectionUtils.isNotEmpty(event.getScenes())) {
            return MapUtils.getString(event.getScenes().get(0), "sceneCode");
        }else{
            return MapUtils.getString(data, "sceneCode");
        }
    }

    /**
     * 获取语义策略服务枚举
     * @param event 前端传递
     * @return 枚举
     */
    public String getServiceNameByMessageType(AthenaMessageEvent event) {
        String serviceName = event.getMessageType();
        switch (serviceName){
            case "0":
            case "1":
            case "3":
            case "5":
            case "6":
                serviceName = SchemaDataEnum.DESIGNER.getCode();
                break;
            case "2":
                serviceName = SchemaDataEnum.DEBUG.getCode();
                break;
            case "4":
                serviceName = SchemaDataEnum.PARAM.getCode();
                break;
//            case "5" :
//                serviceName = SchemaDataEnum.METRIC.getCode();
//                break;
            case "8" :
                serviceName = SchemaDataEnum.MOCK.getCode();
                break;
            case "9" :
                serviceName = SchemaDataEnum.SPECIAL.getCode();
                break;
            case "10" :
                serviceName = SchemaDataEnum.SPECIAL_LITE.getCode();
                break;
            case "11" :
                serviceName = SchemaDataEnum.METRIC_DEBUG.getCode();
                break;
            case "12" :
                serviceName = SchemaDataEnum.PPT_ANALYZE.getCode();
                break;
            case "13" :
                serviceName = SchemaDataEnum.PPT_GENERATE.getCode();
                break;
            case "14" :
                serviceName = SchemaDataEnum.AUTODATA_REPORT.getCode();
                break;
            case "15" :
                serviceName = SchemaDataEnum.AUTO_DATA_QUESTION.getCode();
                break;
            case "16" :
                serviceName = SchemaDataEnum.AI_BOARD_QUESTION.getCode();
                break;
            case "17" :
                serviceName = SchemaDataEnum.ATTRIBUTION_SUMMARY.getCode();
                break;
            default:
                return SchemaDataEnum.DESIGNER.getCode();
        }
        return serviceName;
    }

    /**
     * 根据语义作用类型获取调用api名称
     * @param schemaEnum adt 语义分类
     * @return url
     */
    public String getSseSchemaUrl(String schemaEnum) {
        String url = "";
        switch (schemaEnum){
            case "debug":
            case "param":
            case "metric":
            case "metric_app":
            case "designer":
            case "mock":
            case "metric_debug":
                url = adtEnvProperties.getChatBIUrl()+"/scrumbi/stream/chat";
                break;
            case "special":
                url = adtEnvProperties.getChatBIUrl()+"/scrumbi/chat/demo";
                break;
            case "special_lite":
                url = adtEnvProperties.getChatBIUrl()+"/scrumbi/chat/lite";
                break;
            case "ppt_Analyze":
                url = adtEnvProperties.getChatBIUrl()+"/scrumbi/question/planning";
                break;
            case "ppt_Generate":
                url = adtEnvProperties.getChatBIUrl()+"/scrumbi/ppt/genarate";
                break;
            case "autoData_report":
                url = adtEnvProperties.getChatBIUrl()+"/scrumbi/report/create";
                break;
            case "auto_data_question":
                url = adtEnvProperties.getChatBIUrl()+ "";
                break;
            case "ai_board_question":
                url = adtEnvProperties.getChatBIUrl()+"/scrumbi/chart/plan";
                break;
            case "attribution_summary":
                url = adtEnvProperties.getChatBIUrl()+"/scrumbi/dataAttribution/report";
                break;
            default:
                return adtEnvProperties.getChatBIUrl()+"scrumbi/stream/chat";
        }
        return url;
    }

    /**
     * 根据语义返回method 判断ade服务类
     * @param router 语义method路由转换
     * @return ade策略枚举
     */
    public String getServiceNameByQuerySchema(String router) {
        String serviceName = AgileDataEnum.DESIGNER.getCode();
        switch (router){
            case "metric":
                serviceName = AgileDataEnum.METRIC.getCode();
                break;
            case "dataFlow":
                serviceName = AgileDataEnum.DESIGNER.getCode();
                break;
            case "mock":
                serviceName = AgileDataEnum.MOCK.getCode();
                break;
            case "special":
                serviceName = AgileDataEnum.SPECIAL.getCode();
                break;
            case "dataset":
                serviceName = AgileDataEnum.DATASET.getCode();
                break;
            case "ppt_Analyze":
                serviceName = AgileDataEnum.PPT_ANALYZE.getCode();
                break;
            case "ppt_process":
                serviceName = AgileDataEnum.PPT_PROCESS.getCode();
                break;
            case "autoData_report":
                serviceName = AgileDataEnum.AUTODATA_REPORT.getCode();
                break;
            case "ai_board_analyze":
                serviceName = AgileDataEnum.AI_BOARD_ANALYZE.getCode();
                break;
            case "ai_board_question":
                serviceName = AgileDataEnum.AI_BOARD_QUESTION.getCode();
                break;
            default:
                return serviceName;
        }
        return serviceName;
    }

    /**
     * catch 异常Log 记录方法
     * @param event 上下文
     * @param tipMessage 记录信息
     */
    public void saveAbnormal(AthenaMessageEvent event,String tipMessage,Integer success){
        AgileDataLogEvent agileDataLogEvent = new AgileDataLogEvent();
        String question = event.getQuestion();
        String combinationQuestion = event.getCombinationQuestion();
        Long generateSerialNo = event.getGenerateSerialNo();
        //尽量使用组合问句作为日志记录
        if(StringUtils.isNotEmpty(event.getCombinationQuestion())){
            question = combinationQuestion;
        }
        agileDataLogEvent.setQuestion(question);
        agileDataLogEvent.setGenerateSerialNo(generateSerialNo);
        agileDataLogEvent.setErrorAnswer(tipMessage);
        agileDataLogEvent.setAuthoredUser(event.getUser());
        agileDataLogEvent.setTerminal(event.getTerminal());
        agileDataLogEvent.setSuccess(success);
        agileDataLogEvent.setType(1);
        agileDataLogEvent.setAskTime(event.getAskTime());
        agileDataLogEvent.setAnswerTime(LocalDateTime.now());
        agileDataLogEvent.setAdeScencDTO(event.getAdeScencDTO());
        agileDataLogEvent.setPtxId(event.getPtxId());
        agileDataLogEvent.setSourceCode(event.getSourceCode());
        agileDataLogEvent.setSourceName(event.getSourceName());
        agileDataLogEvent.setApplicationCode(event.getAppCode());
        agileDataLogEvent.setApplicationName(event.getAppName());
        agileDataLogEvent.setLogSaveSuccess("N");
        agileDataLogEvent.setReason("");
        agileDataLogEvent.setProductVersion(event.getProductVersion());
        agileDataLogService.saveAgileDataLog(agileDataLogEvent);
    }

    /**
     * 记录问题结果数据
     * @param data
     * @param event
     * @param question
     * @param generateSerialNo
     */
    public void saveQuestionData(Map<String,Object> data,AthenaMessageEvent event,String question,String combinationQuestion,Long generateSerialNo){
        AgileDataLogEvent agileDataLogEvent = new AgileDataLogEvent();
        agileDataLogEvent.setAuthoredUser(event.getUser());
        //尽量使用组合问句，作为日志记录
        if(StringUtils.isNotEmpty(combinationQuestion)){
            question = combinationQuestion;
        }
        agileDataLogEvent.setQuestion(question);
        agileDataLogEvent.setGenerateSerialNo(generateSerialNo);
        agileDataLogEvent.setTerminal(event.getTerminal());
        agileDataLogEvent.setLocal(event.getLang());
        agileDataLogEvent.setAskTime(event.getAskTime());
        agileDataLogEvent.setAnswerTime(LocalDateTime.now());
        agileDataLogEvent.setAdeScencDTO(event.getAdeScencDTO());
        agileDataLogEvent.setPtxId(event.getPtxId());
        agileDataLogEvent.setSourceCode(event.getSourceCode());
        agileDataLogEvent.setSourceName(event.getSourceName());
        agileDataLogEvent.setApplicationCode(event.getAppCode());
        agileDataLogEvent.setApplicationName(event.getAppName());
        agileDataLogEvent.setLogSaveSuccess("N");
        agileDataLogEvent.setReason("");
        agileDataLogEvent.setProductVersion(event.getProductVersion());
        agileDataLogService.saveQuestionData(data,agileDataLogEvent);
    }

    /**
     * 记录警告success = 2日志
     * @param event 上下文
     */
    public void saveWarnQuestionData(AthenaMessageEvent event,String msg){
        AgileDataLog agileDataLog = new AgileDataLog();
        agileDataLog.setUserId(event.getUser().getUserId());
        agileDataLog.setUserName(event.getUser().getUserName());
        agileDataLog.setTenantId(event.getUser().getTenantId());
        agileDataLog.setTenantName(event.getUser().getTenantName());
        agileDataLog.setTerminal(event.getTerminal());
        agileDataLog.setGenerateSerialNo(event.getGenerateSerialNo());
        agileDataLog.setQuestion(event.getQuestion());
        agileDataLog.setSuccess(2);
        agileDataLog.setSourceCode(event.getSourceCode());
        agileDataLog.setSourceName(event.getSourceName());
        agileDataLog.setLogSaveSuccess("N");
        agileDataLog.setErrorAnswer(msg);
        agileDataLog.setCreateTime(event.getAskTime());
        agileDataLog.setAnswerTime(LocalDateTime.now());
        agileDataLog.setResSecTime(DateUtils.calculateAndSetDuration(event.getAskTime(),
                agileDataLog.getAnswerTime()));
        agileDataLog.setProductVersion(event.getProductVersion());
        agileDataLogMapper.insert(agileDataLog);
    }

    /**
     * 发送娜娜和记录日志
     * @param event 上下文
     * @param businessType 业务日志类型
     */
    public void sendMessageAndSaveLogByCost(AthenaMessageEvent event,
                                      String businessType){
        Map<String, Object> msgBody = new HashMap<>();
        String warnStr = messageUtil.getMessageByLangName(businessType, event.getLang());
        msgBody.put("prompt", warnStr);
        this.saveAbnormalLog(event,warnStr,1,2);
        if(event.isSendNana()){
            semcService.sendMessageToGpt(event,msgBody);
        }
    }

    /**
     * 存入分类关系
     * @param event
     * @param response
     */
    public void saveBuiltClassifications(AthenaMessageEvent event, Map<String, Object> response) {
        Map<String, Object> snapshot = MapUtils.getMap(response, "data");
        if (snapshot != null) {
            Map<String, Object> contextMap = MapUtils.getMap(snapshot, "context");
            if (contextMap != null) {
                Map<String, Object> bizParams = MapUtils.getMap(contextMap, "bizParams");
                if (bizParams != null) {
                    List<Map<String, Object>> classifications = Objects.isNull(bizParams.get("builtClassifications")) ? new ArrayList<>() :
                             (List<Map<String, Object>>) bizParams.get("builtClassifications");
                    if(CollectionUtils.isNotEmpty(classifications)){
                        for (int i = 0; i < classifications.size(); i++){
                            AgileDataClassifications agileDataClassifications = new AgileDataClassifications();
                            agileDataClassifications.setClassName(String.valueOf(classifications.get(i).get("name")));
                            agileDataClassifications.setClassCode(String.valueOf(classifications.get(i).get("code")));
                            agileDataClassifications.setCreateDate(new Date());
                            agileDataClassifications.setMessageId(event.getGenerateSerialNo());
                            agileDataClassifications.setUserId(event.getUser().getUserId());
                            agileDataClassifications.setTenantId(event.getUser().getTenantId());
                            agileDataClassifications.setSort(i+1);
                            agileDataClassificationsService.insert(agileDataClassifications,event.getUser());
                        }
                    }
                }
            }
        }
    }

    /**
     * 大数据量优化呈现-判断是否从echo获取消息
     * @param data ade返回数据
     */
    public Pair<Boolean, Integer> isDataProcessAction(Map<String, Object> data){
        SnapShotDTO snapShotDTO = JsonUtils.jsonToObject(JsonUtils.objectToString(data),SnapShotDTO.class);
        if(Objects.nonNull(snapShotDTO) && Objects.nonNull(snapShotDTO.getContext())
                && Objects.nonNull(snapShotDTO.getContext().getBizParams())
                && StringUtils.isNotEmpty(snapShotDTO.getContext().getBizParams().getDataProcessAction())
                && "interception".equals(snapShotDTO.getContext().getBizParams().getDataProcessAction())
                && Objects.nonNull(snapShotDTO.getContext().getBizParams().getInterceptSize())){
            return Pair.of(true, snapShotDTO.getContext().getBizParams().getInterceptSize());
        }
        return Pair.of(false, 0);
    }


    /**
     * 获取ade sse 解题要素识别
     * @param event 上下文信息
     * @param param 入参
     */
    public void getEssentialFactorAndSendSseMessage(AthenaMessageEvent event,
                                                    Map<String,Object> param,
                                                    AtomicBoolean state,
                                                    CountDownLatch returnResultLatch){
        if(!event.isSseMessage()){
            returnResultLatch.countDown();
            return;
        }
        String sseProductVersion = event.getSseProductVersion();
        try {
            Map<String, Object> response = adeService.getEssentialFactor(event,param);
            if(response == null || Objects.isNull(response.get("msg"))){
                EventData.EventMessage toolResponse = EventData.EventMessage.builderMessageClose(
                        "tool_response",event.getSseCallId(),"fail");
                this.sendEventDataMessage(SseAniaEventEnum.MESSAGE_COMPLETED.getEvent()
                        ,event.getSseEmitter(),toolResponse);
                //开始执行取数会话
                this.sendSseGetSnapshotMessage(event);
                returnResultLatch.countDown();
                return;
            }
            if(VERSION_1.equals(sseProductVersion)){
                this.essentialFactorSendByV1(event,response,returnResultLatch,state);
            }else{
                this.essentialFactorSendByV2(event,response,returnResultLatch,state);
            }

        } catch (Exception e) {
            log.error("ade message getEssentialFactor error: {}" ,e.getMessage());
            Thread.currentThread().interrupt();
        }
    }

    /**
     * 流式输出要素识别V1版本
     * @param event e
     * @param response r
     * @param returnResultLatch c
     * @param state s
     */
    private void essentialFactorSendByV1(AthenaMessageEvent event,
                                         Map<String, Object> response,
                                         CountDownLatch returnResultLatch,
                                         AtomicBoolean state) {
        try {
            // 发送sse 消息
            SseEmitter sseEmitter = event.getSseEmitter();
            Object msg = response.get("data");
            String msgStr = String.valueOf(msg);
            String[] answerSplit = msgStr.split("");
            if(StringUtils.isEmpty(msgStr)){
                EventData.EventMessage toolResponse = EventData.EventMessage.builderMessageClose(
                        "tool_response",event.getSseCallId(),"fail");
                this.sendEventDataMessage(SseAniaEventEnum.MESSAGE_COMPLETED.getEvent(),sseEmitter,toolResponse);
                //开始执行取数会话
                this.sendSseGetSnapshotMessage(event);
                returnResultLatch.countDown();
                return;
            }
            for (int i = 0; i < msgStr.length(); i++) {
                if (state.get()) {
                    EventData.EventMessage msgSub = EventData.EventMessage.builderMessage(
                            "function_answer",msgStr.substring(i),event.getSseCallId());
                    this.sendEventDataMessage(SseAniaEventEnum.MESSAGE_DELTA.getEvent(),sseEmitter,msgSub);
                    EventData.EventMessage toolResponse = EventData.EventMessage.builderMessageClose(
                            "tool_response",event.getSseCallId(),"success");
                    this.sendEventDataMessage(SseAniaEventEnum.MESSAGE_COMPLETED.getEvent(),sseEmitter,toolResponse);
                    //开始执行取数会话
                    this.sendSseGetSnapshotMessage(event);
                    returnResultLatch.countDown();
                    break;
                } else {
                    Thread.sleep(30);
                    EventData.EventMessage msgSplit = EventData.EventMessage.builderMessage(
                            "function_answer",answerSplit[i],event.getSseCallId());
                    this.sendEventDataMessage(SseAniaEventEnum.MESSAGE_DELTA.getEvent(),sseEmitter,msgSplit);
                    if (i == msgStr.length() - 1) {
                        EventData.EventMessage toolResponse = EventData.EventMessage.builderMessageClose(
                                "tool_response",event.getSseCallId(),"success");
                        this.sendEventDataMessage(SseAniaEventEnum.MESSAGE_COMPLETED.getEvent(),sseEmitter,toolResponse);
                        //开始执行取数会话
                        this.sendSseGetSnapshotMessage(event);
                        returnResultLatch.countDown();
                    }
                }
            }
        } catch (Exception e) {
            log.error("ade message getEssentialFactor error: {}" ,e.getMessage());
            Thread.currentThread().interrupt();
        }
    }

    /**
     * 流式输出要素识别V2版本
     * @param event e
     * @param response r
     * @param returnResultLatch c
     * @param state s
     */
    private void essentialFactorSendByV2(AthenaMessageEvent event,
                                         Map<String, Object> response,
                                         CountDownLatch returnResultLatch,
                                         AtomicBoolean state) {
        try {
            // 发送sse 消息
            SseEmitter sseEmitter = event.getSseEmitter();
            Object msg = response.get("data");
            String msgStr = String.valueOf(msg);
            String[] answerSplit = msgStr.split("");
            if(StringUtils.isEmpty(msgStr)){
                EventData.EventMessage toolResponse = EventData.EventMessage.builderMessageCloseV2(
                        "tool_response",event.getSseCallId(),"fail");
                this.sendEventDataMessage(SseAniaEventEnum.MESSAGE_COMPLETED.getEvent(),sseEmitter,toolResponse);
                //开始执行取数会话
                this.sendSseGetSnapshotMessage(event);
                returnResultLatch.countDown();
                return;
            }
            for (int i = 0; i < msgStr.length(); i++) {
                if (state.get()) {
                    EventData.EventMessage msgSub = EventData.EventMessage.builderMessageV2(
                            "function_answer",msgStr.substring(i),event.getSseCallId());
                    this.sendEventDataMessage(SseAniaEventEnum.MESSAGE_DELTA.getEvent(),sseEmitter,msgSub);
                    EventData.EventMessage toolResponse = EventData.EventMessage.builderMessageCloseV2(
                            "tool_response",event.getSseCallId(),"success");
                    this.sendEventDataMessage(SseAniaEventEnum.MESSAGE_COMPLETED.getEvent(),sseEmitter,toolResponse);
                    //开始执行取数会话
                    this.sendSseGetSnapshotMessage(event);
                    returnResultLatch.countDown();
                    break;
                } else {
                    Thread.sleep(30);
                    EventData.EventMessage msgSplit = EventData.EventMessage.builderMessageV2(
                            "function_answer",answerSplit[i],event.getSseCallId());
                    this.sendEventDataMessage(SseAniaEventEnum.MESSAGE_DELTA.getEvent(),sseEmitter,msgSplit);
                    if (i == msgStr.length() - 1) {
                        EventData.EventMessage toolResponse = EventData.EventMessage.builderMessageCloseV2(
                                "tool_response",event.getSseCallId(),"success");
                        this.sendEventDataMessage(SseAniaEventEnum.MESSAGE_COMPLETED.getEvent(),sseEmitter,toolResponse);
                        //开始执行取数会话
                        this.sendSseGetSnapshotMessage(event);
                        returnResultLatch.countDown();
                    }
                }
            }
        } catch (Exception e) {
            log.error("ade message getEssentialFactor error: {}" ,e.getMessage());
            Thread.currentThread().interrupt();
        }
    }


    /**
     * 开始执行取数会话
     * @param event 上下文
     */
    public void sendSseGetSnapshotMessage(AthenaMessageEvent event){
        log.info("send 取数会话开始");
        // 开始执行取数会话
        Map<String ,Object> messageObj = new HashMap<>();
        String messageTitle = messageUtil
                .getMessageByLangNameWithFormat("message.sse.getSnapshotMessage", event.getLang());
        messageObj.put("name",messageTitle);
        messageObj.put("arguments ",new HashMap<>());
        // sse组装消息体
        EventData.EventMessage eventMessage;
        if(VERSION_1.equals(event.getSseProductVersion())){
            eventMessage = EventData.EventMessage.builderNewEventMessage(
                    "function_call",messageObj,event);
        }else{
            eventMessage = EventData.EventMessage.builderNewEventMessageV2(
                    "function_call",messageTitle,event);
        }
        this.sendEventDataMessage(SseAniaEventEnum.MESSAGE_COMPLETED.getEvent(),event.getSseEmitter(),eventMessage);
    }

    /**
     * 发送娜娜消息
     * @param sendEvent 发送事件名称
     * @param sseEmitter sse链接
     * @param eventMessage 发送消息体
     */
    public void sendEventDataMessage(String sendEvent,SseEmitter sseEmitter,EventData.EventMessage eventMessage){
        EventData eventData = new EventData(sendEvent,eventMessage);
        if (Objects.nonNull(sseEmitter)) {
            SseEmitterUtils.send(sseEmitter, eventData);
        }
    }

    /**
     * 校验当前提问用户是否存在敏捷数据应用
     * @param event 上下文
     * @return 是否
     */
    public boolean checkUserAppPermission(AthenaMessageEvent event,String messageType) {
        // 只校验敏数入口
        if( !"asada".equals(event.getAsaCode())
                || SchemaDataEnum.SPECIAL.getCode().equals(messageType)
                || SchemaDataEnum.SPECIAL_LITE.getCode().equals(messageType)){
            return false;
        }
        String token = event.getUser().getToken();
        String tenantId = event.getUser().getTenantId();
        String userId = event.getUser().getUserId();
        List<KMPurchaseModelResDTO> apps = kmService.queryKmPurchaseList(1,token,tenantId,userId);
        return CollectionUtils.isEmpty(apps);
    }

    /**
     * 根据上下文版本号返回ade请求入参
     * @param responseData 语义返回
     * @param event 上下文
     * @return 请求 object
     */
    public Map<String, Object> parameterBasedOnVersion(QuerySchemaResDTO responseData,AthenaMessageEvent event) {
        String productVersion = event.getProductVersion();
        // 1.0 指标
        QuerySchemaMetricDTO metric = responseData.getData().getMetric();
        // 2.0 数据集
        QuerySchemaDatasetDTO dataset = responseData.getData().getDataset();
        if(VERSION_1.equals(productVersion)){
            if(Objects.nonNull(metric)){
                return CommonUtil.convertObjectToMap(AdeMetricReqDTO.BuildMetricAdeReq(event,metric));
            }else{
                return CommonUtil.convertObjectToMap(AdeDatasetReqDTO.BuildDatasetAdeReq(event,dataset));
            }
        }else if(BusinessConstants.VERSION_2.equals(productVersion)){
            return CommonUtil.convertObjectToMap(AdeDatasetReqDTO.BuildDatasetAdeReq(event,dataset));
        }else{
            // 默认V1
            return CommonUtil.convertObjectToMap(AdeMetricReqDTO.BuildMetricAdeReq(event,metric));
        }
    }

    /**
     * 根据元数据获取数据总结字段存入上下文
     * @param response ade dataset
     * @param event 上下文
     */
    public void setDataSummary(Map<String, Object> response, AthenaMessageEvent event) {

        SnapShotDTO snapShotDTO = JsonUtils.jsonToObject(JsonUtils.objectToString(response),SnapShotDTO.class);
        Boolean exceedSummarizeSize = snapShotDTO.getContext().getBizParams().getExceedSummarizeSize();
        if(Objects.nonNull(exceedSummarizeSize)){
            event.setExceedSummarizeSize(exceedSummarizeSize);
        }
    }

    public void executeStepAnalysis(AthenaMessageEvent event, Map<String, Object> response) {
        List<Map<String, Object>> steps = Lists.newArrayList();
        Map<String,Object> contextMap = MapUtils.getMap(MapUtils.getMap(response, "data"), "context");
        Object step = contextMap.get("stepAnalysis");
        if(Objects.nonNull(step)){
            steps.addAll((Collection<? extends Map<String, Object>>) step);
        }
        if(CollectionUtils.isNotEmpty(steps)){
            echoService.echoMongodbSubmit(EchoSubmitReq.builderStepAnalysis(event,steps)
                    ,event.getUser().getToken(),event.getUser().getTenantId());
        }
    }

    /**
     * 数据筛选
     * @param response ade res
     * @param event e
     */
    public void setDataScreening(AthenaMessageEvent event, Map<String, Object> response) {
        Map<String,Object> contextMap = MapUtils.getMap(MapUtils.getMap(response, "data"), "context");
        Optional.ofNullable(contextMap)
                .map(ctx -> ctx.get("stepFilterAnalysis"))
                .filter(obj -> obj instanceof List<?>)
                .map(obj -> (List<Map<String, Object>>) obj)
                .filter(CollectionUtils::isNotEmpty)
                .ifPresent(steps -> {
                    echoService.echoMongodbSubmit(
                            EchoSubmitReq.builderStepFilterAnalysis(event, steps),
                            event.getUser().getToken(),
                            event.getUser().getTenantId()
                    );
                });
    }

    /**
     * ppt 上下文组装
     * @param appCode a
     * @param question q
     * @param user u
     * @return t
     */
    public AthenaMessageEvent initEvent(String appCode,
                                                String question,
                                                List<String> datasetIds,
                                                AuthoredUser user) {

        Map<String,Object> athenaMessage = new HashMap<>();
        Map<String,Object> extData = new HashMap<>();
        extData.put("msg",question);
        extData.put("asaCode",appCode);
        extData.put("probe",false);
        extData.put("datasetIdList",datasetIds);

        athenaMessage.put("msgExt",extData);
        Map<String,Object> msgBody = new HashMap<>();
        msgBody.put("text",question);
        athenaMessage.put("msgBody",msgBody);
        AthenaMessageDTO athenaMessageDTO =
                CommonUtil.mapToObject(athenaMessage,AthenaMessageDTO.class);
        athenaMessageDTO.setAppCode(appCode);
        return init(null,athenaMessageDTO,user);
    }


    /**
     * 更新ppt执行步骤和状态
     * @param event e
     * @param processStreamData 执行步骤
     * @param errorQueryData errorMsg
     */
    public void updatePPTDataByError(AthenaMessageEvent event, Map<String,Object> processStreamData,Map<String,Object> errorQueryData) {
        AgileDataFileDTO agileDataFileDTO = new AgileDataFileDTO();
        agileDataFileDTO.setMessageId(Long.valueOf(event.getMsgExt().get("pptMessageId").toString()));
        List<AgileDataFileDTO.AgileDataFileQuestionDTO> questionDTOS = new ArrayList<>();
        AgileDataFileDTO.AgileDataFileQuestionDTO questionDTO = new AgileDataFileDTO.AgileDataFileQuestionDTO();
        questionDTO.setQuestionId(Long.valueOf(event.getMsgExt().get("pptQuestionId").toString()));
        questionDTO.setAnalyzeStatus(1);
        questionDTO.setAnalyzeMessageId(event.getGenerateSerialNo());
        AgileDataFileDTO.AgileDataFileAnalyzeDTO analyzeDTO = new AgileDataFileDTO.AgileDataFileAnalyzeDTO();
        if(MapUtils.isNotEmpty(processStreamData)){
            analyzeDTO = JsonUtils.jsonToObject(
                    JsonUtils.objectToString(processStreamData),AgileDataFileDTO.AgileDataFileAnalyzeDTO.class);
            if(MapUtils.isNotEmpty(errorQueryData)){
                analyzeDTO.setQueryData(errorQueryData);
            }
            questionDTO.setAnalyzeResult(analyzeDTO);
        }else{
            analyzeDTO.setQueryData(errorQueryData);
            questionDTO.setAnalyzeResult(analyzeDTO);
        }
        questionDTOS.add(questionDTO);
        agileDataFileDTO.setQuestionInfo(questionDTOS);
        log.info("更新ppt执行步骤和状态入参 eventQuestionId :{},: data:{}",event.getGenerateSerialNo(),
                JsonUtils.objectToString(agileDataFileDTO));
        echoService.updateAgileDataFile(agileDataFileDTO,event.getUser().getToken(),event.getUser().getTenantId());
    }

    /**
     * 更新对话看板执行步骤和状态
     * @param event e
     * @param processStreamData 执行步骤
     * @param errorQueryData errorMsg
     */
    public void updateAIBoardDataByError(AthenaMessageEvent event, Map<String,Object> processStreamData,Map<String,Object> errorQueryData) {
        AIBoardDTO aiBoardDTO = new AIBoardDTO();
        aiBoardDTO.setMessageId(Long.valueOf(event.getMsgExt().get("aiBoardMessageId").toString()));
        List<AIBoardDTO.AIBoardQuestionDTO> questionDTOS = new ArrayList<>();
        AIBoardDTO.AIBoardQuestionDTO questionDTO = new AIBoardDTO.AIBoardQuestionDTO();
        questionDTO.setQuestionId(Long.valueOf(event.getMsgExt().get("aiBoardQuestionId").toString()));
        questionDTO.setAnalyzeStatus(1);
        questionDTO.setAnalyzeMessageId(event.getGenerateSerialNo());
        AIBoardDTO.AIBoardAnalyzeDTO analyzeDTO = new AIBoardDTO.AIBoardAnalyzeDTO();
        if(MapUtils.isNotEmpty(processStreamData)){
            analyzeDTO = JsonUtils.jsonToObject(
                    JsonUtils.objectToString(processStreamData),AIBoardDTO.AIBoardAnalyzeDTO.class);
            if(MapUtils.isNotEmpty(errorQueryData)){
                analyzeDTO.setQueryData(errorQueryData);
            }
            questionDTO.setAnalyzeResult(analyzeDTO);
        }else{
            analyzeDTO.setQueryData(errorQueryData);
            questionDTO.setAnalyzeResult(analyzeDTO);
        }
        questionDTOS.add(questionDTO);
        aiBoardDTO.setQuestionInfo(questionDTOS);
        log.info("更新对话看板执行步骤和状态入参 eventQuestionId :{},: data:{}",event.getGenerateSerialNo(),
                JsonUtils.objectToString(aiBoardDTO));
        echoService.updateAIBoard(aiBoardDTO,event.getUser().getToken(),event.getUser().getTenantId());
    }
}
