package com.digiwin.athena.adt.agileReport.service.impl.process.agileData;

import com.digiwin.athena.adt.agileReport.constant.AgileDataEnum;
import com.digiwin.athena.adt.agileReport.controller.dto.ScrumBIBoardDTO;
import com.digiwin.athena.adt.agileReport.event.domain.SendMessageDTO;
import com.digiwin.athena.adt.agileReport.eventbus.AthenaMessageEvent;
import com.digiwin.athena.adt.agileReport.interfaces.AgileDataType;
import com.digiwin.athena.adt.agileReport.service.AgileDataProcessService;
import com.digiwin.athena.adt.domain.dto.aiBoard.AIBoardDTO;
import com.digiwin.athena.adt.domain.dto.schema.QuerySchemaDataDTO;
import com.digiwin.athena.adt.domain.dto.schema.QuerySchemaResDTO;
import com.digiwin.athena.adt.domain.echo.EchoService;
import com.digiwin.athena.adt.domain.semc.SemcService;
import com.digiwin.athena.adt.sse.domain.EventData;
import com.digiwin.athena.adt.sse.domain.SseAniaEventEnum;
import com.digiwin.athena.adt.util.MessageUtil;
import com.digiwin.athena.appcore.util.SnowflakeIdWorker;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.security.NoSuchAlgorithmException;
import java.security.SecureRandom;
import java.util.List;
import java.util.Map;
import java.util.Random;

@Slf4j
@Service
@AgileDataType(value = AgileDataEnum.AI_BOARD_QUESTION)
public class AgileDataProcessAIBoardQuestionServiceImpl extends AbsAgileDataProcess implements AgileDataProcessService {

    @Autowired
    private SemcService semcService;

    @Autowired
    private MessageUtil messageUtil;

    @Autowired
    private EchoService echoService;

    @Override
    public void process(AthenaMessageEvent event, QuerySchemaResDTO querySchemaResDTO) {
        Map<String,Object> msgBody = Maps.newHashMap();
        QuerySchemaDataDTO querySchemaDataDTO = querySchemaResDTO.getData();
        if(querySchemaDataDTO == null){
            msgBody.put("prompt",messageUtil.getMessageByLangNameWithFormat(
                    "message.adt.aiBoard.error",event.getLang()));
            msgBody.put("answerModel", "replay");
            semcService.sendMessageToGpt(event, msgBody);
            return;
        }
        //解析语义结构
        ScrumBIBoardDTO scrumBIBoardDTO  = new ScrumBIBoardDTO();
        scrumBIBoardDTO.setBoardTitle(querySchemaDataDTO.getBoardTitle());
        scrumBIBoardDTO.setBoardDesign(querySchemaDataDTO.getBoardDesign());
        scrumBIBoardDTO.setTips(querySchemaDataDTO.getTips());
        scrumBIBoardDTO.setTitle(querySchemaDataDTO.getTitle());

        this.analyzeAIBoardData(event,scrumBIBoardDTO);
        //对话完结
        msgBody.put("analyzeResult",true);
        msgBody.put("dataTipMessage",scrumBIBoardDTO.getTips());
        msgBody.put("boardData",scrumBIBoardDTO);
        msgBody.put("answerModel", "analyzeResult");
        semcService.sendMessageToGpt(event, msgBody);
    }

    /**
     * 解析语义结构
     * @param event
     * @param scrumBIBoardDTO
     */
    private void analyzeAIBoardData(AthenaMessageEvent event, ScrumBIBoardDTO scrumBIBoardDTO) {
        AIBoardDTO aiBoardDTO = new AIBoardDTO();
        if(StringUtils.isNotEmpty(scrumBIBoardDTO.getTitle())){
            EventData.EventMessage eventMessage = EventData.EventMessage.builderAnswerTextMessage(scrumBIBoardDTO.getTitle());
            this.sendEventDataMessage(SseAniaEventEnum.MESSAGE_DELTA.getEvent(),event.getSseEmitter(),eventMessage);
        }
        //看板标题
        aiBoardDTO.setBoardTitle(scrumBIBoardDTO.getBoardTitle());
        StringBuilder markDownTitle = new StringBuilder("\n### **");
        markDownTitle.append(scrumBIBoardDTO.getBoardTitle());
        markDownTitle.append("**");
        markDownTitle.append("\n");
        EventData.EventMessage eventMessage = EventData.EventMessage.builderAnswerTextMessage(markDownTitle.toString());
        this.sendEventDataMessage(SseAniaEventEnum.MESSAGE_DELTA.getEvent(),event.getSseEmitter(),eventMessage);
        //看板内容
        List<AIBoardDTO.AIBoardQuestionDTO> questionInfos = Lists.newArrayList();
        aiBoardDTO.setAppCode(event.getAppCode());
        aiBoardDTO.setMessageId(event.getGenerateSerialNo());
        aiBoardDTO.setQuestionInfo(questionInfos);
        aiBoardDTO.setMessageStatus(0);
        if(CollectionUtils.isNotEmpty(scrumBIBoardDTO.getBoardDesign())){
            scrumBIBoardDTO.getBoardDesign().stream().forEach(boardDesign -> {
                AIBoardDTO.AIBoardQuestionDTO questionDTO = this.buildAIBoardQuestionDTO(boardDesign);
                //存储对象
                questionInfos.add(questionDTO);
                //文本消息输出
                this.contentStream(event, boardDesign.getShowContent());
                //card消息输出
                Map<String,Object> msgBody = Maps.newHashMap();
                msgBody.put("questionId",questionDTO.getQuestionId());
                msgBody.put("answerModel", "questionAnalyze");
                SendMessageDTO sendMessageDTO  = semcService.getSendMessageDTO(event, msgBody);
                EventData.EventMessage cardMessage  = EventData.EventMessage.builderAnswerMessageV2(sendMessageDTO);
                this.sendEventDataMessage(SseAniaEventEnum.MESSAGE_DELTA.getEvent(),event.getSseEmitter(),cardMessage);
            });
        }
        //记录看板问句
        echoService.addAIBoard(aiBoardDTO, event.getUser().getToken(), event.getUser().getTenantId());
    }

    /**
     * 流式输出markdown
     * @param event
     * @param content
     */
    private void contentStream(AthenaMessageEvent event,String content) {
        Random random = new Random();
        int index = 0;  // 当前处理的位置
        int totalLength = content.length();

        while (index < totalLength) {
            // 生成本次截取长度：1~10，但不能超过剩余字符数
            int remaining = totalLength - index;
            if (random == null) {
                break;
            }
            int chunkSize = random.nextInt(Math.min(10, remaining)) + 1;

            // 截取当前块
            String chunk = content.substring(index, index + chunkSize);
            //流式输出
            EventData.EventMessage eventMessage = EventData.EventMessage.builderAnswerTextMessage(chunk);
            this.sendEventDataMessage(SseAniaEventEnum.MESSAGE_DELTA.getEvent(), event.getSseEmitter(), eventMessage);
            //模拟流式输出
            try {
                Thread.sleep(50L + random.nextInt(100)); // 50ms ~ 150ms 随机延迟
            } catch (InterruptedException e) {
                Thread.currentThread().interrupt();
                break;
            }
            // 更新位置
            index += chunkSize;
        }
    }

    /**
     * 构建问句对象
     * @param boardDesign
     * @return
     */
    private AIBoardDTO.AIBoardQuestionDTO buildAIBoardQuestionDTO(ScrumBIBoardDTO.BoardDesign boardDesign){
        AIBoardDTO.AIBoardQuestionDTO aiBoardQuestionDTO = new AIBoardDTO.AIBoardQuestionDTO();
        aiBoardQuestionDTO.setQuestionId(SnowflakeIdWorker.getInstance().newId());
        aiBoardQuestionDTO.setQuestion(boardDesign.getQuestion());
        aiBoardQuestionDTO.setChartTitle(boardDesign.getChartTitle());
        aiBoardQuestionDTO.setShowType(boardDesign.getChartType());
        aiBoardQuestionDTO.setDatasetIds(boardDesign.getDataSetId());
        aiBoardQuestionDTO.setChartAttribute(boardDesign.getChartAttribute());
        return aiBoardQuestionDTO;
    }
}
