package com.digiwin.athena.adt.agileReport.service.impl.process.agileData;

import com.digiwin.athena.adt.agileReport.constant.AgileDataEnum;
import com.digiwin.athena.adt.agileReport.constant.TroubleToolCodeEnum;
import com.digiwin.athena.adt.agileReport.eventbus.AthenaMessageEvent;
import com.digiwin.athena.adt.agileReport.interfaces.AgileDataType;
import com.digiwin.athena.adt.agileReport.service.AgileDataCalculateCostService;
import com.digiwin.athena.adt.agileReport.service.AgileDataCostDetailService;
import com.digiwin.athena.adt.agileReport.service.AgileDataProcessService;
import com.digiwin.athena.adt.agileReport.service.AgileReportService;
import com.digiwin.athena.adt.domain.ade.ADEService;
import com.digiwin.athena.adt.domain.dto.ade.AdeMetricReqDTO;
import com.digiwin.athena.adt.domain.dto.echo.EchoSubmitReq;
import com.digiwin.athena.adt.domain.dto.schema.QuerySchemaMetricDTO;
import com.digiwin.athena.adt.domain.dto.schema.QuerySchemaResDTO;
import com.digiwin.athena.adt.domain.echo.EchoService;
import com.digiwin.athena.adt.domain.semc.SemcService;
import com.digiwin.athena.adt.sse.domain.EventData;
import com.digiwin.athena.adt.sse.domain.SseAniaEventEnum;
import com.digiwin.athena.adt.util.CommonUtil;
import com.digiwin.athena.adt.util.LogUtils;
import com.digiwin.athena.adt.util.agileData.AgileDataAnalysisUtils;
import com.digiwin.athena.appcore.domain.log.LogDto;
import com.digiwin.athena.appcore.util.JsonUtils;
import com.digiwin.athena.adt.util.MessageUtil;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.compress.utils.Lists;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.util.*;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;

/**
 * @Author: SunHong
 * @Date: 2024/6/3 13:38
 * @Description: 执行问句-指标
 */
@Slf4j
@Service
@AgileDataType(value = AgileDataEnum.METRIC)
public class AgileDataProcessMetricServiceImpl extends AbsAgileDataProcess implements AgileDataProcessService{

    @Resource
    private ADEService adeService;

    @Autowired
    private SemcService semcService;

    @Autowired
    private MessageUtil messageUtil;

    @Autowired
    private AgileDataCalculateCostService agileDataCalculateCostService;

    @Autowired
    private EchoService echoService;

    @Autowired
    private AgileReportService agileReportService;

    @Autowired
    private AgileDataCostDetailService agileDataCostDetailService;

    @Override
    public void process(AthenaMessageEvent event, QuerySchemaResDTO responseData) {

        try {
            // 前置
            Map<String,Object> req = this.processBefore(event,responseData);
            // 后置
            this.processAfter(event,req,responseData);
         } catch (Exception e) {
            Map<String, Object> msgBody = new HashMap<>();
            log.error("ade message sendMessageToAde error: {}" ,e.getMessage());
            String tipMessage = messageUtil.getMessageByLangNameWithFormat("message.adt.ade.error",event.getLang());
            msgBody.put("prompt",tipMessage);
            this.saveAbnormal(event,tipMessage,0);
            semcService.sendMessageToGpt(event,msgBody);
            Thread.currentThread().interrupt();
        }
    }

    /**
     * 调用ade 前 前置输出要素识别等信息
     * @param event 上下文
     * @param responseData 语义返回
     * @return ade 入参
     * @throws ExecutionException e
     * @throws InterruptedException i
     */
    public Map<String,Object> processBefore(AthenaMessageEvent event, QuerySchemaResDTO responseData) throws ExecutionException, InterruptedException {
        AtomicBoolean state = new AtomicBoolean(false);
        CountDownLatch returnResultLatch = new CountDownLatch(1);
        // 1.0 指标
        QuerySchemaMetricDTO metric = responseData.getData().getMetric();
        // ade req
        Map<String,Object> req = CommonUtil.convertObjectToMap(AdeMetricReqDTO.BuildMetricAdeReq(event,metric));
        if(event.isSseMessage()){
            log.info("send 要素识别会话开始");
            Map<String ,Object> messageObj = new HashMap<>();
            // 解析要素识别
            String messageTitle = messageUtil
                    .getMessageByLangNameWithFormat("message.sse.essentialFactor", event.getLang());
            messageObj.put("name",messageTitle);
            messageObj.put("arguments ",new HashMap<>());
            // sse组装消息体
            EventData.EventMessage eventMessage = EventData.EventMessage.builderNewEventMessage(
                    "function_call",messageObj,event);
            this.sendEventDataMessage(SseAniaEventEnum.MESSAGE_COMPLETED.getEvent(),event.getSseEmitter(),eventMessage);
        }
        CompletableFuture<Boolean> getEssentialFactor = CompletableFuture.supplyAsync(() -> {
            // sse 解题要素识别
            this.getEssentialFactorAndSendSseMessage(event,req,state,returnResultLatch);
            return true;
        });
        CompletableFuture<Map<String,Object>> adeResult = CompletableFuture.supplyAsync(() -> {
            Map<String, Object> res = adeService.reqMetricSnapShotData(req, event);
            state.set(true);
            try {
                boolean awaitResult = returnResultLatch.await(60, TimeUnit.SECONDS);
                if(awaitResult){
                    log.info("Condition is satisfied");
                }else{
                    log.info("Condition is not satisfied");
                }
            } catch (InterruptedException e) {
                Thread.currentThread().interrupt();
//                throw new RuntimeException(e);
            }
            if(res == null || MapUtils.getMap(res, "data") == null) {
                if(event.isSseMessage()){
                    EventData.EventMessage toolResponse = EventData.EventMessage.builderMessageClose(
                            "tool_response",event.getSseCallId(),"fail");
                    this.sendEventDataMessage(SseAniaEventEnum.MESSAGE_COMPLETED.getEvent(),event.getSseEmitter(),toolResponse);
                }
                String errorMsg = MapUtils.getString(res, "msg");
                Map<String,Object> sendMap = Maps.newHashMap();
                sendMap.put("prompt", errorMsg);
                //记录异常日志
                String tipMessage = messageUtil.getMessageByLangNameWithFormat("message.adt.ade.error", event.getLang());
                this.saveAbnormal(event, tipMessage, 0);
                // 非计费正常发送娜娜
                semcService.sendMessageToGpt(event,sendMap);
                return null;
            }
            return res;
        });
        getEssentialFactor.get();
        log.info("send 取数结束");
        if(event.isSseMessage()){
            EventData.EventMessage toolResponse = EventData.EventMessage.builderMessageClose(
                    "tool_response",event.getSseCallId(),"success");
            this.sendEventDataMessage(SseAniaEventEnum.MESSAGE_COMPLETED.getEvent(),event.getSseEmitter(),toolResponse);
        }
        return adeResult.get();
    }

    /**
     * ade 返回消息后置处理
     * @param event 上下文
     * @param response 返回实体
     * @param responseData 语义返回实体
     */
    public void processAfter(AthenaMessageEvent event, Map<String,Object> response,QuerySchemaResDTO responseData) {
        //记录回答结果
        QuerySchemaMetricDTO metric = responseData.getData().getMetric();
        AgileDataAnalysisUtils.transAnalysisParams(responseData, AdeMetricReqDTO.BuildMetricAdeReq(event,metric), event);
        if(Objects.isNull(response)){
            return;
        }
        String question = event.getQuestion();
        String combinationQuestion = event.getCombinationQuestion();
        Long generateSerialNo = event.getGenerateSerialNo();
        try{
            Map<String,Object> sendMap = Maps.newHashMap();
            Map<String,Object> data = MapUtils.getMap(response, "data");
            // 大数据量优化呈现
            Pair<Boolean, Integer> interception = isDataProcessAction(data);
            if(interception.getLeft()){
                echoService.echoMongodbSubmit(EchoSubmitReq.builderInterception(event, messageUtil,interception.getRight())
                        ,event.getUser().getToken(),event.getUser().getTenantId());
            }
            String snapshotId = MapUtils.getString(data, "snapshotId");
            sendMap.put("snapshotId",snapshotId);
            //指标debug信息封装
            this.executeDebugInfo(event, response,responseData);
            //指标要素识别信息
            this.executeStepAnalysis(event,response);
            // 数据总结标识
            this.setDataSummary(data,event);
            // 数据筛选
            this.setDataScreening(event,response);
            //图表转换呈现信息
            AgileDataAnalysisUtils.getTransShowInfo(data,sendMap,null);
            //转换业务数据，存入历史i消息
            agileReportService.executeAgileDataByTokenSize(event.getUser(),event, data);
            // 记录问答
            this.saveQuestionData(MapUtils.getMap(response, "data"),event,question,combinationQuestion,generateSerialNo);
            // 分类关系
            this.saveBuiltClassifications(event,response);
            if(StringUtils.isNotEmpty(snapshotId)){
                // 保存计费平台数据
                agileDataCostDetailService.saveAgileDataCostDetail(data,event.getUser(),event.getPtxId()
                        ,event.getAppCode(),event.getAppName(),event.getSourceName());
            }
            // 计费组合实现 返回false 记录日志和娜娜 - 根据上下文是否发送娜娜字段判断
            boolean isCheck = agileDataCalculateCostService.combinedBillingCalculation(event,data,sendMap);
            if(!isCheck){
                return;
            }
            // 非计费正常发送娜娜
            semcService.sendMessageToGpt(event,sendMap);
        }catch (Exception e){
            Map<String,Object> sendMap = Maps.newHashMap();
            sendMap.put("prompt",e.getMessage());
            //记录异常日志
            String tipMessage = messageUtil.getMessageByLangNameWithFormat("message.adt.ade.error",event.getLang());
            semcService.sendMessageToGpt(event,sendMap);
            this.saveAbnormal(event,tipMessage,0);
        }
    }


    /**
     * 获取指标debug执行sql
     * @param event
     * @param response
     */
    public void executeDebugInfo(AthenaMessageEvent event ,Map<String,Object> response ,QuerySchemaResDTO responseData) {
        List<Map<String, Object>> debugList = Lists.newArrayList();
        try{
            //指标debug模式，获取debug信息
            if(event.isDebug() && StringUtils.equals("metric",event.getMethod())){
                Map<String,Object> contextMap = MapUtils.getMap(MapUtils.getMap(response, "data"), "context");
                //指标debug模式，通过上下文字段输出执行sql及应用信息
                StringBuilder sqlBuilder = new StringBuilder();
                Object debugObj = contextMap.get("debug");
                if(debugObj instanceof List){
                    debugList.addAll((Collection<? extends Map<String, Object>>) debugObj);
                } else {
                    debugList.add((Map<String, Object>) debugObj);
                }
                if(CollectionUtils.isNotEmpty(debugList)){
                    debugList.stream().forEach(debugMap -> {
                        sqlBuilder.append(messageUtil.getMessageByLangName("message.adt.metricDebug.message",event.getLang()));
                        sqlBuilder.append(MapUtils.getString(debugMap,"applicationCode"));
                        sqlBuilder.append("\n");
                        sqlBuilder.append(messageUtil.getMessageByLangName("message.adt.metricDebug.sql.message",event.getLang()));
                        sqlBuilder.append(MapUtils.getString(debugMap,"sql"));
                    });
                }
                if(StringUtils.isNotEmpty(sqlBuilder.toString())){
                    event.setQuestionUnderstand(sqlBuilder.toString());
                }
                if(CollectionUtils.isNotEmpty(debugList)){
                    //metric debug check
                    this.sendMetricCheckMessage(debugList.get(0),event,responseData);
                }
            }
        }catch (Exception e){
            LogUtils.buildAgileLog(LogUtils.MODULE_ADT, "debugInfo", TroubleToolCodeEnum.ADT_901_0123.getErrCode(),
                    JsonUtils.objectToString(debugList),TroubleToolCodeEnum.ADT_901_0123.getErrMsg(),
                    TroubleToolCodeEnum.ADT_901_0123.getSuggestion());
        }
    }

    /**
     * 上报metric check 集合
     * @param contextMap ade 返参
     */
    public void sendMetricCheckMessage(Map<String, Object> contextMap,AthenaMessageEvent event,QuerySchemaResDTO responseData) {
        contextMap.put("messageId",event.getGenerateSerialNo());
        contextMap.put("appCode",event.getAppCode());
        if(CollectionUtils.isNotEmpty(event.getMetricList())){
            contextMap.put("metricId",event.getMetricList().get(0).get("metricId"));
        }
        contextMap.put("userLang",event.getLang());
        contextMap.put("debugInfo",responseData.getData().getMetric().getDebug());
        contextMap.put("appName",responseData.getData().getMetric().getApplicationList().get(0).get("appName"));
        String token = event.getUser().getToken();
        String tenantId = event.getUser().getTenantId();
        echoService.sendMetricDebugMessage(contextMap,token,tenantId);
    }


}
