package com.digiwin.athena.adt.agileReport.service.impl.process.agileData;

import com.digiwin.athena.adt.agileReport.constant.AgileDataEnum;
import com.digiwin.athena.adt.agileReport.controller.dto.AgileDataSummarizeReqDTO;
import com.digiwin.athena.adt.agileReport.controller.dto.AgileDataSummarizeResDTO;
import com.digiwin.athena.adt.agileReport.eventbus.AthenaMessageEvent;
import com.digiwin.athena.adt.agileReport.interfaces.AgileDataType;
import com.digiwin.athena.adt.agileReport.service.AgileDataProcessService;
import com.digiwin.athena.adt.domain.ade.ADEService;
import com.digiwin.athena.adt.domain.dto.agileReport.SnapShotDTO;
import com.digiwin.athena.adt.domain.dto.echo.AgileDataFileDTO;
import com.digiwin.athena.adt.domain.dto.schema.QuerySchemaResDTO;
import com.digiwin.athena.adt.domain.echo.EchoService;
import com.digiwin.athena.adt.domain.report.AgileReportDataService;
import com.digiwin.athena.adt.util.MessageUtil;
import com.digiwin.athena.appcore.domain.log.LogDto;
import com.digiwin.athena.appcore.util.JsonUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.util.*;

/**
 * @Author: SunHong
 * @Date: 2025/5/19 10:38
 * @Description: ppt 问数
 */
@Slf4j
@Service
@AgileDataType(value = AgileDataEnum.PPT_PROCESS)
public class AgileDataProcessPPTServiceImpl extends AbsAgileDataProcess implements AgileDataProcessService{

    @Resource
    private ADEService adeService;

    @Autowired
    private MessageUtil messageUtil;

    @Autowired
    private EchoService echoService;

    @Autowired
    private AgileReportDataService agileReportDataService;

    @Override
    public void process(AthenaMessageEvent event, QuerySchemaResDTO responseData) {
        try {
            // 前置
            Map<String,Object> req = this.processBefore(event,responseData);
            // 后置
            this.processAfter(event,req,responseData);
         } catch (Exception e) {
            String tipMessage = messageUtil.getMessageByLangNameWithFormat("message.adt.ade.error",event.getLang());
            Map<String,Object> queryData = new HashMap<>();
            queryData.put("title","取数失败");
            queryData.put("content",tipMessage);
            queryData.put("status",false);
            this.updatePPTDataByError(event,responseData.getData().getDataset().getProcessStreamData(),queryData);
            log.error("ade message sendMessageToAde error: {}" ,JsonUtils.objectToString(e));
            this.saveAbnormal(event,tipMessage,0);
            Thread.currentThread().interrupt();
        }
    }

    /**
     * 调用ade 前 前置输出要素识别等信息
     * @param event 上下文
     * @param responseData 语义返回
     * @return ade 入参
     */
    public Map<String,Object> processBefore(AthenaMessageEvent event, QuerySchemaResDTO responseData) {
        // 根据版本号V1 or V2 返回ADE请求入参
        Map<String,Object> req = this.parameterBasedOnVersion(responseData,event);
        // 更新ppt要素识别信息
        this.updateProcessStreamDataByEF(event,req,responseData);
        Map<String, Object> res = adeService.reqMetricSnapShotData(req, event);
        Map<String, Object> queryData = new HashMap<>();
        if(res == null || MapUtils.getMap(res, "data") == null) {
            // 取数步骤
            String errorMsg = MapUtils.getString(res, "msg");
            queryData.put("title","取数失败");
            queryData.put("content",errorMsg);
            queryData.put("status",false);
            // 更新ppt执行步骤取数信息
            this.updateProcessStreamDataByQD(responseData,queryData);
            //记录异常日志
            String tipMessage = messageUtil.getMessageByLangNameWithFormat("message.adt.ade.error", event.getLang());
            this.saveAbnormal(event, tipMessage, 0);
            return null;
        }
        return res;
    }

    /**
     * ade 返回消息后置处理
     * @param event 上下文
     * @param response 返回实体
     * @param responseData 语义返回实体
     */
    public void processAfter(AthenaMessageEvent event, Map<String,Object> response,QuerySchemaResDTO responseData) {
        if(Objects.isNull(response)){
            // 更新失败执行步骤 0 成功 1 失败 2 执行中
            log.info("更新ppt执行步骤 ADE取数为空");
            this.updatePPTDataFile(event,responseData,1,"","");
            return;
        }
        SnapShotDTO snapShotDTO = JsonUtils.jsonToObject(JsonUtils.objectToString(response.get("data")),SnapShotDTO.class);
        boolean queryDataFlag = snapShotDTO.getContext().getBizParams().isQueryDataFlag();
        Map<String, Object> queryData = new HashMap<>();
        if(queryDataFlag){
            queryData.put("title","取数成功");
            queryData.put("status",true);
            queryData.put("content","");
        }else{
            queryData.put("title","取数失败");
            queryData.put("status",false);
            queryData.put("content","无数据");
        }
        //更新执行步骤取数信息
        this.updateProcessStreamDataByQD(responseData,queryData);
        String question = event.getQuestion();
        String combinationQuestion = event.getCombinationQuestion();
        Long generateSerialNo = event.getGenerateSerialNo();
        try{
            Map<String,Object> data = MapUtils.getMap(response, "data");
            String snapshotId = MapUtils.getString(data, "snapshotId");
            // 数据总结标识
            this.setDataSummary(data,event);
            // 更新ppt执行步骤 数据总结
            String summarizeData = this.getProcessSummarizeData(event,snapShotDTO);
            // 更新成功执行步骤
            this.updatePPTDataFile(event,responseData,queryDataFlag ? 0 : 1 ,summarizeData,snapshotId);
            // 记录问答
            this.saveQuestionData(MapUtils.getMap(response, "data"),event,question,combinationQuestion,generateSerialNo);
        }catch (Exception e){
            log.info("更新ppt执行步骤 processAfter error");
            queryData.put("title","取数失败");
            queryData.put("content","process after error");
            queryData.put("status",false);
            // 更新ppt执行步骤取数信息
            this.updateProcessStreamDataByQD(responseData,queryData);
            this.updatePPTDataFile(event,responseData,1,"","");
            //记录异常日志
            String tipMessage = messageUtil.getMessageByLangNameWithFormat("message.adt.ade.error",event.getLang());
            this.saveAbnormal(event,tipMessage,0);
        }
    }

    /**
     * 取数阶段的ppt执行步骤
     * @param responseData scrumBi res
     */
    public void updateProcessStreamDataByQD(QuerySchemaResDTO responseData,
                                            Map<String, Object> queryData) {

        Map<String,Object> processStreamData = responseData.getData().getDataset().getProcessStreamData();
        processStreamData.put("queryData",queryData);
        log.info("取数阶段的ppt执行步骤 processStreamData: {}",JsonUtils.objectToString(processStreamData));
    }

    /**
     * 更新要素识别信息
     * @param event e
     * @param req r
     * @param responseData res
     */
    public void updateProcessStreamDataByEF(AthenaMessageEvent event,
                                        Map<String,Object> req,
                                        QuerySchemaResDTO responseData) {
        Map<String,Object> processStreamData = responseData.getData().getDataset().getProcessStreamData();
        log.info("ppt 更新要素识别语义入参 processStreamData: {}",JsonUtils.objectToString(processStreamData));
        // sse 解题要素识别
        Map<String, Object> essentialFactor = new HashMap<>();
        essentialFactor.put("title","解题要素识别");
        Map<String, Object> response = adeService.getEssentialFactor(event,req);
        if(response == null || Objects.isNull(response.get("msg"))){
            assert response != null;
            essentialFactor.put("content",response.get("msg"));
            essentialFactor.put("status",false);
        }else{
            essentialFactor.put("content",response.get("data"));
            essentialFactor.put("status",true);
        }
        processStreamData.put("stepAnasisly",essentialFactor);
    }

    public String getProcessSummarizeData(AthenaMessageEvent event,SnapShotDTO snapShotDTO) {
        // 无数据
        boolean queryDataFlag = snapShotDTO.getContext().getBizParams().isQueryDataFlag();
        if(!queryDataFlag){
            return "";
        }
        String snapId = snapShotDTO.getSnapshotId();
        // 是否超出数据总结长度
        Boolean isSummarize = event.getExceedSummarizeSize();
        // 默认总结失败
        String resultMsg = messageUtil.getMessageByLangName("message.adt.data.summarize.info", event.getLang());
        // 没有快照不做总结
        if(Objects.isNull(snapId) || Objects.isNull(isSummarize)){
            return resultMsg;
        }
        // 没有超出字段长度,取到数据-尝试数据总结取数
        if(!isSummarize){
            // 数据总结取数
            AgileDataSummarizeReqDTO req = new AgileDataSummarizeReqDTO();
            req.setSnapshotId(snapId);
            AgileDataSummarizeResDTO summarizeData = agileReportDataService.getAgileDataSummarize(req, event.getUser());
            // 根据快照是否总结和文字
            if(StringUtils.isNotEmpty(summarizeData.getSummarizeData()) && summarizeData.isData()){
                resultMsg = summarizeData.getSummarizeData();
            }else{
                return resultMsg;
            }
        }
        return resultMsg;
    }

    /**
     * 根据状态更新ppt执行步骤信息
     * @param event e
     * @param responseData re
     * @param status s
     */
    public void updatePPTDataFile(AthenaMessageEvent event,QuerySchemaResDTO responseData,Integer status,String summarizeData,String snapshotId) {
        AgileDataFileDTO agileDataFileDTO = new AgileDataFileDTO();
        Map<String,Object> processStreamData = responseData.getData().getDataset().getProcessStreamData();
        agileDataFileDTO.setMessageId(Long.valueOf(event.getMsgExt().get("pptMessageId").toString()));
        List<AgileDataFileDTO.AgileDataFileQuestionDTO> questionDTOS = new ArrayList<>();
        AgileDataFileDTO.AgileDataFileQuestionDTO questionDTO = new AgileDataFileDTO.AgileDataFileQuestionDTO();
        questionDTO.setQuestionId(Long.valueOf(event.getMsgExt().get("pptQuestionId").toString()));
        AgileDataFileDTO.AgileDataFileAnalyzeDTO analyzeDTO = JsonUtils.jsonToObject(
                JsonUtils.objectToString(processStreamData),AgileDataFileDTO.AgileDataFileAnalyzeDTO.class);
        questionDTO.setAnalyzeResult(analyzeDTO);
        questionDTO.setAnalyzeStatus(status);
        questionDTO.setSnapshotId(snapshotId);
        questionDTO.setSummarizeData(summarizeData);
        questionDTO.setAnalyzeMessageId(event.getGenerateSerialNo());
        questionDTOS.add(questionDTO);
        agileDataFileDTO.setQuestionInfo(questionDTOS);
        log.info("更新ppt执行步骤和状态入参 eventQuestionId :{},: data:{}",event.getGenerateSerialNo(),
                JsonUtils.objectToString(agileDataFileDTO));
        echoService.updateAgileDataFile(agileDataFileDTO,event.getUser().getToken(),event.getUser().getTenantId());
    }
}
