package com.digiwin.athena.adt.agileReport.service.impl.process.cost;

import com.digiwin.athena.adt.agileReport.constant.BusinessConstants;
import com.digiwin.athena.adt.agileReport.constant.SchemaConstants;
import com.digiwin.athena.adt.agileReport.constant.TroubleToolCodeEnum;
import com.digiwin.athena.adt.agileReport.eventbus.AthenaMessageEvent;
import com.digiwin.athena.adt.agileReport.service.AgileDataCalculateConfigService;
import com.digiwin.athena.adt.agileReport.service.AgileDataCalculateCostService;
import com.digiwin.athena.adt.agileReport.service.impl.process.agileData.AbsAgileDataProcess;
import com.digiwin.athena.adt.domain.cac.AthenaCacService;
import com.digiwin.athena.adt.domain.dto.agileDataProcess.AgileDataProcessResDTO;
import com.digiwin.athena.adt.domain.dto.agileDataProcess.CalculateCostResDTO;
import com.digiwin.athena.adt.domain.dto.agileReport.SnapShotDTO;
import com.digiwin.athena.adt.domain.dto.cac.CacCurrent;
import com.digiwin.athena.adt.domain.dto.cac.CacDecreaseReq;
import com.digiwin.athena.adt.domain.dto.cac.CacGoods;
import com.digiwin.athena.adt.domain.dto.echo.EchoSubmitReq;
import com.digiwin.athena.adt.domain.dto.km.KMIsBillingGoodsResDTO;
import com.digiwin.athena.adt.domain.dto.schema.QuerySchemaResDTO;
import com.digiwin.athena.adt.domain.dto.tdd.TddThresholdConfig;
import com.digiwin.athena.adt.domain.echo.EchoService;
import com.digiwin.athena.adt.domain.semc.SemcService;
import com.digiwin.athena.adt.domain.tdd.TddService;
import com.digiwin.athena.adt.util.CommonUtil;
import com.digiwin.athena.adt.util.LogUtils;
import com.digiwin.athena.appcore.auth.domain.AuthoredUser;
import com.digiwin.athena.appcore.domain.log.LogDto;
import com.digiwin.athena.appcore.util.JsonUtils;
import com.digiwin.athena.adt.util.MessageUtil;
import com.digiwin.athena.atmc.http.restful.iam.UserService;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;

import java.util.*;
import java.util.stream.Collectors;

import static org.apache.commons.collections.MapUtils.getObject;

/**
 * @Author: SunHong
 * @Date: 2024/7/24 15:31
 * @Description: 计费实现
 */
@Slf4j
@Service
public class AgileDataCalculateCostServiceImpl extends AbsAgileDataProcess implements AgileDataCalculateCostService {

    @Autowired
    private AthenaCacService athenaCacService;

    @Autowired
    private TddService tddService;

    @Autowired
    private AgileDataCalculateConfigService agileDataCalculateConfigService;

    @Autowired
    private EchoService echoService;

    @Autowired
    private MessageUtil messageUtil;

    @Autowired
    private SemcService semcService;

    @Value("${cost.calculateCostValue:}")
    private Integer calculateCostValue;

    @Autowired
    private UserService userService;

    /**
     * 返回当前应用是否被流控
     * @param event 上下文
     * @param querySchemaResDTO 语义
     * @return 是否达到cac流量值
     */
    @Override
    public boolean calculateCostByCode(AthenaMessageEvent event, QuerySchemaResDTO querySchemaResDTO) {
        boolean costFlag = true;
        try{
            String userId = event.getUser().getUserId();
            Map<String,Object> scenesMap = this.getCostMap(querySchemaResDTO);
            if(Objects.isNull(scenesMap)){
                return true;
            }
            Object billingGoodsId = scenesMap.get("billingGoodsId");
            Object appCode = scenesMap.get("appCode");
            LogDto logDto = new LogDto("计费业务计算 入参 scenesMap :{}",JsonUtils.objectToString(scenesMap));
            log.info(logDto.toString());
            if(userId.contains("@digiwin.com") || Objects.isNull(billingGoodsId) || Objects.isNull(appCode)){
                return true;
            }
            AuthoredUser user = event.getUser();
            //根据语义返回appCode and id 查询
            CacGoods cacRes = athenaCacService.getCacCurrentTenantGoods(user,billingGoodsId.toString());
            if(Objects.nonNull(cacRes) && (cacRes.getPaymentType() == 0 ||  cacRes.getPaymentType() == 2) ){
                // 如果是按次计费的 需要确认 该用户是否有流量包应用如果没有拦截
                KMIsBillingGoodsResDTO isBillingDTO = agileDataCalculateConfigService.getIsBilling(user);
                if(Objects.isNull(isBillingDTO) || !isBillingDTO.isBilling()){
                    this.sendMessageAndSaveLogByCost(event,BusinessConstants.ADT_COST_BILLING_MSG);
                    return false;
                }
                event.setAppCode(appCode.toString());
                event.setGoodsCode(billingGoodsId.toString());
                event.setCost(true);
                int reUsage =  cacRes.getRemainingUsage();
                if(reUsage <= 0){
                    this.sendMessageAndSaveLogByCost(event,BusinessConstants.ADT_COST_MSG);
                    return false;
                }
            }
        }catch (Exception e){
            log.info("计费异常：calculateCostByCode：{}",JsonUtils.objectToString(e.getMessage()));
        }
        return costFlag;
    }

    /**
     * 1.扣减流量 2.是否成功 3 扣减剩余次数是否小于阈值告警
     * @param event 上下文event参数
     */
    @Override
    public boolean designerDecreaseCurrentTenant(AthenaMessageEvent event) {
        boolean isGo = true;
        String goodsCode = event.getGoodsCode();
        String appCode = event.getAppCode();
        // 版本号
        String productVersion = event.getProductVersion();
        // 如果不符合计费业务条件 (非 按次流量扣减)
        if(!event.isCost() || StringUtils.isEmpty(goodsCode) ||StringUtils.isEmpty(appCode)){
            return isGo;
        }
        CacDecreaseReq req = CacDecreaseReq.builderCacDecrease(event);
        CacCurrent cacCurrent = new CacCurrent();
        try{
            cacCurrent = athenaCacService.decreaseCurrentTenant(event.getUser(),req);
            if(Objects.nonNull(cacCurrent) && cacCurrent.isSuccess()){
                LogUtils.buildAgileLog(LogUtils.MODULE_ADT, "trafficDeduction", LogUtils.SUCCESS,
                        JsonUtils.objectToString(req),JsonUtils.objectToString(cacCurrent), "");
                // cac 可用流量次数
                int remainingUsage = cacCurrent.getRemainingUsage();
                int thresholdNum = 100;
                // 区分 1.0 / 2.0 流量阈值
                if(StringUtils.isEmpty(productVersion) || BusinessConstants.VERSION_1.equals(productVersion)){
                    TddThresholdConfig tddThresholdConfig = tddService.getTddThresholdConfigByParam(
                            event.getUser().getToken(),event.getUser().getTenantId(),appCode);
                    if(tddThresholdConfig == null){
                        return isGo;
                    }
                    thresholdNum = tddThresholdConfig.getThreshold();
                }else if(BusinessConstants.VERSION_2.equals(productVersion)){
                    thresholdNum = calculateCostValue;
                }
                if(remainingUsage >= thresholdNum){
                    log.info("根据阈值和剩余流量初始化用户配置 remainingUsage:{},thresholdNumL:{} ,userId:{}",remainingUsage
                            ,thresholdNum,event.getUser().getUserId());
                    // 如果未到达告警阈值 刷新用户配置需要告警状态
                    agileDataCalculateConfigService.deleteCalculateTypeByParam(event.getUser().getUserId());
                }
                //是否需要告警
                boolean isAlarm = agileDataCalculateConfigService.queryCalculateConfigByUser(event.getUser());
                // 如果需要告警 并且 阈值达到告警数
                if(isAlarm && remainingUsage <= thresholdNum){
                    //上报echo
                    echoService.echoMongodbSubmit(EchoSubmitReq.builderCostAlarm(event,remainingUsage,thresholdNum)
                            ,event.getUser().getToken(), event.getUser().getTenantId());
                    return false;
                }
            }
            LogUtils.buildAgileLog(LogUtils.MODULE_ADT, "trafficDeduction", LogUtils.SUCCESS,
                    JsonUtils.objectToString(req),JsonUtils.objectToString(cacCurrent), "");
        }catch (Exception e){
            LogUtils.buildAgileLog(LogUtils.MODULE_ADT, "trafficDeduction", TroubleToolCodeEnum.ADT_901_0117.getErrCode(),
                    JsonUtils.objectToString(req),TroubleToolCodeEnum.ADT_901_0117.getErrMsg()
                            + JsonUtils.objectToString(cacCurrent), TroubleToolCodeEnum.ADT_901_0117.getSuggestion());
            log.info("计费异常：designerDecreaseCurrentTenant：{}",JsonUtils.objectToString(e.getMessage()));
        }
        return isGo;
    }

    /**
     * 计费返回-数据看板
     * @param event 上下文
     * @param querySchemaResDTO 语义
     * @param agileDataProcessResDTO 返回数据
     * @return agileDataProcessResDTO
     */
    @Override
    public AgileDataProcessResDTO calculateCostResult(AthenaMessageEvent event,
                                                      QuerySchemaResDTO querySchemaResDTO,
                                                      AgileDataProcessResDTO agileDataProcessResDTO) {

        String warnStr;
        try{
            String userId = event.getUser().getUserId();
            Map<String,Object> scenesMap = this.getCostMap(querySchemaResDTO);
            if(Objects.isNull(scenesMap)){
                return agileDataProcessResDTO;
            }
            LogDto logDto = new LogDto("计费业务计算 入参 scenesMap :{}",JsonUtils.objectToString(scenesMap));
            log.info(logDto.toString());
            Object billingGoodsId = scenesMap.get("billingGoodsId");
            Object appCode = scenesMap.get("appCode");
            if(userId.contains("@digiwin.com") || Objects.isNull(billingGoodsId) || Objects.isNull(appCode)){
                return agileDataProcessResDTO;
            }
            AuthoredUser user = event.getUser();
            //根据语义返回appCode and id 查询
            CacGoods cacRes = athenaCacService.getCacCurrentTenantGoods(user,billingGoodsId.toString());
            if(Objects.nonNull(cacRes) && (cacRes.getPaymentType() == 0 ||  cacRes.getPaymentType() == 2) ){
                KMIsBillingGoodsResDTO isBillingDTO = agileDataCalculateConfigService.getIsBilling(user);
                if(Objects.isNull(isBillingDTO) || !isBillingDTO.isBilling()){
                    warnStr = messageUtil.getMessageByLangName(BusinessConstants.ADT_COST_BILLING_MSG, event.getLang());
                    this.sendMessageAndSaveLogByCost(event,BusinessConstants.ADT_COST_BILLING_MSG);
                    agileDataProcessResDTO.setAlarm(true);
                    agileDataProcessResDTO.setMsg(warnStr);
                    return agileDataProcessResDTO;
                }
                event.setAppCode(appCode.toString());
                event.setGoodsCode(billingGoodsId.toString());
                event.setCost(true);
                agileDataProcessResDTO.setRemainingUsage(agileDataProcessResDTO.getRemainingUsage());
                agileDataProcessResDTO.setTotalUsage(cacRes.getTotalUsage());
                int reUsage =  cacRes.getRemainingUsage();
                if(reUsage <= 0){
                    warnStr = messageUtil.getMessageByLangName(BusinessConstants.ADT_COST_MSG, event.getLang());
                    this.sendMessageAndSaveLogByCost(event,BusinessConstants.ADT_COST_MSG);
                    agileDataProcessResDTO.setAlarm(true);
                    agileDataProcessResDTO.setMsg(warnStr);
                    return agileDataProcessResDTO;
                }
            }
        }catch (Exception e){
            log.info("计费异常：calculateCostByCode：{}",JsonUtils.objectToString(e.getMessage()));
        }
        return agileDataProcessResDTO;
    }

    /**
     * 1.计费扣减二次校验流量次数
     * @param event 上下文
     * @return 是否达到cac流量值
     */
    @Override
    public boolean secondCalculateCostByCode(AthenaMessageEvent event) {
        boolean costFlag = true;
        try{
            String appCode = event.getAppCode();
            String goodsCode = event.getGoodsCode();
            LogDto logDto = new LogDto("计费二次计算是否扣减 入参 appCode :{}, goodsCode :{}",appCode,goodsCode);
            log.info(logDto.toString());
            if(!event.isCost() || StringUtils.isEmpty(goodsCode) ||StringUtils.isEmpty(appCode)){
                return costFlag;
            }
            AuthoredUser user = event.getUser();
            //根据语义返回appCode and id 查询
            CacGoods cacRes = athenaCacService.getCacCurrentTenantGoods(user,goodsCode);
            if(Objects.nonNull(cacRes) && (cacRes.getPaymentType() == 0 || cacRes.getPaymentType() == 2) ){
                int reUsage = cacRes.getRemainingUsage();
                if(reUsage <= 0){
                    return false;
                }
            }
        }catch (Exception e){
            log.info("计费异常：calculateCostByCode：{}",JsonUtils.objectToString(e.getMessage()));
        }
        return costFlag;
    }

    /**
     * 根据语义返回信息获取应用code和id
     * metric/dataflow
     * @param querySchemaResDTO 语义返回
     * @return 应用信息
     */
    public Map<String, Object> getCostMap(QuerySchemaResDTO querySchemaResDTO) {
        String method = querySchemaResDTO.getData().getMethod();
        if (SchemaConstants.METHOD_METRIC.equals(method)) {
            Map<String, Object> result = new HashMap<>();
            result.put("appCode", querySchemaResDTO.getData().getMetric().getAppCode());
            result.put("billingGoodsId", querySchemaResDTO.getData().getMetric().getBillingGoodsId());
            return result;
        } else if (SchemaConstants.METHOD_DATA_FLOW.equals(method)){
            List<Map<String, Object>> scenesList = querySchemaResDTO.getData().getDataflow().stream()
                    .map(data -> {
                        Object scenesObj = getObject(data, "scenes");
                        return scenesObj instanceof List? (List<Map<String, Object>>) (scenesObj) : null;
                    })
                    .filter(Objects::nonNull)
                    .filter(CollectionUtils::isNotEmpty)
                    .flatMap(List::stream)
                    .collect(Collectors.toList());
            return CollectionUtils.isNotEmpty(scenesList)? scenesList.get(0) : null;
        }else if (SchemaConstants.METHOD_DATASET.equals(method)){
            Map<String, Object> result = new HashMap<>();
            result.put("appCode", querySchemaResDTO.getData().getDataset().getAppCode());
            result.put("billingGoodsId", querySchemaResDTO.getData().getDataset().getBillingGoodsId());
            return result;
        }else{
            return null;
        }
    }


    /**
     * 计费组合实现
     * 1.二次校验
     * 2.扣减计费
     * 3.记录日志和娜娜
     * @param event 上下文
     * @param data 这里的data 需要记录log 所以 需要res.data.getData
     * @param sendMap 发送消息体
     * @return 是否通过 为true时 是无需计费流程,以及计费扣减成功流程
     */
    @Override
    public boolean combinedBillingCalculation(AthenaMessageEvent event,
                                              Map<String,Object> data,
                                              Map<String,Object> sendMap) {
        boolean isCostCheck = true;
        // 猜你想问
        List<String> relatedIntentions = new ArrayList<>();
        if(Objects.nonNull(sendMap.get("intentions"))){
            relatedIntentions = CommonUtil.objConvertListString(sendMap.get("intentions"));
            event.setAnswerResult(1);
            event.setSentences(relatedIntentions);

        }
        //二次校验流量次数并记录日志
        boolean isCheck = secondCalculateCostByCode(event);
        if(!isCheck){
            if(event.isSendNana()){
                String msg = messageUtil.getMessageByLangName(BusinessConstants.ADT_COST_MSG, event.getLang());
                Map<String, Object> msgBody = new HashMap<>();
                msgBody.put("prompt", msg);
                this.saveAbnormalLog(event, msg,1,2);
                semcService.sendMessageToGpt(event,msgBody);
            }
            return false;
        }
        // 1.扣减计费
        boolean isReturn = designerDecreaseCurrentTenant(event);
        if(!isReturn && event.isSendNana()){
            // 需要告警
            event.getMsgExt().put("isAlarm",true);
            event.getMsgExt().put("messageId",event.getGenerateSerialNo());
            semcService.sendMessageToGpt(event,sendMap);
            //记录问答数据
            this.saveQuestionData(data,event,event.getQuestion(),event.getCombinationQuestion(),event.getGenerateSerialNo());
            return false;
        }
        return isCostCheck;
    }


    /**
     * 数据看板-计费组合实现
     * 1.二次校验
     * 2.扣减计费
     * 3.记录日志和娜娜
     * @param event 上下文
     * @param snapShotDTO 快照返回
     * @return 是否通过 为true时 是无需计费流程,以及计费扣减成功流程
     */
    @Override
    public AgileDataProcessResDTO combinedBillingCalculationPanel(AthenaMessageEvent event,
                                                                  SnapShotDTO snapShotDTO) {
        AgileDataProcessResDTO result = AgileDataProcessResDTO.init();
        CacDecreaseReq req = new CacDecreaseReq();
        CacCurrent cacCurrent = new CacCurrent();
        try{
            String goodsCode = event.getGoodsCode();
            String appCode = event.getAppCode();
            String msg = messageUtil.getMessageByLangName(BusinessConstants.ADT_COST_MSG, event.getLang());
            String cacMsg = messageUtil.getMessageByLangName(BusinessConstants.ADT_COST_CAC_ERROR_MSG, event.getLang());
            //二次校验流量次数并记录日志
            boolean isCheck = secondCalculateCostByCode(event);
            if(!isCheck){
                this.saveAbnormalLog(event, msg,1,2);
                result.setAlarm(true);
                result.setMsg(msg);
                return result;
            }
            if(StringUtils.isNotEmpty(goodsCode) && StringUtils.isNotEmpty(appCode)){
                // 扣减计费
                req = CacDecreaseReq.builderCacDecrease(event);
                cacCurrent = athenaCacService.decreaseCurrentTenant(event.getUser(),req);
                if(cacCurrent != null && cacCurrent.isSuccess()){
                    result.setTotalUsage(cacCurrent.getTotalUsage());
                    result.setRemainingUsage(cacCurrent.getRemainingUsage());
                }else{
                    result.setAlarm(true);
                    result.setMsg(cacMsg);
                }
            }
            // 大数据量优化呈现判断是否截取上下文标识
            Pair<Boolean, Integer> interception = isDataProcessAction(CommonUtil.convertObjectToMap(snapShotDTO));
            if(interception.getLeft()){
                String dataTipMessage = messageUtil.getMessageByLangNameWithFormat(
                        "message.adt.message.page",event.getLang(),interception.getRight());
                result.setDataTipMessage(dataTipMessage);
            }
            // 2.0 快照结构体
            result.setPanelMap(snapShotDTO);
            result.setSnapshotId(snapShotDTO.getSnapshotId());
            result.setRequestTime(snapShotDTO.getContext().getBizParams().getRequestTime());
        }catch (Exception e){
            LogUtils.buildAgileLog(LogUtils.MODULE_ADT, "trafficDeduction", TroubleToolCodeEnum.ADT_901_0117.getErrCode(),
                    JsonUtils.objectToString(req),TroubleToolCodeEnum.ADT_901_0117.getErrMsg()
                            + JsonUtils.objectToString(cacCurrent), TroubleToolCodeEnum.ADT_901_0117.getSuggestion());
        }
        return result;
    }

    /**
     * 计费交易
     * @param user u
     * @param appCode  a
     * @param billingGoodsId b
     * @return true/false
     */
    @Override
    public CalculateCostResDTO calculateCostByUser(AuthoredUser user, String appCode, String billingGoodsId) {
        CalculateCostResDTO calculateCostResDTO = new CalculateCostResDTO();
        calculateCostResDTO.setCostFlag(true);
        try{
            String userId = user.getUserId();
            if(StringUtils.isEmpty(appCode) || StringUtils.isEmpty(billingGoodsId) || userId.contains("@digiwin.com")){
                return calculateCostResDTO;
            }
            LogDto logDto = new LogDto("计费业务计算 入参 appCode :{} ,billingGoodsId :{}"
                    ,appCode,billingGoodsId);
            log.info(logDto.toString());
            String userLocale = userService.getUserLangNameByUserId(user.getUserId(), user.getTenantId(), user.getToken());
            //根据语义返回appCode and id 查询
            CacGoods cacRes = athenaCacService.getCacCurrentTenantGoods(user,billingGoodsId);
            if(Objects.nonNull(cacRes) && (cacRes.getPaymentType() == 0 ||  cacRes.getPaymentType() == 2) ){
                // 如果是按次计费的 需要确认 该用户是否有流量包应用如果没有拦截
                KMIsBillingGoodsResDTO isBillingDTO = agileDataCalculateConfigService.getIsBilling(user);
                if(Objects.isNull(isBillingDTO) || !isBillingDTO.isBilling()){
                    calculateCostResDTO.setMsg(messageUtil.getMessageByLangName(
                            BusinessConstants.ADT_COST_BILLING_MSG, userLocale));
                    calculateCostResDTO.setCostFlag(false);
                    return calculateCostResDTO;
                }
                int reUsage =  cacRes.getRemainingUsage();
                if(reUsage <= 0){
                    calculateCostResDTO.setMsg(messageUtil.getMessageByLangName(
                            BusinessConstants.ADT_COST_MSG, userLocale));
                    calculateCostResDTO.setCostFlag(false);
                    return calculateCostResDTO;
                }
            }
        }catch (Exception e){
            log.info("计费异常：calculateCostByCode：{}",JsonUtils.objectToString(e.getMessage()));
        }
        return calculateCostResDTO;
    }

}
