package com.digiwin.athena.adt.agileReport.service.impl.process.schema;

import com.digiwin.athena.adt.agileReport.constant.SchemaConstants;
import com.digiwin.athena.adt.agileReport.constant.SchemaDataEnum;
import com.digiwin.athena.adt.agileReport.constant.TroubleToolCodeEnum;
import com.digiwin.athena.adt.agileReport.eventbus.AthenaMessageEvent;
import com.digiwin.athena.adt.agileReport.service.impl.process.agileData.AbsAgileDataProcess;
import com.digiwin.athena.adt.domain.dto.schema.QuerySchemaResDTO;
import com.digiwin.athena.adt.domain.dto.schema.QuerySchemaDebugDTO;
import com.digiwin.athena.adt.domain.dto.echo.EchoSubmitReq;
import com.digiwin.athena.adt.domain.echo.EchoService;
import com.digiwin.athena.adt.util.LogUtils;
import com.digiwin.athena.appcore.util.JsonUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.springframework.beans.factory.annotation.Autowired;

import java.util.*;
import java.util.stream.Collectors;

/**
 * @Author: SunHong
 * @Date: 2024/7/1 13:58
 * @Description: 语义抽象公共类
 */
@Slf4j
public abstract class AbsSchemaDataProcess extends AbsAgileDataProcess {

    @Autowired
    private EchoService echoService;

    @Autowired
    private SchemaDataProcessDesignerServiceImpl schemaDataProcessDesignerService;

    @Autowired
    private SchemaDataProcessMetricServiceImpl schemaDataProcessMetricService;

    /** 1.1
     * 根据约定场景解析问句形成参数
     * @param msg 问句
     * @return 问句解析组装实体
     */
    public QuerySchemaDebugDTO assembleQuestionByMsg(String msg){
        String[] parts = msg.split("\\*");
        List<String> partList = Arrays.stream(parts)
                .map(String::trim)
                .filter(s -> !s.isEmpty())
                .collect(Collectors.toList());
        return QuerySchemaDebugDTO.builderDebugByParams(partList);
    }

    /**
     * param问句解析
     * @param msg
     * @return
     */
    public QuerySchemaDebugDTO paramQuestionByMsg(String msg){
        String[] parts = msg.split("\\*");
        List<String> partList = Arrays.stream(parts)
                .map(String::trim)
                .filter(s -> !s.isEmpty())
                .collect(Collectors.toList());
        return QuerySchemaDebugDTO.builderParamByParams(partList);
    }

    /**
     * dataflow
     * 解析语义内容根据入参枚举获取  List<Map<String, Object>>
     * @param querySchemaResDTO 语义参数
     * @param enumStr 获取key
     * @return 集合 List<Map<String, Object>>
     */
    public List<Map<String, Object>> getListBySchemaData(QuerySchemaResDTO querySchemaResDTO, String enumStr) {
        return querySchemaResDTO.getData().getDataflow().stream()
                .map(data -> (List<Map<String, Object>>) MapUtils.getObject(data, enumStr))
                .filter(CollectionUtils::isNotEmpty)
                .flatMap(Collection::stream)
                .collect(Collectors.toList());
    }

    /**
     * dataflow
     * 解析语义内容根据入参枚举获取  String
     * @param querySchemaResDTO 语义
     * @param enumStr 获取key
     * @return 字符串 String
     */
    public String getStringBySchemaData(QuerySchemaResDTO querySchemaResDTO, String enumStr) {
        return querySchemaResDTO.getData().getDataflow().stream()
                .map(data -> MapUtils.getString(data, enumStr))
                .filter(Objects::nonNull)
                .findFirst()
                .orElse("");
    }

    /**
     * dataflow
     * 多场景/多目标 第一轮返回消息
     * specialMode 目前 1为多目标 3 为多场景
     * @param data 语义实体
     * @param event 上下文
     */
    public void sendSpecialMessage(Map<String, Object> data, AthenaMessageEvent event){
        String specialMode = String.valueOf(data.get("specialMode"));
        try{
            // 1 多目标 3 多场景
            this.setAthenaEventMessageType(event,"1".equals(specialMode) ?"1":"3");
            echoService.echoMongodbSubmit(EchoSubmitReq.builderSchemaGuidance(event,
                            "1".equals(specialMode) ? event.getTargets() : event.getScenes())
                    ,event.getUser().getToken(),event.getUser().getTenantId());
            if("1".equals(specialMode)){
                LogUtils.buildAgileLog(LogUtils.MODULE_ADT, "moreTarget", LogUtils.SUCCESS,
                        JsonUtils.objectToString(data),JsonUtils.objectToString(event.getTargets()), "");
            }else{
                LogUtils.buildAgileLog(LogUtils.MODULE_ADT, "moreScene", LogUtils.SUCCESS,
                        specialMode,JsonUtils.objectToString(event.getScenes()), "");
            }
        }catch (Exception e){
            if("1".equals(specialMode)){
                LogUtils.buildAgileLog(LogUtils.MODULE_ADT, "moreTarget", TroubleToolCodeEnum.ADT_901_0115.getErrCode(),
                        specialMode,TroubleToolCodeEnum.ADT_901_0115.getErrMsg(), TroubleToolCodeEnum.ADT_901_0115.getSuggestion());
            }else{
                LogUtils.buildAgileLog(LogUtils.MODULE_ADT, "moreScene", TroubleToolCodeEnum.ADT_901_0116.getErrCode(),
                        specialMode,TroubleToolCodeEnum.ADT_901_0116.getErrMsg(), TroubleToolCodeEnum.ADT_901_0116.getSuggestion());
            }
        }
    }

    /**
     * 根据语义策略实现的method存入上下文
     * @param event 上下文
     * @param schemaMethod 语义返回method,默认为dataflow
     */
    public void setAthenaEventMethod(AthenaMessageEvent event,String schemaMethod) {
        if(StringUtils.isEmpty(schemaMethod)){
            event.setMethod(SchemaConstants.METHOD_DATA_FLOW);
        }else{
            event.setMethod(schemaMethod);
        }
    }

    /**
     * 1.语义返回校验 dataflow/metric
     * 2.metric 多目标/多场景/多应用/导正 判断返回
     * @param event 上下文
     * @param querySchemaResDTO 语义返回
     * @return 校验后上下文和语义实体
     */
    public Pair<Boolean, QuerySchemaResDTO> absCheckSchemaResponseData(AthenaMessageEvent event,
                                                                       QuerySchemaResDTO querySchemaResDTO) {

        if (querySchemaResDTO == null || StringUtils.isEmpty(querySchemaResDTO.getData().getMethod())
                || (CollectionUtils.isEmpty(querySchemaResDTO.getData().getDataflow())
                && (Objects.isNull(querySchemaResDTO.getData().getMetric())
                && Objects.isNull(querySchemaResDTO.getData().getDataset())))){
            return Pair.of(false, null);
        }

        boolean check;
        String method = querySchemaResDTO.getData().getMethod();
        // 确定存储上下文后续链路类型 dataflow/method
        this.setAthenaEventMethod(event,method);
        Pair<Boolean, QuerySchemaResDTO> result;
        switch (method) {
            case SchemaConstants.METHOD_DATA_FLOW:
                check = schemaDataProcessDesignerService.checkDataFlowResponseData(querySchemaResDTO, event);
                result = check? Pair.of(true, querySchemaResDTO) : Pair.of(false, querySchemaResDTO);
                querySchemaResDTO.setRoute(SchemaDataEnum.DESIGNER.getCode());
                LogUtils.buildAgileLog(LogUtils.MODULE_ADT, "sceneAnalysis", LogUtils.SUCCESS,
                        JsonUtils.objectToString(querySchemaResDTO), JsonUtils.objectToString(result), "");
                break;
            case SchemaConstants.METHOD_METRIC:
                check = schemaDataProcessMetricService.checkMetricResponseData(querySchemaResDTO, event);
                result = check? Pair.of(true, querySchemaResDTO) : Pair.of(false, querySchemaResDTO);
                querySchemaResDTO.setRoute(SchemaDataEnum.METRIC.getCode());
                LogUtils.buildAgileLog(LogUtils.MODULE_ADT, "metricAnalysis", LogUtils.SUCCESS,
                        JsonUtils.objectToString(querySchemaResDTO), JsonUtils.objectToString(result), "");
                break;
            case SchemaConstants.METHOD_DATASET:
                check = schemaDataProcessMetricService.checkDatasetResponseData(querySchemaResDTO, event);
                result = check? Pair.of(true, querySchemaResDTO) : Pair.of(false, querySchemaResDTO);
                querySchemaResDTO.setRoute(SchemaDataEnum.DATASET.getCode());
                LogUtils.buildAgileLog(LogUtils.MODULE_ADT, "datasetAnalysis", LogUtils.SUCCESS,
                        JsonUtils.objectToString(querySchemaResDTO), JsonUtils.objectToString(result), "");
                break;
            default:
                result = Pair.of(false, null);
        }
        return result;
    }
}
