package com.digiwin.athena.adt.domain.cac.impl;

import com.digiwin.athena.adt.agileReport.constant.ErrorCodeEnum;
import com.digiwin.athena.adt.agileReport.constant.TroubleToolCodeEnum;
import com.digiwin.athena.adt.domain.cac.AthenaCacService;
import com.digiwin.athena.adt.domain.dto.cac.CacCurrent;
import com.digiwin.athena.adt.domain.dto.cac.CacDecreaseReq;
import com.digiwin.athena.adt.domain.dto.cac.CacGoods;
import com.digiwin.athena.adt.util.LogUtils;
import com.digiwin.athena.appcore.auth.domain.AuthoredUser;
import com.digiwin.athena.appcore.domain.log.LogDto;
import com.digiwin.athena.appcore.util.JsonUtils;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.core.ParameterizedTypeReference;
import org.springframework.http.*;
import org.springframework.stereotype.Service;
import org.springframework.web.client.ResourceAccessException;
import org.springframework.web.client.RestTemplate;

/**
 * @Author: SunHong
 * @Date: 2024/7/24 14:11
 * @Description: CAC 调用服务
 */
@Slf4j
@Service
public class AthenaCacServiceImpl implements AthenaCacService {

    @Autowired(required = false)
    private RestTemplate restTemplate;

    @Value("${cac.url:}")
    private String cacUrl;

    @Value("${athena.auth.appToken}")
    private String appToken;
    /**
     * 获取当前用户指定应用的授权信息
     * @param user 用户信息
     * @param goodsCode 商品编码
     * @return
     */
    @Override
    public CacGoods getCacCurrentTenantGoods(AuthoredUser user, String goodsCode) {

        long getQuerySchemaTook = System.currentTimeMillis();
        LogDto logDto = new LogDto("查询cac商品信息入参 goodsCode :{} user:{}",goodsCode,JsonUtils.objectToString(user));
        log.info(logDto.toString());
        String url= cacUrl +"/api/cac/v4/authorizations/currentTenant/goods/" + goodsCode;
        HttpHeaders headers = new HttpHeaders();
        headers.set("Digi-Middleware-Auth-User",user.getToken());
        headers.set("Digi-Middleware-Auth-App",appToken);
        HttpEntity<String> requestEntity = new HttpEntity<>(null, headers);
        try {
            ResponseEntity<CacGoods> responseEntity = restTemplate.exchange(url, HttpMethod.GET, requestEntity,
                    new ParameterizedTypeReference<CacGoods>() {
                    });
            log.info("查询 cac商品信息返参：{}" ,JsonUtils.objectToString(responseEntity.getBody()));
            return responseEntity.getBody();
        } catch (ResourceAccessException e){
            log.error("agiledata_ADT_ :耗时:{},问题位置:{},问题描述:{},错误内容:{},可能原因及排错指引导:{}",
                    System.currentTimeMillis() - getQuerySchemaTook,
                    "敏捷数据助理入口(ADT)->获取当前用户指定应用的授权信息(CAC) getCacCurrentTenantGoods",
                    "调用CAC接口超时 ResourceAccessException",
                    goodsCode,
                    "请联系平台处理");
            throw ErrorCodeEnum.CAC_REALTIME_FAIL.getBusinessException();
        } catch (Exception e){
            log.error("{}, 请求body：{}, 报错信息：{}"
                    , ErrorCodeEnum.CAC_GET_FAIL.getErrCode()
                    , goodsCode
                    , e.getMessage());
            throw ErrorCodeEnum.CAC_GET_FAIL.getBusinessException();
        }
    }

    /**
     * cac根据userToken减少按次购买的模组计数器
     * @param user 用户信息
     * @param req 服務id
     * @return
     */
    @Override
    public CacCurrent decreaseCurrentTenant(AuthoredUser user, CacDecreaseReq req) {
        long getQuerySchemaTook = System.currentTimeMillis();
        LogDto logDto = new LogDto("根据应用goodsCode 扣减cac 次数 入参 cacDecreaseReq :{} ",JsonUtils.objectToString(req));
        log.info(logDto.toString());
        String url= cacUrl +"/api/cac/v4/counting/decrease/bycurrenttenant";
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        headers.set("Digi-Middleware-Auth-User",user.getToken());
        headers.set("Digi-Middleware-Auth-App",appToken);
        HttpEntity httpEntity = new HttpEntity(req,headers);
        try {
            ResponseEntity<CacCurrent> response = restTemplate.exchange(url, HttpMethod.POST, httpEntity,
                    new ParameterizedTypeReference<CacCurrent>() {
                    });
            log.info("扣减 cac 返参：{}", JsonUtils.objectToString(response));
            return response.getBody();
        } catch (ResourceAccessException e){
            LogUtils.buildAgileLog(LogUtils.MODULE_ADT, "trafficDeduction", TroubleToolCodeEnum.ADT_901_0118.getErrCode(),
                    JsonUtils.objectToString(req),TroubleToolCodeEnum.ADT_901_0118.getErrMsg(), TroubleToolCodeEnum.ADT_901_0118.getSuggestion());
            log.error("agiledata_ADT_ :耗时:{},问题位置:{},问题描述:{},错误内容:{},可能原因及排错指引导:{}",
                    System.currentTimeMillis() - getQuerySchemaTook,
                    "敏捷数据助理入口(ADT)->扣减cac次数(CAC) decreaseCurrentTenant",
                    "调用ADE接口超时 ResourceAccessException",
                    JsonUtils.objectToString(req),
                    "请联系平台处理");
            throw ErrorCodeEnum.CAC_REALTIME_DECREASE_FAIL.getBusinessException();
        } catch (Exception e){
            LogUtils.buildAgileLog(LogUtils.MODULE_ADT, "trafficDeduction", TroubleToolCodeEnum.ADT_901_0119.getErrCode(),
                    JsonUtils.objectToString(req),TroubleToolCodeEnum.ADT_901_0119.getErrMsg(), TroubleToolCodeEnum.ADT_901_0119.getSuggestion());
            log.error("{}, 请求body：{}, 报错信息：{}"
                    , ErrorCodeEnum.CAC_DECREASE_FAIL.getErrCode()
                    , JsonUtils.objectToString(req)
                    , e.getMessage());
            throw ErrorCodeEnum.CAC_DECREASE_FAIL.getBusinessException();
        }
    }
}
