package com.digiwin.athena.adt.domain.dmc.impl;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.digiwin.athena.adt.agileReport.constant.ErrorCodeEnum;
import com.digiwin.athena.adt.agileReport.service.impl.dmc.CommonInputStreamResource;
import com.digiwin.athena.adt.domain.dmc.DmcService;
import com.digiwin.athena.adt.domain.dto.UploadAttachmentResp;
import com.digiwin.athena.adt.domain.dto.UploadParamDTO;
import com.digiwin.athena.adt.domain.dto.agileReport.SnapShotDTO;
import com.digiwin.athena.appcore.auth.GlobalConstant;
import com.digiwin.athena.appcore.auth.domain.AuthoredUser;
import com.digiwin.athena.appcore.exception.BusinessException;
import com.digiwin.athena.appcore.util.JsonUtils;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.core.ParameterizedTypeReference;
import org.springframework.http.*;
import org.springframework.stereotype.Service;
import org.springframework.util.LinkedMultiValueMap;
import org.springframework.util.MultiValueMap;
import org.springframework.util.StringUtils;
import org.springframework.web.client.RestTemplate;

import java.io.ByteArrayInputStream;
import java.io.InputStream;
import java.nio.charset.StandardCharsets;
import java.security.MessageDigest;
import java.util.*;

/**
 * @Author: SunHong
 * @Date: 2025/5/23 11:26
 * @Description:
 */
@Slf4j
@Service
public class DmcServiceImpl implements DmcService {

    @Autowired(required = false)
    private RestTemplate restTemplate;

    @Value("${dmc.account}")
    private String account;

    @Value("${dmc.password}")
    private String password;

    @Value("${dmc.url:}")
    private String dmcUrl;

    @Value("${dmc.bucket:}")
    private String bucket;

    @Value("${athena.auth.appToken}")
    private String appToken;


    @Override
    public UploadAttachmentResp uploadAgileData(UploadParamDTO uploadParamDTO,AuthoredUser user) {
        try {
            //登录token
            String dmcToken = this.login(user);
            HttpHeaders headers = new HttpHeaders();
            MultiValueMap<String, Object> param = new LinkedMultiValueMap<>();
            String uploadUrl = dmcUrl + "/api/dmc/v2/file/"+ bucket +"/upload";
            Map<String,Object> fileMap = Maps.newHashMap();
            fileMap.put("displayName",uploadParamDTO.getFileName());
            fileMap.put("fileName",uploadParamDTO.getDownloadName());
            fileMap.put("tenantId", user.getTenantId());
            fileMap.put("expireDate",uploadParamDTO.getExpireDate());
            InputStream stream = new ByteArrayInputStream(uploadParamDTO.getFileJson().getBytes());
            uploadParamDTO.setInputStream(stream);
            if (uploadParamDTO.getInputStream().available() <= 0) {
                throw new RuntimeException();
            }
            uploadParamDTO.setDownloadName(uploadParamDTO.getFileName() + ".json");
            //重写inputStreamResource 将file直接转成流
            org.springframework.core.io.Resource inputStreamResource =
                    new CommonInputStreamResource(stream, uploadParamDTO.getDownloadName());
            param.add("file", inputStreamResource);
            param.add("fileInfo",fileMap);
            headers.setContentType(MediaType.MULTIPART_FORM_DATA);
            headers.set(GlobalConstant.IAM_USER_TOKEN, dmcToken);
//            headers.set("tenantId",user.getTenantId());
            HttpEntity<MultiValueMap<String, Object>> uploadHttpEntity = new HttpEntity<>(param, headers);
            ResponseEntity<String> responseEntity = restTemplate.exchange(uploadUrl, HttpMethod.POST, uploadHttpEntity, String.class);
            if (HttpStatus.OK.value() == responseEntity.getStatusCodeValue()) {
                String uploadResult = responseEntity.getBody();
                JSONObject jsonObject = JSON.parseObject(uploadResult);
                return JSONObject.parseObject(JSON.toJSONString(jsonObject.getJSONObject("data")), UploadAttachmentResp.class);
            }
        } catch (Exception e) {
            throw BusinessException.create(ErrorCodeEnum.UPLOAD_DMC_ERROR.getErrCode(),
                    ErrorCodeEnum.UPLOAD_DMC_ERROR.getErrMsg());
        }
        return null;
    }


    public String login(AuthoredUser user) {
        String url = dmcUrl + "/api/dmc/v1/auth/login";
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        Map<String, String> params = new HashMap<>();
        params.put("username", account);
        params.put("pwdhash", this.buildPassword(password));
        HttpEntity<Map> httpEntity = new HttpEntity<>(params, headers);
        try {
            ResponseEntity<Map<String, Object>> respEntity = this.restTemplate
                    .exchange(url, HttpMethod.POST, httpEntity, new ParameterizedTypeReference<Map<String, Object>>() {
            });
            if (HttpStatus.OK.value() == respEntity.getStatusCodeValue()) {
                return String.valueOf(Objects.requireNonNull(respEntity.getBody()).get("userToken"));
            } else {
                log.error("dmcLogin: login dmc failed, params: {}, response: {}", params, respEntity.getBody());
                throw BusinessException.create(respEntity.getStatusCodeValue(), "login dmc failed, error: " + JsonUtils.objectToString(respEntity.getBody()));
            }
        } catch (Exception ex) {
            if (ex instanceof BusinessException) {
                throw ex;
            }
            log.error("dmcLogin: login dmc failed, params: {}, error: {}", params, ex);
            throw BusinessException.create("login dmc failed, error: " + ex.getMessage());
        }
    }

    private String buildPassword(String agileDataPassword) {
        try {
            MessageDigest sha256MessageDigest =
                    MessageDigest.getInstance("SHA-256");
            sha256MessageDigest.update(agileDataPassword.getBytes(StandardCharsets.UTF_8));
            byte[] tempByte = sha256MessageDigest.digest();
            sha256MessageDigest.update(tempByte);
            return new String(Base64.getEncoder().encode(sha256MessageDigest.digest()));
        } catch (Exception e) {
            log.error("dmc密码加密错误，错误原因{}：", e.getMessage());
        }
        return agileDataPassword;
    }

    @Override
    public SnapShotDTO getAgileData(String fileId) {

        HttpHeaders headers = new HttpHeaders();
        String url = dmcUrl + "/api/dmc/v2/file/"+ bucket +"/preview/" + fileId;
        HttpEntity<String> requestEntity = new HttpEntity<>(null, headers);
        ResponseEntity<SnapShotDTO> responseEntity = restTemplate.exchange(url, HttpMethod.GET, requestEntity,
                new ParameterizedTypeReference<SnapShotDTO>() {
                });
        log.info("查询 dmc快照信息返参：{}" ,JsonUtils.objectToString(responseEntity.getBody()));
        if(Objects.nonNull(responseEntity.getBody())
                && !StringUtils.isEmpty(responseEntity.getBody().getSnapshotId())){
            return responseEntity.getBody();
        }else{
            return this.getAthenaAgileData(fileId);
        }

    }

    @Override
    public Map<String, Object> getDmcAgileData(String fileId) {
        HttpHeaders headers = new HttpHeaders();
        String url = dmcUrl + "/api/dmc/v2/file/"+ bucket +"/preview/" + fileId;
        HttpEntity<String> requestEntity = new HttpEntity<>(null, headers);
        ResponseEntity<Map<String,Object>> responseEntity = restTemplate.exchange(url, HttpMethod.GET, requestEntity,
                new ParameterizedTypeReference<Map<String,Object>>() {
                });
        log.info("查询 dmc文件信息返参：{}" ,JsonUtils.objectToString(responseEntity.getBody()));
        if(Objects.nonNull(responseEntity.getBody())){
            return responseEntity.getBody();
        }else{
            return null;
        }
    }

    public SnapShotDTO getAthenaAgileData(String fileId) {

        HttpHeaders headers = new HttpHeaders();
        String url = dmcUrl + "/api/dmc/v2/file/Athena/preview/" + fileId;
        HttpEntity<String> requestEntity = new HttpEntity<>(null, headers);
        ResponseEntity<SnapShotDTO> responseEntity = restTemplate.exchange(url, HttpMethod.GET, requestEntity,
                new ParameterizedTypeReference<SnapShotDTO>() {
                });
        log.info("查询 dmc快照信息返参：{}" ,JsonUtils.objectToString(responseEntity.getBody()));
        return responseEntity.getBody();
    }

    @Override
    public Map<String, Object> deleteAgileData(String fileId) {
        String url = dmcUrl + "/api/dmc/v2/file/"+ bucket +"/delete/"+fileId;
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);


        HttpEntity<Map> httpEntity = new HttpEntity<>(null, headers);
        ResponseEntity<Map> responseEntity =
                restTemplate.exchange(url, HttpMethod.DELETE, httpEntity, new ParameterizedTypeReference<Map>() {
                });
        return responseEntity.getBody();
    }

    @Override
    public void deleteBatchAgileData(Collection<String> dirIds, Collection<String> fileIds, String token) {
        String url = dmcUrl + "/api/dmc/v2/file/"+ bucket +"/delete";

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        headers.set(GlobalConstant.IAM_USER_TOKEN, token);

        Map<String, Collection<String>> param = new HashMap<>();
        param.put("fileIds", fileIds);
        param.put("dirIds", dirIds);
        HttpEntity<Map> httpEntity = new HttpEntity<>(param, headers);
        try {
            ResponseEntity<String> respEntity = this.restTemplate.exchange(url, HttpMethod.POST, httpEntity, new ParameterizedTypeReference<String>() {
            });
            if (HttpStatus.OK.value() == respEntity.getStatusCodeValue()) {
                return;
            } else {
                log.error("deleteAgileData: delete dmc AgileData failed, params: {}, response: {}", param, respEntity.getBody());
                throw BusinessException.create(respEntity.getStatusCodeValue(), "delete dmc AgileData failed, error: " + JsonUtils.objectToString(respEntity.getBody()));
            }
        } catch (Exception ex) {
            if (ex instanceof BusinessException) {
                throw ex;
            }
            log.error("deleteAgileData: delete dmc AgileData failed, params: {}, error: {}", param, ex);
            throw BusinessException.create("delete dmc AgileData failed, error: " + ex.getMessage());
        }
    }

    @Override
    public SnapShotDTO getAgileDataByToken(String fileId, String token) {

        HttpHeaders headers = new HttpHeaders();
        headers.add("token",token);
        String url = dmcUrl + "/api/dmc/v2/file/"+ bucket +"/preview/" + fileId;
        HttpEntity<String> requestEntity = new HttpEntity<>(null, headers);
        ResponseEntity<SnapShotDTO> responseEntity = restTemplate.exchange(url, HttpMethod.GET, requestEntity,
                new ParameterizedTypeReference<SnapShotDTO>() {
                });
        log.info("查询 快照信息返参：{}" ,JsonUtils.objectToString(responseEntity.getBody()));
        if(Objects.nonNull(responseEntity.getBody())
                && !StringUtils.isEmpty(responseEntity.getBody().getSnapshotId())){
            return responseEntity.getBody();
        }else{
            return this.getAthenaAgileData(fileId);
        }
    }
}
