package com.digiwin.athena.adt.domain.report.impl;

import cn.hutool.json.JSONObject;
import com.digiwin.athena.adt.agileReport.constant.AgileDataSourceEnum;
import com.digiwin.athena.adt.agileReport.controller.dto.*;
import com.digiwin.athena.adt.agileReport.controller.dto.excel.ExcelSheet;
import com.digiwin.athena.adt.agileReport.eventbus.AthenaMessageEvent;
import com.digiwin.athena.adt.agileReport.service.AgileDataMessageService;
import com.digiwin.athena.adt.agileReport.service.AgileDataSnapRelationMappingService;
import com.digiwin.athena.adt.agileReport.service.ModulePermissionService;
import com.digiwin.athena.adt.agileReport.service.SnapShotDataService;
import com.digiwin.athena.adt.agileReport.service.impl.process.agileData.AbsAgileDataProcess;
import com.digiwin.athena.adt.domain.ade.ADEService;
import com.digiwin.athena.adt.domain.chatbi.ChatbiService;
import com.digiwin.athena.adt.domain.dmc.DmcService;
import com.digiwin.athena.adt.domain.dto.ade.AdeAnalysisAttributionReqDTO;
import com.digiwin.athena.adt.domain.dto.ade.AdeAnalysisAttributionResDTO;
import com.digiwin.athena.adt.domain.dto.ade.AdeDatasetReqDTO;
import com.digiwin.athena.adt.domain.dto.agileDataProcess.AgileDataMCPDTO;
import com.digiwin.athena.adt.domain.dto.agileDataProcess.AgileDataPanelResult;
import com.digiwin.athena.adt.domain.dto.agileDataProcess.AgileFilterProcessResDTO;
import com.digiwin.athena.adt.domain.dto.aiBoard.AIBoardDTO;
import com.digiwin.athena.adt.domain.dto.echo.AgileDataFileDTO;
import com.digiwin.athena.adt.domain.dto.km.KMCollectStepResDTO;
import com.digiwin.athena.adt.domain.dto.km.KMDatasetCommandIntentionsResDTO;
import com.digiwin.athena.adt.domain.dto.km.ThemeMapBoardDTO;
import com.digiwin.athena.adt.domain.dto.mongo.AgileDataSummarizeMsg;
import com.digiwin.athena.adt.domain.dto.schema.SchemaAnalysisAttributionResDTO;
import com.digiwin.athena.adt.domain.dto.schema.SchemaFluctuateResDTO;
import com.digiwin.athena.adt.domain.dto.schema.SchemaMetricShowDefine;
import com.digiwin.athena.adt.domain.echo.EchoService;
import com.digiwin.athena.adt.domain.file.FileParsingService;
import com.digiwin.athena.adt.app.config.executor.ExecutorConfig;
import com.digiwin.athena.adt.domain.dto.AgileDataPanelResDTO;
import com.digiwin.athena.adt.domain.dto.agileDataProcess.AgileDataProcessResDTO;
import com.digiwin.athena.adt.domain.dto.agileReport.*;
import com.digiwin.athena.adt.domain.dto.UploadAttachmentResp;
import com.digiwin.athena.adt.domain.dto.UploadParamDTO;
import com.digiwin.athena.adt.domain.knowledge.KmService;
import com.digiwin.athena.adt.domain.report.AgileReportDataService;
import com.digiwin.athena.adt.domain.semc.SemcService;
import com.digiwin.athena.adt.domain.uibot.UIBOTService;
import com.digiwin.athena.adt.sse.api.SSEManagerService;
import com.digiwin.athena.adt.sse.domain.EventData;
import com.digiwin.athena.adt.sse.domain.SseAniaEventEnum;
import com.digiwin.athena.adt.sse.dto.SSEBaseEvent;
import com.digiwin.athena.adt.sse.utils.SseEmitterUtils;
import com.digiwin.athena.adt.util.CommonUtil;
import com.digiwin.athena.adt.util.LogUtils;
import com.digiwin.athena.adt.util.MessageUtil;
import com.digiwin.athena.adt.util.agileData.AgileDataSnapShotUtils;
import com.digiwin.athena.appcore.auth.AppAuthContextHolder;
import com.digiwin.athena.appcore.auth.domain.AuthoredUser;
import com.digiwin.athena.appcore.util.JsonUtils;
import com.digiwin.athena.adt.util.MessageUtil;
import com.digiwin.athena.appcore.util.SnowflakeIdWorker;
import com.digiwin.athena.atmc.http.restful.iam.UserService;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.eventbus.AsyncEventBus;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.MDC;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.data.mongodb.core.MongoTemplate;
import org.springframework.data.mongodb.core.query.Criteria;
import org.springframework.data.mongodb.core.query.Query;
import org.springframework.stereotype.Service;
import org.springframework.web.servlet.mvc.method.annotation.SseEmitter;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletRequest;
import java.time.format.DateTimeFormatter;
import java.util.*;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.RejectedExecutionException;
import java.util.concurrent.atomic.AtomicReference;

import static com.digiwin.athena.adt.agileReport.constant.BusinessConstants.VERSION_1;
import static com.digiwin.athena.adt.agileReport.constant.BusinessConstants.VERSION_2;


/**
 * 敏捷数据处理
 */
@Slf4j
@Service
public class AgileReportProcessServiceImpl extends AbsAgileDataProcess implements AgileReportDataService {

    @Autowired
    private AsyncEventBus localEventBus;

    @Resource
    private UIBOTService uibotService;

    @Autowired
    private AgileDataMessageService agileDataMessageService;

    @Resource
    private SSEManagerService sseManagerService;

    @Autowired
    private ExecutorConfig executorConfig;

    private final static String YYYY_MM_DD_HH_MM_SS = "yyyy/MM/dd HH:mm:ss";

    @Autowired
    private SnapShotDataService snapShotDataService;

    @Autowired
    private MessageUtil messageUtil;

    @Autowired
    private FileParsingService fileParsingService;

    @Autowired
    private DmcService dmcService;

    @Autowired
    private ChatbiService chatbiService;

    @Autowired
    private KmService kmService;

    @Autowired
    @Qualifier("mongoAgileDataLogTemplate")
    private MongoTemplate mongoTemplate;

    @Resource
    private ADEService adeService;

    @Autowired
    private AgileDataSnapRelationMappingService agileDataSnapRelationMappingService;

    @Autowired
    private UserService userService;

    @Autowired
    private EchoService echoService;

    @Autowired
    @Qualifier("asyncServiceExecutor")
    private ExecutorService executor;

    @Autowired
    private ModulePermissionService modulePermissionService;

    @Autowired
    private SemcService semcService;

    @Override
    public String processSDData(String processData,String fileName,AuthoredUser user) {
        UploadParamDTO uploadParamDTO = new UploadParamDTO();
        uploadParamDTO.setFileJson(processData);
        uploadParamDTO.setFileName(fileName);
        //文件上传
        UploadAttachmentResp uploadAttachmentResp = dmcService.uploadAgileData(uploadParamDTO,user);
        return uploadAttachmentResp.getId();
    }

    /**
     * 问句主链路
     * @param request request
     * @param athenaMessageDTO 入参
     * @param user 用户
     * @return event
     */
    @Override
    public Map<String,String> processMessage(HttpServletRequest request,AthenaMessageDTO athenaMessageDTO, AuthoredUser user) {
        LogUtils.buildAgileLog(LogUtils.MODULE_ADT, "receiveQuestion", LogUtils.SUCCESS,
                JsonUtils.objectToString(athenaMessageDTO), "娜娜问句请求", "");
        Map<String, String> resultType = new HashMap<>();
        if (athenaMessageDTO.getMsgType() == null || athenaMessageDTO.getMsgBody().isEmpty()){
            resultType.put("resultType","NOTSUPPORT");
        }
        AthenaMessageEvent event = init(request,athenaMessageDTO,user);

        localEventBus.post(event);
        resultType.put("resultType","PROCESS");
        return resultType;
    }



    @Override
    public SnapShotDTO getAgileReportInfo(String dmcId) {
        return dmcService.getAgileData(dmcId);
    }

    /**
     * 补偿指标卡数据
     * @param snapShotDTO
     */
    @Override
    public void compensateAgileData(SnapShotDTO snapShotDTO) {
        if(CollectionUtils.isNotEmpty(snapShotDTO.getContext().getPullData())){
            snapShotDTO.getContext().getPullData().stream().forEach(pullDataDTO -> {
                Map<String,Object> computeMap = JsonUtils.jsonToObject(JsonUtils.objectToString(pullDataDTO),Map.class);
                computeMap.put("method",snapShotDTO.getContext().getBizParams().getMethod());
                computeMap.put("showDefine",snapShotDTO.getContext().getBizParams().getShowDefine());
                Map<String,Object> dataMap = uibotService.compensateAgileData(computeMap);
                if(MapUtils.isNotEmpty(dataMap)){
                    //补偿指标卡数据
                    pullDataDTO.setData(dataMap);
                }
            });
        }
    }

    /**
     * 数据面板请求接口
     * @param request request
     * @param req 入参
     * @param user 用户信息
     * @return 数据面板res
     */
    @Override
    public AgileDataPanelResDTO panelProcessMessage(HttpServletRequest request, AthenaMessageDTO req, AuthoredUser user) {
        LogUtils.buildAgileLog(LogUtils.MODULE_ADT, "receiveQuestion", LogUtils.SUCCESS,
                JsonUtils.objectToString(req), "数据面板收藏请求", "");
        AgileDataProcessResDTO result = AgileDataProcessResDTO.init();
        AgileDataPanelResDTO agileDataPanelResDTO = new AgileDataPanelResDTO();
        AthenaMessageEvent event = init(request,req,user);
        event.setSendNana(false);
        event.setSourceName(AgileDataSourceEnum.PANEL.getName());
        event.setSourceCode(AgileDataSourceEnum.PANEL.getCode());

        String questionId = MapUtils.getString(req.getMsgExt(),"questionId");
        if(StringUtils.isEmpty(questionId)){
            result = agileDataMessageService.processRes(event);
        }else{
            //通过KM获取看板数据
            KMCollectStepResDTO stepResDTO = kmService.getCollectStep(user,questionId);
            if(Objects.isNull(stepResDTO) || MapUtils.isEmpty(stepResDTO.getSolutionStep())
                    || Objects.isNull(stepResDTO.getSnapshotId())){
                result = agileDataMessageService.processRes(event);
                this.setPanelDataResult(result,agileDataPanelResDTO,user,event.getLang());
                return agileDataPanelResDTO;
            }
            // 组装问句event
            this.buildEventByStepQuestion(event,stepResDTO);
            String snapshotId = stepResDTO.getSnapshotId();
            SnapShotDTO snapshotInfo = snapShotDataService
                    .getSnapShotData(snapshotId,user.getTenantId(),user.getUserId());
            if(Objects.nonNull(snapshotInfo) &&
                    Objects.nonNull(stepResDTO.getBoardId()) && MapUtils.isNotEmpty(stepResDTO.getSolutionStep())){
                // 直接取数 指标和数据集
                AgileDataProcessResDTO res = agileDataMessageService.processResV1(event,snapshotInfo,stepResDTO);
                result.setAlarm(false);
                result.setDataTipMessage("");
                result.setMsg(res.getMsg());
                result.setSnapshotId(res.getPanelMap().getSnapshotId());
                result.setPanelMap(res.getPanelMap());
                if(res.getPanelMap().getContext() != null) {
                    result.setRequestTime(res.getPanelMap().getContext().getBizParams().getRequestTime());
                }
            }else{
                // 走语义
                result = agileDataMessageService.processRes(event);
            }
        }
        this.setPanelDataResult(result,agileDataPanelResDTO,user,event.getLang());

        return agileDataPanelResDTO;

    }

    public void setPanelDataResult(AgileDataProcessResDTO result,AgileDataPanelResDTO agileDataPanelResDTO,AuthoredUser user,String lang) {
        DateTimeFormatter sdf = DateTimeFormatter.ofPattern(YYYY_MM_DD_HH_MM_SS);
        if(Objects.nonNull(result) && Objects.nonNull(result.getRequestTime())){
            String formatDate = result.getRequestTime().format(sdf);
            agileDataPanelResDTO.setRequestTime(formatDate);
            agileDataPanelResDTO.setSnapshotId(result.getSnapshotId());
            agileDataPanelResDTO.setMsg(result.getMsg());
            agileDataPanelResDTO.setTotalUsage(result.getTotalUsage());
            agileDataPanelResDTO.setRemainingUsage(result.getRemainingUsage());
            agileDataPanelResDTO.setAlarm(result.isAlarm());
            agileDataPanelResDTO.setDataTipMessage(result.getDataTipMessage());
            if(StringUtils.isEmpty(result.getSnapshotId())){
                agileDataPanelResDTO.setResult(false);
            } else{
                SnapShotDTO snapshotInfo = result.getPanelMap();
                //数据为空
                if(snapshotInfo == null){
                    agileDataPanelResDTO.setResult(false);
                } else {
                    //是否存在数据
                    snapshotInfo.getContext().getPullData().stream().forEach(pullDataDTO -> {
                        if(MapUtils.isEmpty(pullDataDTO.getData())){
                            agileDataPanelResDTO.setTitle(AgileDataSnapShotUtils.getTile(snapshotInfo));
                            agileDataPanelResDTO.setMsg(messageUtil.getMessageByLangNameWithFormat(
                                    "exception.agileReport.empty.data",lang));
                            agileDataPanelResDTO.setResult(false);
                            return;
                        }
                        Object dataListObj = pullDataDTO.getData().get("data");
                        if(dataListObj == null){
                            agileDataPanelResDTO.setTitle(AgileDataSnapShotUtils.getTile(snapshotInfo));
                            agileDataPanelResDTO.setResult(false);
                            agileDataPanelResDTO.setMsg(messageUtil.getMessageByLangNameWithFormat(
                                    "exception.agileReport.empty.data",lang));
                            return;
                        }
                        agileDataPanelResDTO.setResult(true);
                    });
                }
            }
        } else {
            agileDataPanelResDTO.setResult(false);
        }
    }




    /**
     * 看板2.0数据获取
     * @param request res
     * @param aniaAssistantChatReqDTO a
     * @param user u
     * @return r
     */
    @Override
    public AgileDataPanelResult panelFlagShipProcessMessage(HttpServletRequest request, AniaAssistantChatReqDTO aniaAssistantChatReqDTO, AuthoredUser user) {
        AgileDataPanelResult agileDataPanelResult = new AgileDataPanelResult();
        Map<String,Object> extData = aniaAssistantChatReqDTO.getMessage().getExtData();
        String questionId = MapUtils.getString(extData,"questionId");
        AthenaMessageEvent event = this.assemblyVersionInitEvent(request,aniaAssistantChatReqDTO,user);
        SnapShotDTO snapShotDTO = null;
        if(StringUtils.isNotEmpty(questionId)){
            //通过KM获取看板数据
            ThemeMapBoardDTO themeMapBoardDTO = kmService.getThemeMapBoardQuestion(user,questionId);
            //存在取数入参，直接取数
            Map<String,Object> adeParams = MapUtils.getMap(themeMapBoardDTO.getDataParam(), "adeRequest");
            if(MapUtils.isNotEmpty(adeParams)){
                AdeDatasetReqDTO adeDatasetReqDTO = new JSONObject(adeParams).toBean(AdeDatasetReqDTO.class);
                adeDatasetReqDTO.setMessageId(String.valueOf(event.getGenerateSerialNo()));
                AgileDataProcessResDTO res = agileDataMessageService.processResV2(event, null, themeMapBoardDTO, adeDatasetReqDTO);
                agileDataPanelResult.setSnapShotDTO(res.getPanelMap());
                agileDataPanelResult.setMsg(res.getMsg());
                agileDataPanelResult.setTitle(AgileDataSnapShotUtils.getTile(adeDatasetReqDTO.getShowDefine()));
            } else {
                if (Objects.isNull(themeMapBoardDTO) || Objects.isNull(themeMapBoardDTO.getSolutionStep())
                        || Objects.isNull(themeMapBoardDTO.getSnapshotId())) {
                    // 走语义
                    AgileDataProcessResDTO res = agileDataMessageService.processRes(event);
                    // 2.0 返回快照结构体
                    agileDataPanelResult.setSnapShotDTO(res.getPanelMap());
                    agileDataPanelResult.setMsg(res.getMsg());
                    return agileDataPanelResult;
                }
                String snapshotId = themeMapBoardDTO.getSnapshotId();
                SnapShotDTO snapshotInfo = snapShotDataService
                        .getSnapShotData(snapshotId, user.getTenantId(), user.getUserId());
                if (Objects.isNull(snapshotInfo)) {
                    agileDataPanelResult.setSnapShotDTO(snapShotDTO);
                    return agileDataPanelResult;
                }
                // 直接取数
                AgileDataProcessResDTO res = agileDataMessageService.processResV2(event, snapshotInfo, themeMapBoardDTO, null);
                this.buildTitle(res.getPanelMap(), themeMapBoardDTO);
                agileDataPanelResult.setSnapShotDTO(res.getPanelMap());
                agileDataPanelResult.setMsg(res.getMsg());
                agileDataPanelResult.setTitle(AgileDataSnapShotUtils.getTile(res.getPanelMap()));
            }
        } else {
            // 走语义
            AgileDataProcessResDTO res = agileDataMessageService.processRes(event);
            // 2.0 返回快照结构体
            agileDataPanelResult.setSnapShotDTO(res.getPanelMap());
            agileDataPanelResult.setMsg(res.getMsg());
        }
        return agileDataPanelResult;
    }

    @Override
    public AgileDataMCPDTO mcpProcessMessage(HttpServletRequest request, AniaAssistantChatReqDTO req, AuthoredUser user) {
        AgileDataMCPDTO agileDataMCPDTO = new AgileDataMCPDTO();
        agileDataMCPDTO.setResult(true);
        AthenaMessageEvent event = this.assemblyVersionInitEvent(request,req,user);
        //取数
        AgileDataProcessResDTO res = agileDataMessageService.processRes(event);
        SnapShotDTO snapShotDTO = res.getPanelMap();
        if(snapShotDTO == null){
            agileDataMCPDTO.setResult(false);
            agileDataMCPDTO.setMessage(res.getMsg());
            return agileDataMCPDTO;
        }
        agileDataMCPDTO.setSnapShotDTO(snapShotDTO);
        return agileDataMCPDTO;
    }

    /**
     * 支持看板标题修改
     * @param snapShotDTO
     * @param themeMapBoardDTO
     */
    private void buildTitle(SnapShotDTO snapShotDTO,ThemeMapBoardDTO themeMapBoardDTO){
        if(StringUtils.isEmpty(themeMapBoardDTO.getChartTitle())){
            return;
        }
        if(snapShotDTO != null){
            ContextDTO contextDTO = snapShotDTO.getContext();
            if(contextDTO != null){
                BizParamsDTO bizParamsDTO = contextDTO.getBizParams();
                if(bizParamsDTO != null){
                    Map<String,Object> showDefine = bizParamsDTO.getShowDefine();
                    List<Map<String,Object>> showType = (List<Map<String, Object>>) MapUtils.getObject(showDefine,"showType");
                    if(CollectionUtils.isNotEmpty(showType)){
                        showType.stream().forEach(showTypeMap -> {
                            List<Map<String,Object>> type = (List<Map<String, Object>>) MapUtils.getObject(showTypeMap,"type");
                            if(CollectionUtils.isNotEmpty(type)){
                                type.stream().forEach(typeMap -> {
                                    typeMap.put("title",themeMapBoardDTO.getChartTitle());
                                });
                            }
                        });
                    }
                }
            }
        }
    }


    private void buildEventByStepQuestion(AthenaMessageEvent event, KMCollectStepResDTO stepResDTO) {
        event.setDatasetIdList(stepResDTO.getDatasetList());
        event.getMsgExt().put("sceneCode",stepResDTO.getSceneCode());
        event.getMsgExt().put("metricIdList",stepResDTO.getMetricIdList());
    }
    /**
     * 娜娜流式问句 1.0
     * @param request http
     * @param athenaMessageDTO 入参实体
     * @param user 用户信息
     * @return sseEmitter
     */
    @Override
    public SseEmitter sseProcessMessage(HttpServletRequest request,
                                        AthenaMessageDTO athenaMessageDTO,
                                        AuthoredUser user) {
        LogUtils.buildAgileLog(LogUtils.MODULE_ADT, "receiveQuestion", LogUtils.SUCCESS,
                JsonUtils.objectToString(athenaMessageDTO), "娜娜流式V1问句请求", "");
        SSEBaseEvent sseBaseEvent = new SSEBaseEvent();

        SseEmitter sseEmitter = sseManagerService.generateSseEmitter(sseBaseEvent);

        sseBaseEvent.setAniaEmitter(sseEmitter);
        // 1.0 上下文初始化
        AthenaMessageEvent event = init(request,athenaMessageDTO,user);
        event.setSaveTransResult(true);
        event.setSseMessage(true);
        event.setSseEmitter(sseEmitter);
        CompletableFuture.runAsync(() -> {
            // 1.1 异步开启会话
            EventData eventData = new EventData(SseAniaEventEnum.CHAT_CREATED.getEvent(), null);
            if (Objects.nonNull(sseEmitter)) {
                SseEmitterUtils.send(sseEmitter, eventData);
            }
            agileDataMessageService.sseProcess(event,sseBaseEvent);

        }, executorConfig.asyncServiceExecutor());

        return sseEmitter;
    }

    /**
     * 娜娜流式问句 2.0
     * @param request req
     * @param aniaAssistantChatReqDTO 2.0
     * @param user user
     * @return 流式返回
     */
    @Override
    public SseEmitter sseProcessMessageV2(HttpServletRequest request,
                                          AniaAssistantChatReqDTO aniaAssistantChatReqDTO,
                                          AuthoredUser user) {

        LogUtils.buildAgileLog(LogUtils.MODULE_ADT, "receiveQuestion", LogUtils.SUCCESS,
                JsonUtils.objectToString(aniaAssistantChatReqDTO), "娜娜流式V2问句请求", "");
        SSEBaseEvent sseBaseEvent = new SSEBaseEvent();

        SseEmitter sseEmitter = sseManagerService.generateSseEmitter(sseBaseEvent);

        sseBaseEvent.setAniaEmitter(sseEmitter);
        AthenaMessageEvent event = this.assemblyVersionInitEvent(request,aniaAssistantChatReqDTO,user);
        event.setSseMessage(true);
        event.setSseEmitter(sseEmitter);
        event.setAppCode(aniaAssistantChatReqDTO.getAssistantId());
        event.setSaveTransResult(true);

        CompletableFuture.runAsync(() -> {
            // 1.1 异步开启会话
            EventData eventData = new EventData(SseAniaEventEnum.CHAT_CREATED.getEvent(), null);
            if (Objects.nonNull(sseEmitter)) {
                SseEmitterUtils.send(sseEmitter, eventData);
            }

            // 校验模组权限
//            if(!modulePermissionService.checkModulePermission(event.getUser(), event)){
//                Map<String, Object> msgBody = new HashMap<>();
//                msgBody.put("prompt", messageUtil.getMessageByLangNameWithFormat("message.adt.module.error",
//                        event.getLang()));
//                semcService.sendMessageToGpt(event,msgBody);
//            }
            agileDataMessageService.sseProcess(event,sseBaseEvent);

        }, executorConfig.asyncServiceExecutor());

        return sseEmitter;
    }

    @Override
    public SseEmitter sseAgileData(HttpServletRequest request, AniaAssistantChatReqDTO aniaAssistantChatReqDTO, AuthoredUser user) {
        LogUtils.buildAgileLog(LogUtils.MODULE_ADT, "autoDataAnalyze", LogUtils.SUCCESS,
                JsonUtils.objectToString(aniaAssistantChatReqDTO), "生成分析师报告请求", "");
        SSEBaseEvent sseBaseEvent = new SSEBaseEvent();

        SseEmitter sseEmitter = sseManagerService.generateSseEmitter(sseBaseEvent);

        sseBaseEvent.setAniaEmitter(sseEmitter);
        AthenaMessageEvent event = this.assemblyVersionInitEvent(request,aniaAssistantChatReqDTO,user);
        event.setSseMessage(true);
        event.setSseEmitter(sseEmitter);
        event.setAppCode(aniaAssistantChatReqDTO.getAssistantId());

        CompletableFuture.runAsync(() -> {
            // 1.1 异步开启会话
            EventData eventData = new EventData(SseAniaEventEnum.CHAT_CREATED.getEvent(), null);
            if (Objects.nonNull(sseEmitter)) {
                SseEmitterUtils.send(sseEmitter, eventData);
            }
            MDC.put("traceId", event.getPtxId());
            MDC.put("PtxId", event.getPtxId());
            log.info("initMdc create chat {},mdc get ptxId :{},event PtxId :{}",MDC.get("traceId"),MDC.get("PtxId"),event.getPtxId());
            agileDataMessageService.sseProcess(event,sseBaseEvent);

        }, executorConfig.asyncServiceExecutor());

        return sseEmitter;
    }

    @Override
    public AgileDataDetailUrlDTO getDataDetailUrl(AuthoredUser user, String snapshotId, String locale) {

        AgileDataDetailUrlDTO agileDataDetailUrlDTO = new AgileDataDetailUrlDTO();
//        //通过数据快照，获取ADE入参
        SnapShotDTO snapshotInfo = snapShotDataService.getSnapShotData(snapshotId,user.getTenantId(),user.getUserId());
        if(Objects.isNull(snapshotInfo)){
            this.buildErrorMessage(agileDataDetailUrlDTO,locale);
            return agileDataDetailUrlDTO;
        }
        Map<String,Object> params = this.buildParams(snapshotInfo);
        if(Objects.isNull(params)){
            return agileDataDetailUrlDTO;
        }
        AgileDataEngineFileDownModel adeFileDown = this.getFileDownUrl(user, params, locale);
        if(adeFileDown == null){
            this.buildErrorMessage(agileDataDetailUrlDTO,locale);
            return agileDataDetailUrlDTO;
        }
        agileDataDetailUrlDTO.setDataUrl(adeFileDown.getData());
        agileDataDetailUrlDTO.setResult(StringUtils.equals("200", adeFileDown.getCode()));
        agileDataDetailUrlDTO.setErrorMessage(adeFileDown.getMsg());
        return agileDataDetailUrlDTO;
    }

    public AgileDataEngineFileDownModel getFileDownUrl(AuthoredUser user,
                                                       Map<String, Object> params,
                                                       String locale) {
        AgileDataEngineFileDownModel model = new AgileDataEngineFileDownModel();
        ArrayList<String> header = (ArrayList<String>) params.get("descriptionsList");
        ArrayList<String> dataKey = (ArrayList<String>) params.get("dataNamesList");
        List data = (List) params.get("data");
        Map<String, Object> excelRequest = new HashMap<>(2);
        StringBuilder sb = new StringBuilder();
        List<ExcelSheet> excelSheets = new ArrayList<>(1);
        String sheetName = "图表数据";
        String fileName = "数据下载";
        if("zh_TW".equals(locale)){
            sheetName = "圖表數據";
            fileName = "數據下載";
        }
        excelSheets.add(ExcelSheet.build(sheetName, header, dataKey, data));
        excelRequest.put("fileName", fileName);
        excelRequest.put("fileData", excelSheets);
        excelRequest.put("tenantId", user.getTenantId());
        AgileDataEngineFileDownModel res = fileParsingService.getDmcShareUrl(user.getTenantId(),
                user.getToken(), locale, excelRequest);
        if(Objects.nonNull(res)){
            return res;
        }
        return model;
    }

    /**
     * 构建隐藏信息
     * @param agileDataDetailUrlDTO dto
     */
    private void buildErrorMessage(AgileDataDetailUrlDTO agileDataDetailUrlDTO,String locale){
        agileDataDetailUrlDTO.setResult(false);
        agileDataDetailUrlDTO.setErrorMessage(messageUtil.getMessageByLangName("message.ade.dataUrl.error.msg",locale));
    }

    /**
     * 构建excel入参
     * @param snapshotInfo i
     * @return r
     */
    private Map<String,Object> buildParams(SnapShotDTO snapshotInfo){
        Map<String,Object> res = new HashMap<>();
        ContextDTO context = snapshotInfo.getContext();
        if(Objects.isNull(context)){
            return null;
        }
        List<PullDataDTO> pullData = context.getPullData();
        if(CollectionUtils.isEmpty(pullData)){
            return null;
        }
        if(CollectionUtils.isNotEmpty(pullData)){
            PullDataDTO pullDataDTO = pullData.get(0);
            res.put("data",pullDataDTO.getData().get("data"));

            Map<String,Object> response =
                    Optional.ofNullable((Map<String, Object>) pullDataDTO.getMetadata().get("response"))
                            .orElse(Collections.emptyMap());

            Map<String,Object> data = Optional.ofNullable((Map<String, Object>) response.get("data"))
                    .orElse(Collections.emptyMap());
          List<Map<String,Object>> fields = (List<Map<String, Object>>) data.get("field");
            List<String> descriptionsList = new ArrayList<>();
            List<String> dataNamesList = new ArrayList<>();
          for(Map<String,Object> field :fields){
                  descriptionsList.add(String.valueOf(field.get("description")));
                  dataNamesList.add(String.valueOf(field.get("data_name")));
          }
            res.put("descriptionsList",descriptionsList);
            res.put("dataNamesList",dataNamesList);
        }
        return res;
    }

    @Override
    public AgileDataSummarizeResDTO getAgileDataSummarize(AgileDataSummarizeReqDTO agileDataSummarizeReqDTO,
                                                          AuthoredUser user) {

        AgileDataSummarizeResDTO agileDataSummarizeResDTO = new AgileDataSummarizeResDTO();
//        String mongoSummarizeData = this.getMongoSummarizeMsg(agileDataSummarizeReqDTO.getSnapshotId());
//        if(StringUtils.isNotEmpty(mongoSummarizeData)){
//            agileDataSummarizeResDTO.setSummarizeData(mongoSummarizeData);
//            agileDataSummarizeResDTO.setData(true);
//            return agileDataSummarizeResDTO;
//        }
        String snapshotId = agileDataSummarizeReqDTO.getSnapshotId();
        //通过数据快照，获取ADE入参
        SnapShotDTO snapshotInfo = snapShotDataService.getSnapShotData(snapshotId,user.getTenantId(),user.getUserId());
        // 判断是否有数据做数据总结
        boolean queryDataFlag = snapshotInfo.getContext().getBizParams().isQueryDataFlag();
        if(!queryDataFlag){
            agileDataSummarizeResDTO.setData(false);
            agileDataSummarizeResDTO.setSummarizeMsg("");
            return agileDataSummarizeResDTO;
        }
        String summarizeData = snapshotInfo.getContext().getPullData().get(0).getSummarizeData();
        String question = snapshotInfo.getContext().getBizParams().getQuestion();
        Map<String, Object> requestParam = new HashMap<>();
        requestParam.put("question",question);
        requestParam.put("resultData",summarizeData);
        requestParam.put("snapshotId",snapshotId);
        requestParam.put("userPromptList", new ArrayList<>());
        Map<String,Object> analysis = chatbiService.getResultAnalysis(requestParam,user);
        if(Objects.isNull(analysis) || Objects.isNull(analysis.get("data"))){
            agileDataSummarizeResDTO.setSummarizeMsg("获取数据总结异常");
            agileDataSummarizeResDTO.setData(false);
            return agileDataSummarizeResDTO;
        }
        String summarizeMsg = String.valueOf(analysis.get("data"));
        agileDataSummarizeResDTO.setSummarizeData(summarizeMsg);
        agileDataSummarizeResDTO.setData(true);
        // 保存mongo db 每条快照的数据总结
        this.saveAgileDataSummarizeById(snapshotId,summarizeMsg);
        return agileDataSummarizeResDTO;
    }

    /**
     * 保存数据总结快照数据
     * @param snapshotId 快照Id
     * @param summarizeMsg 数据总结
     */
    public void saveAgileDataSummarizeById(String snapshotId, String summarizeMsg) {
        Query query = Query.query(Criteria.where("snapshotId").is(snapshotId));
        List<AgileDataSummarizeMsg> results = mongoTemplate.find(query, AgileDataSummarizeMsg.class);
        if(CollectionUtils.isEmpty(results)){
            AgileDataSummarizeMsg msg = new AgileDataSummarizeMsg();
            msg.setSummarizeMsg(summarizeMsg);
            msg.setSnapshotId(snapshotId);
            mongoTemplate.insert(msg);
        }
    }

    /**
     * 根据快照id 获取缓存的数据总结
     * @param snapshotId 快照id
     * @return str
     */
    public String getMongoSummarizeMsg(String snapshotId) {
        Query query = Query.query(Criteria.where("snapshotId").is(snapshotId));
        List<AgileDataSummarizeMsg> results = mongoTemplate.find(query, AgileDataSummarizeMsg.class);
        if(CollectionUtils.isNotEmpty(results)){
            return results.get(0).getSummarizeMsg();
        }
        return null;
    }

    /**
     * 根据 appCode 查询版本号
     * @param user u
     * @param appCode 应用编码
     * @return v1 v2
     */
    @Override
    public Map<String,Object> getAppVersion(AuthoredUser user, String appCode) {
        AthenaMessageEvent event = new AthenaMessageEvent();
        Map<String,Object> res = new HashMap<>();
        event.setUser(user);
        if("asada".equals(appCode)){
            res.put("version", VERSION_1);
            res.put("isAgileDataApp", true);
            return res;
        }
        if (!StringUtils.isEmpty(appCode)) {
            KMDatasetCommandIntentionsResDTO cIData = kmService.dataSetCommandIntentions(event.getUser(),appCode,"");
            if (Objects.nonNull(cIData) && Objects.nonNull(cIData.getAppType())) {
                int appType = cIData.getAppType();
                // 根据应用类型设置版本
                String version = appType == 12 ? VERSION_2 : VERSION_1;
                res.put("version", version);
                // 判断是否为AileData应用
                boolean isAileDataApp = appType == 12 || appType == 6;
                res.put("isAgileDataApp", isAileDataApp);
            }else{
                res.put("version", "");
                res.put("isAgileDataApp", false);
            }
        }
       return res;
    }

    /**
     * 数据筛选
     * @param request req
     * @param req req
     * @param user u
     * @return process res dto
     */
    @Override
    public AgileFilterProcessResDTO filterProcessMessage(HttpServletRequest request, AniaAssistantChatReqDTO req, AuthoredUser user) {
        AgileFilterProcessResDTO res = new AgileFilterProcessResDTO();
        LogUtils.buildAgileLog(LogUtils.MODULE_ADT, "filterQuestion", LogUtils.SUCCESS,
                JsonUtils.objectToString(req), "数据筛选请求", "");
        try{
            Map<String,Object> extData = req.getMessage().getExtData();
            AthenaMessageEvent event = this.assemblyVersionInitEvent(request,req,user);
            event.setSaveTransResult(true);
            String productVersion = event.getProductVersion();
            List<Map<String,Object>> stepFilterAnalysis = (List<Map<String, Object>>) req.getMessage()
                    .getExtData().get("stepFilterAnalysis");
            Object showType = req.getMessage().getExtData().get("showType");
            int filterShowType = 0 ;
            if(Objects.nonNull(showType)){
                filterShowType = Integer.parseInt(showType.toString());
            }
            String snapshotId = MapUtils.getString(extData,"snapshotId");
            SnapShotDTO snapshotInfo = snapShotDataService
                    .getSnapShotData(snapshotId,user.getTenantId(),user.getUserId());
            SnapShotDTO snapShotDTO ;
            if(Objects.isNull(snapshotInfo)){
                res.setMsg("未查询到快照");
                return res;
            }
            if(VERSION_1.equals(productVersion)){
                KMCollectStepResDTO stepResDTO = KMCollectStepResDTO
                        .builderCollectStepBySnapShotInfo(snapshotInfo,filterShowType);
                stepResDTO.setStepFilterAnalysis(stepFilterAnalysis);
                stepResDTO.setFilterAnalysis(true);
                AgileDataProcessResDTO resDTO = agileDataMessageService.processResV1(event,snapshotInfo,stepResDTO);
                snapShotDTO = resDTO.getPanelMap();
            }else{
                // version 2
                ThemeMapBoardDTO themeMapBoardDTO = ThemeMapBoardDTO.
                        builderThemeMapBoardBySnapShotInfo(snapshotInfo,filterShowType);
                themeMapBoardDTO.setStepFilterAnalysis(stepFilterAnalysis);
                themeMapBoardDTO.setFilterAnalysis(true);
                AgileDataProcessResDTO result =  agileDataMessageService.processResV2(event,snapshotInfo,themeMapBoardDTO,null);
                snapShotDTO = result.getPanelMap();
            }
            if(Objects.nonNull(snapShotDTO) && StringUtils.isNotEmpty(snapShotDTO.getSnapshotId())){
                res.setSnapshotId(snapShotDTO.getSnapshotId());
            }
            if(Objects.nonNull(snapShotDTO) && Objects.nonNull(snapShotDTO.getContext()) && Objects.nonNull(snapShotDTO.getContext().getBizParams())){
                res.setTransIds(snapShotDTO.getContext().getBizParams().getTransIds());
                Map<String,Object> showDefine = snapShotDTO.getContext().getBizParams().getShowDefine();
                if(MapUtils.isNotEmpty(showDefine)){
                    List<Map<String,Object>> showTypeList = (List<Map<String, Object>>) MapUtils.getObject(showDefine, "showType");
                    if(CollectionUtils.isNotEmpty(showTypeList)){
                        Map<String, Object> typeMap = showTypeList.get(0);
                        List<Map<String,Object>> typeList = (List<Map<String, Object>>) typeMap.get("type");
                        if(CollectionUtils.isNotEmpty(typeList)){
                            res.setShowType(MapUtils.getString(typeList.get(0), "value"));
                        }
                    }
                }
            }
            res.setOrgSnapshotId(snapshotId);
            String filterSnapshotId = res.getSnapshotId();
            // 保存筛选过后的新老快照映射关系
            agileDataSnapRelationMappingService.saveSnapshotIdMappingRelation(snapshotId,filterSnapshotId);

        }catch (Exception e){
            log.error("filterProcessMessage is error :{}",e.getMessage());
        }
        LogUtils.buildAgileLog(LogUtils.MODULE_ADT, "filterQuestion", LogUtils.SUCCESS,
                "数据筛选返回", JsonUtils.objectToString(res), "");
        return res;

    }


    /**
     * 归因-波动数据
     * @param request r
     * @param req r
     * @param user u
     * @return res
     */
    @Override
    public AnalysisFluctuateResDTO analysisFluctuateProcess(HttpServletRequest request,
                                                     AnalysisFluctuateReqDTO req, AuthoredUser user) {
        log.info("数据波动-入参：{}",JsonUtils.objectToString(req));
        AnalysisFluctuateResDTO res = new AnalysisFluctuateResDTO();
        AthenaMessageEvent event = new AthenaMessageEvent();
        Long messageId = SnowflakeIdWorker.getInstance().newId();
        event.setUser(user);
        event.setGenerateSerialNo(messageId);
        event.setPtxId(MDC.get("PtxId"));
        try{
            String snapshotId = req.getSnapshotId();
            SnapShotDTO orgSnapshotDTO = snapShotDataService
                    .getSnapShotData(snapshotId,user.getTenantId(),user.getUserId());
            if(Objects.isNull(orgSnapshotDTO)){
                res.setMsg("未查询到快照");
                return res;
            }
            // 获取用户语言别
            String userLocale = userService.getUserLangNameByUserId(user.getUserId(), user.getTenantId(), user.getToken());
            event.setLang(userLocale);

            AnalysisFluctuateReqDTO scrumReq = AnalysisFluctuateReqDTO.builderQuery(req,orgSnapshotDTO);
            SchemaFluctuateResDTO scrumAnalysis = chatbiService.getFluctuateAnalysis(event,scrumReq);
            //语义无法解析，输出提示语
            if(scrumAnalysis == null){
                res.setMsg("AI分析异常");
                return res;
            }
            if(scrumAnalysis != null && scrumAnalysis.getData() != null && StringUtils.equals("1",scrumAnalysis.getData().getCode())){
                res.setMsg(scrumAnalysis.getData().getReply());
                return res;
            }

            AdeDatasetReqDTO params = AdeDatasetReqDTO.BuildAnalysisFluctuateReq(scrumAnalysis,orgSnapshotDTO,user);
            params.setLocale(event.getLang());
            params.setMessageId(String.valueOf(messageId));
            Map<String,Object> response = adeService.reqMetricSnapShotData(CommonUtil.convertObjectToMap(params),event);
            if(response == null || "-1".equals(response.get("code"))) {
                String errorMsg = MapUtils.getString(response, "msg");
                res.setMsg(errorMsg);
                return res;
            }
            SnapShotDTO snapShotDTO = JsonUtils.jsonToObject(JsonUtils.objectToString(response.get("data")),SnapShotDTO.class);
            res.setMsg("");
            res.setStep(scrumAnalysis.getData().getStep());
            res.setSolutionStep(scrumAnalysis.getData().getSolutionStep());
            res.setPullData(snapShotDTO.getContext().getPullData());
            res.setOrgOutPutSql(scrumAnalysis.getData().getOrgOutPutSql());
            res.setDatasetIds(scrumReq.getDatasetIds());
            res.setQuestion(scrumAnalysis.getData().getQuestion());
            res.setQuestionDate(scrumReq.getQuestionDate());
        }catch (Exception e){
            log.error("获取归因-波动数据失败:{}",e.getMessage());
        }
        return res;
    }


    /**
     * 归因 - 取数
     * @param request r
     * @param req r
     * @param user u
     * @return res
     */
    @Override
    public AnalysisAttributionResDTO analysisAttributionProcess(HttpServletRequest request,
                                                                AnalysisAttributionReqDTO req,
                                             AuthoredUser user) {
        log.info("归因分析入参：{}",JsonUtils.objectToString(req));
        AnalysisAttributionResDTO res = new AnalysisAttributionResDTO();
        AthenaMessageEvent event = new AthenaMessageEvent();
        Long messageId = SnowflakeIdWorker.getInstance().newId();
        event.setUser(user);
        event.setGenerateSerialNo(messageId);
        event.setPtxId(MDC.get("PtxId"));
        SchemaAnalysisAttributionResDTO scrumAnalysis = chatbiService.getAttributionAnalysis(event,req);
        if(scrumAnalysis == null || scrumAnalysis.getData() == null || !"0".equals(scrumAnalysis.getData().getCode())){
            assert Objects.requireNonNull(scrumAnalysis).getData() != null;
            String errorMsg = scrumAnalysis.getData().getReply();
            res.setMsg(errorMsg);
            return res;
        }
            String snapshotId = req.getSnapshotId();
        SnapShotDTO orgSnapshotDTO = snapShotDataService
                .getSnapShotData(snapshotId,user.getTenantId(),user.getUserId());
        if(Objects.isNull(orgSnapshotDTO)){
            res.setMsg("未查询到快照");
            return res;
        }
        AdeAnalysisAttributionReqDTO params = AdeAnalysisAttributionReqDTO
                .BuildAnalysisAttributionReq(scrumAnalysis,req,orgSnapshotDTO,user);
        // 获取用户语言别
        String userLocale = userService.getUserLangNameByUserId(user.getUserId(), user.getTenantId(), user.getToken());
        params.setLocale(userLocale);
        AdeAnalysisAttributionResDTO response = adeService.getAttributionDimension(event,params);
        if(response == null || "-1".equals(response.getCode())) {
            assert response != null;
            String errorMsg = response.getMsg();
            res.setMsg(errorMsg);
            return res;
        }
        res.setGroupedSolutionSteps(response.getData().getGroupedSolutionSteps());
        return res;
    }

    /**
     * ppt 问句解析
     * @param request r
     * @param req r
     * @param user u
     */
    @Override
    public void chatPPts(HttpServletRequest request, ChatReqPPTReqDTO req, AuthoredUser user) {
        Map<String, Object> params = Maps.newHashMap();
        Long pptMessageId = req.getMessageId();
        params.put("messageId", pptMessageId);
        AgileDataFileDTO pptDTO;

        try {
            // 1.0 获取PPT问句信息
            pptDTO = echoService.getAgileDataFile(params, user.getToken(), user.getTenantId());
            log.info("执行 ppt 解析 获取问句信息:{}",JsonUtils.objectToString(pptDTO));
            if (pptDTO == null || CollectionUtils.isEmpty(pptDTO.getQuestionInfo())) {
                log.warn("未获取到PPT问句信息，messageId：{}", pptMessageId);
                return;
            }
            String appCode = pptDTO.getAppCode();
            // 1.1 更新PPT解析状态
            this.updatePPTStatusByMessageId(pptDTO, user, req.getMessageId());
            CompletableFuture.runAsync(() -> {
                // 2.0 异步执行问数
                this.processChatPPT(appCode, user, pptDTO,
                        agileDataMessageService::processPPT,
                        () -> agileDataMessageService.processPPTAfter(user, pptMessageId)
                );
            }, executorConfig.asyncServiceExecutor());
        }catch (Exception e) {
            log.error("PPT问句解析未知异常，messageId：{}", req.getMessageId(), e);
        }
    }

    /**
     * 解析对话看板
     * @param request r
     * @param req r
     * @param user u
     */
    @Override
    public void chatAIBoard(HttpServletRequest request, ChatReqAIBoardDTO req, AuthoredUser user) {
        Map<String, Object> params = Maps.newHashMap();
        Long aiBoardMessageId = req.getMessageId();
        params.put("messageId", aiBoardMessageId);
        AIBoardDTO aiBoardDTO;
        log.info("开始执行解析对话看板,入参:{}",JsonUtils.objectToString(req));
        try {
            // 获取看板信息
            aiBoardDTO = echoService.getAIBoard(params, user.getToken(), user.getTenantId());
            log.info("执行 对话看板 获取问句信息:{}",JsonUtils.objectToString(aiBoardDTO));
            if (aiBoardDTO == null || CollectionUtils.isEmpty(aiBoardDTO.getQuestionInfo())) {
                log.warn("未获取到对话看板问句信息，messageId：{}", aiBoardMessageId);
                return;
            }
            String appCode = aiBoardDTO.getAppCode();
            // 1.1 更新对话看板解析状态
            this.updateAIBoardStatusByMessageId(aiBoardDTO, user, req.getMessageId());
            CompletableFuture.runAsync(() -> {
                // 2.0 异步执行问句解析
                this.processChatAIBoard(appCode, user, aiBoardDTO,
                        agileDataMessageService::processAIBoard,
                        () -> agileDataMessageService.processAIBoardAfter(user, aiBoardMessageId)
                );
            }, executorConfig.asyncServiceExecutor());
        }catch (Exception e) {
            log.error("对话看板解析未知异常，messageId：{}", req.getMessageId(), e);
        }
    }

    @FunctionalInterface
    public interface QuestionProcessor {
        void process(AthenaMessageEvent event);
    }

    /**
     * 异步执行ppt问数
     * @param pptBean questions
     * @param processor process
     * @param after after
     */
    public void processChatPPT(String appCode,
                                   AuthoredUser user,
                                   AgileDataFileDTO pptBean,
                                   QuestionProcessor processor,
                                   Runnable after) {

        List<AgileDataFileDTO.AgileDataFileQuestionDTO> questions = pptBean.getQuestionInfo();
        if (CollectionUtils.isEmpty(questions)) {
            log.warn("PPT问句列表为空，无需处理，messageId：{}", pptBean.getMessageId());
            after.run();
            return;
        }
        try {
            CompletableFuture<?>[] futures = questions.stream()
                    .map(question -> CompletableFuture.runAsync(
                            () -> {
                                AthenaMessageEvent event = new AthenaMessageEvent();
                                try {
                                    event = this.initEvent(appCode, question.getQuestion(),question.getDatasetIds(), user);
                                    event.setSendNana(false);
                                    event.getMsgExt().put("pptMessageId",pptBean.getMessageId());
                                    event.getMsgExt().put("pptQuestionId",question.getQuestionId());
                                    processor.process(event);
                                } catch (Exception e) {
                                    Map<String,Object> queryData = new HashMap<>();
                                    queryData.put("title","系统异常");
                                    queryData.put("content","");
                                    queryData.put("status",false);
                                    this.updatePPTDataByError(event,null,queryData);
                                    log.error("异步处理PPT问句失败，questionId：{}", question.getQuestion(), e);
                                } finally {
                                    AppAuthContextHolder.clearContext();
                                }
                            },
                            executor
                    )).toArray(CompletableFuture[]::new);
            //部分任务失败也执行after
            CompletableFuture.allOf(futures)
                    .exceptionally(ex -> {
                        log.error("PPT问句异步处理批次失败，messageId：{}", pptBean.getMessageId(), ex);
                        return null;
                    })
                    .thenRun(after)
                    .join();
        } catch (RejectedExecutionException e) {
            log.error("线程池[asyncServiceExecutor]拒绝接收任务，messageId：{}", pptBean.getMessageId(), e);
        } catch (Exception e) {
            log.error("PPT问句异步处理框架异常，messageId：{}", pptBean.getMessageId(), e);
        }
    }

    public void processChatAIBoard(String appCode,
                               AuthoredUser user,
                               AIBoardDTO aiBoardDTO,
                               QuestionProcessor processor,
                               Runnable after) {

        List<AIBoardDTO.AIBoardQuestionDTO> questions = aiBoardDTO.getQuestionInfo();
        if (CollectionUtils.isEmpty(questions)) {
            log.warn("对话看板问句列表为空，无需处理，messageId：{}", aiBoardDTO.getMessageId());
            after.run();
            return;
        }
        try {
            CompletableFuture<?>[] futures = questions.stream()
                    .map(question -> CompletableFuture.runAsync(
                            () -> {
                                AthenaMessageEvent event = new AthenaMessageEvent();
                                try {
                                    event = this.initEvent(appCode, question.getQuestion(),question.getDatasetIds(), user);
                                    event.setSendNana(false);
                                    event.getMsgExt().put("aiBoardMessageId",aiBoardDTO.getMessageId());
                                    event.getMsgExt().put("aiBoardQuestionId",question.getQuestionId());
                                    processor.process(event);
                                } catch (Exception e) {
                                    Map<String,Object> queryData = new HashMap<>();
                                    queryData.put("title","系统异常");
                                    queryData.put("content","");
                                    queryData.put("status",false);
                                    this.updatePPTDataByError(event,null,queryData);
                                    log.error("异步处理对话看板问句失败，questionId：{}", question.getQuestion(), e);
                                } finally {
                                    AppAuthContextHolder.clearContext();
                                }
                            },
                            executor
                    )).toArray(CompletableFuture[]::new);
            //部分任务失败也执行after
            CompletableFuture.allOf(futures)
                    .exceptionally(ex -> {
                        log.error("对话看板问句异步处理批次失败，messageId：{}", aiBoardDTO.getMessageId(), ex);
                        return null;
                    })
                    .thenRun(after)
                    .join();
        } catch (RejectedExecutionException e) {
            log.error("线程池[asyncServiceExecutor]拒绝接收任务，messageId：{}", aiBoardDTO.getMessageId(), e);
        } catch (Exception e) {
            log.error("对话看板问句异步处理框架异常，messageId：{}", aiBoardDTO.getMessageId(), e);
        }
    }

    /**
     * 更新ppt状态为解析汇总
     * @param user u
     * @param messageId messageId
     */
    public void updatePPTStatusByMessageId(AgileDataFileDTO agileDataFileDTO,AuthoredUser user,Long messageId) {
        AgileDataFileDTO updateStatusDTO = new AgileDataFileDTO();
        agileDataFileDTO.setMessageId(messageId);
        agileDataFileDTO.setMessageStatus(2);
        updateStatusDTO.setMessageId(messageId);
        updateStatusDTO.setMessageStatus(2);
        List<AgileDataFileDTO.AgileDataFileQuestionDTO> updateQuestions = Lists.newArrayList();
        updateStatusDTO.setQuestionInfo(updateQuestions);
        List<AgileDataFileDTO.AgileDataFileQuestionDTO> questionInfos = agileDataFileDTO.getQuestionInfo();
        Iterator<AgileDataFileDTO.AgileDataFileQuestionDTO> iterator = questionInfos.iterator();
        while (iterator.hasNext()) {
            AgileDataFileDTO.AgileDataFileQuestionDTO dto = iterator.next();
            Integer analyzeStatus = dto.getAnalyzeStatus();
            if (analyzeStatus != null && analyzeStatus == 0) {
                iterator.remove();
            } else {
                AgileDataFileDTO.AgileDataFileQuestionDTO questionDTO = new AgileDataFileDTO.AgileDataFileQuestionDTO();
                questionDTO.setQuestionId(dto.getQuestionId());
                questionDTO.setAnalyzeStatus(2);
                updateQuestions.add(questionDTO);
                dto.setAnalyzeStatus(2);
            }
        }
        echoService.updateAgileDataFile(updateStatusDTO,user.getToken(),user.getTenantId());
    }

    public void updateAIBoardStatusByMessageId(AIBoardDTO aiBoardDTO,AuthoredUser user,Long messageId) {
        aiBoardDTO.setMessageId(messageId);
        aiBoardDTO.setMessageStatus(2);
        List<AIBoardDTO.AIBoardQuestionDTO> questionInfos = aiBoardDTO.getQuestionInfo();
        Iterator<AIBoardDTO.AIBoardQuestionDTO> iterator = questionInfos.iterator();
        while (iterator.hasNext()) {
            AIBoardDTO.AIBoardQuestionDTO dto = iterator.next();
            Integer analyzeStatus = dto.getAnalyzeStatus();
            if (analyzeStatus != null && analyzeStatus == 0) {
                iterator.remove();
            } else {
                dto.setAnalyzeStatus(2);
            }
        }
        echoService.updateAIBoard(aiBoardDTO,user.getToken(),user.getTenantId());
    }
}
