package com.digiwin.athena.adt.agileReport.service;

import com.digiwin.athena.appcore.auth.AppAuthContextHolder;
import com.digiwin.athena.appcore.util.JsonUtils;
import com.digiwin.athena.atmc.http.env.EnvProperties;
import com.rabbitmq.client.Channel;
import lombok.extern.slf4j.Slf4j;
import org.springframework.amqp.rabbit.core.RabbitTemplate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.i18n.LocaleContextHolder;
import org.springframework.data.mongodb.core.MongoTemplate;
import org.springframework.data.mongodb.core.query.Criteria;
import org.springframework.data.mongodb.core.query.Query;
import org.springframework.data.mongodb.core.query.Update;
import org.springframework.stereotype.Service;

import java.io.IOException;
import java.time.LocalDateTime;
import java.util.Locale;
import java.util.Map;

@Slf4j
@Service
public class EventBaseService {

    @Autowired
    protected RabbitTemplate rabbitTemplate;

    @Autowired
    protected EnvProperties envProperties;

    @Autowired
    @Qualifier("mongoTemplate")
    private MongoTemplate apiLogMongoTemplate;


    private static final String MONGODB_COLLECTION_NAME = "AbtEvent_Log";
    private static final String EXECPTION_RETRY_COUNT = "execption-retry-count";

    /**
     * 接收消息后处理
     *
     * @param data
     * @param queueName
     * @param channel
     * @param tag
     * @throws IOException
     */
    public void receiveOrder(String data, String queueName, Channel channel, long tag,
                             Map<String, Object> headers) throws IOException {
        log.info("[athena.event.message.adt-data]: queueName = {} , processMessage data = {}", queueName, data);
        boolean successed;
        Integer retryCount = 0;
        try {
            // 获取语言别
            if ("TRADITIONAL_CHINESE".equals(envProperties.getDefaultLanguage())) {
                LocaleContextHolder.setLocale(Locale.TRADITIONAL_CHINESE);
            } else {
                LocaleContextHolder.setLocale(Locale.SIMPLIFIED_CHINESE);
            }
            // 处理消息
            processMessage(data, queueName,headers);
            successed = true;
        } catch (Exception e) {//NOSONAR
            //如果处理失败
            log.error("[{}.error] error:{}, stack:{}, data:{}", queueName, e, e.getStackTrace(), data);
            successed = false;
            retryCount = retryBpmMessage(data, e);
        } finally {
            AppAuthContextHolder.clearContext();
        }
        if (successed || retryCount >= envProperties.getBpmMqRetryCount()) {
            channel.basicAck(tag, false);
        } else {
            //消息重回队列
            channel.basicReject(tag, true);
        }
    }

    /**
     * 处理mq消息
     */
    public void processMessage(String data, String  queueName,Map<String,Object> headers) {

    }

    /**
     * 记录重试次数
     *
     * @param data
     * @param e
     * @return
     */
    private Integer retryBpmMessage(String data, Exception e) {
        Integer retryCount = 1;
        try {
            Object exceptionMessage = e.toString();

            int messageId = data.hashCode();
            Query query = Query.query(Criteria.where("execption-message-id").is(messageId));
            query.fields().include(EXECPTION_RETRY_COUNT);
            Map log = apiLogMongoTemplate.findOne(query, Map.class, MONGODB_COLLECTION_NAME);
            if (log != null) {
                retryCount = (Integer) log.get(EXECPTION_RETRY_COUNT);
                retryCount = retryCount + 1;

                query = new Query(Criteria.where("_id").is(log.get("_id")));
                Update update = new Update()
                        .set("execption-message", exceptionMessage)
                        .set(EXECPTION_RETRY_COUNT, retryCount)
                        .set("execption-modify-time", LocalDateTime.now());
                apiLogMongoTemplate.updateFirst(query, update, MONGODB_COLLECTION_NAME);
            } else {
                log = JsonUtils.jsonToObject(data, Map.class);
                log.put("execption-message", exceptionMessage);
                log.put("execption-message-id", messageId);
                log.put(EXECPTION_RETRY_COUNT, retryCount);
                log.put("execption-create-time", LocalDateTime.now());
                log.put("execption-modify-time", LocalDateTime.now());
                apiLogMongoTemplate.insert(log, MONGODB_COLLECTION_NAME);
            }
        } catch (Exception ex) {
            log.error("[athena_event_message_abt_error] error: {}", ex);
            //如果retry异常，则不再重试
            retryCount = envProperties.getBpmMqRetryCount() + 1;
        }

        return retryCount;
    }

}
