package com.digiwin.athena.adt.agileReport.service.impl.ania;

import com.digiwin.athena.adt.agileReport.service.AniaAssistantService;
import com.digiwin.athena.adt.domain.dto.ania.*;
import com.digiwin.athena.adt.domain.dto.km.KMDatasetCommandIntentionsResDTO;
import com.digiwin.athena.adt.domain.dto.mongo.AgileDataCustomConfig;
import com.digiwin.athena.adt.domain.knowledge.KmService;
import com.digiwin.athena.appcore.auth.domain.AuthoredUser;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.data.mongodb.core.MongoTemplate;
import org.springframework.data.mongodb.core.query.Criteria;
import org.springframework.data.mongodb.core.query.Query;
import org.springframework.data.mongodb.core.query.Update;
import org.springframework.stereotype.Service;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Objects;

/**
 * @Author: SunHong
 * @Date: 2025/5/13 9:50
 * @Description: 2.0 助理应用信息定义
 */
@Service
public class AniaAssistantServiceImpl implements AniaAssistantService {

    @Autowired
    private KmService kmService;

    @Autowired
    @Qualifier("mongoAgileDataLogTemplate")
    private MongoTemplate mongoTemplate;

    @Value("${adt.url:}")
    private String adtUrl;


    /**
     * 根据助理id查询 敏数2.0 助理信息
     * @param req 助理ids
     * @param user 用户信息
     * @return 助理信息
     */
    @Override
    public AniaAssistantResDTO getPublishedInfo(AniaAssistantReqDTO req,AuthoredUser user) {
        // 请求助理集合
        List<AniaAssistantReqDTO.Assistant> assistantReq =  req.getAssistant();
        AniaAssistantResDTO res = new AniaAssistantResDTO();
        List<AniaAssistantResDTO.Assistant> assistantList = new ArrayList<>();
        for(AniaAssistantReqDTO.Assistant dto : assistantReq){
            String appCode = dto.getAssistantId();
            String version = dto.getVersion();
            AgileDataCustomConfig config = new AgileDataCustomConfig();
            // km获取指令集和开场白
            KMDatasetCommandIntentionsResDTO cIData = kmService.dataSetCommandIntentions(user,appCode);
            int appType = 6;
            if(Objects.nonNull(cIData) && Objects.nonNull(cIData.getAppType())){
                appType = cIData.getAppType();
            }
            AgileDataCustomConfig configReq = new AgileDataCustomConfig();
            configReq.setVersion(appType == 12 ? "V2" : "V1");
            // 获取动态前端组件配置
            List<AgileDataCustomConfig> customConfigs = this.getMongoCustomConfig(configReq,user);
            if(CollectionUtils.isNotEmpty(customConfigs)){
                config = customConfigs.get(0);
            }
            // 组装应用助理信息
            AniaAssistantResDTO.Assistant assistant =  AniaAssistantResDTO.Assistant
                    .builderAssistant(version,appCode,cIData,config,adtUrl);
            assistantList.add(assistant);
        }
        res.setAssistant(assistantList);
        //1.0
        return res;
    }

    /**
     * 助理应用前端组件信息定义
     * @param agileDataCustomConfig 入参
     * @param user token
     */
    @Override
    public void saveCustomConfig(AgileDataCustomConfig agileDataCustomConfig, AuthoredUser user) {
        String version = agileDataCustomConfig.getVersion();
        Query query = Query.query(Criteria.where("version").is(version));
        List<AgileDataCustomConfig> results = mongoTemplate.find(query, AgileDataCustomConfig.class);
        if(CollectionUtils.isNotEmpty(results)){
            Update update = new Update();
            update.set("pluginCustoms",agileDataCustomConfig.getPluginCustoms());
            mongoTemplate.updateFirst(query,update,AgileDataCustomConfig.class);
        }else{
            mongoTemplate.insert(agileDataCustomConfig);
        }
    }

    /**
     * 查询助理前端组件信息
     * @param agileDataCustomConfig bean
     * @param user token
     * @return 助理前端组件信息
     */
    @Override
    public List<AgileDataCustomConfig> getMongoCustomConfig(AgileDataCustomConfig agileDataCustomConfig, AuthoredUser user) {
        String version = agileDataCustomConfig.getVersion();
        Query query = Query.query(Criteria.where("version").is(version));
        return mongoTemplate.find(query, AgileDataCustomConfig.class);
    }

    @Override
    public AniaVersionResDTO getVersion(AniaVersionReqDTO req, AuthoredUser user) {
        AniaVersionResDTO res = new AniaVersionResDTO();
        List<AniaAssistantResDTO.Assistant> assistant = new ArrayList<>();
        String version = kmService.getVersion(user);
        if(Objects.nonNull(version)){
            AniaAssistantResDTO.Assistant versionBean = new AniaAssistantResDTO.Assistant();
            versionBean.setAssistantId(req.getAssistantId());
            versionBean.setVersion(version);
            assistant.add(versionBean);
        }
        res.setAssistant(assistant);
        return res;
    }

    @Override
    public AniaAssistantResDTO getInspirationPhrases(AniaAssistantReqDTO req,
                                                              AuthoredUser user) {

        List<AniaAssistantReqDTO.Assistant> assistantReq =  req.getAssistant();
        AniaAssistantResDTO result = new AniaAssistantResDTO();
        List<AniaAssistantResDTO.Assistant> assistantList = new ArrayList<>();
        for(AniaAssistantReqDTO.Assistant dto : assistantReq){
            AniaAssistantResDTO.Assistant assistant = new AniaAssistantResDTO.Assistant();
            String appCode = dto.getAssistantId();
            // km获取指令集和开场白
            KMDatasetCommandIntentionsResDTO cIData = kmService.dataSetCommandIntentions(user,appCode);

            List<Map<String, Object>> commandIntentions = new ArrayList<>();

            if(Objects.nonNull(cIData) && StringUtils.isNotEmpty(cIData.getApplicationName())){
                if(CollectionUtils.isNotEmpty(cIData.getCommandIntentions())){
                    commandIntentions = cIData.getCommandIntentions();
                }
            }
            // 灵感话术 -> 指令集 数据组装
            List<AniaAssistantResDTO.Assistant.InspirationPhrases> phrases =
                    AniaAssistantResDTO.Assistant.InspirationPhrases.builderInspirationPhrases(commandIntentions);
            assistant.setInspirationPhrases(phrases);
            assistant.setAssistantId(appCode);
            assistantList.add(assistant);
        }
        result.setAssistant(assistantList);
        return result;
    }

    @Override
    public void deleteMongoCustomConfig(AgileDataCustomConfig agileDataCustomConfig, AuthoredUser user) {
        String version = agileDataCustomConfig.getVersion();
        Query query = Query.query(Criteria.where("version").is(version));
        mongoTemplate.remove(query,"agile_data_custom_config");
    }
}
