package com.digiwin.athena.adt.agileReport.service.impl.dataset;

import com.alibaba.fastjson.JSONObject;
import com.digiwin.athena.adt.agileReport.controller.dto.AgileDataTableDataResDTO;
import com.digiwin.athena.adt.agileReport.controller.dto.AgileDataTableListResDTO;
import com.digiwin.athena.adt.agileReport.controller.dto.AgileTableDataReqDTO;
import com.digiwin.athena.adt.agileReport.eventbus.AthenaMessageEvent;
import com.digiwin.athena.adt.agileReport.service.AgileDataTableDataService;
import com.digiwin.athena.adt.domain.ade.ADEService;
import com.digiwin.athena.adt.domain.dto.ade.AdeDataSetFieldReqDTO;
import com.digiwin.athena.adt.domain.dto.ade.AdeDatasetReqDTO;
import com.digiwin.athena.adt.domain.dto.km.KMDataSetResDTO;
import com.digiwin.athena.adt.domain.dto.km.KMDatasetCommandIntentionsResDTO;
import com.digiwin.athena.adt.domain.dto.schema.QuerySchemaDatasetDTO;
import com.digiwin.athena.adt.domain.dto.schema.SchemaMetricShowDefine;
import com.digiwin.athena.adt.domain.knowledge.KmService;
import com.digiwin.athena.adt.util.CommonUtil;
import com.digiwin.athena.adt.util.MessageUtil;
import com.digiwin.athena.appcore.auth.domain.AuthoredUser;
import com.digiwin.athena.appcore.util.JsonUtils;
import com.digiwin.athena.appcore.util.SnowflakeIdWorker;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.jugg.agile.framework.core.dapper.log.JaMDC;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.slf4j.MDC;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.util.*;
import java.util.stream.Collectors;

/**
 * @Author: SunHong
 * @Date: 2025/5/21 13:57
 * @Description: 数据集列表实现
 */

@Slf4j
@Service
public class AgileDataTableDataServiceImpl implements AgileDataTableDataService {

    @Autowired
    private KmService kmService;
    @Resource
    private ADEService adeService;
    @Autowired
    private MessageUtil messageUtil;

    @Override
    public AgileDataTableListResDTO getTableList(AuthoredUser user, AgileTableDataReqDTO reqDTO) {

        AgileDataTableListResDTO result = new AgileDataTableListResDTO();
        List<AgileDataTableListResDTO.DatasetTableData> tableList = new ArrayList<>();
        String appCode = reqDTO.getAppCode();
        String userLang = reqDTO.getLang();
        try{
            KMDatasetCommandIntentionsResDTO cIData = kmService.dataSetCommandIntentions(user,appCode);
            if(Objects.isNull(cIData)){
                result.setMsg("未查询到应用");
                result.setTableList(tableList);
                return result;
            }
            Integer appType = cIData.getAppType();
            //获取数据集列表
            List<KMDataSetResDTO> dataSetResDTO = kmService.queryDataSetsByAppCode(user,appCode,userLang);
            if(CollectionUtils.isEmpty(dataSetResDTO)){
                result.setMsg("未查询到数据集");
                result.setTableList(tableList);
                return result;
            }
            for(KMDataSetResDTO dataset : dataSetResDTO){
                // 数据集datasetList
                Map<String,Object> datasetMap = this.assembleDatasetMap(dataset);
                AgileDataTableListResDTO.DatasetTableData  tableData = new AgileDataTableListResDTO.DatasetTableData();
                // 数据集id
                String datasetId = dataset.getCode();
                // 数据集名称
                String datasetName = dataset.getName();
                // 推荐问句
                List<String> sentences = dataset.getQuestions().stream()
                        .map(KMDataSetResDTO.question::getQuestion)
                        .collect(Collectors.toList());
                tableData.setTableId(datasetId);
                tableData.setTableName(datasetName);
                tableData.setSentences(sentences);

                // 根据数据集id查询元数据
                AdeDataSetFieldReqDTO req = AdeDataSetFieldReqDTO.builderReq(user,appCode,datasetId);
                Map<String, Object> snapshotInfo = adeService.getDataSetFieldList(user,req,userLang);
                if(Objects.isNull(snapshotInfo) || "-1".equals(snapshotInfo.get("code"))){
                    if(Objects.isNull(snapshotInfo)){
                        result.setMsg(messageUtil.getMessageByLangName("message.adt.data.dataTable.error", reqDTO.getLang()));
                    } else {
                        result.setMsg(String.valueOf(snapshotInfo.get("msg")));
                    }
                    result.setTableList(tableList);
                    return result;
                }
                Optional.ofNullable(snapshotInfo.get("data"))
                        .map(data -> (Map<String, Object>) data).flatMap(dataMap -> Optional.ofNullable(dataMap.get("context"))
                                .map(context -> (Map<String, Object>) context)).ifPresent(contextMap -> {
                            Object pullDataObj = contextMap.get("pullData");
                            if (pullDataObj instanceof List) {
                                List<Map<String, Object>> pullData = (List<Map<String, Object>>) pullDataObj;
                                for (Map<String, Object> pullDataDTO : pullData) {
                                    // 提取数据及字段
                                    this.parseDataStructure(pullDataDTO, tableData);
                                }
                            } else {
                                log.info("getTableList req : {}", JsonUtils.objectToString(req));
                                result.setMsg(messageUtil.getMessageByLangName("message.adt.data.dataTable.error", reqDTO.getLang()));
                            }
                        });

                if(appType !=null && 12 == appType){
                    tableData.setProductVersion("V2");
                }else{
                    tableData.setProductVersion("V1");
                }
                tableData.setDataset(datasetMap);
                tableList.add(tableData);
            }
            result.setTableList(tableList);
        }catch (Exception e){
            log.error("{}, 请求code：{}, 报错信息：{} ", "获取数据集列表查询解析失败"
                    ,JSONObject.toJSONString(appCode), e.getMessage());
        }
        return result;
    }


    /**
     * 组装数据预览dataset
     * @param kmDataSetResDTO 数据返返回
     * @return dataset
     */
    public Map<String, Object> assembleDatasetMap(KMDataSetResDTO kmDataSetResDTO) {
        Map<String, Object> dataset = new HashMap<>();

        dataset.put("modelId",kmDataSetResDTO.getModelId());
        dataset.put("modelCode",kmDataSetResDTO.getModelCode());
        dataset.put("datasetName",kmDataSetResDTO.getName());
        dataset.put("datasetId",kmDataSetResDTO.getCode());
        dataset.put("lang",kmDataSetResDTO.getLang());
        dataset.put("fieldSchema",buildFieldSchemaList(kmDataSetResDTO));

        dataset.put("applicationCode",kmDataSetResDTO.getAppCode());
        dataset.put("applicationName",kmDataSetResDTO.getAppName());

        dataset.put("mappingFields", buildFieldAndTableFieldMap(kmDataSetResDTO));
        return dataset;
    }

    public static Object buildFieldSchemaList(KMDataSetResDTO kmDataSetResDTO) {
        List<Map<String,Object>> fieldSchema = new ArrayList<>();
        if(CollectionUtils.isNotEmpty(kmDataSetResDTO.getDimensions())){
            for (Map<String,Object> dimension : kmDataSetResDTO.getDimensions()) {
                if(isNonEmptyCollection(dimension.get("enums"))){
                    Map<String,Object> map = new HashMap<>();
                    map.put("name",dimension.get("data_name"));
                    map.put("enums",dimension.get("enums"));
                    fieldSchema.add(map);
                }

            }
        }
        if(CollectionUtils.isNotEmpty(kmDataSetResDTO.getMeasures())){
            for (Map<String,Object> measure : kmDataSetResDTO.getMeasures()) {
                if(isNonEmptyCollection(measure.get("enums"))){
                    Map<String,Object> map = new HashMap<>();
                    map.put("name",measure.get("data_name"));
                    map.put("enums",measure.get("enums"));
                    fieldSchema.add(map);
                }
            }
        }
        return fieldSchema;
    }

    public static boolean isNonEmptyCollection(Object obj) {
        // 判断对象是否为 Collection 类型
        if (obj instanceof Collection<?>) {
            // 强制转换为 Collection
            Collection<?> collection = (Collection<?>) obj;
            // 检查集合是否非空
            return !collection.isEmpty();
        }
        return false;
    }

    public static Map<String,Object> buildFieldAndTableFieldMap(KMDataSetResDTO kmDataSetResDTO){
        Map<String,Object> mappingFields = new HashMap<>();

        if(CollectionUtils.isNotEmpty(kmDataSetResDTO.getDimensions())){
            for (Map<String,Object> dimension : kmDataSetResDTO.getDimensions()) {
                mappingFields.put(String.valueOf(dimension.get("data_name")),dimension.get("map_field"));
            }
        }
        if(CollectionUtils.isNotEmpty(kmDataSetResDTO.getMeasures())){
            for (Map<String,Object> measure : kmDataSetResDTO.getMeasures()) {
                mappingFields.put(String.valueOf(measure.get("data_name")),measure.get("map_field"));
            }
        }
        return mappingFields;
    }

    /**
     * 获取快照数据下的数据集字段
     * @param data      元数据
     * @param tableData 返回结构
     */
    public void parseDataStructure(Map<String,Object> data,
                                   AgileDataTableListResDTO.DatasetTableData tableData
                                                                        ) {
        try {
            // 维度
            List<AgileDataTableListResDTO.DatasetTableData.Field> dimensionFields = new ArrayList<>();
            // 度量
            List<AgileDataTableListResDTO.DatasetTableData.Field> measurementFields = new ArrayList<>();

            ObjectMapper mapper = new ObjectMapper();
            JsonNode rootNode = mapper.readTree(JSONObject.toJSONString(data));
            Optional<JsonNode> fieldNodeOptional = Optional.ofNullable(rootNode)
                    .map(node -> node.path("metadata").path("response").path("data"));

            fieldNodeOptional.ifPresent(dataNode -> {
                String dataType = dataNode.path("data_type").asText();
                if ("object".equals(dataType)) {
                    JsonNode fieldNode = dataNode.path("field");
                    if (fieldNode.isArray()) {
                        fieldNode.forEach(node -> {
                            AgileDataTableListResDTO.DatasetTableData.Field field = new AgileDataTableListResDTO.DatasetTableData.Field();
                            String dataName = node.path("data_name").asText();
                            String dataTypeName = node.path("data_type").asText();
                            String description = node.path("description").asText();
                            String feature = node.path("feature").asText();
                            field.setName(dataName);
                            field.setDataType(dataTypeName);
                            field.setDescription(description);
                            // 维度
                            if("dimensions".equals(feature)){
                                dimensionFields.add(field);
                            }else if("measures".equals(feature)){
                                // 度量
                                measurementFields.add(field);
                            }
                        });
                    }
                }
            });
            tableData.setDimensionFields(dimensionFields);
            tableData.setMeasurementFields(measurementFields);
        } catch (Exception e) {
            log.error("{}, 请求code：{}, 报错信息：{} ", "获取数据集列表查询解析失败",
                    JSONObject.toJSONString(tableData), e.getMessage());
        }
    }

    /**
     * 获取数据集数据预览详情
     * @param user user
     * @param reqDTO tableId - 数据集id
     * @return 预览详情
     */
    @Override
    public AgileDataTableDataResDTO getTablePreviewData(AuthoredUser user, AgileTableDataReqDTO reqDTO) {
        AgileDataTableDataResDTO resDTO = new AgileDataTableDataResDTO();
        try{
            String tableId = reqDTO.getTableId();
            String appCode = reqDTO.getAppCode();
            String userLang = reqDTO.getLang();
            // 根据数据集id查询元数据
            AdeDataSetFieldReqDTO datasetFieldReq = AdeDataSetFieldReqDTO.builderReq(user,appCode,tableId);
            Map<String,Object> snapshotInfo = adeService.getDataSetFieldList(user,datasetFieldReq,userLang);
            if(Objects.isNull(snapshotInfo) || "-1".equals(snapshotInfo.get("code"))){
                if(Objects.isNull(snapshotInfo)){
                    resDTO.setMsg(messageUtil.getMessageByLangName("message.adt.data.dataTable.error", reqDTO.getLang()));
                } else {
                    resDTO.setMsg(String.valueOf(snapshotInfo.get("msg")));
                }
                return resDTO;
            }
            // 组装solutionStep
            List<Map<String,Object>> solutionStep = new ArrayList<>();
            Map<String,Object> data = (Map<String, Object>) snapshotInfo.get("data");

            Map<String,Object> context = (Map<String, Object>) data.get("context");
            // 获取pullData字段
            List<Map<String,Object>> pullData = (List<Map<String,Object>>) context.get("pullData");
            for(Map<String,Object> pullDataDTO : pullData){
                List<Map<String,Object>> selectList = new ArrayList<>();
                Map<String,Object> select = new HashMap<>();
                // 提取数据及字段
                ObjectMapper mapper = new ObjectMapper();
                JsonNode rootNode = mapper.readTree(JSONObject.toJSONString(pullDataDTO));
                Optional<JsonNode> fieldNodeOptional = Optional.ofNullable(rootNode)
                        .map(node -> node.path("metadata").path("response").path("data"));
                fieldNodeOptional.ifPresent(dataNode -> {
                    String dataType = dataNode.path("data_type").asText();
                    if ("object".equals(dataType)) {
                        JsonNode fieldNode = dataNode.path("field");
                        if (fieldNode.isArray()) {
                            fieldNode.forEach(node -> {
                                Map<String,Object> dataObjectMap = new HashMap<>();
                                Map<String,Object> dataObject = new HashMap<>();
                                String dataName = node.path("data_name").asText();
                                String dataTypeName = node.path("data_type").asText();
                                String description = node.path("description").asText();
                                dataObject.put("dataType",dataTypeName);
                                dataObject.put("alias",dataName);
                                dataObject.put("language","");
                                dataObject.put("source",tableId);
                                dataObject.put("title",description);
                                dataObject.put("contentType","field");
                                dataObject.put("content",dataName);
                                dataObjectMap.put("dataObject",dataObject);
                                selectList.add(dataObjectMap);
                            });
                        }
                    }
                });
                Map<String,Object> datasetObject = new HashMap<>();
                datasetObject.put("metric",tableId);
                datasetObject.put("alias",tableId);
                datasetObject.put("datasetType","dataset");

                Map<String,Object> limit = new HashMap<>();
                limit.put("startPos",0);
                limit.put("endPos",50);

                select.put("datasetObject",datasetObject);
                select.put("alias",tableId);
                select.put("limit",limit);
                select.put("select",selectList);
                solutionStep.add(select);
            }

            AthenaMessageEvent event = new AthenaMessageEvent();
            String ptxId = JaMDC.getNotNullTraceId();
            MDC.put("traceId", ptxId);
            MDC.put("PtxId", ptxId);
            Long generateSerialNo = SnowflakeIdWorker.getInstance().newId();
            event.setUser(user);
            event.setGenerateSerialNo(generateSerialNo);
            event.setPtxId(ptxId);
            event.setLang(reqDTO.getLang());
            event.setProductVersion(reqDTO.getProductVersion());
            event.setAppCode(reqDTO.getAppCode());

            QuerySchemaDatasetDTO dataset = new QuerySchemaDatasetDTO();
            SchemaMetricShowDefine showDefine = new SchemaMetricShowDefine();
            List<SchemaMetricShowDefine.ShowType> showTypes = new ArrayList<>();
            SchemaMetricShowDefine.ShowType showType = new SchemaMetricShowDefine.ShowType();
            showType.setActionId(tableId);
            showType.setType(new ArrayList<>());
            showTypes.add(showType);
            showDefine.setShowType(showTypes);
            dataset.setShowDefine(showDefine);
            dataset.setSolutionStep(solutionStep);
            // dataset
            List<Map<String,Object>> mappingFields = new ArrayList<>();
            mappingFields.add(reqDTO.getDataset());
            dataset.setDatasetList(mappingFields);
            // 数据预览标识
            dataset.setTablePreviewIf(true);
            Map<String,Object> req = CommonUtil.convertObjectToMap(AdeDatasetReqDTO.BuildDatasetAdeReq(event,dataset));

            Map<String, Object> res = adeService.reqMetricSnapShotData(req, event);
            if(Objects.isNull(res) || "-1".equals(res.get("code"))) {
                if(Objects.isNull(res)){
                    resDTO.setMsg(messageUtil.getMessageByLangName("message.adt.data.dataTable.error", reqDTO.getLang()));
                } else {
                    resDTO.setMsg(String.valueOf(res.get("msg")));
                }
                return resDTO;
            }
            List<Map<String, Object>> dataList = new ArrayList<>();
            List<Map<String, Object>> fieldList = new ArrayList<>();
            Optional.ofNullable((Map<String, Object>) res.get("data"))
                    .map(dataTable -> (Map<String, Object>) dataTable.get("context"))
                    .map(contextTable -> (List<Map<String, Object>>) contextTable.get("pullData"))
                    .ifPresent(pullDataTable -> pullDataTable.forEach(pullDataDTO -> {
                        Map<String, Object> metadata = Optional.ofNullable((Map<String, Object>) pullDataDTO.get("metadata"))
                                .orElse(Collections.emptyMap());
                        Map<String, Object> response = Optional.ofNullable((Map<String, Object>) metadata.get("response"))
                                .orElse(Collections.emptyMap());
                        Map<String, Object> responseData = Optional.ofNullable((Map<String, Object>) response.get("data"))
                                .orElse(Collections.emptyMap());
                        List<Map<String, Object>> fields = Optional.ofNullable((List<Map<String, Object>>) responseData.get("field"))
                                .orElse(Collections.emptyList());
                        Map<String, Object> dataData = Optional.ofNullable((Map<String, Object>) pullDataDTO.get("data"))
                                .orElse(Collections.emptyMap());
                        List<Map<String, Object>> dataArray = Optional.ofNullable((List<Map<String, Object>>) dataData.get("data"))
                                .orElse(Collections.emptyList());
                        dataList.addAll(Optional.of(dataArray).orElse(Collections.emptyList()));
                        fieldList.addAll(Optional.of(fields).orElse(Collections.emptyList()));
                    }));
            if (dataList.isEmpty() || fieldList.isEmpty()) {
                log.info("getTableList req : {}", JsonUtils.objectToString(res));
            }
            resDTO.setData(dataList);
            resDTO.setFields(fieldList);
        }catch (Exception e){
            log.error("{}, 请求code：{}, 报错信息：{} ", "获取数据集数据预览详情解析失败",
                    JSONObject.toJSONString(reqDTO), e.getMessage());
        }
        return resDTO;
    }

}
