package com.digiwin.athena.adt.agileReport.service.impl.log;

import com.digiwin.athena.adt.agileReport.eventbus.AgileDataLogEvent;
import com.digiwin.athena.adt.agileReport.service.AgileReportLogMongoService;
import com.digiwin.athena.adt.domain.dto.AgileDataResultLog;
import com.digiwin.athena.adt.util.DateUtils;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.MapUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.data.mongodb.core.MongoTemplate;
import org.springframework.data.mongodb.core.query.Criteria;
import org.springframework.data.mongodb.core.query.Query;
import org.springframework.data.mongodb.core.query.Update;
import org.springframework.stereotype.Service;

import java.time.LocalDateTime;
import java.util.List;
import java.util.Map;

/**
 * mongo日志存储
 */
@Slf4j
@Service
public class AgileReportLogMongoServiceImpl implements AgileReportLogMongoService {

    @Autowired
    @Qualifier("mongoAgileDataLogTemplate")
    private MongoTemplate mongoAgileDataLogTemplate;

    private static final String MONGODB_COLLECTION_NAME = "agile_data_full_log";


    @Override
    public Boolean saveLog(AgileDataLogEvent agileDataLogEvent,Map<String, Object> schemaMap) {
        Query query = Query.query(Criteria.where("generateSerialNo").is(agileDataLogEvent.getGenerateSerialNo()).and("userId").is(agileDataLogEvent.getAuthoredUser().getUserId()).and("tenantId").is(agileDataLogEvent.getAuthoredUser().getTenantId()));
        AgileDataResultLog agileDataResultLog = mongoAgileDataLogTemplate.findOne(query, AgileDataResultLog.class, MONGODB_COLLECTION_NAME);
        try {
            //新增
            if(agileDataResultLog == null){
                //组装mongo参数
                agileDataResultLog = AgileDataResultLog.builder()
                        .userId(agileDataLogEvent.getAuthoredUser().getUserId())
                        .userName(agileDataLogEvent.getAuthoredUser().getUserName())
                        .tenantId(agileDataLogEvent.getAuthoredUser().getTenantId())
                        .tenantName(agileDataLogEvent.getAuthoredUser().getTenantName())
                        .terminal(agileDataLogEvent.getTerminal())
                        .generateSerialNo(agileDataLogEvent.getGenerateSerialNo())
                        .snapshotId(agileDataLogEvent.getSnapshotId())
                        .success(agileDataLogEvent.getSuccess())
                        .sourceCode(agileDataLogEvent.getSourceCode())
                        .sourceName(agileDataLogEvent.getSourceName())
                        .question(agileDataLogEvent.getQuestion())
                        .errorAnswer(agileDataLogEvent.getErrorAnswer())
                        .answerOpinion(agileDataLogEvent.getAnswerOpinion())
                        .feedbackType(agileDataLogEvent.getFeedbackType())
                        .params(MapUtils.isNotEmpty(agileDataLogEvent.getParams()) ? agileDataLogEvent.getParams() : Maps.newHashMap())
                        .data(MapUtils.isNotEmpty(agileDataLogEvent.getData()) ? agileDataLogEvent.getData() : Maps.newHashMap())
                        .answerOpinion(agileDataLogEvent.getAnswerOpinion())
                        .querySchema(MapUtils.getMap(schemaMap,"sqlSchemaMaps",Maps.newHashMap()))
                        .backContent(agileDataLogEvent.getBackContent())
                        .checkOpinions(agileDataLogEvent.getCheckOpinions())
                        .extendInfo(MapUtils.isNotEmpty(agileDataLogEvent.getExtendInfo()) ? agileDataLogEvent.getExtendInfo() : Maps.newHashMap())
                        .createTime(agileDataLogEvent.getAskTime())
                        .answerTime(agileDataLogEvent.getAnswerTime())
                        .resSecTime(DateUtils.calculateAndSetDuration(agileDataLogEvent.getAskTime(),
                                agileDataLogEvent.getAnswerTime()))
                        .applicationCode(agileDataLogEvent.getApplicationCode())
                        .applicationName(agileDataLogEvent.getApplicationName())
                        .logSaveSuccess(agileDataLogEvent.getLogSaveSuccess())
                        .reason(agileDataLogEvent.getReason())
                        .productVersion(agileDataLogEvent.getProductVersion())
                        .build();
                mongoAgileDataLogTemplate.insert(agileDataResultLog,MONGODB_COLLECTION_NAME);
            } else {
                //更新
                mongoAgileDataLogTemplate.updateFirst(query,this.getUpdateInfo(agileDataLogEvent,agileDataResultLog),MONGODB_COLLECTION_NAME);
            }
        } catch (Exception e){
            log.error("保存问题日志失败，失败原因："+ e.getMessage());
            return false;
        }

        return true;
    }

    @Override
    public Boolean removeLog(AgileDataLogEvent agileDataLogEvent) {
        Query query = Query.query(Criteria.where("generateSerialNo").is(agileDataLogEvent.getGenerateSerialNo()).and("userId").is(agileDataLogEvent.getAuthoredUser().getUserId()).and("tenantId").is(agileDataLogEvent.getAuthoredUser().getTenantId()));
        mongoAgileDataLogTemplate.remove(query,MONGODB_COLLECTION_NAME);
        return true;
    }

    /**
     * 根据场景获取更新字段
     * @param agileDataLogEvent
     * @return
     */
    private Update getUpdateInfo(AgileDataLogEvent agileDataLogEvent,AgileDataResultLog agileDataResultLog){
        Update update = new Update();
        switch (agileDataLogEvent.getType()){
            //数据日志
            case 0:
                update.set("data",agileDataLogEvent.getData());
                break;
            //异常日志
            case 1:
                update.set("errorAnswer",agileDataLogEvent.getErrorAnswer());
                update.set("params",agileDataLogEvent.getParams());
                update.set("success",agileDataLogEvent.getSuccess());
                break;
            //用户点赞/点踩
            case 2:
                update.set("feedbackType",agileDataLogEvent.getFeedbackType());
                update.set("extendInfo",agileDataLogEvent.getExtendInfo());
                if(agileDataResultLog.getCreateBackTime() == null) {
                    update.set("createBackTime", LocalDateTime.now());
                }
                break;
            //用户反馈
            case 3:
                update.set("answerOpinion",agileDataLogEvent.getAnswerOpinion());
                update.set("backContent",agileDataLogEvent.getBackContent());
                update.set("checkOpinions",agileDataLogEvent.getCheckOpinions());
                //只有点踩后才有反馈，所以此处增加判定
                if(agileDataResultLog.getExtendInfo() == null) {
                    update.set("extendInfo", agileDataLogEvent.getExtendInfo());
                }
                if(agileDataResultLog.getCreateBackTime() == null) {
                    update.set("createBackTime", LocalDateTime.now());
                }
                break;
            //用户取消点赞/点踩
            case 4:
                update.set("feedbackType",null);
                update.set("answerOpinion",agileDataLogEvent.getAnswerOpinion());
                update.set("backContent",agileDataLogEvent.getBackContent());
                update.set("checkOpinions",agileDataLogEvent.getCheckOpinions());
                update.set("createBackTime", null);
                break;
            default:
                break;
        }
        update.set("modifyTime", LocalDateTime.now());
        return update;
    }

}
