package com.digiwin.athena.adt.agileReport.service.impl.process.schema;

import com.alibaba.fastjson.JSONObject;
import com.digiwin.athena.adt.agileReport.constant.SchemaConstants;
import com.digiwin.athena.adt.agileReport.constant.SchemaDataEnum;
import com.digiwin.athena.adt.agileReport.constant.TroubleToolCodeEnum;
import com.digiwin.athena.adt.agileReport.eventbus.AthenaMessageEvent;
import com.digiwin.athena.adt.agileReport.interfaces.SchemaDataType;
import com.digiwin.athena.adt.agileReport.service.SchemaDataProcessService;
import com.digiwin.athena.adt.domain.dto.echo.EchoSubmitReq;
import com.digiwin.athena.adt.domain.dto.schema.QuerySchemaDatasetDTO;
import com.digiwin.athena.adt.domain.dto.schema.QuerySchemaMetricDTO;
import com.digiwin.athena.adt.domain.dto.schema.QuerySchemaResDTO;
import com.digiwin.athena.adt.domain.echo.EchoService;
import com.digiwin.athena.adt.domain.semc.SemcService;
import com.digiwin.athena.adt.util.LogUtils;
import com.digiwin.athena.appcore.util.JsonUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.*;

/**
 * @Author: SunHong
 * @Date: 2024/7/1 14:46
 * @Description: 指标/数据集 数据解析
 */
@Service
@Slf4j
@SchemaDataType(value = SchemaDataEnum.METRIC)
public class SchemaDataProcessMetricServiceImpl extends AbsSchemaDataProcess implements SchemaDataProcessService {

    @Autowired
    private SemcService semcService;

    @Autowired
    private EchoService echoService;

    /**
     * 目前语义设计器和指标 新老架构走统一接口
     *
     * @param event 上下文
     * @return 暂未使用到策略-可开启根据租户路由
     */
    @Override
    public Pair<Boolean, QuerySchemaResDTO> process(AthenaMessageEvent event) {
        return Pair.of(false, null);
    }

    /**
     * 同 走 designer 方法
     *
     * @param event 上下文
     * @return new
     */
    @Override
    public Map<String, Object> getQuerySchemaReq(AthenaMessageEvent event) {
        return new HashMap<>();
    }

    /**
     * 新架构指标语义识别返回数据校验
     * 多轮导正判断 这里语义返回的
     * code ： 0 正常解析sql 1 reply导正 2 多应用导正 3 多指标导正
     *
     * @param querySchemaResDTO 语义返回
     * @param event             消息体
     * @return 是否返回
     */
    public boolean checkMetricResponseData(QuerySchemaResDTO querySchemaResDTO, AthenaMessageEvent event) {
        QuerySchemaMetricDTO metric = querySchemaResDTO.getData().getMetric();
        boolean isCheck = false;
        try {
            Object code = metric.getCode();
            if (Objects.isNull(code)) {
                return false;
            }
            // 组合问句
            event.setCombinationQuestion(metric.getCombinationQuestion());
            // 多产品线上下文
            event.setProductLineInfo(metric.getProductLineInfo());
            // 猜你想问/相关推荐
            event.setAnswerResult(metric.getSentenceType());
            event.setSentences(metric.getSentences());
            switch (code.toString()) {
                // 成功执行sql
                case SchemaConstants.METRIC_SQL:
                    // metric
                    List<Map<String, Object>> metricList = metric.getMetricList();
                    if (CollectionUtils.isNotEmpty(metricList)) {
                        event.setMetricList(metricList);
                    }
                    //应用信息
                    this.setAthenaEventAppCodeAndName(event, metric.getApplicationList());
                    this.setAthenaEventMessageType(event, SchemaDataEnum.METRIC.getValue());
                    isCheck = true;
                    break;
                // 异常引导语句
                case SchemaConstants.SCHEMA_REPLY:
                    this.sendMessageAndSaveLogReply(event, metric);
                    break;
                case SchemaConstants.METRIC_APP:
                    try {
                        // 上下文消息类型
                        this.setAthenaEventMessageType(event, SchemaDataEnum.METRIC_APP.getValue());
                        // 多应用导正echo
                        echoService.echoMongodbSubmit(EchoSubmitReq.builderSchemaGuidance(event, metric.getApplicationList())
                                , event.getUser().getToken(), event.getUser().getTenantId());
                        this.sendMessageAndSaveLogReply(event, metric);
                        LogUtils.buildAgileLog(LogUtils.MODULE_ADT, "moreApplication", LogUtils.SUCCESS,
                                JsonUtils.objectToString(metric.getApplicationList()), "", "");
                    } catch (Exception e) {
                        LogUtils.buildAgileLog(LogUtils.MODULE_ADT, "moreApplication", TroubleToolCodeEnum.ADT_901_0114.getErrCode(),
                                JsonUtils.objectToString(metric.getApplicationList()),
                                TroubleToolCodeEnum.ADT_901_0114.getErrMsg(), TroubleToolCodeEnum.ADT_901_0114.getSuggestion());
                    }
                    break;
                case SchemaConstants.METRIC_TARGET:
                    // 暂无多指标导正
                    this.setAthenaEventMessageType(event, SchemaDataEnum.METRIC.getValue());
                    break;
                default:
                    break;
            }
        } catch (Exception e) {
            LogUtils.buildAgileLog(LogUtils.MODULE_ADT, "metricAnalysis", TroubleToolCodeEnum.ADT_901_0112.getErrCode(),
                    JsonUtils.objectToString(querySchemaResDTO),
                    TroubleToolCodeEnum.ADT_901_0112.getErrMsg(),
                    TroubleToolCodeEnum.ADT_901_0112.getSuggestion());
        }
        return isCheck;
    }

    /**
     * 成功后解析 metric的applicationList
     *
     * @param event           上下文
     * @param applicationList 应用信息
     */
    public void setAthenaEventAppCodeAndName(AthenaMessageEvent event, List<Map<String, Object>> applicationList) {
        if (CollectionUtils.isNotEmpty(applicationList)) {
            Object appCode = applicationList.get(0).get("appCode");
            Object appName = applicationList.get(0).get("appName");
            // 上下文应用code
            event.setAppCode(Objects.isNull(appCode) ? "" : String.valueOf(appCode));
            // 应用名称
            event.setAppName(Objects.isNull(appName) ? "" : String.valueOf(appName));
        }
    }

    /**
     * metric
     * 语义非异常上报导正消息和记录日志
     *
     * @param event      上下文
     * @param schemaData 语义返回
     */
    public void sendMessageAndSaveLogReply(AthenaMessageEvent event,
                                           QuerySchemaMetricDTO schemaData) {
        String reply = schemaData.getReply();
        Map<String, Object> msgBody = new HashMap<>();
        msgBody.put("prompt", reply);
        msgBody.put("scrumbiQuestion", event.getQuestion());
        if (event.isSendNana()) {
            semcService.sendMessageToGpt(event, msgBody);
            this.saveAbnormalLog(event, reply, 1, 0);
        }
    }

    @Override
    public Pair<Boolean, QuerySchemaResDTO> sseProcess(AthenaMessageEvent event, JSONObject jsonObject) {
        return Pair.of(false, null);
    }

    /**
     * dataset 语义返回校验
     * @param querySchemaResDTO 语义返回实体
     * @param event 上下文
     * @return f/t
     */
    public boolean checkDatasetResponseData(QuerySchemaResDTO querySchemaResDTO, AthenaMessageEvent event) {

        QuerySchemaDatasetDTO dataset = querySchemaResDTO.getData().getDataset();
        boolean isCheck = false;
        try {
            Object code = dataset.getCode();
            if (Objects.isNull(code)) {
                return false;
            }
            // 组合问句
            event.setCombinationQuestion(dataset.getCombinationQuestion());
            // 多产品线上下文
            event.setProductLineInfo(dataset.getProductLineInfo());
            // 猜你想问/相关推荐
            event.setAnswerResult(dataset.getSentenceType());
            event.setSentences(dataset.getSentences());
            switch (code.toString()) {
                // 成功执行sql
                case SchemaConstants.METRIC_SQL:
                    // metric
                    List<Map<String, Object>> datasetList = dataset.getDatasetList();
                    if (CollectionUtils.isNotEmpty(datasetList)) {
                        event.setDatasetList(datasetList);
                    }
                    //应用信息
                    this.setAthenaEventAppCodeAndName(event, dataset.getApplicationList());
                    this.setAthenaEventMessageType(event, SchemaDataEnum.DATASET.getValue());
                    isCheck = true;
                    break;
                // 异常引导语句
                case SchemaConstants.SCHEMA_REPLY:
                    this.sendMessageAndSaveLogReplyByDataset(event, dataset);
                    break;
                case SchemaConstants.METRIC_APP:
                    try {
                        // 上下文消息类型
                        this.setAthenaEventMessageType(event, SchemaDataEnum.METRIC_APP.getValue());
                        // 多应用导正echo
                        echoService.echoMongodbSubmit(EchoSubmitReq.builderSchemaGuidance(event, dataset.getApplicationList())
                                , event.getUser().getToken(), event.getUser().getTenantId());
                        this.sendMessageAndSaveLogReplyByDataset(event, dataset);
                        LogUtils.buildAgileLog(LogUtils.MODULE_ADT, "moreApplication", LogUtils.SUCCESS,
                                JsonUtils.objectToString(dataset.getApplicationList()), "", "");
                    } catch (Exception e) {
                        LogUtils.buildAgileLog(LogUtils.MODULE_ADT, "moreApplication", TroubleToolCodeEnum.ADT_901_0114.getErrCode(),
                                JsonUtils.objectToString(dataset.getApplicationList()),
                                TroubleToolCodeEnum.ADT_901_0114.getErrMsg(), TroubleToolCodeEnum.ADT_901_0114.getSuggestion());
                    }
                    break;
                case SchemaConstants.METRIC_TARGET:
                    // 暂无多指标导正
                    this.setAthenaEventMessageType(event, SchemaDataEnum.DATASET.getValue());
                    break;
                default:
                    break;
            }
        } catch (Exception e) {
            LogUtils.buildAgileLog(LogUtils.MODULE_ADT, "datasetAnalysis", TroubleToolCodeEnum.ADT_901_0129.getErrCode(),
                    JsonUtils.objectToString(querySchemaResDTO),
                    TroubleToolCodeEnum.ADT_901_0129.getErrMsg(),
                    TroubleToolCodeEnum.ADT_901_0129.getSuggestion());
        }
        return isCheck;
    }


    /**
     * dataset
     * 语义非异常上报导正消息和记录日志
     *
     * @param event      上下文
     * @param dataset 语义返回
     */
    public void sendMessageAndSaveLogReplyByDataset(AthenaMessageEvent event,
                                                    QuerySchemaDatasetDTO dataset) {
        String reply = dataset.getReply();
        Map<String, Object> msgBody = new HashMap<>();
        msgBody.put("prompt", reply);
        msgBody.put("scrumbiQuestion", event.getQuestion());
        if (event.isSendNana()) {
            semcService.sendMessageToGpt(event, msgBody);
            this.saveAbnormalLog(event, reply, 1, 0);
        }
    }
}
