package com.digiwin.athena.adt.agileReport.service.impl.snap;

import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.digiwin.athena.adt.agileReport.constant.ErrorCodeEnum;
import com.digiwin.athena.adt.agileReport.controller.dto.*;
import com.digiwin.athena.adt.agileReport.dao.SnapShotDataMapper;
import com.digiwin.athena.adt.agileReport.service.AgileApcDataService;
import com.digiwin.athena.adt.agileReport.service.SnapShotDataService;
import com.digiwin.athena.adt.domain.ade.ADEService;
import com.digiwin.athena.adt.domain.dmc.DmcService;
import com.digiwin.athena.adt.domain.dto.agileReport.*;
import com.digiwin.athena.adt.domain.dto.SDScencDTO;
import com.digiwin.athena.adt.domain.dto.SnapDataInfoDTO;
import com.digiwin.athena.adt.domain.dto.apc.AgileApcDataEocMap;
import com.digiwin.athena.adt.domain.dto.schema.SchemaMetricShowDefine;
import com.digiwin.athena.adt.domain.po.SnapData;
import com.digiwin.athena.appcore.auth.domain.AuthoredUser;
import com.digiwin.athena.appcore.exception.BusinessException;
import com.digiwin.athena.appcore.util.JsonUtils;
import com.digiwin.athena.adt.util.MessageUtil;
import com.digiwin.athena.atmc.http.restful.eoc.EocService;
import com.digiwin.athena.atmc.http.restful.eoc.model.EocDeptEmployeeDTO;
import com.digiwin.athena.atmc.http.restful.eoc.model.EocDirectEmployeeDTO;
import com.digiwin.athena.atmc.http.restful.eoc.model.EocDutyEmployeeDTO;
import com.digiwin.athena.atmc.http.restful.eoc.model.EocEmployeeDTO;
import com.digiwin.athena.atmc.http.restful.thememap.model.TmActionParameterDTO;
import com.digiwin.athena.atmc.http.restful.thememap.model.TmUserResponseDto;
import com.github.pagehelper.PageInfo;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections4.MapUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.redis.core.StringRedisTemplate;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.time.LocalDateTime;
import java.util.*;
import java.util.stream.Collectors;

@Slf4j
@Service
public class SnapShotDataServiceImpl implements SnapShotDataService {

    @Autowired
    private SnapShotDataMapper snapShotDataMapper;
    @Resource
    private ADEService adeService;

    @Autowired
    private EocService eocService;

    @Autowired
    private StringRedisTemplate stringRedisTemplate;

    @Autowired
    private AgileApcDataService agileApcDataService;

    @Autowired
    private MessageUtil messageUtil;

    @Autowired
    private DmcService dmcService;

    @Autowired
    private SnapShotDataService snapShotDataService;

    private static final String TENANT_ID = "tenantId";
    private static final String PERFORMER_ID = "performerId";
    private static final String SNAPSHOT_ID = "snapshotId";
    private static final String CODE = "code";

    /**
     * 存储快照信息
     * @param snapShotDTO
     */
    @Override
    public void saveSnapShotData(SnapShotDTO snapShotDTO, String uploadFileId, List<TmUserResponseDto> userResponses) {
        Objects.requireNonNull(snapShotDTO, "传入的实体类为空！");

        List<SnapData> snapDataList = new ArrayList<>();
        String snapShotId = snapShotDTO.getSnapshotId();
        String detectionId = snapShotDTO.getContext().getBizParams().getMonitor().getMonitorRuleId();
        String tenantId = snapShotDTO.getContext().getBizParams().getTenantId();
        String code = snapShotDTO.getContext().getBizParams().getSceneCode();
        if(StringUtils.isEmpty(code)){
            code = snapShotDTO.getContext().getScene().getCode();
        }
        String title = "";
        //存在场景code，从图谱获取
//        if(StringUtils.isNotEmpty(detectionId) && StringUtils.isNotEmpty(code)) {

//            KMScencDTO kmScencDTO = kmService.getKmSceneInfo(event,snapShotDTO.getContext().getScene().getCode(),null,null,tenantId);
//            title = kmScencDTO.getSceneTitle();
//        }
        //不存在从快照获取
        if(MapUtils.isNotEmpty(snapShotDTO.getContext().getBizParams().getQuerySchema())){
            title = MapUtils.getString(snapShotDTO.getContext().getBizParams().getQuerySchema(),"sceneTitle");
        }
        Map<String, Object> map = snapShotDTO.getContext().getBizParams().getParam();
        log.info("snapShotDTO.getContext().getBizParams().getParam() : {}",JsonUtils.objectToString(map));
        int defaultValue = StringUtils.equals("instant", snapShotDTO.getContext().getBizParams().getRequestType()) ? 1 : 0;
        // 数据订阅类型
        int entry = MapUtils.getIntValue(map, "type", defaultValue);
        // 消息类型
        String messageType = MapUtils.getString(map,"messageType");
        LocalDateTime requestTime = snapShotDTO.getContext().getBizParams().getRequestTime();

        for (TmUserResponseDto userResponse : userResponses) {
            SnapData snapData = new SnapData();
            snapData.setSnapshotId(snapShotId);
            snapData.setPerformerId(userResponse.getUserId());
            snapData.setTenantId(tenantId);
            snapData.setUserName(userResponse.getUserName());
            snapData.setCode(code);
            snapData.setDetectionId(detectionId);
            snapData.setRequestTime(requestTime);
            snapData.setDmcId(uploadFileId);
            snapData.setEntry(entry);
            snapData.setMessageType(messageType);
            snapData.setTitle(title);
            snapData.setReadCount(0);
            snapData.setStatus(0);
            snapData.setCreateDate(LocalDateTime.now());
            if(MapUtils.isNotEmpty(snapShotDTO.getContext().getBizParams().getQuerySchema())){
                snapData.setQueryType("1");
            } else {
                snapData.setQueryType("0");
            }
            snapDataList.add(snapData);
        }
        // 调用批量插入方法，将 snapDataList 中的数据批量插入数据库
        if (!snapDataList.isEmpty()) {
            snapShotDataMapper.insertBatchSomeColumn(snapDataList);
        }
    }

    /**
     * 获取人员信息
     * @param snapShotDTO
     */
    @Override
    public List<TmUserResponseDto> getUserResponse(SnapShotDTO snapShotDTO) {
        List<TmUserResponseDto> retList = new ArrayList<>();
        List<SceneUserCommonDTO> whoList = snapShotDTO.getContext().getScene().getWho();

        if (whoList == null || whoList.isEmpty()) {
            throw ErrorCodeEnum.WHO_IS_EMPTY.getBusinessException();
        }

        for (SceneUserCommonDTO sceneCommonDTO : whoList) {
            String type = sceneCommonDTO.getType();
            List<String> valueList = sceneCommonDTO.getValue().stream()
                    .map(userMap -> MapUtils.getString(userMap, "id"))
                    .collect(Collectors.toList());

            processValueByType(type, valueList, retList);
        }
        return retList;
    }

    private void processValueByType(String type, List<String> valueList, List<TmUserResponseDto> retList) {
        for (String value : valueList) {
            switch (type) {
                case "user":
                    EocEmployeeDTO userDto = eocService.getEmployeeByUserId(value);
                    if (userDto != null) {
                        retList.addAll(convertFromEmployee(userDto));
                    }
                    break;
                case "deptDirector":
                    EocDirectEmployeeDTO directorDto = eocService.getDirectorEmployee(value);
                    if (directorDto != null) {
                        retList.addAll(convertFromDirector(directorDto));
                    }
                    break;
                case "duty":
                    List<EocDutyEmployeeDTO> dutyDtoList = eocService.getDutyEmployees(value, null, null);
                    if (dutyDtoList != null) {
                        retList.addAll(convertFromDuty(dutyDtoList));
                    }
                    break;
                case "deptUser":
                    List<EocDeptEmployeeDTO> deptDtoList = eocService.getDepartEmployees(value);
                    if (deptDtoList != null) {
                        retList.addAll(convertFromDept(deptDtoList));
                    }
                    break;
                default:
                    break;
            }
        }
    }


    @Override
    public List<SnapData> getSnapShotDataList(String tenantId, String userId, PageReqDTO pageReqDTO) {
        List<SnapData> dataList;
        QueryWrapper<SnapData> queryWrapper = new QueryWrapper<>();
        Page<SnapData> page = new Page<>();
        queryWrapper.eq(TENANT_ID, tenantId).eq(PERFORMER_ID, userId).eq("entry","0").orderByDesc("requestTime");

        if (pageReqDTO.getIsRead() != null) {
            if (pageReqDTO.getIsRead()) {
                // 如果 isRead 为 true，则查询 readCount 不为 0 的数据
                queryWrapper.ne("readCount", 0);
            } else {
                // 如果 isRead 为 false，则查询 readCount 为 0 的数据
                queryWrapper.eq("readCount", 0);
            }
        }

        // 如果 currentPage 和 pageSize 都不为 null，则进行分页查询
        if (pageReqDTO.getCurrentPage() != null && pageReqDTO.getPageSize() != null) {
            page.setCurrent(pageReqDTO.getCurrentPage());
            page.setSize(pageReqDTO.getPageSize());
            page =  snapShotDataMapper.selectPage(page,queryWrapper);
            dataList = page.getRecords();
        } else {
            // 没有传入分页参数，则执行不分页查询
            dataList = snapShotDataMapper.selectList(queryWrapper);
        }
        return dataList;
    }

    /**
     * 获取快照数据
     * @param tenantId
     * @param userId
     * @param snapshotId
     * @return
     */
    @Override
    public SnapData getSnapDataBySnapshotId(String tenantId, String userId, String snapshotId) {
        QueryWrapper<SnapData> queryWrapper = new QueryWrapper<>();
        queryWrapper.eq(TENANT_ID, tenantId).eq(PERFORMER_ID, userId).eq(SNAPSHOT_ID,snapshotId);
        return snapShotDataMapper.selectOne(queryWrapper);
    }

    /**
     * 获取敏捷报表信息
     * @param tenantId
     * @param userId
     * @param snapshotId
     * @return
     */
    @Override
    public SnapDataInfoDTO getSnapDataInfoBySnapshotId(String tenantId, String userId, String snapshotId) {
        SnapDataInfoDTO snapDataInfoDTO = new SnapDataInfoDTO();
        QueryWrapper<SnapData> queryWrapper = new QueryWrapper<>();
        queryWrapper.eq(TENANT_ID, tenantId).eq(PERFORMER_ID, userId).eq(SNAPSHOT_ID,snapshotId);
        SnapData snapData = snapShotDataMapper.selectOne(queryWrapper);
        if(snapData == null){
            return snapDataInfoDTO;
        }
        return this.getSnapDataInfo(snapData);
    }


    /**
     * 敏捷数据-大屏快照
     * @param tenantId
     * @param userId
     * @param sceneCode
     * @return
     */
    @Override
    public SnapDataInfoDTO getScreenSnapData(String tenantId, String userId, String sceneCode) {
        SnapDataInfoDTO snapDataInfoDTO = new SnapDataInfoDTO();
        QueryWrapper<SnapData> queryWrapper = new QueryWrapper<>();
        queryWrapper.eq(TENANT_ID, tenantId).eq(PERFORMER_ID, userId).eq(CODE,sceneCode);
        queryWrapper.orderByDesc("requestTime");
        queryWrapper.last("LIMIT 1");
        SnapData snapData = snapShotDataMapper.selectOne(queryWrapper);
        if(snapData == null){
            return snapDataInfoDTO;
        }
        snapDataInfoDTO = this.getSnapDataInfo(snapData);
        //默认走场景链路
        if(StringUtils.isEmpty(snapDataInfoDTO.getSnapData().getQueryType())) {
            snapDataInfoDTO.getSnapData().setQueryType("0");
        }
        return snapDataInfoDTO;
    }

    /**
     * 实时获取大屏数据
     * @param user
     * @param agileDataMoreActionDTO
     * @param local
     * @return
     */
    @Override
    public SnapDataInfoDTO getScreenSnapSyncData(AuthoredUser user, AgileDataMoreActionDTO agileDataMoreActionDTO,String local) {
        SnapDataInfoDTO snapDataInfoDTO = new SnapDataInfoDTO();
        AthenaApcMessageReqDTO athenaApcMessageReqDTO = new AthenaApcMessageReqDTO();
        athenaApcMessageReqDTO.setSceneCode(agileDataMoreActionDTO.getSceneCode());

        AgileApcDataEocMap eocMap = new AgileApcDataEocMap();
        eocMap.setEocCompanyId(MapUtils.getString(agileDataMoreActionDTO.getEocMap(),"eocCompanyId"));
        eocMap.setEocSiteId(MapUtils.getString(agileDataMoreActionDTO.getEocMap(),"eocSiteId"));
        athenaApcMessageReqDTO.setEocMaps(Lists.newArrayList(eocMap));
        athenaApcMessageReqDTO.setRequestType(agileDataMoreActionDTO.getRequestType());
        athenaApcMessageReqDTO.setFixedParams(agileDataMoreActionDTO.getFixedParams());

        AthenaApcMessageResDTO data = (AthenaApcMessageResDTO) agileApcDataService.processMessage(null,athenaApcMessageReqDTO,user,local);

        if(data == null || data.isError()){
            log.error("获取大屏信息失败");
            return snapDataInfoDTO;
        }
        SnapDataDTO snapDataDTO = new SnapDataDTO();
        snapDataDTO.setQueryType("0");
        snapDataDTO.setCode(data.getSnapShotDTO().getContext().getBizParams().getSceneCode());
        snapDataDTO.setPerformerId(user.getUserId());
        snapDataDTO.setTenantId(user.getTenantId());
        snapDataInfoDTO.setSnapData(snapDataDTO);
        //获取快照详情
        this.getSnapShotDetail(snapDataInfoDTO,data.getSnapShotDTO());
        //默认走场景链路
        if(StringUtils.isEmpty(snapDataInfoDTO.getSnapData().getQueryType())) {
            snapDataInfoDTO.getSnapData().setQueryType("0");
        }
        return snapDataInfoDTO;
    }

    /**
     * 构建前端所需快照结构
     * @param snapData
     * @return
     */
    private SnapDataInfoDTO getSnapDataInfo(SnapData snapData) {
        SnapDataInfoDTO snapDataInfoDTO = new SnapDataInfoDTO();
        if(snapData == null){
            return snapDataInfoDTO;
        }
        SnapDataDTO snapDataDTO = new SnapDataDTO();
        BeanUtils.copyProperties(snapData,snapDataDTO);
        snapDataInfoDTO.setSnapData(snapDataDTO);
        //查到dmcId调用接口返回快照信息
        SnapShotDTO snapShotDTO = dmcService.getAgileData(snapData.getDmcId());
        if(snapShotDTO == null){
            log.error("获取DMC文件失败");
            return snapDataInfoDTO;
        }
        //获取快照详情
        this.getSnapShotDetail(snapDataInfoDTO,snapShotDTO);
        return snapDataInfoDTO;
    }

    /**
     * 获取快照详情
     * @param snapDataInfoDTO
     * @param snapShotDTO
     */
    private void getSnapShotDetail(SnapDataInfoDTO snapDataInfoDTO,SnapShotDTO snapShotDTO) {
        Map<String, List<PullDataDTO>> listMap = new HashMap<>();
        //存在资产中心查询条件，需返回该条件信息
        if(MapUtils.isNotEmpty(snapShotDTO.getContext().getBizParams().getQuerySchema())){
            snapDataInfoDTO.setQuerySchema(snapShotDTO.getContext().getBizParams().getQuerySchema());
            //按modelId分组
            for (PullDataDTO pullDataDTO : snapShotDTO.getContext().getPullData()) {
                listMap.computeIfAbsent(pullDataDTO.getModelId(), k -> new ArrayList<>()).add(pullDataDTO);
            }
        } else {
            //存在图谱查询条件，需返回该条件信息
            snapDataInfoDTO.setBackRecast(snapShotDTO.getContext().getBizParams().getBackRecast());
            //按actionID分组
            for (PullDataDTO pullDataDTO : snapShotDTO.getContext().getPullData()) {
                listMap.computeIfAbsent(pullDataDTO.getActionId(), k -> new ArrayList<>()).add(pullDataDTO);
            }
        }
        //设计方向
        snapDataInfoDTO.setShowDefine(snapShotDTO.getContext().getBizParams().getShowDefine());
        snapDataInfoDTO.setLink(getLink(snapShotDTO));
        snapDataInfoDTO.setPageData(listMap);
        snapDataInfoDTO.getSnapData().setDimensionInterval(snapShotDTO.getContext().getBizParams().getDimensionInterval());
        snapDataInfoDTO.getSnapData().setMessageId(snapShotDTO.getContext().getBizParams().getMessageId());
        snapDataInfoDTO.getSnapData().setDimensionCnt(snapShotDTO.getContext().getBizParams().getDimensionCnt());
        //粒度
        if(MapUtils.isNotEmpty(snapShotDTO.getContext().getBizParams().getShowDefine())){
            if(snapShotDTO.getContext().getBizParams().getShowDefine().containsKey("aliasGrain")){
                String grainJsonStr = JsonUtils.objectToString(snapShotDTO.getContext().getBizParams().getShowDefine().get("aliasGrain"));
                snapDataInfoDTO.getSnapData().setAliasGrain(JsonUtils.jsonToListObject(grainJsonStr, SchemaMetricShowDefine.AliasGrain.class));
            }
        }
    }

    private String getLink(SnapShotDTO snapShotDTO){
        //lite版本
        String requestType = snapShotDTO.getContext().getBizParams().getRequestType();
        if(StringUtils.isNotEmpty(requestType) && StringUtils.equals("metricInstant",requestType)){
            return "2";
        }
        //指标架构
        String method = snapShotDTO.getContext().getBizParams().getMethod();
        if(StringUtils.isNotEmpty(method) && StringUtils.equals("metric",method)){
            return "2";
        }
        if(StringUtils.isNotEmpty(method) && StringUtils.equals("dataset",method)){
            return "2";
        }
        return "1";
    }

    /**
     * 获取快照数据
     * @param agileReportDetailParamsDTO
     * @param user
     * @return
     */
    @Override
    public List<Map<String,Object>> getSnapShotDetail(AgileReportDetailParamsDTO agileReportDetailParamsDTO, AuthoredUser user) {
        String dmcId = this.getDmcId(agileReportDetailParamsDTO.getReportId(), user.getTenantId(), user.getUserId());
        if(StringUtils.isEmpty(dmcId)){
            return Lists.newArrayList();
        }
        //查到dmcId调用接口返回快照信息
        SnapShotDTO snapShotDTO = dmcService.getAgileData(dmcId);
        if(snapShotDTO == null){
            return Lists.newArrayList();
        }
        return this.getAgileData(snapShotDTO,agileReportDetailParamsDTO);
    }

    /**
     * 获取分页快照数据
     * @param agileReportDetailParamsDTO
     * @param user
     * @return
     */
    @Override
    public Map<String, Object> getSnapPageShotDetail(AgileReportDetailParamsDTO agileReportDetailParamsDTO, AuthoredUser user) {
        String dmcId = this.getDmcId(agileReportDetailParamsDTO.getSnapshotId(), user.getTenantId(), user.getUserId());
        if(StringUtils.isEmpty(dmcId)){
            return Maps.newHashMap();
        }
        //查到dmcId调用接口返回快照信息
        SnapShotDTO snapShotDTO = dmcService.getAgileData(dmcId);
        if(snapShotDTO == null){
            return Maps.newHashMap();
        }
        return this.getPageAgileData(snapShotDTO,agileReportDetailParamsDTO);
    }

    /**
     * 获取敏捷数据
     * @param snapShotDTO
     * @param agileReportDetailParamsDTO
     * @return
     */
    private List<Map<String,Object>> getAgileData(SnapShotDTO snapShotDTO,AgileReportDetailParamsDTO agileReportDetailParamsDTO){
        List<Map<String,Object>> dataList = Lists.newArrayList();
        //按actionID分组
        Map<String, List<PullDataDTO>> listMap = new HashMap<>();
        for (PullDataDTO pullDataDTO : snapShotDTO.getContext().getPullData()) {
            listMap.computeIfAbsent(pullDataDTO.getActionId(), k -> new ArrayList<>()).add(pullDataDTO);
        }
        //获取actionID下的业务数据
        for(String key : listMap.keySet()){
            Map<String,Object> map = Maps.newHashMap();
            List<PullDataDTO> lists = listMap.get(key);
            List<Object> toList = Lists.newArrayList();
            //获取业务数据
            lists.stream().forEach(data -> {
                List<Object> templist = (List<Object>) MapUtils.getObject(data.getData(),"data");
                templist.stream().
                    forEach(monthData -> {
                        if(this.containsData(agileReportDetailParamsDTO,monthData)) {
                            toList.add(monthData);
                        }
                    });
                map.put(key,toList);
                dataList.add(map);
            });
        }
        return dataList;
    }

    /**
     * 获取敏捷数据进行分页展示
     * @param snapShotDTO
     * @param agileReportDetailParamsDTO
     * @return
     */
    private Map<String,Object> getPageAgileData(SnapShotDTO snapShotDTO,AgileReportDetailParamsDTO agileReportDetailParamsDTO){
        Map<String,Object> pageResult = Maps.newHashMap();
        List<Map<String,Object>> dataList = Lists.newArrayList();
        //按actionID分组
        Map<String, List<PullDataDTO>> listMap = new HashMap<>();
        for (PullDataDTO pullDataDTO : snapShotDTO.getContext().getPullData()) {
            listMap.computeIfAbsent(pullDataDTO.getActionId(), k -> new ArrayList<>()).add(pullDataDTO);
        }
        //获取actionID下的业务数据
        Map<String,Object> map = Maps.newHashMap();
        List<PullDataDTO> lists = listMap.get(agileReportDetailParamsDTO.getActionId());
        if(CollectionUtils.isNotEmpty(lists)) {
            List<Object> toList = Lists.newArrayList();
            //获取业务数据
            lists.stream().forEach(data -> {
                List<Object> templist = (List<Object>) MapUtils.getObject(data.getData(),"data");
                templist.stream().
                        forEach(monthData -> {
                            if(this.containsData(agileReportDetailParamsDTO,monthData)) {
                                toList.add(monthData);
                            }
                        });
            });
            pageResult.put(agileReportDetailParamsDTO.getActionId(), toList);
            if (agileReportDetailParamsDTO.getPageSize() != null &&
                    agileReportDetailParamsDTO.getPageNo() != null) {
                PageInfo pageInfo = this.listToPage(toList, agileReportDetailParamsDTO.getPageNo(), agileReportDetailParamsDTO.getPageSize());
                map.put(agileReportDetailParamsDTO.getActionId(), pageInfo.getList());
                dataList.add(map);
                pageResult.put(agileReportDetailParamsDTO.getActionId(),dataList);
                pageResult.put("has_next",pageInfo.isHasNextPage());
                pageResult.put("total_results",pageInfo.getTotal());
            }
        }
        return pageResult;
    }

    /**
     * 分页
     * @param recordList
     * @param page
     * @param pagesize
     */
    public PageInfo listToPage(List<Object> recordList, int page, int pagesize) {
        int total = recordList.size();
        long pageNum = page - 1L;
        long pageSize = pagesize;
        List<Object> pageList = recordList.stream().skip(pageNum * pageSize).limit(pagesize).collect(Collectors.toList());
        PageInfo<Object> pageInfo = new PageInfo<>(pageList);
        pageInfo.setPageNum(page);
        pageInfo.setPageSize(pagesize);
        pageInfo.setList(pageList);
        pageInfo.setPages((total + pagesize -1) / pagesize);
        pageInfo.setTotal(total);

        return pageInfo;
    }

    /**
     * 场景敏捷数据
     * @param sdScencDTO
     * @param user
     * @return
     */
    @Override
    public List<Map<String, Object>> getScencDetail(SDScencDTO sdScencDTO, AuthoredUser user) {
        SnapShotDTO snapShotDTO = adeService.getSnapShotData(sdScencDTO);
        return this.getAgileData(snapShotDTO,new AgileReportDetailParamsDTO());
    }

    /**
     * 是否满足入参要求
     * @return
     */
    private boolean containsData(AgileReportDetailParamsDTO agileReportDetailParamsDTO,Object responseDataDTO){
        boolean checkResult = true;
        if(CollectionUtils.isEmpty(agileReportDetailParamsDTO.getParams())){
            return checkResult;
        }
        Map<String,Object> pullMap = JsonUtils.jsonToObject(JsonUtils.objectToString(responseDataDTO),Map.class);
        for(TmActionParameterDTO tmActionParameterDTO : agileReportDetailParamsDTO.getParams()){
            //不存在参数，无需校验
            if(pullMap.containsKey(tmActionParameterDTO.getName())){
                continue;
            }
            //参数存在且匹配失败
            if(!Objects.equals(tmActionParameterDTO.getValue(),pullMap.get(tmActionParameterDTO.getName()))){
                checkResult = false;
            }
        }
        return checkResult;
    }


    @Override
    public List<SnapData> getSnapShotUserList(String snapshotId) {
        QueryWrapper<SnapData> queryWrapper = new QueryWrapper<>();
        queryWrapper.eq(SNAPSHOT_ID, snapshotId);
        return snapShotDataMapper.selectList(queryWrapper);
    }

    /**
     * 从文档中心获取快照数据
     * @param tenantId
     * @param userId
     * @return
     */
    public SnapShotDTO getSnapShotData(String snapshotId,String tenantId, String userId) {
        String dmcId = this.getDmcId(snapshotId, tenantId, userId);
        if(StringUtils.isEmpty(dmcId)) {
            throw ErrorCodeEnum.DMCID_CAN_NOT_FIND.getBusinessExceptionWithArgs(snapshotId,tenantId,userId);
        }
        return dmcService.getAgileData(dmcId);
    }

    /**
     * 从agile_data表中，按tenantId和userId获取dmcId
     * @param tenantId
     * @param userId
     * @return dmcId
     */
    @Override
    public String getDmcId(String snapshotId, String tenantId,String userId) {
        return snapShotDataMapper.getDmcId(snapshotId, tenantId, userId);
    }

    /**
     * 更新读取次数，调用一次+1
     * @param snapshotId
     * @param tenantId
     * @param userId
     */
    @Override
    public void updateReadCount(String snapshotId, String tenantId, String userId) {
        snapShotDataMapper.updateReadCount(snapshotId ,tenantId, userId);
    }

    /**
     * 删除文档中心和数据库的快照信息
     * @param snapshotId
     * @param tenantId
     * @param userId
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteSnap(String snapshotId, String tenantId, String userId) {
        try {
            SnapData snapData = this.getSnapDataBySnapshotId(tenantId,userId,snapshotId);
            if (null == snapData){throw ErrorCodeEnum.DELETE_SNAP_IS_EMPTY.getBusinessException();}
            QueryWrapper<SnapData> queryWrapper = new QueryWrapper<>();
            queryWrapper.eq(SNAPSHOT_ID, snapshotId).eq(TENANT_ID, tenantId).eq(PERFORMER_ID, userId);
            snapShotDataMapper.delete(queryWrapper);
            dmcService.deleteAgileData(snapData.getDmcId());
        } catch (Exception e) {
            throw BusinessException.create("删除过程中出现异常： " + e.getMessage());
        }
    }

    /**
     *  更新数据库快照状态
     * @param snapshotId
     * @param tenantId
     * @param userId
     */
    @Override
    public Integer updateSnapsStatue(String snapshotId, String tenantId, String userId) {
        return snapShotDataMapper.updateSnapsStatue(snapshotId,tenantId,userId);
    }

    /**
     * 转化员工信息
     *
     * @param employeeDTO employeeDTO
     * @return 结果
     */
    private List<TmUserResponseDto> convertFromEmployee(EocEmployeeDTO employeeDTO) {
        List<TmUserResponseDto> retList = new ArrayList<>();
        TmUserResponseDto dto = new TmUserResponseDto();
        if (null != employeeDTO.getDepts() && !employeeDTO.getDepts().isEmpty()) {
            //设置部门信息
            dto.setDeptId(employeeDTO.getDepts().get(0).getId());
            //设置部门名称
            dto.setDeptName(employeeDTO.getDepts().get(0).getName());
        }
        //设置员工id
        dto.setEmpId(employeeDTO.getId());
        //设置员工名称
        dto.setEmpName(employeeDTO.getName());
        if (null != employeeDTO.getUser()) {
            //设置用户id
            dto.setUserId(employeeDTO.getUser().getId());
            //设置用户名称
            dto.setUserName(employeeDTO.getUser().getName());
        }
        retList.add(dto);
        return retList;
    }


    /**
     * 处理部门主管对象
     *
     * @param directorDto 部门主管信息
     * @return 处理结果
     */
    private List<TmUserResponseDto> convertFromDirector(EocDirectEmployeeDTO directorDto) {
        List<TmUserResponseDto> retList = new ArrayList<>();
        TmUserResponseDto dto = new TmUserResponseDto();
        //设置部门信息
        dto.setDeptId(directorDto.getDeptId());
        //设置部门名称
        dto.setDeptName(directorDto.getDeptName());

        //设置员工id
        dto.setEmpId(directorDto.getId());
        //设置员工名称
        dto.setEmpName(directorDto.getName());

        //设置用户id
        dto.setUserId(directorDto.getUserId());
        //设置用户名称
        dto.setUserName(directorDto.getUserName());

        retList.add(dto);
        return retList;
    }


    /**
     * 处理职责下的员工信息
     *
     * @param dutyDtoList 职责员工信息
     * @return 处理结果
     */
    private List<TmUserResponseDto> convertFromDuty(List<EocDutyEmployeeDTO> dutyDtoList) {
        List<TmUserResponseDto> retList = new ArrayList<>();

        for (EocDutyEmployeeDTO tempDto : dutyDtoList) {
            TmUserResponseDto dto = new TmUserResponseDto();
            //设置部门信息
            dto.setDeptId(tempDto.getDeptId());
            //设置部门名称
            dto.setDeptName(tempDto.getDeptName());

            //设置员工id
            dto.setEmpId(tempDto.getId());
            //设置员工名称
            dto.setEmpName(tempDto.getName());

            //设置用户id
            dto.setUserId(tempDto.getUserId());
            //设置用户名称
            dto.setUserName(tempDto.getUserName());

            retList.add(dto);
        }
        return retList;
    }


    /**
     * 处理部门下的员工信息
     *
     * @param deptDtoList 部门员工信息
     * @return 处理结果
     */
    private List<TmUserResponseDto> convertFromDept(List<EocDeptEmployeeDTO> deptDtoList) {
        List<TmUserResponseDto> retList = new ArrayList<>();

        for (EocDeptEmployeeDTO tempDto : deptDtoList) {
            TmUserResponseDto dto = new TmUserResponseDto();
            //设置部门信息
            dto.setDeptId(tempDto.getDeptId());
            //设置部门名称
            dto.setDeptName(tempDto.getDeptName());

            //设置员工id
            dto.setEmpId(tempDto.getEmpId());
            //设置员工名称
            dto.setEmpName(tempDto.getEmpName());

            //设置用户id
            dto.setUserId(tempDto.getUserId());
            //设置用户名称
            dto.setUserName(tempDto.getUserName());

            retList.add(dto);
        }
        return retList;
    }

    /**
     * 根据 snapshotId 和用户信息获取dmcId 调用aam 查询信息
     * @param snapshotId 快照id
     * @param tenantId 租户id
     * @param userId 用户id
     * @return aamData
     */
    @Override
    public AamDataResDTO getAamDataBySnapshotId(String snapshotId, String tenantId, String userId) {
        AamDataResDTO resDTO = new AamDataResDTO();
        String dmcId = getDmcId(snapshotId, tenantId, userId);
        if(StringUtils.isEmpty(dmcId)){
            return resDTO;
        }
        log.info("adt getAamDataBySnapshotId snapshotId{},dmcId{}:",snapshotId,dmcId);
        String redisSceneCode = stringRedisTemplate.opsForValue().get(snapshotId);
        if(StringUtils.isNotEmpty(redisSceneCode)){
            resDTO.setSceneCode(redisSceneCode);
            return resDTO;
        }else{
            return AamDataResDTO.builderAamDataBySnapShotDTO(dmcService.getAgileData(dmcId), stringRedisTemplate);
        }
    }

    /**
     * 敏数提供第三方数据查询
     * @param snapshotId
     * @param authoredUser
     * @return
     */
    @Override
    public Map<String, Object> getThirdPartyAgileData(String snapshotId, AuthoredUser authoredUser) {
        Map<String,Object> snapshotResult = new HashMap<>();
        String dmcId = this.getDmcId(snapshotId, authoredUser.getTenantId(), authoredUser.getUserId());
        if(StringUtils.isEmpty(dmcId)) {
            return snapshotResult;
        }
        SnapShotDTO snapShotDTO = dmcService.getAgileData(dmcId);
        if(snapShotDTO != null){
            String method = snapShotDTO.getContext().getBizParams().getMethod();
            Object undeleteTable = snapShotDTO.getContext().getBizParams().getParam().get("undeletable");
            String question = snapShotDTO.getContext().getBizParams().getQuestion();
            if(CollectionUtils.isNotEmpty(snapShotDTO.getContext().getPullData())){
                List<Map<String,Object>> metricList = snapShotDTO.getContext().getPullData().get(0).getMetricList();
                List<String> metricIdList = new ArrayList<>();
                if(CollectionUtils.isNotEmpty(metricList)){
                    for (Map<String, Object> map : metricList) {
                        if (map.containsKey("metricId")) {
                            Object metricIdObj = map.get("metricId");
                            if (metricIdObj instanceof String) {
                                metricIdList.add((String) metricIdObj);
                            }
                        }
                    }
                }
                /**
                 * 指标编码
                 */
                snapshotResult.put("metricIdList", metricIdList);
            }
            /**
             * 架构标识
             */
            snapshotResult.put("method",method);
            /**
             * 问句
             */
            snapshotResult.put("question",question);
            /**
             * 前端订阅标识
             */
            snapshotResult.put("undeletable",!Objects.isNull(undeleteTable) && (boolean) undeleteTable);
        }
        return snapshotResult;
    }

    /**
     * 按要求转换原始数据
     * @param agileTransDataDTO
     * @param authoredUser
     * @param lang
     * @return
     */
    @Override
    public SnapDataInfoDTO transAgileData(AgileTransDataDTO agileTransDataDTO, AuthoredUser authoredUser, String lang) {
        //获取快照信息
        SnapDataInfoDTO snapDataInfoDTO = new SnapDataInfoDTO();
        QueryWrapper<SnapData> queryWrapper = new QueryWrapper<>();
        queryWrapper.eq(TENANT_ID, authoredUser.getTenantId()).eq(PERFORMER_ID, authoredUser.getUserId()).eq(SNAPSHOT_ID,agileTransDataDTO.getSnapshotId());
        SnapData snapData = snapShotDataMapper.selectOne(queryWrapper);
        if(snapData == null){
            log.error("无快照信息");
            throw BusinessException.create(ErrorCodeEnum.TRANS_DATA_ERROR.getErrCode(), messageUtil.getMessage("agileData.getTransData.error.info"));
        }
        SnapDataDTO snapDataDTO = new SnapDataDTO();
        BeanUtils.copyProperties(snapData,snapDataDTO);
        snapDataInfoDTO.setSnapData(snapDataDTO);
        //查到dmcId调用接口返回快照信息
        SnapShotDTO snapShotDTO = dmcService.getAgileData(snapData.getDmcId());
        if(snapShotDTO == null){
            log.error("获取DMC文件失败");
            throw BusinessException.create(ErrorCodeEnum.TRANS_DATA_ERROR.getErrCode(), messageUtil.getMessage("agileData.getTransData.error.info"));
        }
        //获取原始快照数据ID
        String originalId = snapShotDTO.getContext().getBizParams().getOriginalSnapshotDataFileId();
        //获取转换结果
        Map<String,Object> transData;
        //通过版本号，判断是否为历史数据,历史数据无版本号信息
        if(StringUtils.isEmpty(originalId)){
            //历史数据通过历史快照，进行数据转换
            transData = adeService.transPresentAgileData(authoredUser, agileTransDataDTO.getTransId(),null,snapShotDTO,lang);
        } else {
            //非历史数据，通过ADE存储的原始数据，进行转换
            transData = adeService.transPresentAgileData(authoredUser, agileTransDataDTO.getTransId(),originalId,null,lang);
        }

        if(transData == null){
            log.error("转换原始数据失败");
            throw BusinessException.create(ErrorCodeEnum.TRANS_DATA_ERROR.getErrCode(), messageUtil.getMessage("agileData.getTransData.error.info"));
        }
        log.info("转换后的快照数据：{}",JsonUtils.objectToString(transData));
        //替换pullData
        Map<String,Object> data = (Map<String, Object>) MapUtils.getMap(transData,"data");
        Map<String,Object> context = (Map<String, Object>) MapUtils.getMap(data,"context");
        List<Map<String,Object>> pullData = (List<Map<String,Object>>) MapUtils.getObject(context,"pullData");
        if(CollectionUtils.isNotEmpty(pullData)){
            List<PullDataDTO> snapPullData = Lists.newArrayList();
            pullData.stream().forEach(transAgileData -> {
                PullDataDTO pullDataDTO = new PullDataDTO();
                pullDataDTO.setActionId(MapUtils.getString(transAgileData,"actionId"));
                pullDataDTO.setModelId(MapUtils.getString(transAgileData,"modelId"));
                pullDataDTO.setSceneCode(MapUtils.getString(transAgileData,"sceneCode"));
                pullDataDTO.setStartTime(MapUtils.getString(transAgileData,"startTime"));
                pullDataDTO.setEndTime(MapUtils.getString(transAgileData,"endTime"));
                pullDataDTO.setData((Map<String, Object>) MapUtils.getMap(transAgileData,"data"));
                pullDataDTO.setMetadata((Map<String, Object>) MapUtils.getMap(transAgileData,"metadata"));
                pullDataDTO.setDataTag((Map<String, Object>) MapUtils.getMap(transAgileData,"dataTag"));
                pullDataDTO.setMetricList((List<Map<String, Object>>) MapUtils.getObject(transAgileData,"metricList"));
                snapPullData.add(pullDataDTO);
            });
            snapShotDTO.getContext().setPullData(snapPullData);
        }
        //替换呈现定义
        Map<String,Object> bizParams = (Map<String, Object>) MapUtils.getMap(context,"bizParams");
        Map<String,Object> showDefine = (Map<String, Object>) MapUtils.getObject(bizParams, "showDefine");
        if(MapUtils.isNotEmpty(showDefine)){
            List<Map<String,Object>> showTypeList = (List<Map<String, Object>>) MapUtils.getObject(showDefine, "showType");
            if(CollectionUtils.isNotEmpty(showTypeList)) {
                Map<String,Object> showType = showTypeList.get(0);
                List<Map<String, Object>> typeList = (List<Map<String, Object>>) showType.get("type");
                if (CollectionUtils.isNotEmpty(showTypeList)) {
                    Map<String, Object> typeMap = typeList.get(0);
                    typeMap.put("value", agileTransDataDTO.getTransId());
                }
            }
        }

        snapShotDTO.getContext().getBizParams()
                .setShowDefine(showDefine);
        //替换提示文本
        String transTips = MapUtils.getString(bizParams, "transTips");
        if(StringUtils.isNotEmpty(transTips)){
            snapDataDTO.setTransTips(transTips);
        }
        //封装快照结果
        this.getSnapShotDetail(snapDataInfoDTO,snapShotDTO);
        return snapDataInfoDTO;
    }

    /**
     * 获取可转换图形列表
     * @param agileTransDataDTO
     * @param authoredUser
     * @param lang
     * @return
     */
    @Override
    public List<String> getTransList(AgileTransDataDTO agileTransDataDTO, AuthoredUser authoredUser, String lang) {
        //获取数据快照
        String dmcId = this.getDmcId(agileTransDataDTO.getSnapshotId(), authoredUser.getTenantId(), authoredUser.getUserId());
        if(StringUtils.isEmpty(dmcId)){
            return Lists.newArrayList();
        }
        //查到dmcId调用接口返回快照信息
        SnapShotDTO snapShotDTO = dmcService.getAgileData(dmcId);
        if(snapShotDTO == null){
            return Lists.newArrayList();
        }
        //通过快照获取数据信息
        return adeService.getTransList(snapShotDTO, authoredUser, lang);
    }

    /**
     * 根据 快照id 返回solutionStep
     * @param snapshotId s
     * @param tenantId t
     * @param userId u
     * @return s
     */
    @Override
    public Map<String, Object> getSolutionStepBySnapshotId(String snapshotId, String tenantId, String userId) {

        // 参数非空检查
        if (snapshotId == null || tenantId == null || userId == null) {
            throw new IllegalArgumentException("snapshotId, tenantId, and userId cannot be null");
        }

        // 获取快照信息并检查是否为空
        SnapShotDTO snapshotInfo = snapShotDataService.getSnapShotData(snapshotId, tenantId, userId);
        if (snapshotInfo == null) {
            throw new RuntimeException("Snapshot not found for id: " + snapshotId);
        }

        // 检查context是否存在
        ContextDTO context = snapshotInfo.getContext();
        if (context == null) {
            throw new RuntimeException("Context not found in snapshot: " + snapshotId);
        }

        // 检查bizParams是否存在
        BizParamsDTO bizParams = context.getBizParams();
        if (bizParams == null) {
            throw new RuntimeException("BizParams not found in context");
        }
        String method = bizParams.getMethod();
        if (method == null) {
            throw new RuntimeException("Method is null");
        }
        Map<String, Object> datasetMap = Maps.newHashMap();
        if(StringUtils.equals("dataset",method)) {
            // 检查datasetList是否存在且不为空
            List<Map<String, Object>> datasetList = bizParams.getDatasetList();
            if (datasetList == null || datasetList.isEmpty()) {
                throw new RuntimeException("DatasetList is empty or null");
            }

            // 获取第一个dataset并检查是否存在datasetId
            Map<String, Object> firstDataset = datasetList.get(0);
            if (firstDataset == null || firstDataset.get("datasetId") == null) {
                throw new RuntimeException("DatasetId not found in first dataset");
            }

            // 检查dataset是否存在
            datasetMap = bizParams.getDataset();
            if (datasetMap == null) {
                throw new RuntimeException("Dataset map is null");
            }
        }
        if(StringUtils.equals("metric",method)) {
            // 获取metric语义解析结果
            datasetMap = bizParams.getMetric();
            if (datasetMap == null) {
                throw new RuntimeException("Dataset map is null");
            }
        }

        return datasetMap;
    }

}
