package com.digiwin.athena.adt.domain.dto.ania;

import com.digiwin.athena.adt.domain.dto.km.KMDatasetCommandIntentionsResDTO;
import com.digiwin.athena.adt.domain.dto.mongo.AgileDataCustomConfig;
import lombok.Data;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;

import java.io.Serializable;
import java.util.*;
import java.util.stream.Collectors;

/**
 * @Author: SunHong
 * @Date: 2025/5/12 9:45
 * @Description: 返回给娜娜的助理信息详情
 */
@Data
public class AniaAssistantResDTO implements Serializable {

    private List<Assistant> assistant;

    @Data
    public static class Assistant {

        /**
         * 助理ID
         */
        private String assistantId;

        /**
         * 助理名称
         */
        private String name;

        /**
         * 助理请求基地址
         */
        private String baseURL;

        /**
         * 助理的头像地址
         */
        private String iconUrl;

        /**
         * 助理的开场白配置
         */
        private OnBoardingInfo onboardingInfo;

        /**
         * 助理的灵感话术
         */
        private List<InspirationPhrases> inspirationPhrases;

        /**
         * 助理的定制组件package包配置，用于JS预加载；默认为空
         */
        private CustomConfig customConfig;

        /**
         * 助理吊顶卡片的组件代号，用于固定显示于悬停显示在助理会话列表顶部；默认为空
         */
        private String ceilingCard;

        /**
         * 助理自定义的输入框操作组件代号，用于显示于输入框上方；默认为空
         */
        private List<String> inputCustomComponents;

        /**
         * 用户输入类型；默认仅支持文字输入
         */
        private InputConfig inputConfig;

        /**
         * 助理对话模式；默认长对话模式
         */
        private DialogueMode dialogueMode;

        /**
         * 助理支持的端类型。取值为：
         * - WEB：WEB端（包含PCAPP及PCWEB）。
         * - APP：移动端（包含移动APP及移动WEB[钉钉等手机版]）。
         */
        private List<String> clientType;

        /**
         * 助理的版本号
         */
        private String version;

        /**
         * 多语言信息。每层涉及多语言都需返回
         */
        private String lang;

        /**
         * 前端组件信息
         */
        private List<Map<String,Object>> pluginCustoms;

        @Data
        public static class OnBoardingInfo {

            /**
             * 助理配置的开场白内容。 默认：assistant.iconUrl Hi~ assistant.name  为您服务
             * 支持以下固定占位符：用户名称：{{user_name}}
             */
            private String prologue;

            /**
             * 助理配置的推荐问题列表。若无需建议时，不返回此字段
             */
            private List<SuggestedQuestions> suggestedQuestions;

            @Data
            public static class SuggestedQuestions {

                /**
                 * 对于多技能的助理，此参数返回推荐问题的分组名称。
                 */
                private String groupName;

                /**
                 * 助理配置的灵感话术内容
                 */
                private List<Questions> questions;


                @Data
                public static class Questions {

                    /**
                     * 助理配置的推荐问题
                     */
                    private String question;

                    /**
                     * 开场白推荐问题的扩展信息，会带入助理对话时的extData中
                     */
                    private Map<String,Object> ext;

                }
            }

        }

        @Data
        public static class InspirationPhrases {

            /**
             * 对于多技能的助理，此参数返回快捷指令的分组名称
             */
            private String groupName;

            /**
             * 数据集code 做权限认证
             */
            private String moduleCode;

            /**
             * 助理配置的灵感话术内容
             */
            private List<QueryTemplates> queryTemplates;

            /**
             * 灵感话术内容的变量列表
             */
            private List<InspirationPhraseComponent> components;

            public static List<InspirationPhrases> builderInspirationPhrases(List<Map<String, Object>> commandIntentions) {
//                List<Map<String, Object>> testData = testCommandIntentionsData(commandIntentions);
//                commandIntentions.clear();
//                commandIntentions = testData;
                List<InspirationPhrases> phrases = new ArrayList<>();
                if(CollectionUtils.isNotEmpty(commandIntentions)){
                    for (Map<String, Object> commandIntention : commandIntentions) {
                        InspirationPhrases inspirationPhrases = new InspirationPhrases();
                        inspirationPhrases.setComponents(new ArrayList<>());
                        inspirationPhrases.setGroupName(String.valueOf(commandIntention.get("groupName")));
                        inspirationPhrases.setModuleCode(String.valueOf(commandIntention.get("code")));
                        List<Map<String, Object>> questions = (List<Map<String, Object>>) commandIntention.get("questions");
                        List<InspirationPhrases.QueryTemplates> queryTemplates = questions.stream()
                                .map(question -> {
                                    InspirationPhrases.QueryTemplates queryTemplate = new InspirationPhrases.QueryTemplates();
                                    queryTemplate.setContent(String.valueOf(question.get("question")));
                                    queryTemplate.setExt((Map<String, Object>) question.get("lang"));
                                    return queryTemplate;
                                })
                                .collect(Collectors.toList());
                        inspirationPhrases.setQueryTemplates(queryTemplates);
                        if(Objects.nonNull(commandIntention.get("groupName"))){
                            phrases.add(inspirationPhrases);
                        }
                    }
                }
                return phrases;
            }

            private static List<Map<String, Object>> testCommandIntentionsData(List<Map<String, Object>> commandIntentions) {

                if (commandIntentions == null || commandIntentions.isEmpty()) {
                    System.out.println("输入数据为空，无法进行扩展");
                    return new ArrayList<>();
                }
                List<Map<String, Object>> expandedData = new ArrayList<>(commandIntentions.size() * 1200);

                // 数据扩展2000倍
                for (int i = 0; i < 1200; i++) {
                    for (Map<String, Object> original : commandIntentions) {
                        // 深拷贝Map内容，避免引用问题
                        Map<String, Object> copy = new HashMap<>(original.size());
                        for (Map.Entry<String, Object> entry : original.entrySet()) {
                            Object value = entry.getValue();
                            // 基本类型直接复制，集合类需要深拷贝
                            if (value instanceof String || value instanceof Number || value instanceof Boolean) {
                                copy.put(entry.getKey(), value);
                            } else if (value instanceof List) {
                                copy.put(entry.getKey(), new ArrayList<>((List<?>) value));
                            } else if (value instanceof Map) {
                                copy.put(entry.getKey(), new HashMap<>((Map<?, ?>) value));
                            } else {
                                // 其他类型默认使用原始引用，可根据需要扩展
                                copy.put(entry.getKey(), value);
                            }
                        }
                        expandedData.add(copy);
                    }
                }
                return expandedData;
            }

            @Data
            public static class QueryTemplates {

                /**
                 * 若包含{{components.name}}，则会基于变量定义控制{{}}中的内容输入
                 */
                private String content;

                /**
                 * 灵感话术的扩展信息，若用户未修改{{}}外的内容，则会带入助理对话时的extData中
                 */
                private Map<String,Object> ext;
            }

            @Data
            public static class InspirationPhraseComponent {

                /**
                 * 变量的名称
                 */
                private String name;

                /**
                 * 变量的类型。取值为：
                 * - text：文本。
                 */
                private String type;

                /**
                 * 变量的默认值
                 */
                private String defaultValue;
            }


        }

        @Data
        public static class CustomConfig {

            /**
             * package插件名称，正常同助理ID，需保证唯一
             */
            private String name;

            /**
             * 定制包路径
             */
            private String path;

            /**
             * 定制包打包后的chunk路径，一般为打包后js的文件名
             */
            private String chunkName;

            /**
             * 微前端中被导出的模块名称
             */
            private String remoteName;

            /**
             *微前端中被导出的模块路径
             */
            private String exposedModule;
        }

        @Data
        public static class InputConfig {

            /**
             * 输入提示
             */
            private String placeholder;

            /**
             * 变量的类型。取值为：
             * - text：文字输入。
             * - file：上传文件
             */
            private List<String> type;

            /**
             * type 包含 file 时，options 为支持的文件类型。取值为：
             * - image：图片。
             * - doc：文档。
             * - table：表格。
             */
            private List<String> options;

            /**
             * 钩子函数配置信息
             */
            private List<Hooks> hooks;



            @Data
            public static class Hooks {

                /**
                 * 入口暴露的组件id
                 */
                private String eventSource;

                /**
                 * 初始化后执行的钩子函数
                 */
                private String afterInitHook;

                /**
                 * 初始化之前的钩子函数
                 */
                private String beforeHook;
            }
        }

        @Data
        public static class DialogueMode {

            /**
             * 对话模式。取值为：
             * - longDialogueMode：长对话模式。
             * - topicBasedDialogueMode：topic对话模式。
             */
            private String mode;
        }

        /**
         * 组装助理信息
         * @param version 版本 娜娜
         * @param appCode 应用
         * @param cIData km 指令集信息
         * @param customConfig 前端组件存储信息
         * @param baseURL 服务调用base地址
         * @return 助理信息实体
         */
        public static Assistant builderAssistant(String version,
                                                 String appCode,
                                                 KMDatasetCommandIntentionsResDTO cIData,
                                                 AgileDataCustomConfig customConfig,
                                                 String baseURL) {
            String appName = "";
            String introduction = "";
            String prompt = "";
            String iconUrl = "";
            List<Map<String, Object>> commandIntentions = new ArrayList<>();
            if(Objects.nonNull(cIData) && StringUtils.isNotEmpty(cIData.getApplicationName())){
                appName = cIData.getApplicationName();
                if(StringUtils.isNotEmpty(cIData.getIntroduction())){
                    introduction = cIData.getIntroduction();
                }
                if(CollectionUtils.isNotEmpty(cIData.getCommandIntentions())){
                    commandIntentions = cIData.getCommandIntentions();
                }
                if(StringUtils.isNotEmpty(cIData.getPrompt())){
                    prompt = cIData.getPrompt();
                }
                if(StringUtils.isNotEmpty(cIData.getIconUrl())){
                    iconUrl = cIData.getIconUrl();
                }
            }
            Assistant assistant = new Assistant();
            assistant.setAssistantId(appCode);
            assistant.setName(appName);
            assistant.setBaseURL(baseURL+"/api/ai/agileData/assistant");
            assistant.setIconUrl(iconUrl);
            // 开场白
            OnBoardingInfo onBoardingInfo = new OnBoardingInfo();
            onBoardingInfo.setPrologue(introduction);
            onBoardingInfo.setSuggestedQuestions(new ArrayList<>());
            assistant.setOnboardingInfo(onBoardingInfo);
            // 灵感话术 -> 指令集 数据组装
            List<InspirationPhrases> phrases = InspirationPhrases.builderInspirationPhrases(commandIntentions);
            if(CollectionUtils.isNotEmpty(phrases)){
                assistant.setInspirationPhrases(phrases);
            }
            // 助理的定制组件package包配置
            CustomConfig config = new CustomConfig();
//            if(StringUtils.isNotEmpty(customConfig.getId())){
//                config.setName(String.valueOf(customConfig.getCustomConfig().get("name")));
//                config.setChunkName(String.valueOf(customConfig.getCustomConfig().get("chunkName")));
//                config.setPath(String.valueOf(customConfig.getCustomConfig().get("path")));
//                config.setRemoteName(String.valueOf(customConfig.getCustomConfig().get("remoteName")));
//                config.setExposedModule(String.valueOf(customConfig.getCustomConfig().get("exposedModule")));
//            }
            assistant.setCustomConfig(config);
            // 助理吊顶卡片的组件代号
            assistant.setCeilingCard("");
            // 助理自定义的输入框操作组件代号
            assistant.setInputCustomComponents(new ArrayList<>());
            // 用户输入类型
            InputConfig inputConfig = new InputConfig();
            inputConfig.setPlaceholder(prompt);
            assistant.setInputConfig(inputConfig);
            // 对话模式
            DialogueMode dialogueMode = new DialogueMode();
            dialogueMode.setMode("topicBasedDialogueMode");
            assistant.setDialogueMode(dialogueMode);
            // 助理支持的端类型
            List<String> clientTypes = new ArrayList<>();
            clientTypes.add("WEB");
            clientTypes.add("APP");
            assistant.setClientType(clientTypes);
            assistant.setVersion(version);
            assistant.setLang("");
            // 前端组件集合
            assistant.setPluginCustoms(customConfig.getPluginCustoms());
            return assistant;
        }
    }
}
