package com.digiwin.athena.adt.domain.report.impl;

import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.digiwin.athena.adt.agileReport.constant.AgileDataSourceEnum;
import com.digiwin.athena.adt.agileReport.constant.AthenaMessageEnum;
import com.digiwin.athena.adt.agileReport.constant.TroubleToolCodeEnum;
import com.digiwin.athena.adt.agileReport.controller.dto.AthenaMessageDTO;
import com.digiwin.athena.adt.agileReport.dao.AgileDataConfigMapper;
import com.digiwin.athena.adt.agileReport.eventbus.AthenaMessageEvent;
import com.digiwin.athena.adt.domain.dto.agileReport.SceneCommonDTO;
import com.digiwin.athena.adt.domain.dto.agileReport.SceneDTO;
import com.digiwin.athena.adt.domain.dto.agileReport.SceneUserCommonDTO;
import com.digiwin.athena.adt.domain.dto.km.KMDatasetCommandIntentionsResDTO;
import com.digiwin.athena.adt.domain.dto.schema.QuerySchemaReqDTO;
import com.digiwin.athena.adt.domain.knowledge.KmService;
import com.digiwin.athena.adt.domain.po.AgileDataConfig;
import com.digiwin.athena.adt.util.CommonUtil;
import com.digiwin.athena.adt.util.LogUtils;
import com.digiwin.athena.appcore.auth.domain.AuthoredUser;
import com.digiwin.athena.appcore.domain.log.LogDto;
import com.digiwin.athena.appcore.util.JsonUtils;
import com.digiwin.athena.appcore.util.SnowflakeIdWorker;
import com.digiwin.athena.atmc.http.restful.iam.UserService;
import com.jugg.agile.framework.core.dapper.log.JaMDC;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.compress.utils.Lists;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.MDC;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletRequest;
import java.time.LocalDateTime;
import java.util.*;

import static com.digiwin.athena.adt.agileReport.constant.BusinessConstants.VERSION_1;
import static com.digiwin.athena.adt.agileReport.constant.BusinessConstants.VERSION_2;

/**
 * @Author: SunHong
 * @Date: 2024/7/1 10:07
 * @Description: 上下文抽象公共类
 */
@Slf4j
public abstract class AbsAgileEventProcess {

    @Autowired
    private UserService userService;

    @Resource
    private AgileDataConfigMapper agileDataConfigMapper;

    @Autowired
    private KmService kmService;

    private static final String DESIGNER = "designer";

    private static final String METRIC = "metric";

    private static final String SPECIAL = "special";

    private static final String SPECIAL_LITE = "special_lite";

    private static final String TENANT_ID = "tenant_id";

    /**
     * 初始化敏捷数据adt上下文信息
     * @param request 请求
     * @param athenaMessageDTO 请求入参
     * @param user 用户
     * @return 上下文信息
     */
    public AthenaMessageEvent init(HttpServletRequest request,AthenaMessageDTO athenaMessageDTO, AuthoredUser user) {
            AthenaMessageEvent event = new AthenaMessageEvent();
        try{
            // 1.初始化用户信息上下文
            this.initUserInfo(request,athenaMessageDTO,event,user);
            // 2.初始化上下文业务信息
            this.initEventInfo(athenaMessageDTO,event);
            // 3.初始化消息类型
            event.setMessageType(this.getMessageType(event,user));
            // 4.初始化助理编码
            this.analyzeAsaCode(athenaMessageDTO,event);
            // 5.初始化特定业务类型的上下文字段
            this.initEventMsgExt(event,athenaMessageDTO);
            // 6.初始化语义相关入参
            this.initSchemaReq(event,athenaMessageDTO);
            // 7.初始化问句来源信息
            this.setSourceDataByMessageType(event);
        }catch (Exception e){
            LogUtils.buildAgileLog(LogUtils.MODULE_ADT, "layeredQuestion", TroubleToolCodeEnum.ADT_901_0101.getErrCode(),
                    JsonUtils.objectToString(athenaMessageDTO), TroubleToolCodeEnum.ADT_901_0101.getErrMsg(),
                    TroubleToolCodeEnum.ADT_901_0101.getSuggestion());
        }
        LogUtils.buildAgileLog(LogUtils.MODULE_ADT, "layeredQuestion", LogUtils.SUCCESS,
                JsonUtils.objectToString(athenaMessageDTO), JsonUtils.objectToString(event), "");
        return event;
    }

    /**
     * 流式需要初始化语义入参
     * @param event 上下文信息
     * @param athenaMessageDTO 入参
     */
    private void initSchemaReq(AthenaMessageEvent event,AthenaMessageDTO athenaMessageDTO) {
        QuerySchemaReqDTO querySchemaReqDTO = QuerySchemaReqDTO.builderSchemaReq(event);
        querySchemaReqDTO.setMetricIdList(athenaMessageDTO.getIntentCode());
        querySchemaReqDTO.setCombinationMsg(athenaMessageDTO.getCombinationMsg());
        querySchemaReqDTO.setProductVersion(event.getProductVersion());
        querySchemaReqDTO.setDatasetIdList(event.getDatasetIdList());
        querySchemaReqDTO.setProbe(event.getProbe());
        querySchemaReqDTO.setMultiDialogue(event.getProbe());
        querySchemaReqDTO.setAppCode(athenaMessageDTO.getAppCode());
        querySchemaReqDTO.setMessageId4Ania(athenaMessageDTO.getMessageId());
        querySchemaReqDTO.setSectionId(athenaMessageDTO.getSectionId());
        querySchemaReqDTO.setConversationId(athenaMessageDTO.getConversationId());
        event.setQuerySchemaReqDTO(querySchemaReqDTO);
    }

    private void initEventMsgExt(AthenaMessageEvent event,AthenaMessageDTO athenaMessageDTO) {
        if(Objects.nonNull(event.getMsgExt())){
            String sessionId = Objects.nonNull(athenaMessageDTO.getMsgExt().get("sessionId")) ?
                    athenaMessageDTO.getMsgExt().get("sessionId").toString() : null;
            event.setSessionId(sessionId);
            event.getMsgBody().put("sessionId",sessionId);
            event.getMsgExt().put("isAlarm",false);
            event.getMsgExt().put("messageId",event.getGenerateSerialNo());
            event.getMsgExt().put("sessionId",sessionId);
            if(event.getMsgExt() != null && event.getMsgExt().containsKey("sceneCode")
                    && StringUtils.isNotEmpty(String.valueOf(event.getMsgExt().get("sceneCode")))){
                // 确定场景问句
                event.setDebug(true);
            }
            Object undeleteTable = athenaMessageDTO.getMsgExt().get("undeletable");
            event.setUndeletable(Objects.nonNull(undeleteTable) && (boolean) undeleteTable);
            // 根据应用编码区分数据源版本号 productVersion
            String productVersion = this.getProductVersionByAppCode(athenaMessageDTO.getAppCode(),event);
            event.setProductVersion(productVersion);
            // 数据集id
            Object datasetIdList = athenaMessageDTO.getMsgExt().get("datasetIdList");
            event.setDatasetIdList(Objects.nonNull(datasetIdList)? (List<String>) datasetIdList : new ArrayList<>());
            // 追问标识
            Object probe = event.getMsgExt().get("probe");
            event.setProbe(Objects.nonNull(probe)? (Boolean) probe : false);
            // 精准/自由模式
            String conversationMode = Objects.nonNull(event.getMsgBody().get("conversationMode")) ?
                    String.valueOf(event.getMsgBody().get("conversationMode")) : "" ;
            event.setConversationMode(conversationMode);
        }
    }

    /**
     * 根据appCode 查询km 应用信息
     * @param appCode 应用编码
     * @return 版本号
     */
    public String getProductVersionByAppCode(String appCode,AthenaMessageEvent event) {
        if (!StringUtils.isEmpty(appCode)) {
            KMDatasetCommandIntentionsResDTO cIData = kmService.dataSetCommandIntentions(event.getUser(),appCode);
            if (Objects.nonNull(cIData) && Objects.nonNull(cIData.getAppType())) {
                Integer appType = cIData.getAppType();
                if (12 == appType) {
                    return VERSION_2;
                } else {
                    return VERSION_1;
                }
            }
        }
        return VERSION_1;
    }

    /**
     * 初始化上下文业务信息字段
     * @param athenaMessageDTO 入参
     * @param event 上下文
     */
    private void initEventInfo(AthenaMessageDTO athenaMessageDTO,AthenaMessageEvent event) {

        // 相关推荐/猜你想问默认1
        event.setAnswerResult(1);
        // 默认发送娜娜
        event.setSendNana(true);
        // 链路id
        String ptxId = JaMDC.getNotNullTraceId();
        MDC.put("traceId", ptxId);
        MDC.put("PtxId", ptxId);
        log.info("traceId : {}",ptxId);
        log.info("mdc PtxId : {}",MDC.get("PtxId"));
        event.setPtxId(ptxId);
        event.setMsgBody(athenaMessageDTO.getMsgBody());
        event.setMsgExt(athenaMessageDTO.getMsgExt());
        event.setAskTime(LocalDateTime.now());
        // 唯一序列顶替actionId
        Long generateSerialNo = SnowflakeIdWorker.getInstance().newId();
        String question = MapUtils.getString(event.getMsgBody(), "text");
        event.setQuestion(question);
        event.setGenerateSerialNo(generateSerialNo);
        log.info("agiledata_adt_" + generateSerialNo);
    }

    /**
     * 初始化用户信息
     * @param request 请求入参
     * @param athenaMessageDTO 求求入参
     * @param event 上下文
     * @param user 用户信息token
     */
    private void initUserInfo(HttpServletRequest request,AthenaMessageDTO athenaMessageDTO, AthenaMessageEvent event,AuthoredUser user) {
        // 区分普通调用和订阅调用
        AuthoredUser adtUser = this.setMqUserInfoByAthena(athenaMessageDTO.getMsgExt(),user);
        // 区分移动端/web端
        String clientAgent = getClientAgent(request,athenaMessageDTO.getMsgExt());
        List<String> terminalList = getTerminalList(clientAgent);
        // 历史默认入参
        SceneDTO sceneDTO = builderSceneDTO(user,terminalList);
        // 获取user lang
        String userLangName = getUserLangByParam(athenaMessageDTO.getMsgExt());
        if(StringUtils.isEmpty(userLangName)){
            userLangName = userService.getUserLangNameByUserId(adtUser.getUserId(), adtUser.getTenantId(), adtUser.getToken());
            LogDto logDto = new LogDto("未从娜娜获取用户多语言信息，从IAM获取结果：" + userLangName);
            logDto.toString();
        }
        LogDto logDto = new LogDto("获取用户多语言信息:" + userLangName);
        logDto.toString();
        event.setSceneDTO(sceneDTO);
        event.setUser(adtUser);
        event.setLang(userLangName);
        event.setTerminal(clientAgent);
    }

    /**
     * 根据不通场景解析asaCode助理编码
     * 优先级 asaCode > assistantCode > athenaMessageDTO.getAsaCode();
     * @param athenaMessageDTO 入参
     * @param event 上下文
     */
    public void analyzeAsaCode(AthenaMessageDTO athenaMessageDTO,AthenaMessageEvent event) {
        if(Objects.isNull(athenaMessageDTO.getMsgExt())){
            event.setAsaCode(athenaMessageDTO.getAsaCode());
            return;
        }
        String asaCode = Optional.ofNullable(athenaMessageDTO.getMsgExt().get("asaCode"))
                .map(Object::toString)
                .filter(StringUtils::isNotEmpty)
                .orElseGet(() -> Optional.ofNullable(athenaMessageDTO.getMsgExt().get("assistantCode"))
                        .map(Object::toString)
                        .filter(StringUtils::isNotEmpty)
                        .orElse(athenaMessageDTO.getAsaCode()));
        event.setAsaCode(asaCode);
    }

    public void setSourceDataByMessageType(AthenaMessageEvent event) {
        String messageType = event.getMessageType();
        if(AthenaMessageEnum.DEBUG.getValue().equals(messageType)
                || AthenaMessageEnum.PARAM.getValue().equals(messageType)
                || AthenaMessageEnum.METRIC_DEBUG.getValue().equals(messageType)){

            event.setSourceName(AgileDataSourceEnum.DEBUG_MODEL.getName());
            event.setSourceCode(AgileDataSourceEnum.DEBUG_MODEL.getCode());
        }else{
            event.setSourceName(AgileDataSourceEnum.QUERY.getName());
            event.setSourceCode(AgileDataSourceEnum.QUERY.getCode());
        }
    }

    public String getUserLangByParam(Map<String, Object> msgExt) {
        return Optional.ofNullable(msgExt)
                .map(ext -> ext.get("language"))
                .map(Object::toString)
                .orElse("");
    }

    public List<SceneCommonDTO> buildWhat() {
        SceneCommonDTO whatSe = new SceneCommonDTO();
        whatSe.setType("realTimeQuery");
        return Collections.singletonList(whatSe);
    }

    public List<SceneCommonDTO> buildWhen() {
        SceneCommonDTO whenSe = new SceneCommonDTO();
        whenSe.setType("currentTime");
        whenSe.setValue(LocalDateTime.now().toString());
        return Collections.singletonList(whenSe);
    }

    public List<SceneUserCommonDTO> buildWho(AuthoredUser user) {
        List<Map<String,Object>> value = Lists.newArrayList();
        Map<String, Object> whoValue = new HashMap<>();
        whoValue.put("id", user.getUserId());
        whoValue.put("name", user.getUserName());
        whoValue.put("roles", user.getRoles());
        whoValue.put("sid", user.getSid());
        value.add(whoValue);

        SceneUserCommonDTO whoSe = new SceneUserCommonDTO();
        whoSe.setType("user");
        whoSe.setValue(value);
        return Collections.singletonList(whoSe);
    }

    public SceneDTO builderSceneDTO(AuthoredUser user, List<String> terminalList) {
        SceneDTO sceneDTO = new SceneDTO();
        sceneDTO.setWho(buildWho(user));
        sceneDTO.setWhat(buildWhat());
        sceneDTO.setWhen(buildWhen());
        sceneDTO.setTerminal(new ArrayList<>());
        return sceneDTO;
    }

    public String getClientAgent(HttpServletRequest request, Map<String, Object> msgExt) {
        if(request == null){
            return "";
        }
        return Optional.ofNullable(msgExt)
                .map(ext -> ext.get("client-agent"))
                .map(Object::toString)
                .orElseGet(() -> {
                    String agentHeader = request.getHeader("client-agent");
                    return StringUtils.defaultIfEmpty(agentHeader, request.getHeader("Client-Agent"));
                });
    }

    public static List<String> getTerminalList(String clientAgent) {
        return StringUtils.isEmpty(clientAgent) ? Collections.emptyList() : Collections.singletonList(clientAgent);
    }


    /**
     * 订阅相关用户信息逻辑
     * @param msgExt 入参
     * @param user 用户信息
     * @return 用户
     */
    public AuthoredUser setMqUserInfoByAthena(Map<String, Object> msgExt, AuthoredUser user) {
        AuthoredUser authoredUser = new AuthoredUser();
        BeanUtils.copyProperties(user, authoredUser);
        if (MapUtils.isNotEmpty(msgExt)) {
            if (!Objects.isNull(msgExt.get("userId")) && !Objects.isNull(msgExt.get("userName"))) {
                authoredUser.setUserId(String.valueOf(msgExt.get("userId")));
                authoredUser.setUserName(String.valueOf(msgExt.get("userName")));
            }
        }
        return authoredUser;
    }

    /**
     * 确认语义消息类型 获取消息类型
     * @ messageType 0 单轮会话 1 多目标 2 debug 3 多场景 4 param
     * @return 业务消息类型
     */
    public String getMessageType(AthenaMessageEvent event, AuthoredUser user){

        String messageType = "0";
        String question = event.getQuestion();
        if(StringUtils.isEmpty(question)){
            return messageType;
        }
        // 租户策略
        String tenantServiceName = getAgileDataByUserConfig(user);

        if(AthenaMessageEnum.METRIC.getCode().equals(tenantServiceName)){
            return AthenaMessageEnum.METRIC.getValue();
        }else if(AthenaMessageEnum.SPECIAL.getCode().equals(tenantServiceName)){
            return AthenaMessageEnum.SPECIAL.getValue();
        }else if(AthenaMessageEnum.SPECIAL_LITE.getCode().equals(tenantServiceName)){
            return AthenaMessageEnum.SPECIAL_LITE.getValue();
        }else if(AthenaMessageEnum.MOCK.getCode().equals(tenantServiceName)){
            return AthenaMessageEnum.MOCK.getValue();
        }
        //debug标识
        if(question.contains("*debug*") || question.contains("*pr-debug*") || question.contains("*test*") ){
            return AthenaMessageEnum.DEBUG.getValue();
        }
        //param标识
        if(question.contains("*param*") || question.contains("*pr-param*")){
            return AthenaMessageEnum.PARAM.getValue();
        }
        //metric-debug标识
        if(question.contains("*metric*")){
            return AthenaMessageEnum.METRIC_DEBUG.getValue();
        }
        return messageType;
    }

    public String getAgileDataByUserConfig(AuthoredUser user) {
        QueryWrapper<AgileDataConfig> queryWrapper = new QueryWrapper<>();
        queryWrapper.eq(TENANT_ID, user.getTenantId());
        AgileDataConfig agileDataConfig = agileDataConfigMapper.selectOne(queryWrapper);
        String serviceName = DESIGNER;
        if(agileDataConfig != null && StringUtils.isNotEmpty(agileDataConfig.getType())
                && (METRIC.equals(agileDataConfig.getType()) || DESIGNER.equals(agileDataConfig.getType())
                || SPECIAL.equals(agileDataConfig.getType())
                || SPECIAL_LITE.equals(agileDataConfig.getType()))){
            serviceName = agileDataConfig.getType();
        }
        log.info("user agile data serviceName : {}", serviceName);
        return serviceName;
    }

    /**
     * 根据语义返回set消息类型
     * 后续可统一改造
     * @param event 上下文
     * @param messageType 消息类型
     */
    public void setAthenaEventMessageType(AthenaMessageEvent event,String messageType){
        event.setMessageType(messageType);
    }

    /**
     * 校验当前问句的租户是否存在响应的场景编码和指标
     * @param event 上下文
     * @return 是否
     */
    public boolean checkUserScene(AthenaMessageEvent event) {
        String sceneCode = MapUtils.getString(event.getMsgExt(), "sceneCode");
        List<String> metricIdListObj = CommonUtil.objConvertListString(event.getMsgExt().get("metricIdList"));
        if (StringUtils.isEmpty(sceneCode) && CollectionUtils.isEmpty(metricIdListObj)) {
            return true;
        }
        List<String> codes = kmService.querySceneByTenantId(event.getUser().getToken(), event.getUser().getTenantId());
        if(codes == null || codes.isEmpty()){
            return false;
        }
        List<String> queryCodes = new ArrayList<>();
        if(StringUtils.isNotEmpty(sceneCode)){
            queryCodes.add(sceneCode);
        }
        if(CollectionUtils.isNotEmpty(metricIdListObj)){
            queryCodes.addAll(metricIdListObj);
        }
        return new HashSet<>(codes).containsAll(queryCodes);
    }
}
