package com.digiwin.athena.adt.domain.report.impl;

import com.digiwin.athena.adt.agileReport.constant.AgileDataSourceEnum;
import com.digiwin.athena.adt.agileReport.controller.dto.*;
import com.digiwin.athena.adt.agileReport.controller.dto.excel.ExcelSheet;
import com.digiwin.athena.adt.agileReport.eventbus.AthenaMessageEvent;
import com.digiwin.athena.adt.agileReport.service.AgileDataMessageService;
import com.digiwin.athena.adt.agileReport.service.SnapShotDataService;
import com.digiwin.athena.adt.agileReport.service.impl.process.agileData.AbsAgileDataProcess;
import com.digiwin.athena.adt.domain.chatbi.ChatbiService;
import com.digiwin.athena.adt.domain.dmc.DmcService;
import com.digiwin.athena.adt.domain.dto.km.KMCollectStepResDTO;
import com.digiwin.athena.adt.domain.dto.km.KMDatasetCommandIntentionsResDTO;
import com.digiwin.athena.adt.domain.dto.km.ThemeMapBoardDTO;
import com.digiwin.athena.adt.domain.dto.mongo.AgileDataSummarizeMsg;
import com.digiwin.athena.adt.domain.file.FileParsingService;
import com.digiwin.athena.adt.app.config.executor.ExecutorConfig;
import com.digiwin.athena.adt.domain.dto.AgileDataPanelResDTO;
import com.digiwin.athena.adt.domain.dto.agileDataProcess.AgileDataProcessResDTO;
import com.digiwin.athena.adt.domain.dto.agileReport.*;
import com.digiwin.athena.adt.domain.dto.UploadAttachmentResp;
import com.digiwin.athena.adt.domain.dto.UploadParamDTO;
import com.digiwin.athena.adt.domain.knowledge.KmService;
import com.digiwin.athena.adt.domain.report.AgileReportDataService;
import com.digiwin.athena.adt.domain.uibot.UIBOTService;
import com.digiwin.athena.adt.sse.api.SSEManagerService;
import com.digiwin.athena.adt.sse.domain.EventData;
import com.digiwin.athena.adt.sse.domain.SseAniaEventEnum;
import com.digiwin.athena.adt.sse.dto.SSEBaseEvent;
import com.digiwin.athena.adt.sse.utils.SseEmitterUtils;
import com.digiwin.athena.adt.util.CommonUtil;
import com.digiwin.athena.adt.util.LogUtils;
import com.digiwin.athena.adt.util.MessageUtil;
import com.digiwin.athena.appcore.auth.domain.AuthoredUser;
import com.digiwin.athena.appcore.util.JsonUtils;
import com.google.common.eventbus.AsyncEventBus;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.data.mongodb.core.MongoTemplate;
import org.springframework.data.mongodb.core.query.Criteria;
import org.springframework.data.mongodb.core.query.Query;
import org.springframework.stereotype.Service;
import org.springframework.web.servlet.mvc.method.annotation.SseEmitter;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletRequest;
import java.time.format.DateTimeFormatter;
import java.util.*;
import java.util.concurrent.CompletableFuture;

import static com.digiwin.athena.adt.agileReport.constant.BusinessConstants.VERSION_1;
import static com.digiwin.athena.adt.agileReport.constant.BusinessConstants.VERSION_2;


/**
 * 敏捷数据处理
 */
@Slf4j
@Service
public class AgileReportProcessServiceImpl extends AbsAgileDataProcess implements AgileReportDataService {

    @Autowired
    private AsyncEventBus localEventBus;

    @Resource
    private UIBOTService uibotService;

    @Autowired
    private AgileDataMessageService agileDataMessageService;

    @Resource
    private SSEManagerService sseManagerService;

    @Autowired
    private ExecutorConfig executorConfig;

    private final static String YYYY_MM_DD_HH_MM_SS = "yyyy/MM/dd HH:mm:ss";

    @Autowired
    private SnapShotDataService snapShotDataService;

    @Autowired
    private MessageUtil messageUtil;

    @Autowired
    private FileParsingService fileParsingService;

    @Autowired
    private DmcService dmcService;

    @Autowired
    private ChatbiService chatbiService;

    @Autowired
    private KmService kmService;

    @Autowired
    @Qualifier("mongoAgileDataLogTemplate")
    private MongoTemplate mongoTemplate;

    @Override
    public String processSDData(String processData,String fileName,AuthoredUser user) {
        UploadParamDTO uploadParamDTO = new UploadParamDTO();
        uploadParamDTO.setFileJson(processData);
        uploadParamDTO.setFileName(fileName);
        //文件上传
        UploadAttachmentResp uploadAttachmentResp = dmcService.uploadAgileData(uploadParamDTO,user);
        return uploadAttachmentResp.getId();
    }

    /**
     * 问句主链路
     * @param request request
     * @param athenaMessageDTO 入参
     * @param user 用户
     * @return event
     */
    @Override
    public Map<String,String> processMessage(HttpServletRequest request,AthenaMessageDTO athenaMessageDTO, AuthoredUser user) {
        LogUtils.buildAgileLog(LogUtils.MODULE_ADT, "receiveQuestion", LogUtils.SUCCESS,
                JsonUtils.objectToString(athenaMessageDTO), "娜娜问句请求", "");
        Map<String, String> resultType = new HashMap<>();
        if (athenaMessageDTO.getMsgType() == null || athenaMessageDTO.getMsgBody().isEmpty()){
            resultType.put("resultType","NOTSUPPORT");
        }
        AthenaMessageEvent event = init(request,athenaMessageDTO,user);

        localEventBus.post(event);
        resultType.put("resultType","PROCESS");
        return resultType;
    }



    @Override
    public SnapShotDTO getAgileReportInfo(String dmcId) {
        return dmcService.getAgileData(dmcId);
    }

    /**
     * 补偿指标卡数据
     * @param snapShotDTO
     */
    @Override
    public void compensateAgileData(SnapShotDTO snapShotDTO) {
        if(CollectionUtils.isNotEmpty(snapShotDTO.getContext().getPullData())){
            snapShotDTO.getContext().getPullData().stream().forEach(pullDataDTO -> {
                Map<String,Object> computeMap = JsonUtils.jsonToObject(JsonUtils.objectToString(pullDataDTO),Map.class);
                computeMap.put("method",snapShotDTO.getContext().getBizParams().getMethod());
                computeMap.put("showDefine",snapShotDTO.getContext().getBizParams().getShowDefine());
                Map<String,Object> dataMap = uibotService.compensateAgileData(computeMap);
                if(MapUtils.isNotEmpty(dataMap)){
                    //补偿指标卡数据
                    pullDataDTO.setData(dataMap);
                }
            });
        }
    }

    /**
     * 数据面板请求接口
     * @param request request
     * @param req 入参
     * @param user 用户信息
     * @return 数据面板res
     */
    @Override
    public AgileDataPanelResDTO panelProcessMessage(HttpServletRequest request, AthenaMessageDTO req, AuthoredUser user) {
        LogUtils.buildAgileLog(LogUtils.MODULE_ADT, "receiveQuestion", LogUtils.SUCCESS,
                JsonUtils.objectToString(req), "数据面板收藏请求", "");
        AgileDataProcessResDTO result = AgileDataProcessResDTO.init();
        AgileDataPanelResDTO agileDataPanelResDTO = new AgileDataPanelResDTO();
        AthenaMessageEvent event = init(request,req,user);
        event.setSendNana(false);
        event.setSourceName(AgileDataSourceEnum.PANEL.getName());
        event.setSourceCode(AgileDataSourceEnum.PANEL.getCode());

        String questionId = MapUtils.getString(req.getMsgExt(),"questionId");
        if(StringUtils.isEmpty(questionId)){
            result = agileDataMessageService.processRes(event);
        }else{
            //通过KM获取看板数据
            KMCollectStepResDTO stepResDTO = kmService.getCollectStep(user,questionId);
            if(Objects.isNull(stepResDTO) || MapUtils.isEmpty(stepResDTO.getSolutionStep())
                    || Objects.isNull(stepResDTO.getSnapshotId())){
                result = agileDataMessageService.processRes(event);
                this.setPanelDataResult(result,agileDataPanelResDTO);
                return agileDataPanelResDTO;
            }
            // 组装问句event
            this.buildEventByStepQuestion(event,stepResDTO);
            if(Objects.nonNull(stepResDTO.getBoardId()) && MapUtils.isNotEmpty(stepResDTO.getSolutionStep())){
                // 直接取数 指标和数据集
                SnapShotDTO snapShotDTO = agileDataMessageService.processResV1(event,stepResDTO);
                result.setAlarm(false);
                result.setDataTipMessage("");
                result.setSnapshotId(snapShotDTO.getSnapshotId());
                result.setPanelMap(snapShotDTO);
                result.setRequestTime(snapShotDTO.getContext().getBizParams().getRequestTime());
            }else{
                // 走语义
                result = agileDataMessageService.processRes(event);
            }
        }
        this.setPanelDataResult(result,agileDataPanelResDTO);

        return agileDataPanelResDTO;

    }

    public void setPanelDataResult(AgileDataProcessResDTO result,AgileDataPanelResDTO agileDataPanelResDTO) {
        DateTimeFormatter sdf = DateTimeFormatter.ofPattern(YYYY_MM_DD_HH_MM_SS);
        if(Objects.nonNull(result) && Objects.nonNull(result.getRequestTime())){
            String formatDate = result.getRequestTime().format(sdf);
            agileDataPanelResDTO.setRequestTime(formatDate);
            agileDataPanelResDTO.setSnapshotId(result.getSnapshotId());
            agileDataPanelResDTO.setMsg(result.getMsg());
            agileDataPanelResDTO.setTotalUsage(result.getTotalUsage());
            agileDataPanelResDTO.setRemainingUsage(result.getRemainingUsage());
            agileDataPanelResDTO.setAlarm(result.isAlarm());
            agileDataPanelResDTO.setDataTipMessage(result.getDataTipMessage());
        }
    }


    /**
     * 看板2.0数据获取
     * @param request res
     * @param aniaAssistantChatReqDTO a
     * @param user u
     * @return r
     */
    @Override
    public Map<String, Object> panelFlagShipProcessMessage(HttpServletRequest request, AniaAssistantChatReqDTO aniaAssistantChatReqDTO, AuthoredUser user) {
        Map<String,Object> extData = aniaAssistantChatReqDTO.getMessage().getExtData();
        String questionId = MapUtils.getString(extData,"questionId");
        Map<String,Object> athenaMessage = new HashMap<>();
        athenaMessage.put("msgExt",aniaAssistantChatReqDTO.getMessage().getExtData());
        Map<String,Object> msgBody = new HashMap<>();
        msgBody.put("text",aniaAssistantChatReqDTO.getMessage().getExtData().get("msg"));
        athenaMessage.put("msgBody",msgBody);
        AthenaMessageDTO athenaMessageDTO =
                CommonUtil.mapToObject(athenaMessage,AthenaMessageDTO.class);
        athenaMessageDTO.setMessageId(aniaAssistantChatReqDTO.getMessage().getMessageId());
        athenaMessageDTO.setSectionId(aniaAssistantChatReqDTO.getMessage().getSectionId());
        athenaMessageDTO.setConversationId(aniaAssistantChatReqDTO.getConversationId());
        athenaMessageDTO.setAppCode(aniaAssistantChatReqDTO.getAssistantId());
        AthenaMessageEvent event = init(request,athenaMessageDTO,user);
        if(StringUtils.isNotEmpty(questionId)){
            //通过KM获取看板数据
            ThemeMapBoardDTO themeMapBoardDTO = kmService.getThemeMapBoardQuestion(user,questionId);
            if(Objects.isNull(themeMapBoardDTO) || Objects.isNull(themeMapBoardDTO.getSolutionStep())){
                // 走语义
                AgileDataProcessResDTO res = agileDataMessageService.processRes(event);
                // 2.0 返回快照结构体
                return CommonUtil.convertObjectToMap(res.getPanelMap());
            }
            // 直接取数
            return agileDataMessageService.processResV2(event,themeMapBoardDTO);
        }else{
            // 走语义
            AgileDataProcessResDTO res = agileDataMessageService.processRes(event);
            // 2.0 返回快照结构体
            return CommonUtil.convertObjectToMap(res.getPanelMap());
        }
    }


    private void buildEventByStepQuestion(AthenaMessageEvent event, KMCollectStepResDTO stepResDTO) {
        event.setDatasetIdList(stepResDTO.getDatasetList());
        event.getMsgExt().put("sceneCode",stepResDTO.getSceneCode());
        event.getMsgExt().put("metricIdList",stepResDTO.getMetricIdList());
    }
    /**
     * 娜娜流式问句 1.0
     * @param request http
     * @param athenaMessageDTO 入参实体
     * @param user 用户信息
     * @return sseEmitter
     */
    @Override
    public SseEmitter sseProcessMessage(HttpServletRequest request,
                                        AthenaMessageDTO athenaMessageDTO,
                                        AuthoredUser user) {
        LogUtils.buildAgileLog(LogUtils.MODULE_ADT, "receiveQuestion", LogUtils.SUCCESS,
                JsonUtils.objectToString(athenaMessageDTO), "娜娜流式V1问句请求", "");
        SSEBaseEvent sseBaseEvent = new SSEBaseEvent();

        SseEmitter sseEmitter = sseManagerService.generateSseEmitter(sseBaseEvent);

        sseBaseEvent.setAniaEmitter(sseEmitter);
        // 1.0 上下文初始化
        AthenaMessageEvent event = init(request,athenaMessageDTO,user);
        event.setSseMessage(true);
        event.setSseEmitter(sseEmitter);
        CompletableFuture.runAsync(() -> {
            // 1.1 异步开启会话
            EventData eventData = new EventData(SseAniaEventEnum.CHAT_CREATED.getEvent(), null);
            if (Objects.nonNull(sseEmitter)) {
                SseEmitterUtils.send(sseEmitter, eventData);
            }
            agileDataMessageService.sseProcess(event,sseBaseEvent);

        }, executorConfig.asyncServiceExecutor());

        return sseEmitter;
    }

    /**
     * 娜娜流式问句 2.0
     * @param request req
     * @param aniaAssistantChatReqDTO 2.0
     * @param user user
     * @return 流式返回
     */
    @Override
    public SseEmitter sseProcessMessageV2(HttpServletRequest request,
                                          AniaAssistantChatReqDTO aniaAssistantChatReqDTO,
                                          AuthoredUser user) {

        LogUtils.buildAgileLog(LogUtils.MODULE_ADT, "receiveQuestion", LogUtils.SUCCESS,
                JsonUtils.objectToString(aniaAssistantChatReqDTO), "娜娜流式V2问句请求", "");
        SSEBaseEvent sseBaseEvent = new SSEBaseEvent();

        SseEmitter sseEmitter = sseManagerService.generateSseEmitter(sseBaseEvent);

        sseBaseEvent.setAniaEmitter(sseEmitter);
        Map<String,Object> athenaMessage = new HashMap<>();
        athenaMessage.put("msgExt",aniaAssistantChatReqDTO.getMessage().getExtData());
        Map<String,Object> msgBody = new HashMap<>();
        msgBody.put("text",aniaAssistantChatReqDTO.getMessage().getExtData().get("msg"));
        athenaMessage.put("msgBody",msgBody);
        //2.0 入参构造
        AthenaMessageDTO athenaMessageDTO =
                CommonUtil.mapToObject(athenaMessage,AthenaMessageDTO.class);
        athenaMessageDTO.setMessageId(aniaAssistantChatReqDTO.getMessage().getMessageId());
        athenaMessageDTO.setSectionId(aniaAssistantChatReqDTO.getMessage().getSectionId());
        athenaMessageDTO.setConversationId(aniaAssistantChatReqDTO.getConversationId());
        athenaMessageDTO.setAppCode(aniaAssistantChatReqDTO.getAssistantId());
        // 1.0 上下文初始化
        AthenaMessageEvent event = init(request,athenaMessageDTO,user);
        event.setSseMessage(true);
        event.setSseEmitter(sseEmitter);
        event.setAppCode(aniaAssistantChatReqDTO.getAssistantId());

        CompletableFuture.runAsync(() -> {
            // 1.1 异步开启会话
            EventData eventData = new EventData(SseAniaEventEnum.CHAT_CREATED.getEvent(), null);
            if (Objects.nonNull(sseEmitter)) {
                SseEmitterUtils.send(sseEmitter, eventData);
            }
            agileDataMessageService.sseProcess(event,sseBaseEvent);

        }, executorConfig.asyncServiceExecutor());

        return sseEmitter;
    }

    @Override
    public AgileDataDetailUrlDTO getDataDetailUrl(AuthoredUser user, String snapshotId, String locale) {

        AgileDataDetailUrlDTO agileDataDetailUrlDTO = new AgileDataDetailUrlDTO();
//        //通过数据快照，获取ADE入参
        SnapShotDTO snapshotInfo = snapShotDataService.getSnapShotData(snapshotId,user.getTenantId(),user.getUserId());
        if(Objects.isNull(snapshotInfo)){
            this.buildErrorMessage(agileDataDetailUrlDTO,locale);
            return agileDataDetailUrlDTO;
        }
        Map<String,Object> params = this.buildParams(snapshotInfo);
        if(Objects.isNull(params)){
            return agileDataDetailUrlDTO;
        }
        AgileDataEngineFileDownModel adeFileDown = this.getFileDownUrl(user, params, locale);
        if(adeFileDown == null){
            this.buildErrorMessage(agileDataDetailUrlDTO,locale);
            return agileDataDetailUrlDTO;
        }
        agileDataDetailUrlDTO.setDataUrl(adeFileDown.getData());
        agileDataDetailUrlDTO.setResult(StringUtils.equals("200", adeFileDown.getCode()));
        agileDataDetailUrlDTO.setErrorMessage(adeFileDown.getMsg());
        return agileDataDetailUrlDTO;
    }

    public AgileDataEngineFileDownModel getFileDownUrl(AuthoredUser user,
                                                       Map<String, Object> params,
                                                       String locale) {
        AgileDataEngineFileDownModel model = new AgileDataEngineFileDownModel();
        ArrayList<String> header = (ArrayList<String>) params.get("descriptionsList");
        ArrayList<String> dataKey = (ArrayList<String>) params.get("dataNamesList");
        List data = (List) params.get("data");
        Map<String, Object> excelRequest = new HashMap<>(2);
        StringBuilder sb = new StringBuilder();
        List<ExcelSheet> excelSheets = new ArrayList<>(1);
        excelSheets.add(ExcelSheet.build("图表数据", header, dataKey, data));
        excelRequest.put("fileName", "数据下载");
        excelRequest.put("fileData", excelSheets);
        excelRequest.put("tenantId", user.getTenantId());
        AgileDataEngineFileDownModel res = fileParsingService.getDmcShareUrl(user.getTenantId(),
                user.getToken(), locale, excelRequest);
        if(Objects.nonNull(res)){
            return res;
        }
        return model;
    }

    /**
     * 构建隐藏信息
     * @param agileDataDetailUrlDTO dto
     */
    private void buildErrorMessage(AgileDataDetailUrlDTO agileDataDetailUrlDTO,String locale){
        agileDataDetailUrlDTO.setResult(false);
        agileDataDetailUrlDTO.setErrorMessage(messageUtil.getMessageByLangName("message.ade.dataUrl.error.msg",locale));
    }

    /**
     * 构建excel入参
     * @param snapshotInfo i
     * @return r
     */
    private Map<String,Object> buildParams(SnapShotDTO snapshotInfo){
        Map<String,Object> res = new HashMap<>();
        ContextDTO context = snapshotInfo.getContext();
        if(Objects.isNull(context)){
            return null;
        }
        List<PullDataDTO> pullData = context.getPullData();
        if(CollectionUtils.isEmpty(pullData)){
            return null;
        }
        if(CollectionUtils.isNotEmpty(pullData)){
            PullDataDTO pullDataDTO = pullData.get(0);
            res.put("data",pullDataDTO.getData().get("data"));

            Map<String,Object> response =
                    Optional.ofNullable((Map<String, Object>) pullDataDTO.getMetadata().get("response"))
                            .orElse(Collections.emptyMap());

            Map<String,Object> data = Optional.ofNullable((Map<String, Object>) response.get("data"))
                    .orElse(Collections.emptyMap());
          List<Map<String,Object>> fields = (List<Map<String, Object>>) data.get("field");
            List<String> descriptionsList = new ArrayList<>();
            List<String> dataNamesList = new ArrayList<>();
          for(Map<String,Object> field :fields){
                  descriptionsList.add(String.valueOf(field.get("description")));
                  dataNamesList.add(String.valueOf(field.get("data_name")));
          }
            res.put("descriptionsList",descriptionsList);
            res.put("dataNamesList",dataNamesList);
        }
        return res;
    }

    @Override
    public AgileDataSummarizeResDTO getAgileDataSummarize(AgileDataSummarizeReqDTO agileDataSummarizeReqDTO,
                                                          AuthoredUser user) {

        AgileDataSummarizeResDTO agileDataSummarizeResDTO = new AgileDataSummarizeResDTO();
        String mongoSummarizeData = this.getMongoSummarizeMsg(agileDataSummarizeReqDTO.getSnapshotId());
        if(StringUtils.isNotEmpty(mongoSummarizeData)){
            agileDataSummarizeResDTO.setSummarizeData(mongoSummarizeData);
            agileDataSummarizeResDTO.setData(true);
            return agileDataSummarizeResDTO;
        }
        String snapshotId = agileDataSummarizeReqDTO.getSnapshotId();
        //通过数据快照，获取ADE入参
        SnapShotDTO snapshotInfo = snapShotDataService.getSnapShotData(snapshotId,user.getTenantId(),user.getUserId());
        // 判断是否有数据做数据总结
        boolean queryDataFlag = snapshotInfo.getContext().getBizParams().isQueryDataFlag();
        if(!queryDataFlag){
            agileDataSummarizeResDTO.setData(false);
            agileDataSummarizeResDTO.setSummarizeMsg("");
            return agileDataSummarizeResDTO;
        }
        String summarizeData = snapshotInfo.getContext().getPullData().get(0).getSummarizeData();
        String question = snapshotInfo.getContext().getBizParams().getQuestion();
        Map<String, Object> requestParam = new HashMap<>();
        requestParam.put("question",question);
        requestParam.put("resultData",summarizeData);
        requestParam.put("snapshotId",snapshotId);
        requestParam.put("userPromptList", new ArrayList<>());
        Map<String,Object> analysis = chatbiService.getResultAnalysis(requestParam,user);
        if(Objects.isNull(analysis) || Objects.isNull(analysis.get("data"))){
            agileDataSummarizeResDTO.setSummarizeMsg("获取数据总结异常");
            agileDataSummarizeResDTO.setData(false);
            return agileDataSummarizeResDTO;
        }
        String summarizeMsg = String.valueOf(analysis.get("data"));
        agileDataSummarizeResDTO.setSummarizeData(summarizeMsg);
        agileDataSummarizeResDTO.setData(true);
        // 保存mongo db 每条快照的数据总结
        this.saveAgileDataSummarizeById(snapshotId,summarizeMsg);
        return agileDataSummarizeResDTO;
    }

    /**
     * 保存数据总结快照数据
     * @param snapshotId 快照Id
     * @param summarizeMsg 数据总结
     */
    public void saveAgileDataSummarizeById(String snapshotId, String summarizeMsg) {
        Query query = Query.query(Criteria.where("snapshotId").is(snapshotId));
        List<AgileDataSummarizeMsg> results = mongoTemplate.find(query, AgileDataSummarizeMsg.class);
        if(CollectionUtils.isEmpty(results)){
            AgileDataSummarizeMsg msg = new AgileDataSummarizeMsg();
            msg.setSummarizeMsg(summarizeMsg);
            msg.setSnapshotId(snapshotId);
            mongoTemplate.insert(msg);
        }
    }

    /**
     * 根据快照id 获取缓存的数据总结
     * @param snapshotId 快照id
     * @return str
     */
    public String getMongoSummarizeMsg(String snapshotId) {
        Query query = Query.query(Criteria.where("snapshotId").is(snapshotId));
        List<AgileDataSummarizeMsg> results = mongoTemplate.find(query, AgileDataSummarizeMsg.class);
        if(CollectionUtils.isNotEmpty(results)){
            return results.get(0).getSummarizeMsg();
        }
        return null;
    }

    /**
     * 根据 appCode 查询版本号
     * @param user u
     * @param appCode 应用编码
     * @return v1 v2
     */
    @Override
    public Map<String,Object> getAppVersion(AuthoredUser user, String appCode) {
        AthenaMessageEvent event = new AthenaMessageEvent();
        Map<String,Object> res = new HashMap<>();
        event.setUser(user);
        if (!StringUtils.isEmpty(appCode)) {
            KMDatasetCommandIntentionsResDTO cIData = kmService.dataSetCommandIntentions(event.getUser(),appCode);
            if (Objects.nonNull(cIData) && Objects.nonNull(cIData.getAppType())) {
                int appType = cIData.getAppType();
                // 根据应用类型设置版本
                String version = appType == 12 ? VERSION_2 : VERSION_1;
                res.put("version", version);
                // 判断是否为AileData应用
                boolean isAileDataApp = appType == 12 || appType == 6;
                res.put("isAgileDataApp", isAileDataApp);
            }
        }
       return res;
    }
}
