package com.digiwin.athena.adt.domain.semc.impl;

import com.digiwin.athena.adt.agileReport.constant.BusinessConstants;
import com.digiwin.athena.adt.agileReport.constant.ErrorCodeEnum;
import com.digiwin.athena.adt.agileReport.constant.SchemaConstants;
import com.digiwin.athena.adt.agileReport.controller.dto.AgileDataSummarizeReqDTO;
import com.digiwin.athena.adt.agileReport.controller.dto.AgileDataSummarizeResDTO;
import com.digiwin.athena.adt.agileReport.event.domain.SendMessageDTO;
import com.digiwin.athena.adt.agileReport.event.domain.SendMessageReqDTO;
import com.digiwin.athena.adt.agileReport.eventbus.AthenaMessageEvent;
import com.digiwin.athena.adt.app.env.AdtEnvProperties;
import com.digiwin.athena.adt.domain.report.AgileReportDataService;
import com.digiwin.athena.adt.domain.semc.SemcService;
import com.digiwin.athena.adt.sse.domain.EventData;
import com.digiwin.athena.adt.sse.domain.SseAniaEventEnum;
import com.digiwin.athena.adt.sse.utils.SseEmitterUtils;
import com.digiwin.athena.adt.util.CommonUtil;
import com.digiwin.athena.appcore.auth.AppAuthContextHolder;
import com.digiwin.athena.appcore.util.ExceptionUtil;
import com.digiwin.athena.appcore.util.JsonUtils;
import com.digiwin.athena.adt.util.MessageUtil;
import com.digiwin.athena.atmc.http.constant.GlobalConstant;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.*;
import org.springframework.stereotype.Service;
import org.springframework.web.client.RestTemplate;
import org.springframework.web.servlet.mvc.method.annotation.SseEmitter;

import java.util.Map;
import java.util.Objects;


@Slf4j
@Service
public class SemcServiceImpl implements SemcService {

    @Autowired
    private AdtEnvProperties adtEnvProperties;

    @Autowired
    private RestTemplate restTemplate;

    @Autowired
    private MessageUtil messageUtil;

    @Autowired
    private AgileReportDataService agileReportDataService;

    public Map sendMessage(SendMessageDTO sendMessageDTO,String token,String tenantId){
        String securityToken = Objects.isNull(AppAuthContextHolder.getContext()) ? null : AppAuthContextHolder.getContext().getSecurityToken();
        log.info("发送----娜娜 messageId:{}, app core securityToken : {}",sendMessageDTO.getMessageId(),securityToken);
        String url= adtEnvProperties.getSemcUrl()+"/gpt/message/sendMessage";
        HttpHeaders headers = new HttpHeaders();
        headers.add(GlobalConstant.IAM_USER_TOKEN, token);
        headers.add("token",token);
        headers.add("routerkey",tenantId);
        HttpEntity httpEntity = new HttpEntity(sendMessageDTO,headers);
        try {
            ResponseEntity<Map> response = restTemplate.exchange(url, HttpMethod.POST, httpEntity, Map.class);
            return response.getBody();
        } catch (Exception e){
            log.error("{}, 请求body：{}, 报错信息：{} ", ErrorCodeEnum.SEMC_SEND_MESSAGE_FAIL.getErrCode(),JsonUtils.objectToString(sendMessageDTO), e);
            throw ExceptionUtil.wrap(ErrorCodeEnum.SEMC_SEND_MESSAGE_FAIL.getErrCode(),e);
        }
    }

    /**
     * 组装发送娜娜日志消息
     * @param event 上下文
     * @param messageBody 其他信息
     * @return 发送的消息体
     */
    @Override
    public Map<String,Object> sendMessageToGpt(AthenaMessageEvent event,
                                               Map<String, Object> messageBody) {

        SendMessageReqDTO sendMessageReqDTO = SendMessageReqDTO.
                builderMessageAndType(event,messageBody);

        SendMessageDTO sendMessageDTO = sendMessageReqDTO.getSendMessageDTO();
        String question = sendMessageReqDTO.getQuestion();
        String combinationQuestion = sendMessageReqDTO.getCombinationQuestion();
        String questionUnderstand = sendMessageReqDTO.getQuestionUnderstand();
        Long generateSerialNo = sendMessageReqDTO.getGenerateSerialNo();
        sendMessageDTO.setSkillType("1");
        sendMessageDTO.setMsgType("CUSTOM");

        if(MapUtils.isNotEmpty(sendMessageDTO.getMsgBody())){
            sendMessageDTO.getMsgBody().put("getMessageInfo", sendMessageDTO.isGetMessageInfo());
            sendMessageDTO.getMsgBody().put("agileDataMessageType",sendMessageReqDTO.getMessageType());
            // dataflow / method
            sendMessageDTO.getMsgBody().put("method",StringUtils.isEmpty(sendMessageDTO.getMethod()) ?
                    SchemaConstants.METHOD_DATA_FLOW : sendMessageDTO.getMethod());
            sendMessageDTO.getMsgBody().put("metricList",sendMessageDTO.getMetricList());
            sendMessageDTO.getMsgBody().put("datasetList",event.getDatasetList());
            // debug model
            sendMessageDTO.getMsgBody().put("debugModel", sendMessageDTO.getDebugModel());

            if(StringUtils.isNotEmpty(question)) {
                sendMessageDTO.getMsgBody().put("question", question);
            }
            if(StringUtils.isNotEmpty(combinationQuestion)) {
                sendMessageDTO.getMsgBody().put("scrumbiQuestion", combinationQuestion);
            }
            if(StringUtils.isNotEmpty(questionUnderstand)) {
                sendMessageDTO.getMsgBody().put("questionUnderstand", questionUnderstand);
            }
            if(generateSerialNo != null) {
                sendMessageDTO.getMsgBody().put("generateSerialNo", generateSerialNo);
            }
            // 猜你想问/相关推荐标识和数据集
            sendMessageDTO.getMsgBody().put("answerResult", event.getAnswerResult());
            sendMessageDTO.getMsgBody().put("intentions", event.getSentences());
            //SSE消息，需要传递业务数据（云信无法存储较大数据量）
            if(event.isSseMessage()){
                sendMessageDTO.getMsgBody().put("historyData", event.getHistoryData());
                sendMessageDTO.getMsgBody().put("tokenSize", event.getTokenSize());
                sendMessageDTO.getMsgBody().put("exceedTokenThreshold", event.getExceedTokenThreshold());
                // 数据总结标识
                sendMessageDTO.getMsgBody().put("exceedSummarizeSize", event.getExceedSummarizeSize());
            }
        }
        //是否需要透传订阅消息
        if(MapUtils.isNotEmpty(sendMessageDTO.getMsgExt()) &&
            sendMessageDTO.getMsgExt().containsKey("type")){
            sendMessageDTO.getMsgBody().put("type",MapUtils.getString(sendMessageDTO.getMsgExt(),"type"));
        }
        // 是否发送b娜云信
        if(!event.isSseMessage() &&
                (MapUtils.isEmpty(sendMessageDTO.getMsgExt()) || !"agileSubscribe".equals(sendMessageDTO.getMsgExt().get("type")))){
            log.info("Sending Message to B娜 sendMessageDTO: {},sendMessageReqDTO :{}", JsonUtils.objectToString(sendMessageDTO)
                    ,JsonUtils.objectToString(sendMessageReqDTO));
            sendMessage(sendMessageDTO,sendMessageReqDTO.getToken(),sendMessageReqDTO.getTenantId());
        }else if (event.isSseMessage()){
            // 流式输出发送消息
            if(BusinessConstants.VERSION_1.equals(event.getProductVersion())){
                this.sendMessageToGptByV1(sendMessageDTO,event);
            }else if (BusinessConstants.VERSION_2.equals(event.getProductVersion())){
                this.sendMessageToGptByV2(sendMessageDTO,event);
            }else{
                this.sendMessageToGptByV1(sendMessageDTO,event);
            }
        }
        return CommonUtil.convertObjectToMap(sendMessageDTO);
    }

    /**
     * 数据总结消息体
     * @param sendMessageDTO send
     * @param event e
     */
    public void setSummarizeData(SendMessageDTO sendMessageDTO, AthenaMessageEvent event) {

        SseEmitter sseEmitter = event.getSseEmitter();
        // 是否超出数据总结长度
        Boolean isSummarize = event.getExceedSummarizeSize();
        // 默认总结失败
        String resultMsg = messageUtil.getMessageByLangName("message.adt.data.summarize.info", event.getLang());
        Object snapId = sendMessageDTO.getMsgBody().get("snapshotId");
        // 没有快照不做总结
        if(Objects.isNull(snapId) || Objects.isNull(isSummarize)){
            return;
        }
        // 没有超出字段长度尝试数据总结取数
        if(!isSummarize){
            // 数据总结取数
            AgileDataSummarizeReqDTO req = new AgileDataSummarizeReqDTO();
            req.setSnapshotId(String.valueOf(snapId));
            AgileDataSummarizeResDTO summarizeData = agileReportDataService.getAgileDataSummarize(req, event.getUser());
            // 根据快照是否总结和文字
            if(StringUtils.isNotEmpty(summarizeData.getSummarizeData()) && summarizeData.isData()){
                resultMsg = summarizeData.getSummarizeData();
            }else{
                return;
            }
        }
        try{
//            String[] answerSplit = resultMsg.split("");
//            for (int i = 0; i < resultMsg.length(); i++) {
//                Thread.sleep(15);
                EventData.EventMessage eventMessage  = EventData.EventMessage.builderAnswerTextMessage(resultMsg);
                this.sendEventDataMessage(SseAniaEventEnum.MESSAGE_DELTA.getEvent(),sseEmitter,eventMessage);
//            }
        }catch (Exception e){
            log.error("send message setSummarizeData error: {}" ,e.getMessage());
            Thread.currentThread().interrupt();
        }

    }


    /**
     * 1.0 流式输出
     * @param sendMessageDTO 数据
     * @param event 上下文
     */
    private void sendMessageToGptByV1(SendMessageDTO sendMessageDTO,AthenaMessageEvent event) {
        SseEmitter sseEmitter = event.getSseEmitter();
        EventData.EventMessage eventMessage  = EventData.EventMessage.builderAnswerMessage(sendMessageDTO);
        this.sendEventDataMessage(SseAniaEventEnum.MESSAGE_DELTA.getEvent(),sseEmitter,eventMessage);
//        this.setSummarizeData(sendMessageDTO,event);
//        try {
//            Thread.sleep(1000);
//        }catch (InterruptedException e) {
//            throw new RuntimeException(e);
//        }
        log.info("Sending Message to B娜 sse message:{}", JsonUtils.objectToString(sendMessageDTO));
        this.sendEventDataMessage(SseAniaEventEnum.CHAT_COMPLETED.getEvent(),sseEmitter,new EventData.EventMessage());
        this.sendEventDataMessage(SseAniaEventEnum.CHAT_DONE.getEvent(),sseEmitter,new EventData.EventMessage());
        sseEmitter.complete();
    }

    /**
     * 2.0 标准流式输出
     * @param sendMessageDTO 敏数发送实体
     * @param event 上下文
     */
    private void sendMessageToGptByV2(SendMessageDTO sendMessageDTO, AthenaMessageEvent event) {

        SseEmitter sseEmitter = event.getSseEmitter();
        EventData.EventMessage eventMessage  = EventData.EventMessage.builderAnswerMessageV2(sendMessageDTO);
        this.sendEventDataMessage(SseAniaEventEnum.MESSAGE_DELTA.getEvent(),sseEmitter,eventMessage);
        log.info("Sending Message to B娜 sse message:{}", JsonUtils.objectToString(sendMessageDTO));
        // 引导语句- 猜你想问/相关推荐
        EventData.EventMessage followUp  = EventData.EventMessage.builderCompletedFollowUpMessageV2(event);
        this.sendEventDataMessage(SseAniaEventEnum.MESSAGE_COMPLETED.getEvent(),sseEmitter,followUp);
        // 数据总结
        this.setSummarizeData(sendMessageDTO,event);
        this.sendEventDataMessage(SseAniaEventEnum.CHAT_COMPLETED.getEvent(),sseEmitter,new EventData.EventMessage());
        this.sendEventDataMessage(SseAniaEventEnum.CHAT_DONE.getEvent(),sseEmitter,new EventData.EventMessage());
        sseEmitter.complete();
    }

    /**
     * 发送娜娜消息
     * @param sendEvent 发送事件名称
     * @param sseEmitter sse链接
     * @param eventMessage 发送消息体
     */
    public void sendEventDataMessage(String sendEvent,SseEmitter sseEmitter,EventData.EventMessage eventMessage){
        EventData eventData = new EventData(sendEvent,eventMessage);
        if (Objects.nonNull(sseEmitter)) {
            SseEmitterUtils.send(sseEmitter, eventData);
        }
    }
}
