package com.digiwin.athena.adt.util;

import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.json.JSONNull;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.alibaba.fastjson.TypeReference;
import com.digiwin.athena.appcore.exception.BusinessException;
import com.digiwin.dap.middleware.lmc.common.serializer.*;
import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.core.JsonGenerator;
import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.*;
import com.fasterxml.jackson.databind.module.SimpleModule;
import com.fasterxml.jackson.databind.ser.std.NullSerializer;
import com.fasterxml.jackson.datatype.jsr310.JavaTimeModule;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.http.converter.json.Jackson2ObjectMapperBuilder;

import java.io.IOException;
import java.sql.Timestamp;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.util.*;

import static java.nio.charset.StandardCharsets.UTF_8;

/**
 * json utils
 */
public class JSONUtils {

    private static final Logger logger = LoggerFactory.getLogger(JSONUtils.class);

    /**
     * can use static singleton, inject: just make sure to reuse!
     */
    private static final ObjectMapper objectMapper = new ObjectMapper();
    private static final ObjectMapper newMapper = new ObjectMapper();

    static {
        // 启用美化输出
        newMapper.enable(SerializationFeature.INDENT_OUTPUT);
    }
    private JSONUtils() {
        //Feature that determines whether encountering of unknown properties, false means not analyzer unknown properties
        objectMapper.configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false).setTimeZone(TimeZone.getDefault());
    }

    public static String formatWithIndent(Object obj) throws JsonProcessingException {

        return newMapper.writeValueAsString(obj);
    }

    public static String buildFormattedString(Object data) {
        try {
            return formatWithIndent(data);
        } catch (Exception e) {
            return data.toString();
        }
    }

    /**
     * json representation of object
     *
     * @param object object
     * @return object to json string
     */
    public static String toJson(Object object) {
        try {
            return JSON.toJSONString(object, false);
        } catch (Exception e) {
            logger.error("object to json exception!", e);
        }

        return null;
    }


    /**
     * This method deserializes the specified Json into an object of the specified class. It is not
     * suitable to use if the specified class is a generic type since it will not have the generic
     * type information because of the Type Erasure feature of Java. Therefore, this method should not
     * be used if the desired type is a generic type. Note that this method works fine if the any of
     * the fields of the specified object are generics, just the object itself should not be a
     * generic type.
     *
     * @param json  the string from which the object is to be deserialized
     * @param clazz the class of T
     * @param <T>   T
     * @return an object of type T from the string
     * classOfT
     */
    public static <T> T parseObject(String json, Class<T> clazz) {
        if (StringUtils.isEmpty(json)) {
            return null;
        }

        try {
            return JSON.parseObject(json, clazz);
        } catch (Exception e) {
            logger.error("parse object exception!", e);
        }
        return null;
    }

    /**
     * deserialize
     *
     * @param src   byte array
     * @param clazz class
     * @param <T>   deserialize type
     * @return deserialize type
     */
    public static <T> T parseObject(byte[] src, Class<T> clazz) {
        if (src == null) {
            return null;
        }
        String json = new String(src, UTF_8);
        return parseObject(json, clazz);
    }

    /**
     * deserialize to specified type
     *
     * @param json          json string
     * @param typeReference
     * @param <T>           deserialize type
     * @return deserialize type
     */
    public static <T> T parseObject(String json, TypeReference<T> typeReference) {
        return Optional.ofNullable(json)
                .filter(StringUtils::isNotBlank)
                .map(param -> JSON.parseObject(param, typeReference))
                .orElse(null);
    }

    /**
     * json to list
     *
     * @param json  json string
     * @param clazz class
     * @param <T>   T
     * @return list
     */
    public static <T> List<T> toList(String json, Class<T> clazz) {
        if (StringUtils.isEmpty(json)) {
            return new ArrayList<>();
        }
        try {
            return JSON.parseArray(json, clazz);
        } catch (Exception e) {
            logger.error("JSONArray.parseArray exception!", e);
        }

        return new ArrayList<>();
    }


    /**
     * serialize to json byte
     *
     * @param obj object
     * @param <T> object type
     * @return byte array
     */
    public static <T> byte[] toJsonByteArray(T obj) {
        if (obj == null) {
            return null;
        }
        String json = "";
        try {
            json = toJsonString(obj);
        } catch (Exception e) {
            logger.error("json serialize exception.", e);
        }

        return json.getBytes(UTF_8);
    }

    /**
     * check json object valid
     *
     * @param json json
     * @return true if valid
     */
    public static boolean checkJsonValid(String json) {

        if (StringUtils.isEmpty(json)) {
            return false;
        }

        try {
            objectMapper.readTree(json);
            return true;
        } catch (IOException e) {
            logger.error("check json object valid exception!", e);
        }

        return false;
    }


    /**
     * Method for finding a JSON Object field with specified name in this
     * node or its child nodes, and returning value it has.
     * If no matching field is found in this node or its descendants, returns null.
     *
     * @param jsonNode  json node
     * @param fieldName Name of field to look for
     * @return Value of first matching node found, if any; null if none
     */
    public static String findValue(JsonNode jsonNode, String fieldName) {
        JsonNode node = jsonNode.findValue(fieldName);

        if (node == null) {
            return null;
        }

        return node.toString();
    }


    /**
     * json to map
     * <p>
     * {@link #toMap(String, Class, Class)}
     *
     * @param json json
     * @return json to map
     */
    public static Map<String, String> toMap(String json) {
        if (StringUtils.isEmpty(json)) {
            return null;
        }

        try {
            return JSON.parseObject(json, new TypeReference<HashMap<String, String>>() {
            });
        } catch (Exception e) {
            logger.error("json to map exception!", e);
        }

        return null;
    }

    /**
     * json to map
     *
     * @param json   json
     * @param classK classK
     * @param classV classV
     * @param <K>    K
     * @param <V>    V
     * @return to map
     */
    public static <K, V> Map<K, V> toMap(String json, Class<K> classK, Class<V> classV) {
        if (StringUtils.isEmpty(json)) {
            return null;
        }

        try {
            return JSON.parseObject(json, new TypeReference<HashMap<K, V>>() {
            });
        } catch (Exception e) {
            logger.error("json to map exception!", e);
        }

        return null;
    }

    /**
     * object to json string
     *
     * @param object object
     * @return json string
     */
    public static String toJsonString(Object object) {
        try {
            return JSON.toJSONString(object, false);
        } catch (Exception e) {
            throw new RuntimeException("Object json deserialization exception.", e);
        }
    }

    public static JSONObject parseObject(String text) {
        try {
            return JSON.parseObject(text);
        } catch (Exception e) {
            throw new RuntimeException("String json deserialization exception.", e);
        }
    }

    public static JSONArray parseArray(String text) {
        try {
            return JSON.parseArray(text);
        } catch (Exception e) {
            throw new RuntimeException("Json deserialization exception.", e);
        }
    }


    /**
     * json serializer
     */
    public static class JsonDataSerializer extends JsonSerializer<String> {

        @Override
        public void serialize(String value, JsonGenerator gen, SerializerProvider provider) throws IOException {
            gen.writeRawValue(value);
        }

    }

    /**
     * json data deserializer
     */
    public static class JsonDataDeserializer extends JsonDeserializer<String> {

        @Override
        public String deserialize(JsonParser p, DeserializationContext ctxt) throws IOException {
            JsonNode node = p.getCodec().readTree(p);
            return node.toString();
        }

    }

    /**
     * 深复制Map类型
     * @param maps
     * @param <T>
     * @return
     */
    public static <T> Map<String, T> deepCopyMaps(Map<String, T> maps){
        if(CollectionUtil.isEmpty(maps)){
            return new HashMap<>(1);
        }
        Map<String, T> newMaps = new HashMap<>(maps.size());
        if(maps instanceof Map){
            for(String key : maps.keySet()){
                newMaps.put(key, maps.get(key));
            }
        }
        return newMaps;
    }

    /**
     * 返回JSON 对象
     */
    public static <T> T jsonToObject(String json, Class<T> valueType) throws BusinessException {
        if (null == json) {
            return null;
        }
        try {
            return Instance.objectMapper.readValue(json, valueType);
        } catch (JsonProcessingException e) {
            throw new BusinessException(e.getMessage());
        }
    }

//    /**
//     * 返回JSON 对象
//     */
//    public static <T> T jsonToObject(String json, TypeReference<T> typeReference) {
//        if (null == json) {
//            return null;
//        }
//        try {
//            return Instance.objectMapper.readValue(json, typeReference);
//        } catch (JsonProcessingException e) {
//            throw BusinessException.create(e);
//        }
//    }

    private static class Instance {
        private static final ObjectMapper objectMapper = createObjectMapper();

        private static ObjectMapper createObjectMapper() {
            JavaTimeModule javaTimeModule = new JavaTimeModule();
            javaTimeModule.addSerializer(LocalDate.class, new LocalDateSerializer());
            javaTimeModule.addDeserializer(LocalDate.class, new LocalDateDeserializer());
            javaTimeModule.addSerializer(LocalDateTime.class, new LocalDateTimeSerializer());
            javaTimeModule.addDeserializer(LocalDateTime.class, new LocalDateTimeDeserializer());
            javaTimeModule.addSerializer(LocalTime.class, new LocalTimeSerializer());
            javaTimeModule.addDeserializer(LocalTime.class, new LocalTimeDeserializer());
            javaTimeModule.addSerializer(Timestamp.class, new TimestampSerializer());
            javaTimeModule.addDeserializer(Timestamp.class, new TimestampDeserializer());

            ObjectMapper objectMapper = Jackson2ObjectMapperBuilder.json()
                    .simpleDateFormat("yyyy-MM-dd HH:mm:ss")
                    .failOnUnknownProperties(false)
                    .modules(javaTimeModule)
                    .build();

            objectMapper.setSerializationInclusion(JsonInclude.Include.NON_NULL);
            objectMapper.configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false);
            objectMapper.configure(JsonParser.Feature.ALLOW_SINGLE_QUOTES, true);

            SimpleModule netSfJsonModule = new SimpleModule("net.sf.json");
            netSfJsonModule.addSerializer(JSONNull.class, NullSerializer.instance);
            objectMapper.registerModule(netSfJsonModule);

            return objectMapper;
        }
    }
}
