package com.digiwin.athena.adt.util.agileData;

import com.digiwin.athena.adt.agileReport.eventbus.AthenaMessageEvent;
import com.digiwin.athena.appcore.util.JsonUtils;
import com.google.common.collect.Maps;
import com.knuddels.jtokkit.Encodings;
import com.knuddels.jtokkit.api.*;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.MapUtils;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import java.util.*;
import java.util.stream.Collectors;

/**
 * 处理token量计算
 */
@Slf4j
public class AgileDataExecuteDataRecord {

    private static final EncodingRegistry encodingRegistry = Encodings.newDefaultEncodingRegistry();

    /**
     * 获取token量
     * @param data
     * @return
     */
    private static int getTokenSize(String data) {
        Encoding secondEnc = encodingRegistry.getEncodingForModel(ModelType.TEXT_EMBEDDING_ADA_002);
        return secondEnc.countTokens(data);
    }

    /**
     * 处理加工后的数据
     * @param event
     * @param adeResult
     */
    public static void executeAgileData(AthenaMessageEvent event, Map<String,Object> adeResult,int tokenThreshold){
        //非流失不处理
        if(!event.isSseMessage()){
            return;
        }
        if(MapUtils.isEmpty(adeResult)){
            return;
        }
        Map<String,Object> contextMap = MapUtils.getMap(adeResult,"context");
        //获取上下文结构
        if(MapUtils.isEmpty(contextMap)){
            return;
        }
        //获取数据内容
        Object dataObject = contextMap.get("pullData");
        if(dataObject == null){
            return;
        }
        List<Map<String,Object>> dataList = (List<Map<String, Object>>) dataObject;
        if(CollectionUtils.isEmpty(dataList)){
            return;
        }
        //按actionId分组
        Map<String, List<Map<String, Object>>> groupMap = dataList.stream()
                .collect(Collectors.groupingBy(map -> (String) map.get("actionId")));
        if(MapUtils.isEmpty(groupMap)){
            return;
        }
        //获取业务数据
        Map<String,Object> historyMap = Maps.newHashMap();
        Map<String,Object> data = Maps.newHashMap();
        Map<String,Object> metadata = Maps.newHashMap();
        for(List<Map<String, Object>> v : groupMap.values()){
            Optional<Map<String, Object>> optionalBkMap = v.stream().filter(map -> MapUtils.getObject(map,"data") != null).findFirst();
            if(optionalBkMap.isPresent()) {
                Map<String,Object> bkMap = optionalBkMap.get();
                if (MapUtils.isNotEmpty(bkMap)) {
                    data = MapUtils.getMap(bkMap, "data");
                }
            }
            Optional<Map<String, Object>> optionalMetadataMap = v.stream().filter(map -> MapUtils.getObject(map,"metadata") != null).findFirst();
            if(optionalMetadataMap.isPresent()) {
                Map<String,Object> metadataMap = optionalMetadataMap.get();
                if (MapUtils.isNotEmpty(metadataMap)) {
                    metadata = MapUtils.getMap(metadataMap, "metadata");
                }
            }
        }
        historyMap.putAll(data);
        historyMap.putAll(metadata);
        if(MapUtils.isEmpty(data)){
            return;
        }
        if(MapUtils.isEmpty(metadata)){
            return;
        }
        //token量计算
        int tokenSize = getTokenSize(JsonUtils.objectToString(historyMap));
        event.setTokenSize(tokenSize);
        if(tokenSize > tokenThreshold){
            event.setExceedTokenThreshold(true);
        } else {
            event.setExceedTokenThreshold(false);
            try {
                processData(data,metadata);
                Map<String,Object> newData = Maps.newHashMap();
                newData.put("data", String.valueOf(data));
                event.setHistoryData(newData);
            } catch (Exception e){
                log.error("转换业务数据失败：{}",e.getMessage());
            }
        }
    }

    /**
     * 加工返回的业务数据
     * @param data
     * @param metadata
     * @return
     */
    private static void processData(Map<String,Object> data,Map<String,Object> metadata){
        //获取元数据结构
        Map<String,Object> responseMetadata = MapUtils.getMap(metadata,"response");
        if(MapUtils.isEmpty(responseMetadata)){
            return;
        }
        //元数据敏数默认根节点
        Map<String,Object> responseRoot = MapUtils.getMap(responseMetadata,"data");
        if(MapUtils.isEmpty(responseRoot)){
            return;
        }
        //通过元数据结构替换业务数据名称
        changeData(layoutMetaData(responseRoot),data);
    }

    private static Map<String,String> layoutMetaData(Map<String,Object> metadataMap){
        Map<String,String> layoutMetadataMap = Maps.newHashMap();
        String dataName = MapUtils.getString(metadataMap,"data_name");
        String dataType = MapUtils.getString(metadataMap,"data_type");

        //遍历元数据结构
        if(StringUtils.pathEquals("object", dataType)){
            //元数据结构
            List<Map<String,Object>> metadataList = (List<Map<String, Object>>) MapUtils.getObject(metadataMap,"field");
            //此处进行递归遍历
            metadataList.stream().forEach(map-> layoutMetadataMap.putAll(layoutMetaData(map)));
        } else {
            String description = MapUtils.getString(metadataMap,"description");
            layoutMetadataMap.put(dataName,description);
        }
        return layoutMetadataMap;
    }

    /**
     * 转换原始数据
     * @param layoutMetadataMap
     * @param data
     */
    private static void changeData(Map<String,String> layoutMetadataMap,Map<String,Object> data){
        Map<String, Object> newData = Maps.newHashMap();
        Iterator<Map.Entry<String, Object>> iterator = data.entrySet().iterator();
        while (iterator.hasNext()) {
            Map.Entry<String, Object> entry = iterator.next(); // 确保每次调用next()前都检查hasNext()
            String dataName = entry.getKey();
            Object dataValue = entry.getValue();

            if (dataValue instanceof List) {
                List<Map<String, Object>> tempList = (List<Map<String, Object>>) dataValue;
                tempList.forEach(map -> changeData(layoutMetadataMap, map));
                newData.put(dataName, dataValue); // 如果需要保留原样则添加到新Map中
                continue;
            }

            String description = layoutMetadataMap.get(dataName);
            if (description != null) { // 确保有对应的描述
                newData.put(description, dataValue); // 添加新键值对到新Map中
            }
        }

        // 替换旧的Map为新的Map
        data.clear();
        data.putAll(newData);
    }


}
