package com.digiwin.athena.agiledataecho.controller;

import com.digiwin.athena.agiledataecho.dto.mongodb.MongoDbAddDTO;
import com.digiwin.athena.agiledataecho.dto.mongodb.MongoDbDeleteDTO;
import com.digiwin.athena.agiledataecho.dto.mongodb.MongoDbQueryDTO;
import com.digiwin.athena.agiledataecho.dto.mongodb.MongoDbUpdateDTO;
import com.digiwin.athena.appcore.util.ResponseEntityWrapper;
import com.mongodb.BasicDBObject;
import com.mongodb.DBObject;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.bson.types.ObjectId;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.data.mongodb.core.MongoTemplate;
import org.springframework.data.mongodb.core.query.Criteria;
import org.springframework.data.mongodb.core.query.Query;
import org.springframework.data.mongodb.core.query.Update;
import org.springframework.http.ResponseEntity;
import org.springframework.util.CollectionUtils;
import org.springframework.web.bind.annotation.*;

import javax.validation.Valid;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * mongoDb CURD
 *
 * @author yanfeng
 * @date 2024/10/22
 */
@RestController
@RequestMapping(value = "/api/ai/agile/echo/mongodb")
public class MongoDbController {

    @Autowired
    @Qualifier("mongoTemplate")
    private MongoTemplate mongoTemplate;

    @GetMapping("/query")
    public ResponseEntity<?> query(@Valid MongoDbQueryDTO dto) {
        // 校验口令
        if (StringUtils.isNotBlank(dto.getWordOfCommand()) && !"真嘟假嘟".equals(dto.getWordOfCommand())) {
            return ResponseEntityWrapper.wrapperFail(-1, "口令错误");
        }

        Query query = new Query();
        if (StringUtils.isNotBlank(dto.getId())) {
            query.addCriteria(Criteria.where("_id").is(dto.getId()));
        }
        if (StringUtils.isNotBlank(dto.getIncludeField())) {
            query.fields().include(dto.getIncludeField());
        }
        if (ObjectUtils.isNotEmpty(dto.getDeleted())) {
            query.addCriteria(Criteria.where("deleted").is(false));
        }

        // 查询数据库
        List<Map> mongoDbList = mongoTemplate.find(query, Map.class, dto.getCollectionName());

        if (CollectionUtils.isEmpty(mongoDbList)) {
            return ResponseEntityWrapper.wrapperOk("暂无数据");
        }

        // 处理查询结果
        List<Map<String, Object>> list = new ArrayList<>(mongoDbList.size());
        for (Map map : mongoDbList) {
            if (map.containsKey("_id")) {
                map.put("_id", String.valueOf(map.get("_id")));
            }
            list.add(new HashMap<>(map));
        }
        return ResponseEntityWrapper.wrapperOk(list);
    }

    @PostMapping("/update")
    public ResponseEntity<?> updateLoggingLevel(@RequestBody @Valid MongoDbUpdateDTO dto) {
        // 校验口令
        if (!"真嘟假嘟".equals(dto.getWordOfCommand())) {
            return ResponseEntityWrapper.wrapperFail(-1, "口令错误");
        }
        String id = MapUtils.getString(dto.getMapData(), "_id");
        if (StringUtils.isBlank(id)) {
            return ResponseEntityWrapper.wrapperFail(-1, "ID不能为空");
        }
        // 构建查询条件
        Query query = new Query(Criteria.where("_id").is(id));
        Update update = new Update();
        // 更新数据
        if (ObjectUtils.isNotEmpty(dto.getMapData())) {
            dto.getMapData().forEach(update::set);

            // 执行更新
            mongoTemplate.updateFirst(query, update, dto.getCollectionName());
        }
        // 查询并返回更新后的数据
        Map map = mongoTemplate.findById(id, Map.class, dto.getCollectionName());
        if (map == null) {
            return ResponseEntityWrapper.wrapperFail(-1, "未找到更新后的数据");
        }
        return ResponseEntityWrapper.wrapperOk(map);
    }

    @PostMapping("/add")
    public ResponseEntity<?> add(@RequestBody @Valid MongoDbAddDTO dto) {
        // 校验口令
        if (!"真嘟假嘟".equals(dto.getWordOfCommand())) {
            return ResponseEntityWrapper.wrapperFail(-1, "口令错误");
        }

        // 检查mapData是否为空
        if (ObjectUtils.isEmpty(dto.getMapData())) {
            return ResponseEntityWrapper.wrapperFail(-1, "mapData为空");
        }
        // 检查roleId是否存在并且是否重复
        String roleId = MapUtils.getString(dto.getMapData(), "roleId");
        if (StringUtils.isNotBlank(roleId)) {
            if (mongoTemplate.exists(new Query(Criteria.where("roleId").is(roleId)), dto.getCollectionName())) {
                return ResponseEntityWrapper.wrapperFail(-1, "mapData中对应roleId已存在,请勿重复添加");
            }
        }
        // 添加数据
        try {
            DBObject dbObject = new BasicDBObject(dto.getMapData());
            DBObject result = mongoTemplate.save(dbObject, dto.getCollectionName());
            return ResponseEntityWrapper.wrapperOk(result);
        } catch (Exception e) {
            return ResponseEntityWrapper.wrapperFail(-1, "添加数据失败: " + e.getMessage());
        }
    }

    @PostMapping("/delete")
    public ResponseEntity<?> delete(@RequestBody @Valid MongoDbDeleteDTO dto) {
        //比对口令
        if (StringUtils.isNotBlank(dto.getWordOfCommand())) {
            if (!"真嘟假嘟".equals(dto.getWordOfCommand())) {
                return ResponseEntityWrapper.wrapperFail(-1, "口令错误");
            }
        }
        Map dataMap = mongoTemplate.findById(new ObjectId(dto.getId()), Map.class, dto.getCollectionName());

        if (CollectionUtils.isEmpty(dataMap)) {
            return ResponseEntityWrapper.wrapperFail(-1, "此id对应数据不存在");
        }
        Query query = new Query();
        query.addCriteria(Criteria.where("_id").is(dto.getId()));
        mongoTemplate.remove(query, dto.getCollectionName());
        return ResponseEntityWrapper.wrapperOk();
    }
}