package com.digiwin.athena.agiledataecho.proxy.adt.impl;

import com.digiwin.athena.agiledataecho.app.env.EchoEnvProperties;
import com.digiwin.athena.agiledataecho.constant.ErrorCodeEnum;
import com.digiwin.athena.agiledataecho.proxy.ade.model.AgileDataEngineFileDownModel;
import com.digiwin.athena.agiledataecho.proxy.adt.AdtService;
import com.digiwin.athena.appcore.auth.AppAuthContextHolder;
import com.digiwin.athena.appcore.auth.GlobalConstant;
import com.digiwin.athena.appcore.auth.domain.AuthoredUser;
import com.digiwin.athena.appcore.domain.BaseResultDTO;
import com.digiwin.athena.appcore.util.JsonUtils;
import com.digiwin.athena.show.domain.agileDataDTO.AgileDataHooks;
import com.digiwin.athena.show.domain.agileDataDTO.AgileDataScreenDTO;
import com.digiwin.athena.show.domain.agileDataDTO.AgileDataScreenModel;
import com.digiwin.athena.show.domain.analysis.AnalysisDimensionRes;
import com.digiwin.athena.show.domain.analysis.AnalysisFluctuateRes;
import com.digiwin.athena.show.domain.board.BoardLayoutParamsDTO;
import com.digiwin.athena.show.domain.board.BoardSnapResultDTO;
import com.digiwin.athena.show.domain.board.SnapShotInfo;
import com.digiwin.athena.show.domain.filter.FilterDataDTO;
import com.digiwin.athena.show.domain.queryDefine.SnapDataDTO;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.core.ParameterizedTypeReference;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpMethod;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Service;
import org.springframework.web.client.RestTemplate;

import java.util.Map;

@Slf4j
@Service
public class AdtServiceImpl implements AdtService {

    @Autowired
    private EchoEnvProperties echoEnvProperties;

    @Autowired
    private RestTemplate restTemplate;

    @Value("${appToken:}")
    private String appToken;

    @Override
    public Map<String, Object> getSnapShotInfo(AuthoredUser user, String snapshotId, String locale) {
        String url = echoEnvProperties.getAdtUri() + "/api/ai/agile/report/getData/" + snapshotId;
        HttpHeaders headers = new HttpHeaders();
        headers.add(GlobalConstant.IAM_USER_TOKEN, user.getToken());
        headers.add(GlobalConstant.ROUTER_KEY, user.getTenantId());
        headers.add(GlobalConstant.IAM_APP_TOKEN, appToken);
        headers.add("locale", locale);
        HttpEntity httpEntity = new HttpEntity(headers);
        try {
            log.info("获取快照下载信息入参：{}", snapshotId);
            ResponseEntity<BaseResultDTO<Map<String, Object>>> response = restTemplate.exchange(url, HttpMethod.GET, httpEntity,
                    new ParameterizedTypeReference<BaseResultDTO<Map<String, Object>>>() {
                    });
            log.info("获取快照下载信息返参：{}", JsonUtils.objectToString(response.getBody().getResponse()));
            return response.getBody().getResponse();
        } catch (Exception e) {
            log.error("{}\n,请求url:{}\n, 请求token:{}\n,请求tenantId:{}\n,报错信息：{} ", ErrorCodeEnum.ADE_GET_FILE_URL_FAIL.getErrCode(),
                    url, user.getToken(), user.getTenantId(), e.getMessage());
        }
        return null;
    }

    @Override
    public Integer getChargingUserValue(AuthoredUser user) {
        String url = echoEnvProperties.getAdtUri() + "/api/ai/agile/config/getUserValue";
        HttpHeaders headers = new HttpHeaders();
        headers.add(GlobalConstant.IAM_USER_TOKEN, user.getToken());
        headers.add(GlobalConstant.ROUTER_KEY, user.getTenantId());
        headers.add(GlobalConstant.IAM_APP_TOKEN, appToken);
        HttpEntity httpEntity = new HttpEntity(headers);
        try {
            log.info("获取adt 配置表用户入参：{}", user);
            ResponseEntity<BaseResultDTO<Integer>> response = restTemplate.exchange(url, HttpMethod.GET, httpEntity,
                    new ParameterizedTypeReference<BaseResultDTO<Integer>>() {
                    });
            log.info("获取快照下载信息返参：{}", JsonUtils.objectToString(response.getBody().getResponse()));
            return response.getBody().getResponse();
        } catch (Exception e) {
            log.error("{}\n,请求url:{}\n, 请求token:{}\n,请求tenantId:{}\n,报错信息：{} ", ErrorCodeEnum.ADE_GET_USER_CONFIG_FAIL.getErrCode(),
                    url, user.getToken(), user.getTenantId(), e.getMessage());
        }
        return 0;
    }

    @Override
    public SnapDataDTO getAdtReportBySnapshotId(String reportId, String locale) {
        AuthoredUser authoredUser = AppAuthContextHolder.getContext().getAuthoredUser();
        String url = echoEnvProperties.getAdtUri() + "/api/ai/agile/report/getSnapData/" + reportId;
        try {
            HttpHeaders headers = new HttpHeaders();
            headers.add("locale", locale);
            headers.add("digi-middleware-auth-user", authoredUser.getToken());
            headers.add("proxy_token", authoredUser.getToken());
            headers.add("security-token", null);

            HttpEntity requestEntity = new HttpEntity<>(null, headers);

            ResponseEntity<BaseResultDTO<SnapDataDTO>> resultDTOResponseEntity = restTemplate.exchange(url, HttpMethod.GET, requestEntity,
                    new ParameterizedTypeReference<BaseResultDTO<SnapDataDTO>>() {
                    });
            return resultDTOResponseEntity.getBody().getResponseWithException("");
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return null;
        }
    }

    /**
     * 获取整合数据流实时数据
     *
     * @param agileDataScreenDTO
     * @param locale
     * @return
     */
    @Override
    public SnapDataDTO getAdtSyncReportMoreScene(AgileDataScreenDTO agileDataScreenDTO, String locale) {
        AuthoredUser authoredUser = AppAuthContextHolder.getContext().getAuthoredUser();
        String url = echoEnvProperties.getAdtUri() + "/api/ai/agile/report/getSyncScreenSnapData";
        try {
            HttpHeaders headers = new HttpHeaders();
            headers.add("locale", locale);
            headers.add("digi-middleware-auth-user", authoredUser.getToken());
            headers.add("proxy_token", authoredUser.getToken());
            headers.add("security-token", null);

            Map<String, Object> params = Maps.newHashMap();
            params.put("sceneCode", agileDataScreenDTO.getSceneCode());
            params.put("eocMap", agileDataScreenDTO.getEocMap());
            params.put("fixedParams", agileDataScreenDTO.getFixedParams());
            params.put("requestType", agileDataScreenDTO.getRequestType());
            HttpEntity requestEntity = new HttpEntity<>(params, headers);

            ResponseEntity<BaseResultDTO<SnapDataDTO>> resultDTOResponseEntity = restTemplate.exchange(url, HttpMethod.POST, requestEntity,
                    new ParameterizedTypeReference<BaseResultDTO<SnapDataDTO>>() {
                    });
            return resultDTOResponseEntity.getBody().getResponseWithException("");
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return null;
        }
    }

    @Override
    public SnapDataDTO getAdtReportMoreScene(String sceneCode, String locale) {
        AuthoredUser authoredUser = AppAuthContextHolder.getContext().getAuthoredUser();
        String url = echoEnvProperties.getAdtUri() + "/api/ai/agile/report/getScreenSnapData?sceneCode=" + sceneCode;
        try {
            HttpHeaders headers = new HttpHeaders();
            headers.add("locale", locale);
            headers.add("digi-middleware-auth-user", authoredUser.getToken());
            headers.add("proxy_token", authoredUser.getToken());
            headers.add("security-token", null);

            HttpEntity requestEntity = new HttpEntity<>(null, headers);

            ResponseEntity<BaseResultDTO<SnapDataDTO>> resultDTOResponseEntity = restTemplate.exchange(url, HttpMethod.GET, requestEntity,
                    new ParameterizedTypeReference<BaseResultDTO<SnapDataDTO>>() {
                    });
            return resultDTOResponseEntity.getBody().getResponseWithException("");
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return null;
        }
    }

    @Override
    public AgileDataHooks getAgileDataHooks(String type, String locale) {
        AuthoredUser authoredUser = AppAuthContextHolder.getContext().getAuthoredUser();
        String url = echoEnvProperties.getAdtUri() + "/api/ai/pageView/hooks/query?type=" + type;
        try {
            HttpHeaders headers = new HttpHeaders();
            headers.add("locale", locale);
            headers.add("digi-middleware-auth-user", authoredUser.getToken());
            headers.add("proxy_token", authoredUser.getToken());
            headers.add("security-token", null);

            HttpEntity requestEntity = new HttpEntity<>(null, headers);

            ResponseEntity<BaseResultDTO<AgileDataHooks>> resultDTOResponseEntity = restTemplate.exchange(url, HttpMethod.GET, requestEntity,
                    new ParameterizedTypeReference<BaseResultDTO<AgileDataHooks>>() {
                    });
            return resultDTOResponseEntity.getBody().getResponseWithException("");
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return null;
        }
    }

    @Override
    public AgileDataScreenModel getAgileDataScreenModel(String modelId, String locale) {
        AuthoredUser authoredUser = AppAuthContextHolder.getContext().getAuthoredUser();
        String url = echoEnvProperties.getAdtUri() + "/api/ai/screen/model/query?modelId=" + modelId;
        try {
            HttpHeaders headers = new HttpHeaders();
            headers.add("locale", locale);
            headers.add("digi-middleware-auth-user", authoredUser.getToken());
            headers.add("proxy_token", authoredUser.getToken());
            headers.add("security-token", null);

            HttpEntity requestEntity = new HttpEntity<>(null, headers);

            ResponseEntity<BaseResultDTO<AgileDataScreenModel>> resultDTOResponseEntity = restTemplate.exchange(url, HttpMethod.GET, requestEntity,
                    new ParameterizedTypeReference<BaseResultDTO<AgileDataScreenModel>>() {
                    });
            return resultDTOResponseEntity.getBody().getResponseWithException("");
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return null;
        }
    }

    @Override
    public SnapDataDTO transAgileData(String transId, String snapshotId, String locale) {
        AuthoredUser authoredUser = AppAuthContextHolder.getContext().getAuthoredUser();
        String url = echoEnvProperties.getAdtUri() + "/api/ai/agileData/trans/agileData";
        try {
            HttpHeaders headers = new HttpHeaders();
            headers.add("locale", locale);
            headers.add("digi-middleware-auth-user", authoredUser.getToken());
            headers.add("proxy_token", authoredUser.getToken());
            headers.add("security-token", AppAuthContextHolder.getContext().getSecurityToken());

            Map<String, Object> paramMap = Maps.newHashMap();
            paramMap.put("transId", transId);
            paramMap.put("snapshotId", snapshotId);
            HttpEntity requestEntity = new HttpEntity<>(paramMap, headers);

            ResponseEntity<BaseResultDTO<SnapDataDTO>> resultDTOResponseEntity = restTemplate.exchange(url, HttpMethod.POST, requestEntity,
                    new ParameterizedTypeReference<BaseResultDTO<SnapDataDTO>>() {
                    });
            return resultDTOResponseEntity.getBody().getResponseWithException("");
        } catch (Exception e) {
            log.error("获取转换数据失败，异常信息:{},异常堆栈:{}",e.getMessage(), e);
            return null;
        }
    }

    @Override
    public FilterDataDTO filterAgileData(Map<String, Object> filterParams, String locale) {
        AuthoredUser authoredUser = AppAuthContextHolder.getContext().getAuthoredUser();
        String url = echoEnvProperties.getAdtUri() + "/api/ai/agileData/filter/agileData";
        try {
            HttpHeaders headers = new HttpHeaders();
            headers.add("locale", locale);
            headers.add("digi-middleware-auth-user", authoredUser.getToken());
            headers.add("proxy_token", authoredUser.getToken());
            headers.add("security-token", null);
            HttpEntity requestEntity = new HttpEntity<>(filterParams, headers);

            ResponseEntity<BaseResultDTO<FilterDataDTO>> resultDTOResponseEntity = restTemplate.exchange(url, HttpMethod.POST, requestEntity,
                    new ParameterizedTypeReference<BaseResultDTO<FilterDataDTO>>() {
                    });
            return resultDTOResponseEntity.getBody().getResponseWithException("");
        } catch (Exception e) {
            log.error("获取筛选数据失败，异常信息:{},异常堆栈:{}",e.getMessage(), e);
            return null;
        }
    }

    /**
     * 获取看板数据
     * @param boardLayoutParamsDTO
     * @param locale
     * @return
     */
    @Override
    public BoardSnapResultDTO getBoardData(BoardLayoutParamsDTO boardLayoutParamsDTO,String locale) {
        AuthoredUser authoredUser = AppAuthContextHolder.getContext().getAuthoredUser();
        String url = echoEnvProperties.getAdtUri() + "/api/ai/agileData/v2/dataPanel/processMessage";
        try {
            HttpHeaders headers = new HttpHeaders();
            headers.add("locale", locale);
            headers.add("digi-middleware-auth-user", authoredUser.getToken());
            headers.add("proxy_token", authoredUser.getToken());
            headers.add("security-token", null);
            HttpEntity requestEntity = new HttpEntity<>(boardLayoutParamsDTO, headers);

            ResponseEntity<BaseResultDTO<BoardSnapResultDTO>> resultDTOResponseEntity = restTemplate.exchange(url, HttpMethod.POST, requestEntity,
                    new ParameterizedTypeReference<BaseResultDTO<BoardSnapResultDTO>>() {
                    });
            return resultDTOResponseEntity.getBody().getResponseWithException("");
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return null;
        }
    }

    @Override
    public AnalysisFluctuateRes getAnalysisFluctuateData(Map<String, Object> params, String locale) {
        AuthoredUser authoredUser = AppAuthContextHolder.getContext().getAuthoredUser();
        String url = echoEnvProperties.getAdtUri() + "/api/ai/agileData/v2/analysis/fluctuate";
        try {
            HttpHeaders headers = new HttpHeaders();
            headers.add("locale", locale);
            headers.add("digi-middleware-auth-user", authoredUser.getToken());
            headers.add("proxy_token", authoredUser.getToken());
            headers.add("security-token", null);
            HttpEntity requestEntity = new HttpEntity<>(params, headers);
            log.info("归因分析-数据波动，入参：{}",JsonUtils.objectToString(params));
            ResponseEntity<BaseResultDTO<AnalysisFluctuateRes>> resultDTOResponseEntity = restTemplate.exchange(url, HttpMethod.POST, requestEntity,
                    new ParameterizedTypeReference<BaseResultDTO<AnalysisFluctuateRes>>() {
                    });
            return resultDTOResponseEntity.getBody().getResponseWithException("");
        } catch (Exception e) {
            log.error("获取归因分析-数据波动失败，异常信息:{},异常堆栈:{}",e.getMessage(), e);
            return null;
        }
    }

    @Override
    public AnalysisDimensionRes getAnalysisDataDimension(Map<String, Object> params, String locale) {
        AuthoredUser authoredUser = AppAuthContextHolder.getContext().getAuthoredUser();
        String url = echoEnvProperties.getAdtUri() + "/api/ai/agileData/v2/analysis/attribution";
        try {
            HttpHeaders headers = new HttpHeaders();
            headers.add("locale", locale);
            headers.add("digi-middleware-auth-user", authoredUser.getToken());
            headers.add("proxy_token", authoredUser.getToken());
            headers.add("security-token", null);
            HttpEntity requestEntity = new HttpEntity<>(params, headers);
            log.info("归因分析-维度归因，入参：{}",JsonUtils.objectToString(params));
            ResponseEntity<BaseResultDTO<AnalysisDimensionRes>> resultDTOResponseEntity = restTemplate.exchange(url, HttpMethod.POST, requestEntity,
                    new ParameterizedTypeReference<BaseResultDTO<AnalysisDimensionRes>>() {
                    });
            return resultDTOResponseEntity.getBody().getResponseWithException("");
        } catch (Exception e) {
            log.error("获取归因分析-维度归因失败，异常信息:{},异常堆栈:{}",e.getMessage(), e);
            return null;
        }
    }


}
