package com.digiwin.athena.agiledataecho.service.aiBoard.impl;

import cn.hutool.json.JSONUtil;
import com.digiwin.athena.agiledataecho.dto.aiBoard.AIBoardLayout;
import com.digiwin.athena.agiledataecho.dto.aiBoard.AIBoardQueryReq;
import com.digiwin.athena.agiledataecho.dto.aiBoard.EchoAIBoardDTO;
import com.digiwin.athena.agiledataecho.dto.aiBoard.EchoAIBoardLayoutDTO;
import com.digiwin.athena.agiledataecho.proxy.scrumbi.ScrumBIService;
import com.digiwin.athena.agiledataecho.proxy.scrumbi.dto.ScrumBILayoutReq;
import com.digiwin.athena.agiledataecho.service.aiBoard.EchoAiBoardService;
import com.digiwin.athena.appcore.auth.domain.AuthoredUser;
import com.digiwin.athena.appcore.util.JsonUtils;
import com.digiwin.athena.show.component.gridster.GridGroupDTO;
import com.digiwin.athena.show.domain.agileDataDTO.ConvertAgileDataRelation;
import com.digiwin.athena.show.util.layout.BoardLayoutPlanner;
import com.mongodb.BasicDBObject;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.data.mongodb.core.MongoTemplate;
import org.springframework.data.mongodb.core.query.Criteria;
import org.springframework.data.mongodb.core.query.Query;
import org.springframework.data.mongodb.core.query.Update;
import org.springframework.stereotype.Service;

import java.time.LocalTime;
import java.util.List;
import java.util.Objects;

/**
 * 对话看板
 */
@Slf4j
@Service
public class EchoAiBoardServiceImpl implements EchoAiBoardService {

    @Autowired
    @Qualifier("mongoAgileDataPageViewTemplate")
    private MongoTemplate mongoAgileDataPageViewTemplate;

    @Autowired
    private ScrumBIService scrumBIService;

    private static final String MONGODB_COLLECTION_NAME = "agile_data_aiBoard";

    /**
     * 保存对话看板
     * @param user
     * @param echoAIBoardDTO
     * @return
     */
    @Override
    public Boolean saveAgileDataAIBoard(AuthoredUser user, EchoAIBoardDTO echoAIBoardDTO) {
        if (echoAIBoardDTO == null) {
            return false;
        }
        echoAIBoardDTO.setTenantId(user.getTenantId());
        echoAIBoardDTO.setUserId(user.getUserId());
        echoAIBoardDTO.setCreateTime(LocalTime.now());
        mongoAgileDataPageViewTemplate.insert(echoAIBoardDTO, MONGODB_COLLECTION_NAME);
        return true;
    }

    /**
     * 更新对话看板
     * @param user
     * @param echoAIBoardDTO
     * @return
     */
    @Override
    public Boolean updateAgileDataAIBoard(AuthoredUser user, EchoAIBoardDTO echoAIBoardDTO) {
        if (echoAIBoardDTO == null) {
            return false;
        }
        echoAIBoardDTO.setTenantId(user.getTenantId());
        echoAIBoardDTO.setUserId(user.getUserId());

        //更新规划问句
        if (CollectionUtils.isNotEmpty(echoAIBoardDTO.getQuestionInfo())) {
            //更新规划问句
            echoAIBoardDTO.getQuestionInfo().stream().forEach(questionInfo -> {
                Query query = new Query(Criteria.where("messageId").is(echoAIBoardDTO.getMessageId())
                        .and("tenantId").is(user.getTenantId())
                        .and("questionInfo.questionId").is(questionInfo.getQuestionId())
                        .and("userId").is(user.getUserId()));
                Update update = new Update();
                if (echoAIBoardDTO.getMessageStatus() != null) {
                    update.set("messageStatus", echoAIBoardDTO.getMessageStatus());
                }
                // 构建更新操作
                if (MapUtils.isNotEmpty(questionInfo.getGetDataParams())) {
                    update.set("questionInfo.$.getDataParams", questionInfo.getGetDataParams());
                }
                if (questionInfo.getAnalyzeStatus() != null) {
                    update.set("questionInfo.$.analyzeStatus", questionInfo.getAnalyzeStatus());
                }
                if (questionInfo.getAnalyzeResult() != null) {
                    update.set("questionInfo.$.analyzeResult", questionInfo.getAnalyzeResult());
                }
                if (Objects.nonNull(questionInfo.getAnalyzeMessageId())) {
                    update.set("questionInfo.$.analyzeMessageId", questionInfo.getAnalyzeMessageId());
                }

                update.set("updateTime", LocalTime.now());

                // 执行更新（更新第一个匹配的文档）
                mongoAgileDataPageViewTemplate.updateFirst(query, update, MONGODB_COLLECTION_NAME);
            });
        } else {
            //更新消息状态
            Query query = new Query(Criteria.where("messageId").is(echoAIBoardDTO.getMessageId())
                    .and("tenantId").is(user.getTenantId())
                    .and("userId").is(user.getUserId()));
            Update update = new Update();
            if (echoAIBoardDTO.getMessageStatus() != null) {
                update.set("messageStatus", echoAIBoardDTO.getMessageStatus());
            }
            if (echoAIBoardDTO.getBoardLayout() != null) {
                update.set("boardLayout", echoAIBoardDTO.getBoardLayout());
            }
            if (echoAIBoardDTO.getPreviewLayout() != null) {
                update.set("previewLayout", echoAIBoardDTO.getPreviewLayout());
            }
            if (StringUtils.isNotEmpty(echoAIBoardDTO.getBoardId())) {
                update.set("boardId", echoAIBoardDTO.getBoardId());
            }
            update.set("updateTime", LocalTime.now());
            mongoAgileDataPageViewTemplate.updateFirst(query, update, MONGODB_COLLECTION_NAME);
        }
        return true;
    }

    /**
     * 删除对话看板
     * @param user
     * @param aiBoardQueryReq
     * @return
     */
    @Override
    public Boolean deleteAgileDataAIBoard(AuthoredUser user, AIBoardQueryReq aiBoardQueryReq) {
        if (aiBoardQueryReq == null) {
            return false;
        }
        //根据消息ID查询分析规划结果
        Query query = Query.query(Criteria.where("userId").is(user.getUserId())
                .and("tenantId").is(user.getTenantId())
                .and("messageId").is(aiBoardQueryReq.getMessageId()));
        if (CollectionUtils.isNotEmpty(aiBoardQueryReq.getQuestionIds())) {
            BasicDBObject pullCondition = new BasicDBObject("questionId",
                    new BasicDBObject("$in", aiBoardQueryReq.getQuestionIds())
            );
            Update update = new Update();
            update.pull("questionInfo", pullCondition);

            mongoAgileDataPageViewTemplate.updateFirst(
                    query,
                    update,
                    MONGODB_COLLECTION_NAME
            );
        } else {
            mongoAgileDataPageViewTemplate.remove(query, MONGODB_COLLECTION_NAME);
        }
        return true;
    }

    /**
     * 查询对话看板
     * @param user
     * @param aiBoardQueryReq
     * @return
     */
    @Override
    public EchoAIBoardDTO queryAgileDataAIBoard(AuthoredUser user, AIBoardQueryReq aiBoardQueryReq) {
        if (aiBoardQueryReq == null) {
            return null;
        }
        if (aiBoardQueryReq.getMessageId() == null) {
            return null;
        }
        //根据消息ID查询分析规划结果
        Query query = Query.query(Criteria.where("userId").is(user.getUserId())
                .and("tenantId").is(user.getTenantId())
                .and("messageId").is(aiBoardQueryReq.getMessageId()));
        EchoAIBoardDTO echoAIBoardDTO = mongoAgileDataPageViewTemplate.findOne(query, EchoAIBoardDTO.class, MONGODB_COLLECTION_NAME);
        if (echoAIBoardDTO == null) {
            return null;
        }
        //筛选分析规划结果
        if (CollectionUtils.isNotEmpty(aiBoardQueryReq.getQuestionIds())) {
            if (CollectionUtils.isNotEmpty(echoAIBoardDTO.getQuestionInfo())) {
                echoAIBoardDTO.getQuestionInfo().removeIf(questionInfo ->
                        !aiBoardQueryReq.getQuestionIds().contains(questionInfo.getQuestionId()));
            }
        }
        return echoAIBoardDTO;
    }

    /**
     * 获取预览布局
     * @param user
     * @param aiBoardQueryReq
     * @return
     */
    @Override
    public List<GridGroupDTO> getPreviewLayout(AuthoredUser user, AIBoardQueryReq aiBoardQueryReq, String locale) {
        //获取当前看板数据
        log.info("预览看板布局，入参：{}", JsonUtils.objectToString(aiBoardQueryReq));
        EchoAIBoardDTO echoAIBoardDTO = this.queryAgileDataAIBoard(user, aiBoardQueryReq);
        if (echoAIBoardDTO == null) {
            log.error("获取可生成看板数据为空,{}", aiBoardQueryReq.getMessageId());
            return null;
        }
        //存在看板布局，直接返回
        if (echoAIBoardDTO.getPreviewLayout() != null) {
            //通过布局算法返回布局信息
            return ConvertAgileDataRelation.createAILayout(echoAIBoardDTO.getPreviewLayout());
        }
        //获取语义布局结果
        List<EchoAIBoardLayoutDTO> layoutDTOList = scrumBIService.getAIBoardLayout(user, ScrumBILayoutReq.buildScrumBILayoutReq(echoAIBoardDTO), locale);
        if (CollectionUtils.isEmpty(layoutDTOList)) {
            return null;
        }
        log.info("layoutDTOList:{}", JSONUtil.toJsonStr(layoutDTOList));
        AIBoardLayout aiBoardLayout = echoAIBoardDTO.getBoardLayout();
        aiBoardLayout = BoardLayoutPlanner.getAiBoardLayout(layoutDTOList, echoAIBoardDTO, aiBoardLayout);
        boolean hasLightFields = aiBoardLayout.getCharts().stream().anyMatch(c -> c.getImportance() != null || c.getRow() != null);
        if (hasLightFields) {
            int layoutCols = aiBoardLayout.getOverallLayout() != null && aiBoardLayout.getOverallLayout().getLayoutCols() != null
                    ? aiBoardLayout.getOverallLayout().getLayoutCols() : 12;
            int layoutRows = aiBoardLayout.getOverallLayout() != null && aiBoardLayout.getOverallLayout().getLayoutRows() != null
                    ? aiBoardLayout.getOverallLayout().getLayoutRows() : 18;

            List<BoardLayoutPlanner.Seed> seeds = BoardLayoutPlanner.getSeedList(aiBoardLayout, echoAIBoardDTO);
            if (!seeds.isEmpty()) {
                AIBoardLayout planned = BoardLayoutPlanner.plan(seeds, layoutCols, layoutRows);

                echoAIBoardDTO.setBoardLayout(planned);
                aiBoardLayout = planned;
            }
        }
        //生成看板布局
        List<GridGroupDTO> gridGroupDTOS = ConvertAgileDataRelation.createAILayout(aiBoardLayout);
        if (CollectionUtils.isEmpty(gridGroupDTOS)) {
            return gridGroupDTOS;
        }
        //存储看板布局
        this.updateAgileDataAIBoard(user, this.buildLayoutParams(aiBoardLayout, aiBoardQueryReq));
        return gridGroupDTOS;
    }


    /**
     * 存储预览布局参数
     * @param aiBoardLayout
     * @param aiBoardQueryReq
     * @return
     */
    private EchoAIBoardDTO buildLayoutParams(AIBoardLayout aiBoardLayout, AIBoardQueryReq aiBoardQueryReq) {
        EchoAIBoardDTO echoAIBoardDTO = new EchoAIBoardDTO();
        echoAIBoardDTO.setMessageId(aiBoardQueryReq.getMessageId());
        echoAIBoardDTO.setPreviewLayout(aiBoardLayout);
        return echoAIBoardDTO;
    }
}