package com.digiwin.athena.agiledataecho.service.imp;

import com.digiwin.athena.agiledataecho.constant.TroubleToolCodeEnum;
import com.digiwin.athena.agiledataecho.dao.EchoInformationMapper;
import com.digiwin.athena.agiledataecho.domain.EchoInformation;
import com.digiwin.athena.agiledataecho.dto.*;
import com.digiwin.athena.agiledataecho.dto.mongodb.metric.MetricCheckResDTO;
import com.digiwin.athena.agiledataecho.mongodb.domain.EchoInfoMongoData;
import com.digiwin.athena.agiledataecho.mongodb.domain.EchoSelectMultiple;
import com.digiwin.athena.agiledataecho.mongodb.domain.MetricCheckMongoData;
import com.digiwin.athena.agiledataecho.proxy.ade.AgileDataEngineService;
import com.digiwin.athena.agiledataecho.proxy.ade.model.AgileDataEngineAnalysisModel;
import com.digiwin.athena.agiledataecho.proxy.iam.UserService;
import com.digiwin.athena.agiledataecho.service.AbsEchoService;
import com.digiwin.athena.agiledataecho.service.EchoInformationService;
import com.digiwin.athena.agiledataecho.util.LogUtils;
import com.digiwin.athena.appcore.auth.domain.AuthoredUser;
import com.digiwin.athena.appcore.util.JsonUtils;
import com.digiwin.athena.appcore.util.MessageUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.data.mongodb.core.MongoTemplate;
import org.springframework.data.mongodb.core.query.Criteria;
import org.springframework.data.mongodb.core.query.Query;
import org.springframework.data.mongodb.core.query.Update;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.lang.reflect.Field;
import java.time.LocalDateTime;
import java.util.*;
import java.util.concurrent.CompletableFuture;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * @Author: SunHong
 * @Date: 2024/7/22 15:50
 * @Description: echo 日志submit and query
 */
@Service
@Slf4j
public class EchoInformationServiceImpl extends AbsEchoService implements EchoInformationService {

    @Autowired
    @Qualifier("mongoTemplate")
    private MongoTemplate mongoTemplate;

    @Resource
    private EchoInformationMapper echoInformationMapper;

    @Autowired
    private UserService userService;

    @Autowired
    private MessageUtils messageUtils;

    @Autowired
    private AgileDataEngineService agileDataEngineService;


    @Transactional
    @Override
    public void submit(EchoInformationReqDTO echoInformationReqDTO, AuthoredUser user) {
        EchoInformation echoInformation = EchoInformation.CopySubStrBean(echoInformationReqDTO);
        echoInformation.setCreateTime(new Date());

        TranslationService translationService = new TranslationService();
        // 翻译集合
        List<String> translates = new ArrayList<>();
        translates.add(echoInformation.getDescription());
        translates.add(echoInformation.getContent());
        translates.add(echoInformation.getExtend());
        translates.add(echoInformation.getAppName());
        // 执行翻译
        List<CompletableFuture<String>> translationFutures = translates.stream()
                .map(translationService::translate)
                .collect(Collectors.toList());
        CompletableFuture<Void> allTranslations = CompletableFuture.allOf(translationFutures.toArray(new CompletableFuture[0]));
        allTranslations.thenRun(() -> {
            echoInformation.setComplexDescription(translationFutures.get(0).join());
            echoInformation.setComplexContent(translationFutures.get(1).join());
            echoInformation.setComplexExtend(translationFutures.get(2).join());
            echoInformation.setComplexAppName(translationFutures.get(3).join());
            echoInformationMapper.insert(echoInformation);
        }).join();
    }

    @Override
    public List<EchoInformationResDTO> getExceptionInfoByMessageId(String messageId, AuthoredUser user) {
        List<EchoInformationResDTO> results = new ArrayList<>();
        List<EchoInformation> echoInfos = echoInformationMapper.getExceptionInfoByMessageId(messageId);
        if (CollectionUtils.isEmpty(echoInfos)) {
            return new ArrayList<>();
        }
        String userLangName = userService.getUserLangNameByUserId(
                user.getUserId(), user.getTenantId(), user.getToken());
        for (EchoInformation res : echoInfos) {
            EchoInformationResDTO echo = new EchoInformationResDTO();
            if ("zh_TW".equals(userLangName)) {
                BeanUtils.copyProperties(res, echo);
                echo.setDescription(res.getComplexDescription());
                echo.setContent(res.getComplexContent());
                echo.setExtend(res.getComplexExtend());
                echo.setAppName(res.getComplexAppName());
                results.add(echo);
            } else {
                BeanUtils.copyProperties(res, echo);
                results.add(echo);
            }
        }
        return results;
    }

    @Transactional
    @Override
    public void mongoSubmit(EchoInformationReqDTO echoInformationReqDTO, AuthoredUser user) {
        EchoInfoMongoData echoInformation = new EchoInfoMongoData();
        BeanUtils.copyProperties(echoInformationReqDTO, echoInformation);
        long messageId = echoInformationReqDTO.getMessageId();
        echoInformation.setCreateTime(new Date());
        Query query = Query.query(Criteria.where("messageId").is(messageId));
        List<EchoInfoMongoData> results = mongoTemplate.find(query, EchoInfoMongoData.class);
        if(CollectionUtils.isNotEmpty(results)){
            Update update = new Update();
            if(StringUtils.isEmpty(results.get(0).getExtend())){
                update.set("extend",echoInformation.getExtend());
            }
            if(CollectionUtils.isEmpty(results.get(0).getStepAnalysis())){
                update.set("stepAnalysis",echoInformation.getStepAnalysis());
            }
            if(CollectionUtils.isEmpty(results.get(0).getStepFilterAnalysis())){
                update.set("stepFilterAnalysis",echoInformation.getStepFilterAnalysis());
            }
            update.set("updateTime", LocalDateTime.now());
            mongoTemplate.updateFirst(query,update,EchoInfoMongoData.class);
        }else{
            mongoTemplate.insert(echoInformation);
        }

    }

    @Override
    public List<EchoInfoMongoData> getMongoDbEchoInfoByMessageId(String messageId, AuthoredUser user) {
        Query query = Query.query(Criteria.where("messageId").is(Long.valueOf(messageId)));
        return mongoTemplate.find(query, EchoInfoMongoData.class);
    }

    /**
     * 保存排查提效信息
     * 解析ade返参 组装
     *
     * @param request 入参
     * @param user    用户信息
     */
    @Transactional
    @Override
    public void sendMetricCheck(Map<String, Object> request, AuthoredUser user) {

        log.info("agiledataecho :保存排查提效入参信息：{},user:{}", JsonUtils.objectToString(request)
                , JsonUtils.objectToString(user));
        MetricCheckMongoData metricCheckMongoData = new MetricCheckMongoData();
        try {
            // 消息id
            String messageId = String.valueOf(request.get("messageId"));
            String userLang = String.valueOf(request.get("userLang"));
            // 多语言字段集合
            Map<String, String> multilingualMap = getMultilingualByLang(userLang);
            Object metricRunningList = request.get("metricRunningList");
            Object metricRunningDetail = request.get("metricRunningDetail");
            // 转换 metricRunningList
            List<Map<String, Object>> selectList = MetricCheckResDTO.replaceMetricNamesAndIds(
                    Objects.nonNull(metricRunningList) ? (List<Map<String, Object>>) metricRunningList
                            : new ArrayList<>());
            // 转换 metricRunningDetail
            Map<String, Object> selectValue = Objects.nonNull(metricRunningDetail) ?
                    (Map<String, Object>) request.get("metricRunningDetail") : new HashMap<>();

            MetricCheckResDTO metricCheckResDTO = new MetricCheckResDTO();
            List<MetricCheckResDTO.Tabs> tabs = new ArrayList<>();
            // 组装前端 语义解释
            MetricCheckResDTO.Tabs schemaExplain = MetricCheckResDTO.assembleSchemaExplain(request, multilingualMap);
            // 组装前端 运算流程
            MetricCheckResDTO.Tabs processFlow = MetricCheckResDTO.assembleProcessFlow(selectList, selectValue, multilingualMap);
            // 组装前端 取值SQL
            MetricCheckResDTO.Tabs getSQL = MetricCheckResDTO.assembleGetSQL(selectList, selectValue, multilingualMap);
            tabs.add(schemaExplain);
            tabs.add(processFlow);
            tabs.add(getSQL);
            metricCheckResDTO.setTabs(tabs);

            BeanUtils.copyProperties(metricCheckResDTO, metricCheckMongoData);
            metricCheckMongoData.setMessageId(Long.valueOf(messageId));
            metricCheckMongoData.setCreateTime(new Date());
            // 存入mongodb
            mongoTemplate.insert(metricCheckMongoData);
            LogUtils.buildAgileLog(LogUtils.MODULE_ADT, "debugInfo", LogUtils.SUCCESS,
                    JsonUtils.objectToString(request),JsonUtils.objectToString(metricCheckMongoData),
                    "");
            log.info("agiledataecho :保存排查提效出参信息 messageId：{}", messageId);
        } catch (Exception e) {
            LogUtils.buildAgileLog(LogUtils.MODULE_ADT, "debugInfo", TroubleToolCodeEnum.ECHO_901_0001.getErrCode(),
                    JsonUtils.objectToString(request),TroubleToolCodeEnum.ECHO_901_0001.getErrMsg()
                            + "错误信息:【" + JsonUtils.objectToString(metricCheckMongoData) + "】",
                    TroubleToolCodeEnum.ECHO_901_0001.getSuggestion());
            log.error("agiledataecho :保存排查提效信息报错：{},", JsonUtils.objectToString(e.getMessage()));
        }
    }

    /**
     * 多语言集合
     *
     * @param userLang 用户语言别
     * @return 多语言字段集合
     */
    public Map<String, String> getMultilingualByLang(String userLang) {
        Map<String, String> result = new HashMap<>();
        String schema = messageUtils.getMessageByLangName("message.metric.check.schema", userLang);
        String sameWord = messageUtils.getMessageByLangName("message.metric.check.sameWord", userLang);
        String processFlow = messageUtils.getMessageByLangName("message.metric.check.processFlow", userLang);
        String getSql = messageUtils.getMessageByLangName("message.metric.check.getSql", userLang);
        String appName = messageUtils.getMessageByLangName("message.metric.check.appName", userLang);
        String appCode = messageUtils.getMessageByLangName("message.metric.check.appCode", userLang);
        String valueLogic = messageUtils.getMessageByLangName("message.metric.check.valueLogic", userLang);
        String idea = messageUtils.getMessageByLangName("message.metric.check.idea", userLang);
        String sql = messageUtils.getMessageByLangName("message.metric.check.sql", userLang);
        String matchTip = messageUtils.getMessageByLangName("message.metric.title.match", userLang);
        String entityTip = messageUtils.getMessageByLangName("message.metric.title.entity", userLang);
        String description = messageUtils.getMessageByLangName("message.metric.title.matchTarget", userLang);
        result.put("schema", schema);
        result.put("processFlow", processFlow);
        result.put("sameWord", sameWord);
        result.put("getSql", getSql);
        result.put("appName", appName);
        result.put("appCode", appCode);
        result.put("valueLogic", valueLogic);
        result.put("idea", idea);
        result.put("sql", sql);
        result.put("matchTip", matchTip);
        result.put("entityTip", entityTip);
        result.put("description", description);
        return result;
    }

    @Override
    public List<MetricCheckMongoData> getMongoDbMetricInfoByMessageId(String messageId, AuthoredUser user) {
        Query query = Query.query(Criteria.where("messageId").is(Long.valueOf(messageId)));
        return mongoTemplate.find(query, MetricCheckMongoData.class);
    }

    /**
     * 根据 messageId 或者 pixId 查询 messageId 或者pixId
     * @param echoQueryBackIdReqDTO 入参实体
     * @param user 用户信息
     * @return 返回 信息
     */
    @Override
    public List<Map<String, Object>> queryPixBackendList(EchoQueryBackIdReqDTO echoQueryBackIdReqDTO, AuthoredUser user) {
        List<String> messageIds = echoQueryBackIdReqDTO.getMessageIds();
        List<String> pixIds = echoQueryBackIdReqDTO.getPixIds();
        if (CollectionUtils.isEmpty(messageIds) && CollectionUtils.isEmpty(pixIds)) {
            return new ArrayList<>();
        }
        List<Map<String, Object>> echoInfos = new ArrayList<>();
        if (CollectionUtils.isNotEmpty(messageIds)) {
            echoInfos.addAll(queryInfoByBatch(messageIds, 500,
                    echoInformationMapper::queryInfoMessageIdAndPixIdByMessageIds));
        }else{
            echoInfos.addAll(queryInfoByBatch(pixIds, 500,
                    echoInformationMapper::queryInfoMessageIdAndPixIdByPixIds));
        }
        return echoInfos;
    }

    /**
     * 获取回答分析结果
     * @param params
     * @return
     */
    @Override
    public AgileDataAnalysisDTO getAgileDataAnalysis(Map<String, Object> params,AuthoredUser user,String local) {
        if(MapUtils.isEmpty(params)){
            return null;
        }
        AgileDataAnalysisDTO agileDataAnalysisDTO = new AgileDataAnalysisDTO();
        Map<String,Object> debugInfoData = MapUtils.getMap(params,"aiParams");
        //名词匹配
        if(MapUtils.isNotEmpty(debugInfoData)) {
            // 多语言字段集合
            Map<String, String> multilingualMap = getMultilingualByLang(local);
            MetricCheckResDTO.AreaInfo areaInfo = MetricCheckResDTO.AreaInfo.assembleSameWord(debugInfoData, multilingualMap);
            if(CollectionUtils.isNotEmpty(areaInfo.getValues())){
                MetricCheckResDTO.AreaInfo.ValueInfo valueInfo = areaInfo.getValues().get(0);
                agileDataAnalysisDTO.setNounMatching(valueInfo.getValue());
            }
        }
        Map<String,Object> adeResult = MapUtils.getMap(params,"analysisParams");
        if(MapUtils.isNotEmpty(adeResult)){
            //应用编码
            agileDataAnalysisDTO.setAppCode(MapUtils.getString(adeResult,"appCode"));

            //应用名称
            List<Map<String,Object>> metricList = (List<Map<String, Object>>) MapUtils.getObject(adeResult,"metricList");
            if(CollectionUtils.isNotEmpty(metricList)){
                agileDataAnalysisDTO.setAppName(MapUtils.getString(metricList.get(0),"applicationName"));
            }

            //解题思路
            agileDataAnalysisDTO.setExplain4Gpt(MapUtils.getString(adeResult,"explain4Gpt"));

            //要素识别
            AgileDataEngineAnalysisModel analysisModel = agileDataEngineService.getAnalysisData(user,adeResult,local);
            if(analysisModel != null) {
                agileDataAnalysisDTO.setValueLogic(analysisModel.getData().toString());
            }
        }
        return agileDataAnalysisDTO;
    }

    /**
     * 根据查询数组in 超过 999 后分批查询
     * @param ids 数组集合
     * @param batchSize 指定分批数
     * @param queryFunction 查询方法
     * @return 查询集合
     */
    public List<Map<String, Object>> queryInfoByBatch(List<String> ids,
                                                      int batchSize,
                                                      Function<List<String>, List<Map<String, Object>>> queryFunction) {
        List<Map<String, Object>> result = new ArrayList<>();
        int startIndex = 0;
        int endIndex = Math.min(batchSize, ids.size());

        while (startIndex < ids.size()) {
            List<String> subIds = ids.subList(startIndex, endIndex);
            result.addAll(queryFunction.apply(subIds));
            startIndex = endIndex;
            endIndex = Math.min(startIndex + batchSize, ids.size());
        }
        return result;
    }

    @Transactional
    @Override
    public void mongoMultipleSubmit(EchoSelectMultipleReqDTO echoSelectMultipleReqDTO, AuthoredUser user) {
        EchoSelectMultiple echoSelectMultiple = new EchoSelectMultiple();
        BeanUtils.copyProperties(echoSelectMultipleReqDTO, echoSelectMultiple);
        long messageId = echoSelectMultiple.getMessageId();
        Query query = Query.query(Criteria.where("messageId").is(messageId));
        List<EchoSelectMultiple> results = mongoTemplate.find(query, EchoSelectMultiple.class);
        if(CollectionUtils.isNotEmpty(results)){
            Update update = new Update();
            update.set("selectMultiple",echoSelectMultiple.getSelectMultiple());
            mongoTemplate.updateFirst(query,update,EchoSelectMultiple.class);
        }else{
            mongoTemplate.insert(echoSelectMultiple);
        }
    }

    @Override
    public List<EchoSelectMultiple> getMongoMultipleLogByMessageId(String messageId, AuthoredUser user) {
        Query query = Query.query(Criteria.where("messageId").is(Long.valueOf(messageId)));
        return mongoTemplate.find(query, EchoSelectMultiple.class);
    }

}