package com.digiwin.athena.agiledataecho.service.imp;

import com.digiwin.athena.agiledataecho.dto.EchoDataDetailUrlDTO;
import com.digiwin.athena.agiledataecho.proxy.ade.AgileDataEngineService;
import com.digiwin.athena.agiledataecho.proxy.ade.model.AgileDataEngineFileDownModel;
import com.digiwin.athena.agiledataecho.proxy.adt.AdtService;
import com.digiwin.athena.agiledataecho.service.EchoQueryDataDetailService;
import com.digiwin.athena.appcore.auth.domain.AuthoredUser;
import com.digiwin.athena.appcore.util.JsonUtils;
import com.digiwin.athena.appcore.util.MessageUtils;
import com.digiwin.athena.appcore.util.TimeUtils;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import javax.crypto.interfaces.PBEKey;
import java.util.Date;
import java.util.List;
import java.util.Map;

/**
 * 获取数据明细下载地址
 */
@Slf4j
@Service
public class EchoQueryDataDetailServiceImpl implements EchoQueryDataDetailService {

    @Autowired
    private AdtService adtService;

    @Autowired
    private MessageUtils messageUtils;

    @Autowired
    private AgileDataEngineService agileDataEngineService;

    /**
     * 获取明细下载
     * @param user
     * @param snapshotId
     * @param locale
     * @return
     */
    @Override
    public EchoDataDetailUrlDTO getDataDetailUrl(AuthoredUser user, String snapshotId, String locale) {
        EchoDataDetailUrlDTO echoDataDetailUrlDTO = new EchoDataDetailUrlDTO();
        //通过数据快照，获取ADE入参
        Map<String, Object> snapshotInfo = adtService.getSnapShotInfo(user, snapshotId, locale);
        if(MapUtils.isEmpty(snapshotInfo)){
            buildErrorMessage(echoDataDetailUrlDTO,locale);
            return echoDataDetailUrlDTO;
        }
        Map<String,Object> adeParams = buildAdeParams(user,snapshotInfo,locale);
        String productVersion = MapUtils.getString(adeParams,"productVersion");
        log.info("明细下载 productVersion : {}",productVersion);
        AgileDataEngineFileDownModel adeFileDown;
        if("V1".equals(productVersion)){
            adeFileDown = agileDataEngineService.getFileDownUrl(user, adeParams, locale);
            log.info("明细下载v1 : {}", JsonUtils.objectToString(adeFileDown));
        }else{
            adeFileDown = agileDataEngineService.getFileDownUrlV2(user, adeParams, locale);
            log.info("明细下载v2 : {}",JsonUtils.objectToString(adeFileDown));
        }

        if(adeFileDown == null){
            buildErrorMessage(echoDataDetailUrlDTO,locale);
            return echoDataDetailUrlDTO;
        }
        echoDataDetailUrlDTO.setDataUrl(adeFileDown.getData());
        echoDataDetailUrlDTO.setResult(StringUtils.equals("0",adeFileDown.getCode()) ? true : false);
        echoDataDetailUrlDTO.setErrorMessage(adeFileDown.getMsg());
        return echoDataDetailUrlDTO;
    }

    /**
     * 构建隐藏信息
     * @param echoDataDetailUrlDTO
     */
    private void buildErrorMessage(EchoDataDetailUrlDTO echoDataDetailUrlDTO,String locale){
        echoDataDetailUrlDTO.setResult(false);
        echoDataDetailUrlDTO.setErrorMessage(messageUtils.getMessageByLangName("message.ade.dataUrl.error.msg",locale));
    }

    /**
     * 构建ADE入参
     * @param user
     * @param snapshotInfo
     * @param locale
     * @return
     */
    private Map<String,Object> buildAdeParams(AuthoredUser user,Map<String,Object> snapshotInfo,String locale){
        Map<String,Object> adeParams;
        Map<String,Object> context = MapUtils.getMap(snapshotInfo, "context");
        if(MapUtils.isEmpty(context)){
            return null;
        }
        Map<String,Object> bizParams = MapUtils.getMap(context, "bizParams");
        if(MapUtils.isEmpty(bizParams)){
            return null;
        }
        adeParams = bizParams;
        List<Map<String,Object>> pullData = (List<Map<String, Object>>) MapUtils.getObject(context,"pullData");
        if(CollectionUtils.isNotEmpty(pullData)){
            List<Map<String,Object>> metricList = Lists.newArrayList();
            pullData.stream().forEach(map -> {
                Map<String,Object> metricMap = Maps.newHashMap();
                metricMap.put("actionId",MapUtils.getString(map,"actionId"));
                metricMap.put("metricList",MapUtils.getObject(map,"metricList"));
                metricList.add(metricMap);
            });
            adeParams.put("actionMetricList",metricList);
        }
        adeParams.put("userId",user.getUserId());
        return adeParams;
    }

    private List<Map<String, Object>> getListStructure(Object obj){
        List<Map<String, Object>> resultList = Lists.newArrayList();
        if(obj instanceof List){
            resultList.addAll((List)obj);
        }
        if(obj instanceof Map){
            resultList.add((Map)obj);
        }
        return resultList;
    }
}
