package com.digiwin.athena.show.controller;

import com.digiwin.athena.agiledataecho.constant.ErrorCodeEnum;
import com.digiwin.athena.agiledataecho.domain.AgileDataStandardReq;
import com.digiwin.athena.agiledataecho.proxy.adt.AdtService;
import com.digiwin.athena.appcore.cache.CacheManager;
import com.digiwin.athena.appcore.exception.BusinessException;
import com.digiwin.athena.appcore.util.MessageUtils;
import com.digiwin.athena.appcore.util.ResponseEntityWrapper;
import com.digiwin.athena.show.analyzer.AgileReportPageDefineAnalyzer;
import com.digiwin.athena.show.assistant.AgileDataConstants;
import com.digiwin.athena.show.assistant.EchoShowConstants;
import com.digiwin.athena.show.assistant.ExecuteContext;
import com.digiwin.athena.show.domain.agileDataDTO.AgileReport;
import com.digiwin.athena.show.domain.filter.FilterDataDTO;
import com.digiwin.athena.show.domain.queryDefine.SnapDataDTO;
import com.digiwin.athena.show.domain.showDefine.DynamicAgileData;
import com.digiwin.athena.show.service.AgileReportService;
import com.digiwin.athena.show.util.layout.AgileDataCheck;
import com.google.common.collect.Maps;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.*;

import java.util.Map;

@RestController
@RequestMapping(value = "/api/ai/v1/agile/")
public class AgileReportController {

    @Autowired
    private AgileReportService agileReportService;

    @Autowired
    private AdtService adtService;

    @Autowired
    private AgileReportPageDefineAnalyzer agileReportPageDefineAnalyzer;

    @Autowired
    CacheManager cacheManager;

    /**
     * 敏数呈现DSL结构
     */
    @GetMapping("report/show/{reportId}")
    public ResponseEntity<?> createAgileReportPageTemplate(@PathVariable("reportId") String snapshotId,
                                                           @RequestParam(value = "entry", required = false) String entry,
                                                           @RequestParam(value = "module", required = false) Integer module,
                                                           @RequestParam(value = "title", required = false) String title) {
        ExecuteContext executeContext = ExecuteContext.createByHttpRequest();
        if(StringUtils.isEmpty(entry)){
            entry = "0";
        }
        //获取敏捷报表信息
        SnapDataDTO agileReportDTO = adtService.getAdtReportBySnapshotId(snapshotId, executeContext.getLocale());
        if (AgileDataCheck.sceneBasicCheck(agileReportDTO)) {
            throw BusinessException.create(ErrorCodeEnum.AGILE_REPORT_EMPTY_DATA.getErrCode(), MessageUtils.getMessageByCurrentLanguage(EchoShowConstants.getLanMap("查询结果为空，请确认查询条件是否正确。","查詢結果為空，請確認查詢條件是否正確。","no data")));
        }
        executeContext.appendAgileDataPageInfo(AgileDataConstants.REPORT_PAGE, agileReportDTO,agileReportDTO.getSnapData().getReportCode(),entry,title);
        //解析报表信息
        return ResponseEntityWrapper.wrapperOk(agileReportService.createAgileReportTemplateTable(executeContext, agileReportDTO));
    }

    /**
     * 通过模板生成敏捷报表
     */
    @PostMapping("transPresent")
    public ResponseEntity<?> transAgileData(@RequestBody Map<String,Object> transParams) {
        String snapshotId = transParams.get("snapshotId").toString();
        String transId = transParams.get("transId").toString();
        ExecuteContext executeContext = ExecuteContext.createByHttpRequest();
        executeContext.setTransId(transId);
        //获取敏捷报表信息
        SnapDataDTO agileReportDTO = adtService.transAgileData(transId, snapshotId,executeContext.getLocale());
        if (AgileDataCheck.sceneBasicCheck(agileReportDTO)) {
            throw BusinessException.create(ErrorCodeEnum.AGILE_REPORT_EMPTY_DATA.getErrCode(), MessageUtils.getMessageByCurrentLanguage(EchoShowConstants.getLanMap("查询结果为空，请确认查询条件是否正确。","查詢結果為空，請確認查詢條件是否正確。","no data")));
        }
        executeContext.appendAgileDataPageInfo(AgileDataConstants.REPORT_PAGE, agileReportDTO,null,"0");
        DynamicAgileData dynamicForm = agileReportService.createAgileReportTemplateTable(executeContext, agileReportDTO);
        Map<String,Object> result = Maps.newHashMap();
        result.put("dsl",dynamicForm);
        result.put("dataTipMessage",agileReportDTO.getSnapData().getTransTips());
        return ResponseEntityWrapper.wrapperOk(result);
    }

    /**
     * 获取过滤后的呈现信息
     * @param filterParams
     * @return
     */
    @PostMapping("show/filter")
    public ResponseEntity<?> filterAgileData(@RequestBody Map<String,Object> filterParams) {
        ExecuteContext executeContext = ExecuteContext.createByHttpRequest();
        //获取敏捷报表信息
        FilterDataDTO filterAgileData = adtService.filterAgileData(filterParams,executeContext.getLocale());
        if(StringUtils.isNotEmpty(filterAgileData.getMessage())){
            throw BusinessException.create(ErrorCodeEnum.AGILE_REPORT_EMPTY_DATA.getErrCode(),filterAgileData.getMessage());
        }
        if (AgileDataCheck.sceneBasicCheck(filterAgileData.getSnapDataDTO())) {
            throw BusinessException.create(ErrorCodeEnum.AGILE_REPORT_EMPTY_DATA.getErrCode(), MessageUtils.getMessageByCurrentLanguage(EchoShowConstants.getLanMap("查询结果为空，请确认查询条件是否正确。","查詢結果為空，請確認查詢條件是否正確。","no data")));
        }
        executeContext.appendAgileDataPageInfo(AgileDataConstants.REPORT_PAGE, filterAgileData.getSnapDataDTO(),null,"0");
        DynamicAgileData dynamicForm = agileReportService.createAgileReportTemplateTable(executeContext, filterAgileData.getSnapDataDTO());
        Map<String,Object> result = Maps.newHashMap();
        result.put("orgSnapshotId",filterAgileData.getOrgSnapshotId());
        result.put("snapshotId",filterAgileData.getSnapshotId());
        result.put("dsl",dynamicForm);
        result.put("transIds",filterAgileData.getTransIds());
        result.put("dataTipMessage",filterAgileData.getSnapDataDTO().getSnapData().getTransTips());
        result.put("showType",agileReportService.getShowType(dynamicForm,filterAgileData.getShowType()));
        return ResponseEntityWrapper.wrapperOk(result);
    }


    /**
     * 输出标准呈现结构
     * @param agileDataStandardReq
     * @return
     */
    @PostMapping("standard/agileDataStandard/show")
    public ResponseEntity<?> agileDataStandard(@RequestBody AgileDataStandardReq agileDataStandardReq) {
        return ResponseEntityWrapper.wrapperOk(agileReportService.getAgileDataStandardShow(agileDataStandardReq));
    }


    @GetMapping("clean/agileData/rest")
    public ResponseEntity<?> cleanAgileDataRest(@RequestParam(value = "redisKey", required = false) String redisKey) {
        cacheManager.delete("adt:cache:", redisKey);
        return ResponseEntityWrapper.wrapperOk(true);
    }


    /**
     * 获取场景显示类型
     * @param agileReportDTO
     * @return
     */
    @PostMapping("getSceneShowType")
    public ResponseEntity<?> getSceneShowType(@RequestBody SnapDataDTO agileReportDTO) {
        Integer showType = 0;
        ExecuteContext executeContext = ExecuteContext.createByHttpRequest();
        String entry = "0";
        if (AgileDataCheck.sceneBasicCheck(agileReportDTO)) {
            throw BusinessException.create(ErrorCodeEnum.AGILE_REPORT_EMPTY_DATA.getErrCode(), MessageUtils.getMessageByCurrentLanguage(EchoShowConstants.getLanMap("查询结果为空，请确认查询条件是否正确。","查詢結果為空，請確認查詢條件是否正確。","no data")));
        }
        executeContext.appendAgileDataPageInfo(AgileDataConstants.REPORT_PAGE, agileReportDTO,agileReportDTO.getSnapData().getReportCode(),entry,null);
        AgileReport agileReport = agileReportPageDefineAnalyzer.analysis(executeContext,agileReportDTO);
        if(agileReport != null && CollectionUtils.isNotEmpty(agileReport.getReport())){
            showType = agileReport.getReport().get(0).getReportShow();
        }
        //解析报表信息
        return ResponseEntityWrapper.wrapperOk(showType);
    }


}
