package com.digiwin.athena.show.metadata;

import cn.hutool.core.text.CharPool;
import com.digiwin.athena.show.infrastructure.meta.ApiMetadataConstants;
import com.digiwin.athena.show.metadata.Precision;
import com.fasterxml.jackson.annotation.JsonIgnore;
import lombok.AllArgsConstructor;
import lombok.Builder;
import lombok.Getter;
import lombok.NoArgsConstructor;
import lombok.Setter;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.ArrayUtils;
import org.springframework.util.StringUtils;

import java.io.Serializable;
import java.util.*;

/**
 * @description: 元数字段描述
 * @author: wangminga
 * @create: 2020-05-14 11:14
 **/
@Builder
@Setter
@Getter
@AllArgsConstructor
@NoArgsConstructor
public class MetadataField implements Serializable {
    /**
     * 字段编码
     */
    private String name;

    /**
     * 原始编码
     */
    private String originalName;

    /**
     * 所在路径
     */
    private String path;


    /**
     * 是否必填
     */
    private boolean required;

    /**
     * 字段描述
     */
    private String description;

    /**
     * 字段备注
     */
    private String remark;

    /**
     * 是否主键
     */
    private String dataKey;

    private boolean businessKey;

    /**
     * 数据类型
     */
    private String dataType;

    /**
     * 业务类型
     */
    private String businessType;

    /**
     * 元数据解释
     */
    private String explanation;

    /**
     * 取舍规则
     */
    private String decimalRule;

    /**
     * 单位转换规则
     */
    private String unit;

    /**
     * 单头，原始数据
     */
    private Object originalData;

    /**
     * 是不是简单的字段，非集合，非对象
     *
     * @return
     */
    @JsonIgnore
    public boolean isSimpleField() {
        if (ApiMetadataConstants.METADATA_OBJECT.equals(this.getDataType())) {
            return false;
        }
        return !this.isArray();
    }

    @JsonIgnore
    public boolean isObjectField() {
        return ApiMetadataConstants.METADATA_OBJECT.equals(this.getDataType());
    }

    @JsonIgnore
    public boolean isPercentField() {
        return StringUtils.hasText(this.getPercent()) && ApiMetadataConstants.PERCENT.equals(this.getPercent());
    }

    /**
     * 根据内容计算出来的合理宽度
     */
    @JsonIgnore
    private int displayWidth;

    /**
     * 长度与精度
     */
    private Precision precision;

    private Object objectData;

    /**
     * 可排序字段
     */
    private Boolean canSort;

    /**
     * 排序值
     */
    private String sort;

    /**
     * 是否可筛选
     */
    private Boolean canFilter;

    /**
     * 字段类型
     */
    private String fieldType;

    /**
     * 是否可以编辑
     */
    private boolean canEdit;

    /**
     * 标签的顺序
     */
    private Integer themeMapTagOrder;

    /**
     * 拥有的字段
     */
    private List<MetadataField> subFields;

    /**
     * 拥有的字段
     */
    private List<MetadataField> originalFields;

    @JsonIgnore
    private List<MetadataField> exportTableFields;

    /**
     * 是否多笔数据
     */
    private boolean array;


    /**
     * UIBOT-属性 来源API
     */
    @JsonIgnore
    private String sourceApi;

    @JsonIgnore
    private boolean isHead;

//
//    /**
//     *  UIBOT-属性
//     */
//    private boolean checkItem;

    /**
     * UIBOT-属性
     */
    @JsonIgnore
    private String clientAgent;

    /**
     * 如果字段有下拉列表，则存放下拉列表对应的key
     * 对应元数据中的  enum_key
     */
    private String enumKey;

    /**
     * 把子节点变成带路径的结构
     */
    @JsonIgnore
    private Map<String, MetadataField> fieldMap;

    /**
     * 父字段
     */
    @JsonIgnore
    private MetadataField parentField;

    /**
     * 是否展示百分比/千分位
     */
    private String percent;

    /**
     * 保留小数位
     */
    private Integer decimal;

    /**
     * 是否时间字段
     */
    private Boolean isTime;

    /**
     * 是否分组字段
     */
    private Boolean isGroupBy;

    /**
     * 是否主维度
     */
    private Boolean isMainDimension;


    public MetadataField findByFullPath(String fullName) {
        if (StringUtils.isEmpty(fullName)) {
            return null;
        }
        if (this.getFullName().equals(fullName)) {
            return this;
        }
        if (fullName.startsWith(this.getName() + ".")) {
            String headName = this.getName() + ".";
            String path = fullName.substring(headName.length());
            //String path = fullName.replace(this.getName()+".","");

            return findByPath(path.split("\\."));

        }
        return null;

    }

    public void addField(MetadataField metadataField) {
        if (exist(metadataField.getName())) {
            return;
        }
        this.getSubFields().add(metadataField);
        if (this.getFieldMap() == null) {
            this.setFieldMap(new HashMap<>());
        }
        this.getFieldMap().put(metadataField.getName(), metadataField);

    }

    public MetadataField getField(String name) {
        if (exist(name)) {
            return null;
        }
        return this.getFieldMap().get(name);
    }

    public boolean exist(String name) {
        if (this.getFieldMap() == null) {
            flatMetaData(this.getParentField());
        }
        return this.getFieldMap() != null && this.getFieldMap().containsKey(name);
    }

    /**
     * 创建字段
     *
     * @param sonTagFields
     * @return
     */
    public MetadataField clone(MetadataField parentMetadataField, List<MetadataField> sonTagFields) {
        MetadataField newMetadataField = new MetadataField();
        newMetadataField.setDataType(this.getDataType());
        newMetadataField.setDataKey(this.getDataKey());
        newMetadataField.setRemark(this.getRemark());
        newMetadataField.setDescription(this.getDescription());
        newMetadataField.setRequired(this.isRequired());
        newMetadataField.setName(this.getName());
        newMetadataField.setArray(this.isArray());
        newMetadataField.setCanEdit(this.isCanEdit());
        newMetadataField.setFieldType(this.getFieldType());
        newMetadataField.setSubFields(sonTagFields);
        newMetadataField.setPath(this.getPath());
        newMetadataField.setParentField(parentMetadataField);
        newMetadataField.setEnumKey(this.getEnumKey());
        newMetadataField.setThemeMapTagOrder(this.themeMapTagOrder);
        newMetadataField.setBusinessType(this.businessType);
        return newMetadataField;
    }


    public void flatMetaData(MetadataField parentField) {
        if (CollectionUtils.isEmpty(this.getSubFields())) {
            return;
        }
        Map<String, MetadataField> fieldMap = new LinkedHashMap<>();
        this.setFieldMap(fieldMap);
        this.setParentField(parentField);
        for (MetadataField subField : this.getSubFields()) {
            fieldMap.put(subField.getName(), subField);
            String path = this.getFullName();
            subField.setPath(path);
            subField.flatMetaData(this);
        }
    }

    public void flatMetaDataNotFlatPath(MetadataField parentField) {
        if (CollectionUtils.isEmpty(this.getSubFields())) {
            return;
        }
        Map<String, MetadataField> fieldMap = new HashMap<>();
        this.setFieldMap(fieldMap);
        this.setParentField(parentField);
        for (MetadataField subField : this.getSubFields()) {
            fieldMap.put(subField.getName(), subField);
            subField.setPath(subField.getPath());
            subField.flatMetaDataNotFlatPath(this);
        }
    }

    public MetadataField findByPath(String[] paths) {
        if (paths == null || paths.length == 0) {
            return null;
        }
        String firstPath = paths[0];
        if (this.getFieldMap() == null || !this.getFieldMap().containsKey(firstPath)) {
            return null;
        } else {
            if (paths.length == 1) {
                return this.getFieldMap().get(firstPath);
            }
            return this.getFieldMap().get(firstPath).findByPath(ArrayUtils.subarray(paths, 1, paths.length));
        }
    }

    private String uiBot;

    /**
     * 是否支持跳转
     * true:支持
     * false:不支持
     */
    private Boolean isNavigate = Boolean.FALSE;

    /**
     * 创建集合字段
     *
     * @param name
     * @param path
     * @param canEdit
     * @return
     */
    public static MetadataField createArrayField(String name, String description, String path, boolean canEdit) {
        MetadataField detailField = new MetadataField();
        detailField.setFieldType("object");
        detailField.setName(name);
        detailField.setDescription(description);
        detailField.setDataType("object");
        detailField.setArray(true);
        detailField.setCanEdit(canEdit);
        detailField.setSubFields(new ArrayList<>());
        detailField.setPath(StringUtils.isEmpty(path) ? name : path + "." + name);
        return detailField;
    }

    /**
     * 创建Map字段
     *
     * @param name
     * @param path
     * @param canEdit
     * @return
     */
    public static MetadataField createObjectField(String name, String description, String path, boolean canEdit) {
        MetadataField detailField = new MetadataField();
        detailField.setFieldType("object");
        detailField.setName(name);
        detailField.setDescription(description);
        detailField.setDataType("object");
        detailField.setArray(false);
        detailField.setSubFields(new ArrayList<>());
        detailField.setCanEdit(canEdit);
        detailField.setPath(StringUtils.isEmpty(path) ? name : path + "." + name);
        return detailField;
    }

    /**
     * 创建简单字段字段
     *
     * @param name
     * @param path
     * @param canEdit
     * @return
     */
    public static MetadataField createSimpleField(String name, String description, String type, String path, boolean canEdit) {
        MetadataField detailField = new MetadataField();
        detailField.setFieldType(type);
        detailField.setName(name);
        detailField.setDescription(description);
        detailField.setDataType(type);
        detailField.setArray(false);
        detailField.setCanEdit(canEdit);
        detailField.setPath(StringUtils.isEmpty(path) ? name : path + "." + name);
        return detailField;
    }


    @JsonIgnore
    public String getFullName() {
        if (StringUtils.hasText(this.getPath())) {
            return this.getPath() + CharPool.DOT + this.getName();
        } else {
            return this.getName();
        }
    }

    /**
     * 是否为单头
     *
     * @return
     */
    public boolean isSingleHead() {
        return this.getPath() == null || this.getParentField() == null;
    }

    /**
     * 是否为非单头
     *
     * @return
     */
    public boolean isNotSingleHead() {
        return !isSingleHead();
    }

    public String toString() {
        return this.getName();
    }

}
