package com.digiwin.athena.show.service;

import com.digiwin.athena.show.assistant.AgileDataConstants;
import com.digiwin.athena.show.assistant.ExecuteContext;
import com.digiwin.athena.show.domain.agileDataDTO.*;
import com.digiwin.athena.show.domain.queryDefine.SnapDataDTO;
import com.digiwin.athena.show.domain.showDefine.AgileDataBusinessEnum;
import com.digiwin.athena.show.domain.showDefine.AgileReportKMData;
import com.digiwin.athena.show.domain.showDefine.DataSourceBase;
import com.digiwin.athena.show.domain.showDefine.ThemeMapReport;
import com.digiwin.athena.show.manager.themeMap.ThemeAgileDataService;
import com.digiwin.athena.show.metadata.ApiMetadata;
import com.digiwin.athena.show.metadata.MetadataField;
import com.digiwin.athena.show.util.ApiMetadataUtil;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.util.*;
import java.util.concurrent.atomic.AtomicReference;
import java.util.stream.Collectors;
import java.util.stream.Stream;

/**
 * 敏捷数据定义
 */
@Slf4j
@Service
public abstract class AgileDataPageDefineService {

    @Autowired
    private AgileReportEChartBuildService agileReportEChartBuildService;

    @Autowired
    private ThemeAgileDataService themeAgileDataService;

    private static final List<String> AGILE_RULES = Lists.newArrayList("alertShow", "growthRate");

    public abstract String supportKey();

    /**
     * 获取敏捷数据定义
     *
     * @param executeContext
     * @param snapDataDTO
     * @return
     */
    public AgileReport analysis(ExecuteContext executeContext, SnapDataDTO snapDataDTO) {
        //获取呈现定义
        AgileReportKMData agileReportKMData = getAgileReportDefine(executeContext, snapDataDTO);

        //解析为敏捷数据定义
        return transKMDefine(agileReportKMData, snapDataDTO, executeContext);
    }

    /**
     * 转换为敏捷报表结构
     *
     * @param agileReportKMData
     * @return
     */
    private AgileReport transKMDefine(AgileReportKMData agileReportKMData, SnapDataDTO snapDataDTO, ExecuteContext executeContext) {
        AgileReport agileReport = new AgileReport();
        if (agileReportKMData == null) {
            return agileReport;
        }
        // 时间区间
        agileReport.setDimensionInterval(snapDataDTO.getSnapData().getDimensionInterval());
        //时间粒度
        agileReport.setAliasGrain(snapDataDTO.getSnapData().getAliasGrain());
        //大屏用户自定义布局信息
        this.buildScreenModel(agileReportKMData, executeContext);
        //填充区域对象
        this.buildAgileReport(agileReport, agileReportKMData);
        //获取展现区域
        this.buildThemeMapReport(agileReport, agileReportKMData, snapDataDTO, executeContext);
        return agileReport;
    }

    /**
     * 构建大屏布局信息
     * @param agileReportKMData
     * @param executeContext
     */
    protected abstract void buildScreenModel(AgileReportKMData agileReportKMData,ExecuteContext executeContext);

    /**
     * 获取展现区域配置
     *
     * @param agileReport
     * @param agileReportKMData
     * @param snapDataDTO
     * @param executeContext
     */
    private void buildThemeMapReport(AgileReport agileReport, AgileReportKMData agileReportKMData, SnapDataDTO snapDataDTO, ExecuteContext executeContext) {
        //获取展现区域
        String[] actionIdList = agileReportKMData.getActionId().split(",");
        for (String actionId : actionIdList) {

            //绑定元数据
            ApiMetadata apiMetadata = getApiMetadata(snapDataDTO, executeContext, actionId);

            //补全默认精度
            completionMetadataField(apiMetadata,snapDataDTO,actionId);

            //非大屏模式下，如果元数据为空，则不考虑渲染此场景
            if (this.judgeDefine("metadata", executeContext, apiMetadata, null, null, null)) {
                continue;
            }

            //获取原始数据，数据为空不考虑渲染此场景
            List<Map<String, Object>> agileDataListList = (List<Map<String, Object>>) snapDataDTO.getPageData().get(actionId);

            //非大屏模式下，如果数据为空，则不考虑渲染此场景
            if (this.judgeDefine("agileData", executeContext, null, agileDataListList, null, null)) {
                continue;
            }

            String sourceNamePrefix = actionId;

            TargetDTO targetDTO = this.getTargetDTO(agileReportKMData, actionId);
            if (this.judgeDefine("target", executeContext, null, null, targetDTO, null)) {
                continue;
            }

            //改元数据单头结构,后续展现形式只会存在一种，所以不用担心拆分为区域后，不同的展现形式会调用不同的schema问题
            changeApiData(apiMetadata, sourceNamePrefix);

            //改为动态区域数据结构
            changeData(agileDataListList, sourceNamePrefix, targetDTO);

            //将配置信息映射的字段名称改为动态区域后的名称
            String OriginalSourceName = "";
            Integer sourceShowType = null;
            Optional<AgileReportShow> agileReportShowOptional = agileReportKMData.getShow().stream().filter(agileShow -> Objects.equals(agileShow.getActionId(), actionId)).findFirst();
            if (agileReportShowOptional.isPresent()) {
                AgileReportShow agileReportShow = agileReportShowOptional.get();
                OriginalSourceName = agileReportShow.getTargetSource();
                sourceShowType = agileReportShow.getSourceReportShow();
            }
            changeKMDefine(agileReportKMData, actionId, OriginalSourceName, apiMetadata, targetDTO);
            //元数据名称
            String sourceName = this.getSourceName(apiMetadata.getResponseFields(), StringUtils.EMPTY, targetDTO);
            agileReport.setDataSourceBase(this.buildDataSource(actionId, sourceName));
            //获取展现形式
            AgileReportShowTypeDTO agileReportShowTypeDTO = this.bindShowType(actionId, agileReportKMData);
            //大屏筛选数据流，不走主流程渲染
            if(agileReportShowTypeDTO == null){
                continue;
            }
            //根据展现形式获取呈现区域
            Integer finalSourceShowType = sourceShowType;
            agileReportShowTypeDTO.getType().stream().forEach(agileShowType -> {
                ThemeMapReport themeMapReport = new ThemeMapReport();
                themeMapReport.setActionId(actionId);
                //绑定业务数据
                this.buildReportData(agileDataListList, themeMapReport, apiMetadata, targetDTO);
                //非大屏模式下，如果无数据，则不考虑渲染此场景
                if (this.judgeDefine("pageData", executeContext, null, null, null, themeMapReport.getPageData())) {
                    return;
                }

                //区域添加
                agileReport.getReport().add(themeMapReport);
                //应用编码
                themeMapReport.setAppCode(agileReportKMData.getAppCode());
                //轮播时间
                themeMapReport.setPageTurningFrequency(agileReportKMData.getPageTurningFrequency());
                //展现形式
                themeMapReport.setReportShow(agileShowType.getValue());
                //默认展现形式
                themeMapReport.setSourceReportShow(finalSourceShowType);
                //数据流关系
                themeMapReport.setAgileDataRelation(agileReportKMData.getActionRelation());
                //语义推荐展示信息
                agileShowType.setBiValue(this.getBiValue(agileShowType));
                //绑定元数据
                themeMapReport.setApiMetadata(apiMetadata);
                //绑定数据
                themeMapReport.setPageData(themeMapReport.getPageData());
                //多分类数据直接获取原标题
                themeMapReport.setTitle(getTitle(agileShowType, snapDataDTO, apiMetadata, themeMapReport,executeContext));
                //无论该数据源展现形式存在多种，所使用的数据源应一致
                themeMapReport.setDataSourceName(sourceName);
                //区域唯一ID
                themeMapReport.setCode(sourceName + "_" + themeMapReport.getReportShow());
                //绑定明细信息
                this.buildShow(agileReportKMData, agileReportShowTypeDTO, actionId, themeMapReport, agileShowType, snapDataDTO, executeContext);
            });
        }
    }

    private void completionMetadataField(ApiMetadata apiMetadata,SnapDataDTO snapDataDTO,String actionId){
        if(MapUtils.isNotEmpty(snapDataDTO.getPageData()) && snapDataDTO.getPageData().containsKey(actionId)){
            if(!CollectionUtils.isEmpty(apiMetadata.getResponseFields())){
                apiMetadata.getResponseFields().stream().forEach(metadataField -> completionDecimal(metadataField));
            }
        }
    }

    /**
     * 补全精度
     * @param metadataField
     */
    private void completionDecimal(MetadataField metadataField) {
        if(metadataField.isObjectField()){
            metadataField.getSubFields().stream().forEach(subField -> completionDecimal(subField));
        } else {
            //补齐精度
            if(metadataField.getDecimal() == null && "numeric".equals(metadataField.getDataType())){
                if(AgileDataBusinessEnum.isAmountType(metadataField.getBusinessType())){
                    metadataField.setDecimal(2);
                } else {
                    metadataField.setDecimal(0);
                }
            } else {
                //精度为-1。改为0
                if(metadataField.getDecimal() != null && metadataField.getDecimal() == -1){
                    metadataField.setDecimal(null);
                }
            }
        }
    }

    /**
     * 数据结果是否放行
     * @param judgeCode metadata元数据 agileData原始数据 target根节点 pageData匹配的数据
     * @param executeContext
     * @param apiMetadata
     * @param agileDataListList
     * @param targetDTO
     * @param pageData
     * @return
     */
    protected abstract Boolean judgeDefine(String judgeCode,ExecuteContext executeContext,
                                           ApiMetadata apiMetadata,List<Map<String, Object>> agileDataListList,
                                           TargetDTO targetDTO,List<Map<String,Object>> pageData);

    private String getBiValue(AgileReportShowTypeInfoDTO showType) {
        String biValue = StringUtils.EMPTY;
        //大于2为语义解析结果
        if (showType.getValue() != null && showType.getValue() > 2) {
            switch (showType.getValue()) {
                case 4:
                    biValue = "混合图";
                    break;
                case 5:
                    biValue = "曲线图";
                    break;
                case 6:
                    biValue = "饼图";
                    break;
                case 7:
                    biValue = "横条图";
                    break;
                case 8:
                    biValue = "散点图";
                    break;
                case 9:
                    biValue = "气泡图";
                    break;
                case 3:
                default:
                    biValue = "柱形图";
                    break;
            }
        }
        return biValue;
    }

    /**
     * 绑定明细，不存在明细默认展现表格
     *
     * @param agileReportKMData
     * @param actionId
     * @param themeMapReport
     */
    private void buildShow(AgileReportKMData agileReportKMData, AgileReportShowTypeDTO agileReportShowTypeDTO
            , String actionId, ThemeMapReport themeMapReport, AgileReportShowTypeInfoDTO showType, SnapDataDTO snapDataDTO, ExecuteContext executeContext) {
        //获取具体的明细定义
        Optional<AgileReportShow> agileReportShowOptional = agileReportKMData.getShow().stream().filter(agileShow -> Objects.equals(agileShow.getActionId(), actionId)).findFirst();
        AgileReportShow agileReportShow = new AgileReportShow();
        //拿定义进行封装
        if (agileReportShowOptional.isPresent()) {
            agileReportShow = agileReportShowOptional.get();
            //规则
            themeMapReport.setAgileRule(agileReportShow.getAgileRule());
        } else {
            //不存在默认使用表格
            themeMapReport.setReportShow(0);
        }
        //数据源
        themeMapReport.setDataSource(agileReportShow.getTargetSource());
        //绑定数据信息
        switch (ConvertAgileDataUtils.transType(themeMapReport.getReportShow())) {
            case "0":
                //获取表格呈现定义
                this.bindTable(themeMapReport, agileReportShow);
                break;
            case "chart":
                //获取图形呈现定义
                this.bindEChart(agileReportShow, agileReportShowTypeDTO, themeMapReport.getApiMetadata(), themeMapReport, themeMapReport.getPageData(), showType);
                break;
//            case 7:
//                //获取卡片呈现定义
//                this.bindCard(agileReportKMData, agileReportShow, themeMapReport, showType, executeContext, snapDataDTO);
//                break;
            case "2":
                //获取指标定义
                this.bindIndex(agileReportShow, themeMapReport);
                break;
            default:
                this.bindTable(themeMapReport, agileReportShow);
                break;
        }
    }

    private void bindEChart(AgileReportShow agileReportShow, AgileReportShowTypeDTO agileReportShowTypeDTO,
                            ApiMetadata apiMetadata, ThemeMapReport themeMapReport, List<Map<String, Object>> pageList, AgileReportShowTypeInfoDTO showType) {
        //获取图表呈现定义
        AgileReportECharts agileReportECharts = new AgileReportECharts();
        //构建呈现定义
        List<AgileReportECharts.EChartsData> eChartsDataList = Lists.newArrayList();
        for (AgileReportEChartDefineDTO eChartsDefine : agileReportShow.getEChart()) {
            //图表基础属性
            AgileReportECharts.EChartsData eChartsData = new AgileReportECharts.EChartsData();
            //图表展示字段
            List<AgileReportECharts.EChartsDataDefine> points = Lists.newArrayList();
            //图表展示数据
            List<AgileReportECharts.EChartsDataDefine> values = Lists.newArrayList();
            eChartsDefine.getXAxis().stream().forEach(x -> {
                AgileReportECharts.EChartsDataDefine point = new AgileReportECharts.EChartsDataDefine();
                point.setApplyToField(x.getApplyToField());
                points.add(point);
            });
            List<String> ruleFields = Lists.newArrayList();
            if (agileReportShow.getAgileRule() != null) {
                List<AgileReportRuleDTO> rules = agileReportShow.getAgileRule().stream().filter(x -> AGILE_RULES.contains(x.getType())).collect(Collectors.toList());
                if (!CollectionUtils.isEmpty(rules)) {
                    rules.stream().forEach(rule -> {
                        switch (rule.getType()) {
                            case "growthRate":
                                rule.getRules().stream().forEach(ruleField -> ruleFields.add(ruleField.getApplyToField()));
                                break;
                            default:
                                break;
                        }
                    });
                }
            }
            eChartsDefine.getSeries().stream().forEach(eChartSeries -> {
                if (!ruleFields.contains(eChartSeries.getApplyToField())) {
                    AgileReportECharts.EChartsDataDefine eChartsDataDefine = new AgileReportECharts.EChartsDataDefine();
                    eChartsDataDefine.setApplyToField(eChartSeries.getApplyToField());
                    eChartsDataDefine.setChart(eChartSeries.getChart());
                    values.add(eChartsDataDefine);
                }
            });

            //组装呈现信息
            eChartsData.setPoints(Lists.newArrayList(points));
            eChartsData.setValues(values);
            eChartsDataList.add(eChartsData);
        }
        agileReportECharts.setData(eChartsDataList);
        agileReportECharts.setTitle(themeMapReport.getTitle());

        //图表类型选择
        String chartType = "";
        //为空代表图谱定义，通过呈现图谱获取信息
        if (StringUtils.isEmpty(showType.getBiValue())) {
            Map<String, Object> eChartParams = agileReportEChartBuildService.getChartParams(agileReportShow, agileReportShowTypeDTO, apiMetadata, pageList);
            //调用KM服务获取图表类型
            chartType = themeAgileDataService.findLastNodeShapeType(eChartParams);
            if (StringUtils.isEmpty(chartType)) {
                chartType = "柱形图";
            }
        } else {
            //不为空，由语义提供呈现形式
            chartType = showType.getBiValue();
        }
        //最终输出结果为多级结构，使用多轴图用mixed定义类型
        if (!CollectionUtils.isEmpty(agileReportShow.getEChart())
                && agileReportShow.getEChart().size() > 1) {
            chartType = "混合图";
        }

        agileReportECharts.setSourceChartType(themeMapReport.getSourceReportShow());
        agileReportECharts.setChartType(AgileReportChartTypeEnum.getType(chartType));
        //图谱定义需改变原有呈现类型
        if (StringUtils.isEmpty(showType.getBiValue())) {
            themeMapReport.setReportShow(AgileReportChartTypeEnum.getValueByType(agileReportECharts.getChartType()));
            agileReportECharts.setSourceChartType(AgileReportChartTypeEnum.getValueByType(agileReportECharts.getChartType()));
        }
        themeMapReport.setECharts(agileReportECharts);
    }

    /**
     * 构建卡片定义
     *
     * @param agileReportShow
     * @param themeMapReport
     * @param showType
     */
    private void bindCard(AgileReportKMData agileReportKMData, AgileReportShow agileReportShow,
                          ThemeMapReport themeMapReport, AgileReportShowTypeInfoDTO showType, ExecuteContext executeContext, SnapDataDTO snapDataDTO) {
        AgileReportCard agileReportCard = new AgileReportCard();
        agileReportCard.setCardDefine(agileReportShow.getCard());
        //是否存在过滤器
        if (!CollectionUtils.isEmpty(showType.getFilter())) {
            //呈现字段规则
            Optional<AgileDataFilter> agileDataFilterOptional = showType.getFilter().stream().findFirst();
            if (agileDataFilterOptional.isPresent()) {
                AgileDataFilter agileDataFilter = agileDataFilterOptional.get();
                if (StringUtils.isNotEmpty(agileDataFilter.getActionId())) {
                    Optional<AgileReportRuleDTO> filterOptional = agileReportKMData.getShow().stream()
                            .filter(show -> StringUtils.equals(agileDataFilter.getActionId(), show.getActionId()))
                            .flatMap(show -> {
                                if (show.getAgileRule() == null || show.getAgileRule().isEmpty()) {
                                    return Stream.empty();  // 如果 agileRule 为空，则返回一个空的 Stream
                                } else {
                                    return show.getAgileRule().stream().filter(rule -> StringUtils.equals("filter", rule.getType()));
                                }
                            })
                            .findFirst();
                    if (filterOptional.isPresent()) {
                        agileDataFilter.setAgileReportRule(filterOptional.get());
                    }
                    agileDataFilter.setApiMetadata(getApiMetadata(snapDataDTO, executeContext, agileDataFilter.getActionId()));

                    Object filterData = snapDataDTO.getPageData().get(agileDataFilter.getActionId());
                    List<Map<String, Object>> pageDataList = Lists.newArrayList();
                    if (filterData instanceof List) {
                        pageDataList = (List<Map<String, Object>>) filterData;
                    } else {
                        pageDataList.add((Map<String, Object>) filterData);
                    }

                    List<Map<String, Object>> dataList = Lists.newArrayList();
                    MetadataField metadataField = ConvertGetAgileData.getRootMetadataField(agileDataFilter.getApiMetadata());
                    if (!CollectionUtils.isEmpty(pageDataList)) {
                        pageDataList.stream().forEach(data -> {
                            Object dataObj = data.get("data");
                            if (dataObj != null) {
                                if (dataObj instanceof Map) {
                                    Map<String, Object> dataMap = (Map<String, Object>) dataObj;
                                    Object value = dataMap.get(metadataField.getName());
                                    if (value != null) {
                                        if (value instanceof List) {
                                            dataList.addAll((List<Map<String, Object>>) value);
                                        } else {
                                            dataList.add((Map<String, Object>) value);
                                        }
                                    }
                                }

                            }
                        });
                        agileDataFilter.setPageData(dataList);
                    }
                    agileReportCard.setFilter(agileDataFilter);
                }
            }
        }
        //获取图表呈现定义
        themeMapReport.setCard(agileReportCard);
    }

    private void bindIndex(AgileReportShow agileReportShow, ThemeMapReport themeMapReport) {
        //获取图表呈现定义
        AgileReportIndex agileReportIndex = new AgileReportIndex();
        agileReportIndex.setIndexes(agileReportShow.getAgileReportIndex().getIndexes());
        themeMapReport.setAgileReportIndex(agileReportIndex);
    }

    /**
     * 表格配置
     *
     * @param themeMapReport
     * @param agileReportShow
     */
    private void bindTable(ThemeMapReport themeMapReport, AgileReportShow agileReportShow) {
        if (agileReportShow.getTable() == null) {
            themeMapReport.setEnablePaging(true);
        }
    }

    /**
     * 绑定业务数据
     *
     * @param agileDataListList
     * @param themeMapReport
     */
    private void buildReportData(List<Map<String, Object>> agileDataListList, ThemeMapReport themeMapReport, ApiMetadata apiMetadata, TargetDTO targetDTO) {
        if(CollectionUtils.isEmpty(agileDataListList)){
            return;
        }
        List<Map<String, Object>> pageList = Lists.newArrayList();
        ApiMetadataUtil.flatMetaData(apiMetadata);
        String name = this.getApiObjName(apiMetadata, targetDTO);
        ApiMetadata masterApi = ConvertGetAgileData.getMasterMetadata(name, apiMetadata);
        Map<String, MetadataField> apiMap = ConvertGetAgileData.getFiledMasterMap(name, masterApi);
        for (Map<String, Object> agileData : agileDataListList) {
            List<Map<String, Object>> dataList = Lists.newArrayList();
            List<Map<String, Object>> tempList = Lists.newArrayList();
            Object dataObj = agileData.get("data");
            if(dataObj == null){
                continue;
            }
            if (dataObj instanceof Collection) {
                tempList.addAll((Collection<? extends Map<String, Object>>) dataObj);
            } else {
                tempList.add((Map<String, Object>) dataObj);
            }
            if (!CollectionUtils.isEmpty(tempList)) {
                tempList.stream().forEach(tempData -> {
                    Map<String, Object> finalTempData = tempData;
                    apiMap.forEach((x, y) -> {
                        if (finalTempData.containsKey(x)) {
                            Object obj = finalTempData.get(x);
                            if (obj instanceof Collection) {
                                dataList.addAll((Collection<? extends Map<String, Object>>) obj);
                            } else {
                                dataList.add((Map<String, Object>) obj);
                            }
                        }
                    });
                });
            }
            pageList.addAll(dataList);
        }

        //绑定数据
        themeMapReport.setPageData(pageList);
    }

    /**
     * 获取展现形式
     *
     * @param actionId
     * @param agileReportKMData
     */
    private AgileReportShowTypeDTO bindShowType(String actionId, AgileReportKMData agileReportKMData) {
        Optional<AgileReportShowTypeDTO> showTypeDTOOptional = agileReportKMData.getShowType().stream().filter(action -> StringUtils.equals(actionId, action.getActionId())).findFirst();
        AgileReportShowTypeDTO agileReportShowTypeDTO = null;
        if (showTypeDTOOptional.isPresent()) {
            agileReportShowTypeDTO = showTypeDTOOptional.get();
            agileReportShowTypeDTO.getType().stream().forEach(showType -> {
                if(showType.getValue() == null){
                    showType.setValue(0);
                }
            });
        }
        return agileReportShowTypeDTO;
    }

    /**
     * 默认配置
     *
     * @param apiMetadata
     * @return
     */
    private Integer getShowType(ApiMetadata apiMetadata) {
        Integer showType = 0;
        if (StringUtils.isNotEmpty(apiMetadata.getShowType())) {
            showType = Integer.valueOf(apiMetadata.getShowType());
        }
        return showType;
    }

    /**
     * 通过展现形式决定所需数据源
     *
     * @param actionId
     * @return
     */
    private DataSourceBase buildDataSource(String actionId, String sourceName) {
        DataSourceBase dataSourceBase = new DataSourceBase();
        dataSourceBase.setActionId(actionId);
        return dataSourceBase;
    }

    /**
     * 获取元数据名称
     *
     * @param responseFields
     * @return
     */
    private String getSourceName(List<MetadataField> responseFields, String sourceName, TargetDTO targetDTO) {
        String name = sourceName;
        for (MetadataField metadataField : responseFields) {
            if (metadataField.isObjectField() && targetDTO != null && StringUtils.equals(targetDTO.getName(), metadataField.getOriginalName())) {
                name = metadataField.getName();
                if (!CollectionUtils.isEmpty(metadataField.getSubFields())) {
                    name = this.getSourceName(metadataField.getSubFields(), name, targetDTO.getSubTarget());
                }
            }
        }
        return name;
    }

    /**
     * 原始定义适配动态结构
     *
     * @param agileReportKMData
     * @param actionId
     * @param sourceName
     * @param apiMetadata
     * @param targetDTO
     */
    private void changeKMDefine(AgileReportKMData agileReportKMData, String actionId, String sourceName, ApiMetadata apiMetadata, TargetDTO targetDTO) {
        if(targetDTO == null){
            return;
        }
        String newSourceName = getApiObjName(apiMetadata, targetDTO);
        //标题替换
        List<AgileReportShowTypeDTO> agileReportShowTypeDTOS = agileReportKMData.getShowType().stream().filter(x -> StringUtils.equals(actionId, x.getActionId())).collect(Collectors.toList());
        if (!CollectionUtils.isEmpty(agileReportShowTypeDTOS)) {
            agileReportShowTypeDTOS.stream().forEach(showType ->
                    showType.getType().stream().forEach(type -> type.setTitleSeat(changeDefineValue(type.getTitleSeat(), sourceName, newSourceName))));
        }
        //展现形式替换
        List<AgileReportShow> agileReportShowList = agileReportKMData.getShow().stream().filter(x -> StringUtils.equals(actionId, x.getActionId())).collect(Collectors.toList());
        if (!CollectionUtils.isEmpty(agileReportShowList)) {
            agileReportShowList.stream().forEach(agileReportShow -> {
                //source替换
                agileReportShow.setTargetSource(changeDefineValue(agileReportShow.getTargetSource(), sourceName, newSourceName));
            });
        }
    }

    private String changeDefineValue(String define, String sourceName, String newSourceName) {
        if (StringUtils.isEmpty(define)) {
            return null;
        }
        return define.replaceFirst(sourceName, newSourceName);
    }

    private String getApiObjName(ApiMetadata apiMetadata, TargetDTO targetDTO) {
        String name = "";
        for (MetadataField metadataField : apiMetadata.getResponseFields()) {
            if (metadataField.isObjectField() && StringUtils.equals(targetDTO.getName(), metadataField.getOriginalName())) {
                name = metadataField.getName();
                if(!CollectionUtils.isEmpty(metadataField.getSubFields())) {
                    name += getSubApiName(metadataField.getSubFields(), targetDTO.getSubTarget());
                }
            }
        }
        return name;
    }

    private String getSubApiName(List<MetadataField> metadataFieldList, TargetDTO targetDTO) {
        String apiName = "";
        for (MetadataField metadataField : metadataFieldList) {
            if (metadataField.isObjectField() && targetDTO != null && StringUtils.equals(targetDTO.getName(), metadataField.getOriginalName())) {
                apiName = "." + metadataField.getName();
                if(!CollectionUtils.isEmpty(metadataField.getSubFields())) {
                    apiName = apiName + getSubApiName(metadataField.getSubFields(), targetDTO.getSubTarget());
                }
            }
        }
        return apiName;
    }

    /**
     * 更改业务数据的节点名称
     *
     * @param agileDataListList
     * @param newSourceName
     * @param targetDTO
     */
    private void changeData(List<Map<String, Object>> agileDataListList, String newSourceName, TargetDTO targetDTO) {
        List<Map<String, Object>> dataList = Lists.newArrayList();
        if(CollectionUtils.isEmpty(agileDataListList)){
            return;
        }
        for (Map<String, Object> agileMap : agileDataListList) {
            //data是pulling结果的固定节点
            Object obj = MapUtils.getObject(agileMap, "data");
            if(obj == null){
                continue;
            }
            List<Map<String, Object>> objList = Lists.newArrayList();
            if (obj instanceof Collection) {
                objList.addAll((Collection<? extends Map<String, Object>>) obj);
            } else {
                objList.add((Map<String, Object>) obj);
            }
            dataList.addAll(objList);
        }
        //拿定义进行封装
        if (targetDTO != null) {

            //变更原始数据
            for (int i = 0; i < dataList.size(); i++) {
                Map<String, Object> flatMap = this.changeSubData(dataList.get(i), newSourceName, 0, targetDTO);
                dataList.get(i).putAll(flatMap);
            }
        }
    }

    private Map<String, Object> changeSubData(Map<String, Object> data, String sourceName, int num, TargetDTO targetDTO) {
        Map<String, Object> flatMap = Maps.newHashMap();
        Iterator<Map.Entry<String, Object>> it = data.entrySet().iterator();
        while (it.hasNext()) {
            Map.Entry<String, Object> entry = it.next();

            if (entry.getValue() instanceof Map && StringUtils.equals(entry.getKey(), targetDTO.getName())) {
                Map<String, Object> tempMap = (Map<String, Object>) entry.getValue();
                if (targetDTO.getSubTarget() != null) {
                    flatMap.putAll(changeSubData(tempMap, sourceName, num, targetDTO.getSubTarget()));
                } else {
//                    flatMap.put(sourceName + "_" + num,tempMap);
                    flatMap.put(sourceName, tempMap);
                }
                num++;
            }
            if (entry.getValue() instanceof Collection && StringUtils.equals(entry.getKey(), targetDTO.getName())) {
//                flatMap.put(sourceName + "_" + num,entry.getValue());
                flatMap.put(sourceName, entry.getValue());
                num++;
            }
        }
        return flatMap;
    }

    /**
     * 更改单头名称，为了区分多表格schema共存问题
     *
     * @param apiMetadata
     */
    private void changeApiData(ApiMetadata apiMetadata, String newSourceName) {
        int i = 0;
        for (MetadataField metadataField : apiMetadata.getResponseFields()) {
            if (AgileDataConstants.DataType.OBJECT.equals(metadataField.getDataType())) {
                metadataField.setOriginalName(metadataField.getName());
                metadataField.setName(newSourceName);
                i++;
                if(!CollectionUtils.isEmpty(metadataField.getSubFields())) {
                    changeSubApiData(metadataField.getSubFields(), newSourceName, i);
                }
            }
        }
    }

    private void changeSubApiData(List<MetadataField> metadataFieldList, String newSourceName, int num) {
        for (MetadataField metadataField : metadataFieldList) {
            if (AgileDataConstants.DataType.OBJECT.equals(metadataField.getDataType())) {
                metadataField.setOriginalName(metadataField.getName());
//                metadataField.setName(newSourceName + "_" + num);
                metadataField.setName(newSourceName);
                num++;
                changeSubApiData(metadataField.getSubFields(), newSourceName, num);
            }
        }
    }

    /**
     * 可遍历数据源节点
     *
     * @param agileReportKMData
     * @param actionId
     * @return
     */
    private TargetDTO getTargetDTO(AgileReportKMData agileReportKMData, String actionId) {
        Optional<AgileReportShow> agileReportShowOptional = agileReportKMData.getShow().stream().filter(agileShow -> Objects.equals(agileShow.getActionId(), actionId)).findFirst();
        if (agileReportShowOptional.isPresent()) {
            AgileReportShow agileReportShow = agileReportShowOptional.get();
            if (StringUtils.isEmpty(agileReportShow.getTargetSource())) {
                return null;
            }
            return this.getTargetDataSource(agileReportShow.getTargetSource());
        }
        return null;
    }

    private TargetDTO getTargetDataSource(String targetSource) {
        TargetDTO targetDTO = new TargetDTO();
        if (!targetSource.contains(".")) {
            targetDTO.setName(targetSource);
            return targetDTO;
        }
        String targetArray[] = targetSource.split("\\.");
        TargetDTO tempTarget = new TargetDTO();
        for (int i = 0; i < targetArray.length; i++) {
            if (i != 0) {
                TargetDTO subTarget = new TargetDTO();
                subTarget.setName(targetArray[i]);
                tempTarget.setSubTarget(subTarget);
                tempTarget = subTarget;
            } else {
                targetDTO.setName(targetArray[i]);
                tempTarget = targetDTO;
            }

        }
        return targetDTO;
    }

    /**
     * 构建场景对象
     *
     * @param agileReport
     * @param agileReportKMData
     */
    private void buildAgileReport(AgileReport agileReport, AgileReportKMData agileReportKMData) {
        List<ThemeMapReport> report = Lists.newArrayList();
        agileReport.setReport(report);
        //场景标题
        agileReport.setTitle(agileReportKMData.getSceneTitle());
        //场景编码
        agileReport.setReportCode(agileReportKMData.getCode());
    }

    /**
     * 获取数据节点
     *
     * @param apiMetadata
     * @return
     */
    protected String getTargetSource(ApiMetadata apiMetadata) {
        String targetSource = "";
        //只支持单数据源的路径
        List<MetadataField> metadataFields = apiMetadata.getResponseFields();
        for (MetadataField metadataField : metadataFields) {
            if (metadataField.isObjectField()) {
                targetSource = metadataField.getName();
                if (!CollectionUtils.isEmpty(metadataField.getSubFields())) {
                    String subName = targetName(metadataField.getSubFields());
                    if (!StringUtils.isEmpty(subName)) {
                        targetSource += "." + subName;
                    }
                }
            }
        }
        //存在单身，取单身中第一层object的路劲
        return targetSource;
    }

    protected String targetName(List<MetadataField> metadataFields) {
        String targetName = "";
        for (MetadataField metadataField : metadataFields) {
            if (metadataField.isObjectField()) {
                targetName = metadataField.getName();
                if (!CollectionUtils.isEmpty(metadataField.getSubFields())) {
                    String subName = targetName(metadataField.getSubFields());
                    if (!StringUtils.isEmpty(subName)) {
                        targetName += "." + subName;
                    }
                }
            }
        }
        return targetName;
    }


    /**
     * 获取敏捷数据定义
     *
     * @param executeContext
     * @param snapDataDTO
     * @return
     */
    protected abstract AgileReportKMData getAgileReportDefine(ExecuteContext executeContext, SnapDataDTO snapDataDTO);

    /**
     * 获取元数据信息
     *
     * @param snapDataDTO
     * @param executeContext
     * @param actionId
     * @return
     */
    protected abstract ApiMetadata getApiMetadata(SnapDataDTO snapDataDTO, ExecuteContext executeContext, String actionId);


    /**
     * 获取元数据TAG
     *
     * @param snapDataDTO
     * @param executeContext
     * @param actionId
     * @return
     */
    protected abstract AgileDataTag getAgileDataTag(SnapDataDTO snapDataDTO, ExecuteContext executeContext, String actionId);

    /**
     * 获取个性化标题
     *
     * @param agileShowType
     * @param snapDataDTO
     * @param apiMetadata
     * @param themeMapReport
     * @return
     */
    protected abstract String getTitle(AgileReportShowTypeInfoDTO agileShowType, SnapDataDTO snapDataDTO, ApiMetadata apiMetadata, ThemeMapReport themeMapReport,ExecuteContext executeContext);

}
