package com.digiwin.athena.show.service.impl;

import cn.hutool.core.io.resource.ResourceUtil;
import cn.hutool.json.JSONUtil;
import com.digiwin.athena.agiledataecho.proxy.adt.AdtService;
import com.digiwin.athena.appcore.util.JsonUtils;
import com.digiwin.athena.appcore.util.MessageUtils;
import com.digiwin.athena.show.assistant.EchoShowConstants;
import com.digiwin.athena.show.component.ComponentFormat;
import com.digiwin.athena.show.domain.agileDataDTO.AgileDataTag;
import com.digiwin.athena.show.domain.agileDataDTO.ConvertAgileDataUtils;
import com.digiwin.athena.show.domain.agileDataDTO.ConvertGetAgileData;
import com.digiwin.athena.show.domain.analysis.*;
import com.digiwin.athena.show.domain.queryDefine.PullDataDTO;
import com.digiwin.athena.show.metadata.ApiMetadata;
import com.digiwin.athena.show.metadata.MetadataField;
import com.digiwin.athena.show.service.AgileDataAnalysisService;
import com.digiwin.athena.show.util.ApiMetadataUtil;
import com.github.houbb.heaven.util.lang.BeanUtil;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.stream.Collectors;

/**
 * 归因分析
 */
@Slf4j
@Service
public class AgileDataAnalysisServiceImpl implements AgileDataAnalysisService {

    @Autowired
    private AdtService adtService;

    private static final List<String> DIMENSION_METADATA_FIELD_NAME = Arrays.asList("DimensionValue","ValueChange","GrowthRate");

    private static final List<String> DIMENSION_QOQ_METADATA_FIELD_NAME = Arrays.asList("CurrentValue","PreValue","GrowthRate");

    /**
     * 获取数据波动信息
     * @param analysisFluctuateReq
     * @return
     */
    @Override
    public AnalysisDataSummaryDTO getAnalysisDataSummary(AnalysisFluctuateReq analysisFluctuateReq, String local) {
        Objects.requireNonNull(analysisFluctuateReq, "数据波动入参为空！");
        AnalysisDataSummaryDTO analysisDataSummaryDTO = new AnalysisDataSummaryDTO();
        //获取波动数据执行结果
        Map<String,Object> fluctuateParams = this.buildParams(analysisFluctuateReq,local);
        AnalysisFluctuateRes analysisFluctuateRes = adtService.getAnalysisFluctuateData(fluctuateParams, local);
        log.info("归因分析-数据波动，返回结果：{}",JsonUtils.objectToString(analysisFluctuateRes));
        if(analysisFluctuateRes == null){
            analysisDataSummaryDTO.setDataResult(false);
            analysisDataSummaryDTO.setMessage(MessageUtils.
                    getMessageByCurrentLanguage(EchoShowConstants.
                            getLanMap("无上期值，无法进行归因分析。","無上期值，無法進行歸因分析。","Without the previous period's value, attribution analysis cannot be conducted.")));
            return analysisDataSummaryDTO;
        }
        //取数异常提示
        if(StringUtils.isNotEmpty(analysisFluctuateRes.getMsg())) {
            analysisDataSummaryDTO.setDataResult(false);
            analysisDataSummaryDTO.setMessage(analysisFluctuateRes.getMsg());
            return analysisDataSummaryDTO;
        }
        //解析执行结果,生成前端呈现结构
        List<AnalysisDataSummaryDTO.AnalysisDataFluctuate> analysisDataFluctuateList = this.parseAnalysisDataFluctuate(analysisFluctuateRes);
        if(CollectionUtils.isEmpty(analysisDataFluctuateList)){
            analysisDataSummaryDTO.setDataResult(false);
            analysisDataSummaryDTO.setMessage(MessageUtils.
                    getMessageByCurrentLanguage(EchoShowConstants.
                            getLanMap("无上期值，无法进行归因分析。","無上期值，無法進行歸因分析。","Without the previous period's value, attribution analysis cannot be conducted.")));
            return analysisDataSummaryDTO;
        }
        analysisDataSummaryDTO.setFluctuateData(analysisDataFluctuateList);
        //包装归因分析入参
        analysisDataSummaryDTO.setAnalysisParams(this.buildAnalysisParams(fluctuateParams,analysisFluctuateRes));
        //输出执行结果
        analysisDataSummaryDTO.setDataResult(true);
        return analysisDataSummaryDTO;
    }

    /**
     * 获取维度归因信息
     * @param analysisFluctuateReq
     * @param local
     * @return
     */
    @Override
    public AnalysisDataDimensionDTO getAnalysisDataDimension(AnalysisFluctuateReq analysisFluctuateReq, String local) {
        Objects.requireNonNull(analysisFluctuateReq, "数据波动入参为空！");
        AnalysisDataDimensionDTO analysisDataDimensionDTO = new AnalysisDataDimensionDTO();
        AnalysisDimensionRes analysisDimensionRes = adtService.getAnalysisDataDimension(this.buildDimensionParams(analysisFluctuateReq,local), local);
        log.info("归因分析-归因分析，返回结果：{}",JsonUtils.objectToString(analysisDimensionRes));

        if(analysisDimensionRes == null || CollectionUtils.isEmpty(analysisDimensionRes.getGroupedSolutionSteps())){
            analysisDataDimensionDTO.setDataResult(false);
            analysisDataDimensionDTO.setMessage(MessageUtils.
                    getMessageByCurrentLanguage(EchoShowConstants.
                            getLanMap("无上期值，无法进行归因分析。","無上期值，無法進行歸因分析。","Without the previous period's value, attribution analysis cannot be conducted.")));
            return analysisDataDimensionDTO;
        }
        //取数异常提示
        if(StringUtils.isNotEmpty(analysisDimensionRes.getMsg())) {
            analysisDataDimensionDTO.setDataResult(false);
            analysisDataDimensionDTO.setMessage(analysisDimensionRes.getMsg());
            return analysisDataDimensionDTO;
        }
        //构建呈现结构
        List<AnalysisDataDimensionDTO.AnalysisDataAscribe> analysisDataAscribeList = this.parseAnalysisDataDimension(analysisDimensionRes);

        analysisDataDimensionDTO.setAscribe(analysisDataAscribeList);
        analysisDataDimensionDTO.setDataResult(true);

        return analysisDataDimensionDTO;
    }

    /**
     * 构建维度归因分析入参
     * @param analysisFluctuateReq
     * @param local
     * @return
     */
    private Map<String,Object> buildDimensionParams(AnalysisFluctuateReq analysisFluctuateReq,String local){
        Map<String,Object> params = Maps.newHashMap();
        params.put("snapshotId",analysisFluctuateReq.getSnapshotId());
        params.put("lang",local);
        if(MapUtils.isNotEmpty(analysisFluctuateReq.getAnalysisParams())){
            analysisFluctuateReq.getAnalysisParams().forEach((key,value)-> params.put(key,value));
        }
        return params;
    }

    /**
     * 归因分析入参
     * @param fluctuateParams
     * @param analysisFluctuateRes
     * @return
     */
    private Map<String,Object> buildAnalysisParams(Map<String,Object> fluctuateParams,AnalysisFluctuateRes analysisFluctuateRes){
        //数据波动解题思路
        fluctuateParams.put("step",analysisFluctuateRes.getStep());
        //数据波动执行结果
        fluctuateParams.put("pullData",analysisFluctuateRes.getPullData());
        //数据集ID
        fluctuateParams.put("datasetIds",analysisFluctuateRes.getDatasetIds());
        //问句
        fluctuateParams.put("question",analysisFluctuateRes.getQuestion());
        //问句日期
        fluctuateParams.put("questionDate",analysisFluctuateRes.getQuestionDate());
        //数据波动执行SQL
        fluctuateParams.put("orgOutPutSql",analysisFluctuateRes.getOrgOutPutSql());
        return fluctuateParams;
    }


    /**
     * 构建取数对象
     * @param analysisFluctuateReq
     * @param local
     * @return
     */
    private Map<String,Object> buildParams(AnalysisFluctuateReq analysisFluctuateReq,String local){
        Map<String,Object> params = BeanUtil.beanToMap(analysisFluctuateReq);
        //度量信息
        if(CollectionUtils.isNotEmpty(analysisFluctuateReq.getMeasurementData())) {
            List<Map<String,Object>> dimensionData = this.transName(analysisFluctuateReq.getMeasurementData());
            if(CollectionUtils.isNotEmpty(dimensionData)) {
                params.put("metric", dimensionData.get(0));
            }
        }
        //维度信息
        if(CollectionUtils.isNotEmpty(analysisFluctuateReq.getDimensionData())) {
            params.put("dimension", this.transName(analysisFluctuateReq.getDimensionData()));
        }
        Map<String,Object> processStreamData = analysisFluctuateReq.getProcessStreamData();
        if(MapUtils.isNotEmpty(processStreamData)) {
            Map<String,Object> questionWay = MapUtils.getMap(processStreamData, "questionWay");
            if(MapUtils.isNotEmpty(questionWay)) {
                params.put("step", MapUtils.getString(questionWay, "content"));
            }
        }
        params.put("lang",local);
        return params;
    }

    /**
     * 适配语义入参
     * @param data
     * @return
     */
    private List<Map<String,Object>> transName(List<AnalysisMetric> data){
        List<Map<String,Object>> dataList = Lists.newArrayList();
        if(CollectionUtils.isNotEmpty(data)){
            data.stream().forEach(field -> {
                Map<String,Object> transField = Maps.newHashMap();
                transField.put("field",field.getName());
                transField.put("value",field.getValue());
                dataList.add(transField);
            });
        }
        return dataList;
    }

    /**
     * 解析当前数据信息
     * @param analysisFluctuateRes
     * @return
     */
    private List<AnalysisDataSummaryDTO.AnalysisDataFluctuate> parseAnalysisDataFluctuate(AnalysisFluctuateRes analysisFluctuateRes) {
        List<AnalysisDataSummaryDTO.AnalysisDataFluctuate> analysisDataFluctuateList = Lists.newArrayList();
        if(analysisFluctuateRes != null && CollectionUtils.isNotEmpty(analysisFluctuateRes.getPullData())) {
            analysisFluctuateRes.getPullData().stream().forEach(pullData -> {
                if(MapUtils.isEmpty(pullData.getMetadata())){
                    return;
                }
                if(MapUtils.isEmpty(pullData.getData())){
                    return;
                }
                //转换为元数据信息
                ApiMetadata apiMetadata = ConvertGetAgileData.transPullData(pullData.getMetadata(),pullData.getActionId());
                if(apiMetadata == null) {
                    return;
                }
                ApiMetadataUtil.flatMetaData(apiMetadata);
                List<MetadataField> responseFields = ConvertGetAgileData.getTargetFiled("data",apiMetadata);
                List<Map<String,Object>> dataList = ConvertGetAgileData.trnasData(pullData.getData());
                if(CollectionUtils.isEmpty(dataList)){
                    return;
                }
                //包装呈现规则
                this.completionDataTag(pullData.getDataTag(),apiMetadata);
                //转换为呈现结构
                responseFields = responseFields.stream().filter(metadataField -> DIMENSION_QOQ_METADATA_FIELD_NAME.contains(metadataField.getName())).collect(Collectors.toList());
                if(CollectionUtils.isEmpty(dataList)){
                    return;
                }
                responseFields.stream().forEach(metadataField -> {
                    AnalysisDataSummaryDTO.AnalysisDataFluctuate fluctuate = new AnalysisDataSummaryDTO.AnalysisDataFluctuate();
                    fluctuate.setDescription(metadataField.getDescription());
                    fluctuate.setFormat(this.buildFormat(metadataField));
                    dataList.stream().forEach(dataMap -> {
                        fluctuate.setValue(dataMap.get(metadataField.getName()));
                        analysisDataFluctuateList.add(fluctuate);
                    });
                });


            });
        }
        return analysisDataFluctuateList;
    }

    /**
     * 包装呈现结构
     * @param dataTag
     * @param apiMetadata
     */
    private void completionDataTag(Map<String,Object> dataTag,ApiMetadata apiMetadata){
        if(MapUtils.isEmpty(dataTag)){
            return;
        }
        AgileDataTag agileDataTag = JsonUtils.jsonToObject(JsonUtils.objectToString(dataTag), AgileDataTag.class);
        MetadataField rootMetadataField = ConvertGetAgileData.getRootMetadataField(apiMetadata);
        Map<String,MetadataField> metadataMap = ConvertGetAgileData.getAgileData(rootMetadataField.getName(),apiMetadata);
        agileDataTag.getFields().stream().forEach(valueField -> {
            String field = MapUtils.getString(valueField,"field",StringUtils.EMPTY);
            MetadataField metadataField = metadataMap.get(field);
            if(metadataField != null){
                metadataField.setBusinessType(MapUtils.getString(valueField,"businessType",StringUtils.EMPTY));
                metadataField.setDecimalRule(MapUtils.getString(valueField,"decimalRule"));
                metadataField.setExplanation(MapUtils.getString(valueField,"explanation",StringUtils.EMPTY));
                metadataField.setUnit(MapUtils.getString(valueField,"dataUnit"));
            }
        });
    }

    /**
     * 生成归因分析页签结构
     * @param analysisDimensionRes
     * @return
     */
    private List<AnalysisDataDimensionDTO.AnalysisDataAscribe> parseAnalysisDataDimension(AnalysisDimensionRes analysisDimensionRes) {
        List<AnalysisDataDimensionDTO.AnalysisDataAscribe> analysisDataAscribes = Lists.newArrayList();
        if(analysisDimensionRes != null && CollectionUtils.isNotEmpty(analysisDimensionRes.getGroupedSolutionSteps())) {
            analysisDimensionRes.getGroupedSolutionSteps().stream().forEach(step -> {
                AnalysisDataDimensionDTO.AnalysisDataAscribe analysisDataAscribe = new AnalysisDataDimensionDTO.AnalysisDataAscribe();
                //页签名称
                analysisDataAscribe.setDimensionName(MapUtils.getString(step,"dimension"));
                Map<String,Object> forwardMap = MapUtils.getMap(step,"forward",MapUtils.EMPTY_MAP);
                Map<String,Object> reverseMap = MapUtils.getMap(step,"reverse",MapUtils.EMPTY_MAP);
                analysisDataAscribe.setForward(this.parseAnalysisDataProgress(forwardMap));
                analysisDataAscribe.setReverse(this.parseAnalysisDataProgress(reverseMap));
                analysisDataAscribes.add(analysisDataAscribe);
            });
        }
        return analysisDataAscribes;
    }

    /**
     * 生成进度条结构
     * @param progressMap
     * @return
     */
    private List<AnalysisDataDimensionDTO.AnalysisDataProgress> parseAnalysisDataProgress(Map<String,Object> progressMap) {
        List<AnalysisDataDimensionDTO.AnalysisDataProgress> progressList = Lists.newArrayList();
        Map<String,Object> metadataMap = MapUtils.getMap(progressMap,"metadata",MapUtils.EMPTY_MAP);
        Map<String,Object> dataMap = MapUtils.getMap(progressMap,"data",MapUtils.EMPTY_MAP);
        Map<String,Object> dataTag = MapUtils.getMap(progressMap,"dataTag",MapUtils.EMPTY_MAP);
        if(MapUtils.isEmpty(metadataMap)){
            return progressList;
        }
        if(MapUtils.isEmpty(dataMap)){
            return progressList;
        }
        //转换为元数据信息
        ApiMetadata apiMetadata = ConvertGetAgileData.transPullData(metadataMap,StringUtils.EMPTY);
        if(apiMetadata == null) {
            return progressList;
        }
        ApiMetadataUtil.flatMetaData(apiMetadata);
        List<MetadataField> responseFields = ConvertGetAgileData.getTargetFiled("data",apiMetadata);
        List<Map<String,Object>> dataList = ConvertGetAgileData.trnasData(dataMap);
        if(CollectionUtils.isEmpty(dataList)){
            return progressList;
        }
        //包装呈现规则
        this.completionDataTag(dataTag,apiMetadata);
        //构建进度条信息
        dataList.stream().forEach(data -> {
            AnalysisDataDimensionDTO.AnalysisDataProgress progress = new AnalysisDataDimensionDTO.AnalysisDataProgress();
            DIMENSION_METADATA_FIELD_NAME.stream().forEach(metadataFieldName -> {
                MetadataField metadataField = responseFields.stream().filter(field -> StringUtils.equals(field.getName(), metadataFieldName)).findFirst().get();
                if(metadataField != null){
                    switch (metadataFieldName){
                        //维度名称
                        case "DimensionValue":
                            AnalysisDataDimensionDTO.AnalysisDataDimensionData description = new AnalysisDataDimensionDTO.AnalysisDataDimensionData();
                            description.setValue(MapUtils.getObject(data,metadataFieldName,StringUtils.EMPTY));
                            description.setFormat(this.buildFormat(metadataField));
                            progress.setDescription(description);
                            break;
                        //维度值
                        case "ValueChange":
                            AnalysisDataDimensionDTO.AnalysisDataDimensionData dimensionValue = new AnalysisDataDimensionDTO.AnalysisDataDimensionData();
                            dimensionValue.setValue(MapUtils.getObject(data,metadataFieldName,StringUtils.EMPTY));
                            dimensionValue.setFormat(this.buildFormat(metadataField));
                            progress.setFieldValue(dimensionValue);
                            break;
                        //维度占比
                        case "GrowthRate":
                            AnalysisDataDimensionDTO.AnalysisDataDimensionData dimensionAccount = new AnalysisDataDimensionDTO.AnalysisDataDimensionData();
                            dimensionAccount.setValue(MapUtils.getObject(data,metadataFieldName,StringUtils.EMPTY));
                            dimensionAccount.setFormat(this.buildFormat(metadataField));
                            progress.setAccount(dimensionAccount);
                            break;
                        default:
                            break;
                    }
                }
            });
            progressList.add(progress);
        });

        return progressList;
    }

    /**
     * 构建数据标签
     * @param metadataField
     */
    private ComponentFormat buildFormat(MetadataField metadataField) {
        ComponentFormat componentFormat = new ComponentFormat();
        if(metadataField.getDecimal() != null && metadataField.getDecimal() == -1){
            metadataField.setDecimal(null);
        }
        componentFormat.setDecimal(metadataField.getDecimal());
        componentFormat.setPercent(metadataField.getPercent());
        componentFormat.setBusinessType(metadataField.getBusinessType());
        componentFormat.setDecimalRule(ConvertAgileDataUtils.getDecimalRule(metadataField.getDecimalRule()));
        componentFormat.setUnit(ConvertAgileDataUtils.getUnitRule(metadataField.getUnit()));
        return componentFormat;
    }
}
