package com.digiwin.athena.show.service.impl;

import com.digiwin.athena.appcore.util.JsonUtils;
import com.digiwin.athena.show.domain.BuildContext;
import com.digiwin.athena.show.domain.agileDataDTO.AgileDataAliasGrainDTO;
import com.digiwin.athena.show.domain.agileDataDTO.AgileDataDefine;
import com.digiwin.athena.show.domain.agileDataDTO.AgileDimensionInterval;
import com.digiwin.athena.show.domain.agileDataDTO.AgileReport;
import com.digiwin.athena.show.domain.showDefine.ThemeMapReport;
import com.digiwin.athena.show.metadata.MetadataField;
import com.digiwin.athena.show.service.AgileDataCompensateDataService;
import com.google.common.collect.Lists;
import digiwin.chartsdk.beans.sdk.chart.ChartBaseSeries;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.springframework.stereotype.Service;

import java.time.LocalDate;
import java.time.format.DateTimeFormatter;
import java.time.format.DateTimeParseException;
import java.util.*;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicReference;
import java.util.stream.Collectors;

/**
 * 补偿业务数据
 */
@Slf4j
@Service
public class AgileDataCompensateDataServiceImpl implements AgileDataCompensateDataService {

    private static final DateTimeFormatter DATE_FORMATTER = DateTimeFormatter.ofPattern("yyyy-MM-dd");

    private static final String[] POSSIBLE_DEFAULT_PATTERN = {"yyyy","yyyy-MM","yyyyMM","yyyy/MM", "yyyy.MM","M",
            "MM","yyyy-MM","MM'月'"};

    private static final String[] POSSIBLE_YEAR_PATTERN = {"yyyy"};

    private static final String[] POSSIBLE_MONTH_PATTERN = {"M","MM","yyyy-MM","yyyyMM","yyyy/MM", "yyyy.MM","yyyy-MM","MM'月'"};

    private static final String[] POSSIBLE_DAY_PATTERN = {"d","dd","yyyy-MM-d","yyyy-MM-dd","yyyyMMdd","yyyy.MM.dd","yyyy/MM/dd"};

    private static final String[] POSSIBLE_WEEK_PATTERN = {"w","ww","yyyy-w","yyyy-ww","yyyy.w","yyyy/w"};

    private static final String[] POSSIBLE_QUARTER_PATTERN = {"Q","yyyy-Q","yyyy-MM","yyyyMM","yyyy/MM", "yyyy.MM","MM","yyyy-MM"};

    private static final List<String> GRAIN_ARRAY = Arrays.asList("y","Q","M","W","d");

    private static final List<String> GRAIN_NAME = Arrays.asList("年","月","日","周","季度","週","季");

    /**
     * 趋势图数据补全
     *
     * @param chartBaseSeriesList
     * @param buildContext
     * @param themeMapReport
     * @return
     */
    @Override
    public List<Map<String, Object>> compensateChartLineAgileData(List<ChartBaseSeries> chartBaseSeriesList, BuildContext buildContext, List<Map<String, Object>> datas,
                                                                  ThemeMapReport themeMapReport, String field) {
        List<Map<String, Object>> compensationData = Lists.newArrayList();
        Object dateType = datas.get(0).get(field);
        List<String> dateList = this.getIntervalNumByIntervalData(buildContext.getAgileReport(), dateType,field,this.getDateFormat(buildContext,datas,field));
        if (datas.size() <= dateList.size() && CollectionUtils.isNotEmpty(dateList)) {
            compensationData = compensationDataByParams(datas, dateList, chartBaseSeriesList);
        }
        return compensationData;
    }

    /**
     * 获取日期格式化信息
     * @param datas
     * @param field
     * @return
     */
    private String getDateFormat(BuildContext buildContext,List<Map<String, Object>> datas,String field) {
        if(CollectionUtils.isEmpty(datas)){
            return StringUtils.EMPTY;
        }
        AgileDataAliasGrainDTO grainDTO = null;
        String link = buildContext.getExecuteContext().getLink();
        if(CollectionUtils.isNotEmpty(buildContext.getAgileReport().getAliasGrain())){
            grainDTO = buildContext.getAgileReport().getAliasGrain().stream().filter(aliasGrain -> StringUtils.equals(aliasGrain.getAlias(),field)).findFirst().orElse(null);
        }
        //指标架构，未传递粒度，不进行补全
        if(StringUtils.equals("2",link) && grainDTO == null){
            return StringUtils.EMPTY;
        }
        AgileDataAliasGrainDTO finalGrainDTO = grainDTO;
        AtomicReference<String> choseFormat = new AtomicReference<>(StringUtils.EMPTY);
        AtomicInteger index = new AtomicInteger(-1);
        datas.stream().forEach(data -> {
            Object dateType = data.get(field);
            //查询数据中的单位（ADE生成）
            String grain  = GRAIN_NAME.stream().filter(s -> dateType.toString().contains(s)).findFirst().orElse(null);
            //去除数据中的单位，适配表达式
            String formatType = dateType.toString();
            if(StringUtils.isNotEmpty(grain)) {
                int lastIndex = formatType.lastIndexOf(grain);
                if (lastIndex != -1) {
                    formatType = formatType.substring(0, lastIndex);
                }
            }
            //查找当前数据的时间格式
            String[] formatArray = choseFormatArray(finalGrainDTO);
            String format = analyzeFormat(formatType, finalGrainDTO);
            int fieldIndex = Arrays.asList(formatArray).indexOf(format);
            if(index.get() == -1 || fieldIndex < index.get()){
                choseFormat.set(format);
                index.set(fieldIndex);
            }
        });
        return choseFormat.get();
    }


    /**
     * 根据入参返回时间区间
     * @param agileReport 入参
     * @return 时间区间
     */
    public List<String> getIntervalNumByIntervalData(AgileReport agileReport, Object dateType, String field, String format) {
        if(dateType == null){
            return new ArrayList<>();
        }
        if (agileReport == null || agileReport.getDimensionInterval() == null) {
            return new ArrayList<>();
        }
        String intervalJson = JsonUtils.objectToString(agileReport.getDimensionInterval());
        log.info("getIntervalNumByIntervalData is param:{}", intervalJson);
        AgileDimensionInterval dimensionInterval = JsonUtils.jsonToObject(intervalJson, AgileDimensionInterval.class);
        if (CollectionUtils.isEmpty(dimensionInterval.getData())) {
            return new ArrayList<>();
        }
        String param = dimensionInterval.getData().get(0).getInterval();
        if (param == null || !param.contains("#")) {
            log.error("Invalid interval format: {}", param);
            return new ArrayList<>();
        }
        String[] dates = param.split("#");
        if (dates.length != 2) {
            log.error("Interval does not contain two dates: {}", param);
            return new ArrayList<>();
        }
        AgileDataAliasGrainDTO grainDTO = null;
        if(CollectionUtils.isNotEmpty(agileReport.getAliasGrain())){
            grainDTO = agileReport.getAliasGrain().stream().filter(aliasGrain -> StringUtils.equals(aliasGrain.getAlias(),field)).findFirst().orElse(null);
        }
        LocalDate startDate;
        LocalDate endDate;
        try {
            startDate = LocalDate.parse(dates[0], DATE_FORMATTER);
            endDate = LocalDate.parse(dates[1], DATE_FORMATTER);
        } catch (Exception e) {
            log.error("Error parsing dates: ", e);
            return new ArrayList<>();
        }
        List<String> months = new ArrayList<>();
        //按补齐区间获取完整日期链
        while (!startDate.isAfter(endDate)) {
            try {
                //查询数据中的单位（ADE生成）
                String grain  = GRAIN_NAME.stream().filter(s -> dateType.toString().contains(s)).findFirst().orElse(null);
                //时间格式转换为空（非标准日期），不处理
                if(StringUtils.isEmpty(format)){
                    months = Lists.newArrayList();
                    break;
                }
                //补齐的数据（默认补齐格式为yyyy-MM-dd），按业务数据给予的日期格式进行转换
                String transDate = startDate.format(DateTimeFormatter.ofPattern(Objects.requireNonNull(format)));
                //将补齐的日期输出（带上ADE转换的单位）
                if(StringUtils.isNotEmpty(grain)){
                    transDate += grain == null ? StringUtils.EMPTY : grain;
                }
                months.add(transDate);
                //按日期粒度递增日期
                startDate = incrementalDate(startDate, grainDTO);
            } catch (Exception e){
                log.error("补齐敏捷数据日期异常：{}",e.getMessage());
                break;
            }

        }
        return months;
    }


    /**
     * 根据月份补齐数据
     * @param params 数据
     * @param intervalList 月份数据
     * @return
     */
    private List<Map<String, Object>> compensationDataByParams(List<Map<String, Object>> params,
                                                               List<String> intervalList,
                                                               List<ChartBaseSeries> chartBaseSeriesList) {
        String xKey = chartBaseSeriesList.get(0).getPoints().get(0).getName();
        List<Map<String, Object>> result = new ArrayList<>();
        Map<String, Map<String, Object>> collectMap = params.stream()
                .collect(Collectors.toMap(
                        map -> String.valueOf(map.get(xKey)),
                        map -> map,
                        (existing, replacement) -> existing
                ));

        for (String monthString : intervalList) {
            Map<String, Object> map = new HashMap<>();
            Map<String, Object> monthData = collectMap.getOrDefault(monthString, null);
            if(monthData == null){
                map.put(xKey,monthString);
                chartBaseSeriesList.stream().forEach(chartBaseSeries -> chartBaseSeries.getValues().stream().forEach(value -> map.put(value.getName(),0D)));
                result.add(map);
            }else{
                result.add(monthData);
            }
        }
        return result;
    }

    /**
     * 分析数据的日期格式
     *
     * @param date
     * @return
     */

    public static String analyzeFormat(Object date,AgileDataAliasGrainDTO aliasGrain) {
        String formatResult = StringUtils.EMPTY;
        for (String pattern : choseFormatArray(aliasGrain)) {
            try {
                DateTimeFormatter.ofPattern(pattern).parse(date.toString());
                formatResult = pattern;
            } catch (DateTimeParseException e) {
                log.error("analyzeFormat error:{}", e.getMessage());
            }
        }
        return formatResult;
    }

    /**
     * 合适的日期格式
     * @param aliasGrain
     * @return
     */
    public static String[] choseFormatArray(AgileDataAliasGrainDTO aliasGrain){
        if(aliasGrain == null || StringUtils.isEmpty(aliasGrain.getAlias()) ||
                !GRAIN_ARRAY.contains(aliasGrain.getGrain())) {
            return POSSIBLE_DEFAULT_PATTERN;
        }
        switch (aliasGrain.getGrain()){
            case "y" :
                return POSSIBLE_YEAR_PATTERN;
            case "d" :
                return POSSIBLE_DAY_PATTERN;
            case "W" :
                return POSSIBLE_WEEK_PATTERN;
            case "Q" :
                return POSSIBLE_QUARTER_PATTERN;
            case "M" :
            default:
                return POSSIBLE_MONTH_PATTERN;
        }
    }

    /**
     * 按粒度进行递增
     * @param startDate
     * @param aliasGrain
     */
    public static LocalDate incrementalDate(LocalDate startDate,AgileDataAliasGrainDTO aliasGrain){
        LocalDate incrementalDate = null;
        if(aliasGrain == null || StringUtils.isEmpty(aliasGrain.getAlias()) ||
                !GRAIN_ARRAY.contains(aliasGrain.getGrain())) {
            return startDate.plusMonths(1).withDayOfMonth(1);
        }
        switch (aliasGrain.getGrain()){
            //按年递增
            case "y" :
                incrementalDate = startDate.plusYears(1);
                break;
            //按日递增
            case "d" :
                incrementalDate = startDate.plusDays(1);
                break;
            //按周递增
            case "W" :
                incrementalDate = startDate.plusWeeks(1);
                break;
            //按季度递增
            case "Q" :
                //先换算为季度，再根据季度补数据
                incrementalDate = startDate.plusMonths(3);
                break;
            //按月递增
            case "M" :
            default:
                incrementalDate = startDate.plusMonths(1).withDayOfMonth(1);
                break;
        }
        return incrementalDate;
    }


}
