package com.digiwin.athena.show.service.impl;

import com.digiwin.athena.show.assistant.AgileDataConstants;
import com.digiwin.athena.show.assistant.ExecuteContext;
import com.digiwin.athena.show.domain.agileDataDTO.*;
import com.digiwin.athena.show.domain.queryDefine.SnapDataDTO;
import com.digiwin.athena.show.domain.showDefine.AgileDataBusinessEnum;
import com.digiwin.athena.show.metadata.ApiMetadata;
import com.digiwin.athena.show.metadata.MetadataField;
import com.digiwin.athena.show.service.AgileDataTransDefineService;
import com.google.common.collect.Lists;
import org.apache.commons.collections.MapUtils;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import java.util.AbstractMap;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * 转换呈现定义
 */
@Service
public class AgileDataTransDefineServiceImpl implements AgileDataTransDefineService {

    private static final List<Integer> TRANS_ECHART_TYPES = Arrays.asList(3,5,7);

    /**
     * 形转换为表格
     * @param showType 呈现类型
     * @param snapDataDTO 业务数据
     * @param apiMetadata 元数据
     * @param actionId 数据ID
     * @param target 元数据指定路径
     */
    @Override
    public void transEchartsDefine(AgileReportShowTypeInfoDTO showType, AgileReportShow show,
                                   SnapDataDTO snapDataDTO, ApiMetadata apiMetadata,
                                   String actionId, String target, ExecuteContext executeContext) {
        //非语义指定呈现链路，不处理
        if(!"2".equals(snapDataDTO.getLink())){
            return;
        }
        //对于图形，涉及到多业务类型字段，需转换为表格呈现
        if(showType == null){
            return;
        }
        if(TRANS_ECHART_TYPES.contains(showType.getValue())){
            //是否满足转换
            List<MetadataField> measureFields = ConvertGetAgileData.getStringField("numeric",target,apiMetadata,true);
            if(!CollectionUtils.isEmpty(measureFields)){
                long businessSize = measureFields.stream().filter(field -> !StringUtils.isEmpty(field.getBusinessType())).map(field -> field.getBusinessType()).distinct().count();
                //构建双轴图呈现定义
                if(businessSize == 2){
                    //ppt转换时转为表格
                    if(AgileDataConstants.STANDARD_PAGE.equals(executeContext.getPageCode())){
                        showType.setValue(0);
                    } else {
                        this.transformationDualDefine(measureFields, show, showType);
                        showType.setValue(4);
                    }
                }
                if(businessSize > 2){
                    showType.setValue(0);
                }
            }
        }
    }

    private void transformationDualDefine(List<MetadataField> measureFields,AgileReportShow show,AgileReportShowTypeInfoDTO showType){
        //按业务类型分组
        Map<String,List<MetadataField>> amountFieldMap = measureFields.stream()
                .collect(Collectors.groupingBy(field -> field.getBusinessType()));
        if(MapUtils.isEmpty(amountFieldMap)){
            return;
        }
        //获取主业务类型
        String leftBusinessKey = amountFieldMap.keySet().stream()
                                    .map(str -> new AbstractMap.SimpleEntry<>(str, AgileDataBusinessEnum.getBusinessType(str))) // 使用自定义方法转换为 Entry<String, Integer>
                                    .filter(entry -> entry.getValue() != null) // 确保我们只处理有效（非空）的整数值
                                    .sorted(Map.Entry.comparingByValue()) // 按照 Integer 进行排序
                                    .findFirst() // 获取第一个元素
                                    .map(Map.Entry::getKey) // 提取 key
                                    .orElse(null); // 如果 map 为空或没有有效的整数值，则返回 null 或者其他默认值
        List<String> leftField = Lists.newArrayList();
        List<String> rightField = Lists.newArrayList();
        amountFieldMap.forEach((key,value) -> {
            if(StringUtils.isEmpty(leftBusinessKey)){
                return;
            }
            if(key.equals(leftBusinessKey)){
                this.getDualFields(amountFieldMap.get(key),"left",leftField,rightField);
            } else {
                this.getDualFields(amountFieldMap.get(key),"right",leftField,rightField);
            }
        });
        if(CollectionUtils.isEmpty(leftField) || CollectionUtils.isEmpty(rightField)){
            return;
        }
        //构建结构
        AgileReportRuleDTO agileReportRuleDTO = new AgileReportRuleDTO();
        agileReportRuleDTO.setType("doubleShaft");
        List<AgileReportRuleInfoDTO> rules = Lists.newArrayList();
        agileReportRuleDTO.setRules(rules);

        AgileDataAreaDefine leftAreaDefine = new AgileDataAreaDefine();
        leftAreaDefine.setApplyToFields(leftField);
        leftAreaDefine.setChart(AgileReportChartTypeEnum.getTypeByValue(showType.getValue()));
        AgileReportRuleInfoDTO leftAgileReportRuleInfo = new AgileReportRuleInfoDTO();
        leftAgileReportRuleInfo.setAreaDefine(Arrays.asList(leftAreaDefine));
        leftAgileReportRuleInfo.setDirection("left");
        rules.add(leftAgileReportRuleInfo);

        AgileDataAreaDefine rightAreaDefine = new AgileDataAreaDefine();
        rightAreaDefine.setApplyToFields(rightField);
        rightAreaDefine.setChart(AgileReportChartTypeEnum.getTypeByValue(showType.getValue()));
        AgileReportRuleInfoDTO rightAgileReportRuleInfo = new AgileReportRuleInfoDTO();
        rightAgileReportRuleInfo.setAreaDefine(Arrays.asList(rightAreaDefine));
        rightAgileReportRuleInfo.setDirection("left");
        rules.add(rightAgileReportRuleInfo);

        if(CollectionUtils.isEmpty(show.getAgileRule())){
            show.setAgileRule(Arrays.asList(agileReportRuleDTO));
        } else {
            show.getAgileRule().add(agileReportRuleDTO);
        }
    }

    /**
     * 获取双轴图呈现字段
     * @param metadataFields
     * @param mode
     * @param leftField
     * @param rightField
     */
    private void getDualFields(List<MetadataField> metadataFields, String mode,List<String> leftField, List<String> rightField) {
        if(!CollectionUtils.isEmpty(metadataFields)){
            if("left".equals(mode)){
                leftField.addAll(metadataFields.stream().map(MetadataField::getName).collect(Collectors.toList()));
            } else {
                rightField.addAll(metadataFields.stream().map(MetadataField::getName).collect(Collectors.toList()));
            }
        }
    }
}
