package com.digiwin.athena.show.service.impl;

import com.digiwin.athena.show.domain.agileDataDTO.*;
import com.digiwin.athena.show.infrastructure.meta.ApiMetadataConstants;
import com.digiwin.athena.show.metadata.ApiMetadata;
import com.digiwin.athena.show.metadata.MetadataField;
import com.digiwin.athena.show.service.AgileReportEChartBuildService;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang.StringUtils;
import org.springframework.stereotype.Service;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * 分析图表入参
 */
@Service
public class AgileReportEChartBuildServiceImpl implements AgileReportEChartBuildService {

    /**
     * 图表分析
     *
     * @param agileReportShow
     * @param apiMetadata
     * @param data
     * @return
     */
    @Override
    public Map<String, Object> getChartParams(AgileReportShow agileReportShow, AgileReportShowTypeDTO agileReportShowTypeDTO, ApiMetadata apiMetadata, List<Map<String, Object>> data) {
        Map<String, Object> chartTypeMap = Maps.newHashMap();
        chartTypeMap.put("reportRelationFir", "");
        chartTypeMap.put("reportRelationSec", "");
        chartTypeMap.put("reportRelationThi", "");
        chartTypeMap.put("reportRelationFou", "");
        String nodeType = agileReportShowTypeDTO.getSceneNodeType();
        //前融之前从action获取的逻辑
        if (StringUtils.isEmpty(nodeType)) {
            nodeType = apiMetadata.getSceneNodeType();
        }
        if (StringUtils.isNotEmpty(nodeType)) {
            switch (nodeType) {
                case "1":
                    //比较分析
                    chartTypeMap = this.getCompareChartParams(chartTypeMap, nodeType, agileReportShow, apiMetadata, data);
                    break;
                case "2":
                    //分布分析
                    chartTypeMap = this.getDistributeChartParams(chartTypeMap, nodeType, agileReportShow);
                    break;
                case "3":
                    //构成分析
                    chartTypeMap = this.getConstituteChartParams(chartTypeMap, nodeType, agileReportShow, apiMetadata, data);
                    break;
                case "4":
                    //联系
                    chartTypeMap = this.getContactChartParams(chartTypeMap, nodeType, agileReportShow);
                    break;
                default:
                    break;
            }
        }
        return chartTypeMap;
    }

    /**
     * 联系分析入参
     *
     * @param agileReportShow
     * @return
     */
    private Map<String, Object> getContactChartParams(Map<String, Object> chartTypeMap, String sceneNodeType, AgileReportShow agileReportShow) {
        //单轴图，只会存在一组信息
        AgileReportEChartDefineDTO agileReportEChartDefineDTO = new AgileReportEChartDefineDTO();
        if (CollectionUtils.isEmpty(agileReportShow.getEChart())) {
            return null;
        }
        agileReportEChartDefineDTO = agileReportShow.getEChart().get(0);
        chartTypeMap.put("nodeType", sceneNodeType);
        List<AgileReportDataFindDTO> agileReportDataFindDTOList = agileReportEChartDefineDTO.getSeries();
        if (StringUtils.isNotEmpty(agileReportEChartDefineDTO.getShowParams().getReportRelationFir())) {
            chartTypeMap.put("reportRelationFir", agileReportEChartDefineDTO.getShowParams().getReportRelationFir());
        } else if (CollectionUtils.isNotEmpty(agileReportDataFindDTOList)) {
            chartTypeMap.put("reportRelationFir", agileReportDataFindDTOList.size() > 2 ? "3个变量" : "2个变量");
        }
        return chartTypeMap;
    }

    /**
     * 构成分析入参
     *
     * @param agileReportShow
     * @return
     */
    private Map<String, Object> getConstituteChartParams(Map<String, Object> chartTypeMap, String sceneNodeType, AgileReportShow agileReportShow, ApiMetadata apiMetadata, List<Map<String, Object>> data) {
        //单轴图，只会存在一组信息
        AgileReportEChartDefineDTO agileReportEChartDefineDTO = new AgileReportEChartDefineDTO();
        if (CollectionUtils.isNotEmpty(agileReportShow.getEChart())) {
            agileReportEChartDefineDTO = agileReportShow.getEChart().get(0);
        }
        chartTypeMap.put("nodeType", sceneNodeType);
        AgileReportEChartParamsDTO agileReportEChartParamsDTO = agileReportEChartDefineDTO.getShowParams();
        //基于分类/时间
        String firType = this.getMetadataType(agileReportShow, apiMetadata.getResponseFields());
        switch (firType) {
            case "date":
                chartTypeMap.put("reportRelationFir", "随时间变化");
                this.buildConstituteDate(chartTypeMap, agileReportShow, data);
                break;
            default:
                chartTypeMap.put("reportRelationFir", "静态");
                chartTypeMap.put("reportRelationSec", agileReportEChartParamsDTO != null ? agileReportEChartParamsDTO.getReportRelationSec() : StringUtils.EMPTY);
                break;
        }
        return chartTypeMap;
    }

    /**
     * 构建构成时间入参
     *
     * @param chartTypeMap
     * @param agileReportShow
     * @param data
     */
    private void buildConstituteDate(Map<String, Object> chartTypeMap, AgileReportShow agileReportShow, List<Map<String, Object>> data) {
        //单轴图，只会存在一组信息
        AgileReportEChartDefineDTO agileReportEChartDefineDTO = new AgileReportEChartDefineDTO();
        if (CollectionUtils.isNotEmpty(agileReportShow.getEChart())) {
            agileReportEChartDefineDTO = agileReportShow.getEChart().get(0);
        }
        AgileReportEChartParamsDTO agileReportEChartParamsDTO = agileReportEChartDefineDTO.getShowParams();
        //基于周期
        //配置多个时间线，表示多周期
        List<AgileReportDataFindDTO> xAxisList = agileReportEChartDefineDTO.getXAxis();
        AgileReportDataFindDTO xAxis = xAxisList.size() > 1 ? xAxisList.get(0) : null;
        if (StringUtils.isNotEmpty(agileReportEChartParamsDTO.getReportRelationSec())) {
            chartTypeMap.put("reportRelationSec", agileReportEChartParamsDTO.getReportRelationSec());
        } else if (xAxis != null) {
            List<Object> dataList = this.getData(agileReportShow.getTargetSource(), xAxis.getApplyToField(), data);
            dataList = dataList.stream().distinct().collect(Collectors.toList());
            chartTypeMap.put("reportRelationSec", dataList.size() > 12 ? "多周期" : "少数周期");
        }
        //差异定义
        if (StringUtils.isNotEmpty(agileReportEChartParamsDTO.getReportRelationThi())) {
            chartTypeMap.put("reportRelationThi", agileReportEChartParamsDTO.getReportRelationThi());
        }
    }


    /**
     * 分布分析入参
     *
     * @param agileReportShow
     * @return
     */
    private Map<String, Object> getDistributeChartParams(Map<String, Object> chartTypeMap, String sceneNodeType, AgileReportShow agileReportShow) {
        //单轴图，只会存在一组信息
        AgileReportEChartDefineDTO agileReportEChartDefineDTO = new AgileReportEChartDefineDTO();
        if (CollectionUtils.isNotEmpty(agileReportShow.getEChart())) {
            agileReportEChartDefineDTO = agileReportShow.getEChart().get(0);
        }
        chartTypeMap.put("nodeType", sceneNodeType);
        AgileReportEChartParamsDTO agileReportEChartParamsDTO = agileReportEChartDefineDTO.getShowParams();
        if (agileReportEChartParamsDTO == null) {
            return null;
        }
        //变量
        List<AgileReportDataFindDTO> seriesList = agileReportEChartDefineDTO.getSeries();
        if (CollectionUtils.isNotEmpty(seriesList)) {
            switch (seriesList.size()) {
                case 0:
                    chartTypeMap.put("reportRelationFir", "单个变量");
                    break;
                case 1:
                    chartTypeMap.put("reportRelationFir", "2个变量");
                    break;
                default:
                    chartTypeMap.put("reportRelationFir", "3个变量");
                    break;
            }
        }
        //据点
        if (StringUtils.isNotEmpty(agileReportEChartParamsDTO.getReportRelationSec())) {
            chartTypeMap.put("reportRelationSec", agileReportEChartParamsDTO.getReportRelationSec());
        }
        return chartTypeMap;
    }

    /**
     * 比较分析入参
     *
     * @param agileReportShow
     * @param apiMetadata
     * @return
     */
    private Map<String, Object> getCompareChartParams(Map<String, Object> chartTypeMap, String sceneNodeType, AgileReportShow agileReportShow, ApiMetadata apiMetadata, List<Map<String, Object>> data) {
        chartTypeMap.put("nodeType", sceneNodeType);
        //基于分类/时间
        String firType = this.getMetadataType(agileReportShow, apiMetadata.getResponseFields());
        if (StringUtils.isEmpty(firType)) {
            return chartTypeMap;
        }
        switch (firType) {
            case "date":
                chartTypeMap.put("reportRelationFir", "基于时间");
                this.buildCompareDate(chartTypeMap, agileReportShow, data);
                break;
            default:
                chartTypeMap.put("reportRelationFir", "基于分类");
                this.buildCompareClassify(chartTypeMap, agileReportShow, data);
                break;
        }

        return chartTypeMap;
    }

    /**
     * 比较-分类线
     *
     * @param chartTypeMap
     * @param agileReportShow
     */
    private void buildCompareClassify(Map<String, Object> chartTypeMap, AgileReportShow agileReportShow, List<Map<String, Object>> data) {
        //单轴图，只会存在一组信息
        AgileReportEChartDefineDTO agileReportEChartDefineDTO = new AgileReportEChartDefineDTO();
        if (CollectionUtils.isNotEmpty(agileReportShow.getEChart())) {
            agileReportEChartDefineDTO = agileReportShow.getEChart().get(0);
        }
        AgileReportEChartParamsDTO agileReportEChartParamsDTO = agileReportEChartDefineDTO.getShowParams();
        //基于变量
        //配置多个y轴展示项，代表  多个变量
        List<AgileReportDataFindDTO> agileReportDataFindDTOList = agileReportEChartDefineDTO.getSeries();
        if (CollectionUtils.isNotEmpty(agileReportDataFindDTOList)) {
            if (agileReportDataFindDTOList.size() > 1) {
                chartTypeMap.put("reportRelationSec", "每项目2个变量");
            } else {
                chartTypeMap.put("reportRelationSec", "每项目1个变量");
                //基于分类数
//                List<Object> dataList = this.getData(agileReportShow.getEChart().getXAxis().getTarget(),agileReportShow.getEChart().getXAxis().getApplyToField(),data);
//                dataList = dataList.stream().distinct().collect(Collectors.toList());
//                chartTypeMap.put("reportRelationThi",dataList.size() > 20 ? "多种分类" : "少数分类");
                chartTypeMap.put("reportRelationThi", "少数分类");
                //基于项目数
//                chartTypeMap.put("reportRelationFou",data.size() > 5 ? "多个项目" : "少数项目");

                chartTypeMap.put("reportRelationFou", "少数项目");
            }
        }

    }

    /**
     * 比较-时间线
     *
     * @param chartTypeMap
     * @param agileReportShow
     */
    private void buildCompareDate(Map<String, Object> chartTypeMap, AgileReportShow agileReportShow, List<Map<String, Object>> data) {
        //单轴图，只会存在一组信息
        AgileReportEChartDefineDTO agileReportEChartDefineDTO = new AgileReportEChartDefineDTO();
        if (CollectionUtils.isNotEmpty(agileReportShow.getEChart())) {
            agileReportEChartDefineDTO = agileReportShow.getEChart().get(0);
        }
        AgileReportEChartParamsDTO agileReportEChartParamsDTO = agileReportEChartDefineDTO.getShowParams();
        //基于周期
        //配置多个时间线，表示多周期
        List<AgileReportDataFindDTO> xAxisList = agileReportEChartDefineDTO.getXAxis();
        AgileReportDataFindDTO xAxis = xAxisList.size() > 1 ? xAxisList.get(0) : null;
        if (agileReportEChartParamsDTO != null && StringUtils.isNotEmpty(agileReportEChartParamsDTO.getReportRelationSec())) {
            chartTypeMap.put("reportRelationSec", agileReportEChartParamsDTO.getReportRelationSec());
        } else if (xAxis != null) {
            List<Object> dataList = this.getData(agileReportShow.getTargetSource(), xAxis.getApplyToField(), data);
            dataList = dataList.stream().distinct().collect(Collectors.toList());
//            chartTypeMap.put("reportRelationSec",dataList.size() > 12 ? "多周期" : "少数周期");
            //日期默认曲线图
            chartTypeMap.put("reportRelationSec", "少数周期");
        }
        //分类定义
        List<AgileReportDataFindDTO> agileReportDataFindDTOList = agileReportEChartDefineDTO.getSeries();
        if (agileReportEChartParamsDTO != null && StringUtils.isNotEmpty(agileReportEChartParamsDTO.getReportRelationThi())) {
            chartTypeMap.put("reportRelationThi", agileReportEChartParamsDTO.getReportRelationThi());
        } else if (CollectionUtils.isNotEmpty(agileReportDataFindDTOList)) {
//            chartTypeMap.put("reportRelationThi",agileReportDataFindDTOList.size() > 1 ? "多种分类" : "单个或少数分类");
            chartTypeMap.put("reportRelationThi", "多种分类");
        }
    }


    /**
     * 获取字段类型
     *
     * @param agileReportShow
     * @param metadataFieldList
     * @return
     */
    private String getMetadataType(AgileReportShow agileReportShow, List<MetadataField> metadataFieldList) {
        String fieldType = "";
        List<AgileReportDataFindDTO> agileReportDataFindDTOList = Lists.newArrayList();
        agileReportShow.getEChart().stream().forEach(echart -> agileReportDataFindDTOList.addAll(echart.getXAxis()));
        for (AgileReportDataFindDTO agileReportDataFindDTO : agileReportDataFindDTOList) {
            fieldType = this.getFieldType(agileReportDataFindDTO.getApplyToField(), metadataFieldList);
        }
        return fieldType;
    }

    /**
     * 获取业务数据
     *
     * @param field
     * @param data
     * @return
     */
    private List<Object> getData(String target, String field, List<Map<String, Object>> data) {
        List<Object> values = Lists.newArrayList();
        List<Map<String, Object>> dataList = Lists.newArrayList();
        if (CollectionUtils.isNotEmpty(data)) {
            data.stream().forEach(mapData -> {
                Object obj = ConvertGetAgileData.targetValue(target, mapData);
                if (obj instanceof Collection) {
                    dataList.addAll((Collection<? extends Map<String, Object>>) obj);
                } else if (obj instanceof Map) {
                    dataList.add((Map<String, Object>) obj);
                }
            });
        }
        if (CollectionUtils.isNotEmpty(dataList)) {
            dataList.stream().forEach(resData -> {
                values.add(MapUtils.getObject(resData, field));
            });
        }
        return values;
    }

    private String getFieldType(String field, List<MetadataField> metadataFieldList) {
        String type = StringUtils.EMPTY;
        for (MetadataField metadataField : metadataFieldList) {
            if (ApiMetadataConstants.METADATA_OBJECT.equals(metadataField.getDataType())) {
                type = getFieldType(field, metadataField.getSubFields());
            } else {
                if (StringUtils.equals(metadataField.getName(), field)) {
                    type = metadataField.getDataType();
                    break;
                }
            }
        }
        return type;
    }
}
