package com.digiwin.athena.show.service.impl;

import com.digiwin.athena.agiledataecho.domain.AgileDataStandardReq;
import com.digiwin.athena.agiledataecho.proxy.adt.AdtService;
import com.digiwin.athena.show.analyzer.AgileReportPageDefineAnalyzer;
import com.digiwin.athena.show.assistant.AgileDataConstants;
import com.digiwin.athena.show.assistant.ExecuteContext;
import com.digiwin.athena.show.component.AbstractComponent;
import com.digiwin.athena.show.component.echarts.EchartsComponent;
import com.digiwin.athena.show.domain.agileDataDTO.AgileReport;
import com.digiwin.athena.show.domain.agileDataDTO.AgileReportChartTypeEnum;
import com.digiwin.athena.show.domain.agileDataDTO.MobileDSLAgileData;
import com.digiwin.athena.show.domain.queryDefine.SnapDataDTO;
import com.digiwin.athena.show.domain.showDefine.AgileDataStandardComponent;
import com.digiwin.athena.show.domain.showDefine.AgileDataStandardShow;
import com.digiwin.athena.show.domain.showDefine.DynamicAgileData;
import com.digiwin.athena.show.domain.showDefine.MobileDSLAgileComponent;
import com.digiwin.athena.show.service.AgileReportService;
import com.digiwin.athena.show.template.AgileReportPageTemplate;
import com.digiwin.athena.show.util.layout.AgileDataCheck;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.compress.utils.Lists;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.List;
import java.util.Map;
import java.util.Objects;

@Slf4j
@Service
public class AgileReportServiceImpl implements AgileReportService {

    @Autowired
    private AgileReportPageDefineAnalyzer agileReportPageDefineAnalyzer;

    @Autowired
    private List<AgileReportPageTemplate> pageDefines;

    @Autowired
    private AdtService adtService;

    @Override
    public DynamicAgileData createAgileReportTemplateTable(ExecuteContext executeContext, SnapDataDTO snapDataDTO) {
        //获取呈现定义
        AgileReport agileReport = agileReportPageDefineAnalyzer.analysis(executeContext,snapDataDTO);
        //组装呈现结构
        return Objects.requireNonNull(getPageDefine(executeContext)).createAgileReport(executeContext,agileReport);
    }

    @Override
    public String getShowType(DynamicAgileData dynamicAgileData,String defaultShowType) {
        String showType = null;
        if(dynamicAgileData != null){
            List<AbstractComponent> layout = dynamicAgileData.getLayout();
            if(CollectionUtils.isEmpty(layout)) {
                return showType;
            }
            for(AbstractComponent abstractComponent :  layout){
                if(abstractComponent instanceof MobileDSLAgileComponent){
                    MobileDSLAgileComponent mobileDSLAgileComponent = (MobileDSLAgileComponent) abstractComponent;
                    if(CollectionUtils.isNotEmpty(mobileDSLAgileComponent.getData())){
                        MobileDSLAgileData mobileDSLAgileData = mobileDSLAgileComponent.getData().get(0);
                        showType = String.valueOf(mobileDSLAgileData.getShowCategory());
                        if(StringUtils.equals("1",showType) && mobileDSLAgileData.getChartRenderSetting() != null){
                            String chartType = mobileDSLAgileData.getChartRenderSetting().getChartType();
                            switch (chartType){
                                case "mixed":
                                    showType = defaultShowType;
                                    break;
                                default:
                                    showType = String.valueOf(AgileReportChartTypeEnum.getValueByType(chartType));
                                    break;
                            }
                        }
                    }
                }else {
                    switch (abstractComponent.getType()) {
                        case "STATISTIC":
                            showType = "2";
                            break;
                        case "TABLE":
                            showType = "0";
                            break;
                        case "ECHARTS":
                            if (abstractComponent instanceof EchartsComponent) {
                                EchartsComponent echartsComponent = (EchartsComponent) abstractComponent;
                                showType = String.valueOf(echartsComponent.getChartType());
                                if (StringUtils.equals("4", showType)) {
                                    showType = defaultShowType;
                                }
                            }
                        default:
                            break;
                    }
                }
            }
        }
        return showType;
    }

    @Override
    public AgileDataStandardShow getAgileDataStandardShow(AgileDataStandardReq agileDataStandardReq) {
        AgileDataStandardShow agileDataStandardShow = new AgileDataStandardShow();
        Map<String,Object> typeMap = Maps.newHashMap();
        agileDataStandardShow.setTypeMap(typeMap);
        List<AgileDataStandardComponent> showDefine = Lists.newArrayList();
        agileDataStandardShow.setShowDefine(showDefine);
        ExecuteContext executeContext = ExecuteContext.createByHttpRequest();
        String entry = "0";
        if(CollectionUtils.isEmpty(agileDataStandardReq.getSnapshotIds())){
            return null;
        }
        agileDataStandardReq.getSnapshotIds().stream().forEach(snapshotId -> {
            SnapDataDTO agileReportDTO = adtService.getAdtReportBySnapshotId(snapshotId, executeContext.getLocale());
            if (AgileDataCheck.sceneBasicCheck(agileReportDTO)) {
                return;
            }
            executeContext.appendAgileDataPageInfo(AgileDataConstants.STANDARD_PAGE, agileReportDTO,agileReportDTO.getSnapData().getReportCode(),entry,null);
            //解析报表信息
            try {
                DynamicAgileData agileData = this.createAgileReportTemplateTable(executeContext, agileReportDTO);
                if(CollectionUtils.isNotEmpty(agileData.getLayout())){
                    agileData.getLayout().stream().forEach(component -> {
                        AgileDataStandardComponent agileDataStandardComponent = new AgileDataStandardComponent();
                        agileDataStandardComponent.setComponentId(component.getId());
                        agileDataStandardComponent.setAbstractComponent(component);
                        agileDataStandardComponent.setSnapshotId(snapshotId);
                        showDefine.add(agileDataStandardComponent);
                        switch (component.getType()){
                            case "STATISTIC":
                                agileDataStandardComponent.setShowType("2");
                                component.setType("2");
                                //表格
                                typeMap.put("2","statistic");
                                break;
                            case "TABLE":
                                agileDataStandardComponent.setShowType("0");
                                component.setType("0");
                                //表格
                                typeMap.put("0","table");
                                break;
                            case "ECHARTS":
                                EchartsComponent echartsComponent = (EchartsComponent) component;
                                component.setType(String.valueOf(echartsComponent.getChartType()));
                                //图形
                                agileDataStandardComponent.setShowType(String.valueOf(echartsComponent.getChartType()));
                                typeMap.put(String.valueOf(echartsComponent.getChartType()),AgileReportChartTypeEnum.getTypeByValue(echartsComponent.getChartType()));
                                break;
                            default:
                                break;
                        }
                    });
                }
            } catch (Exception e){
                log.error("解析呈现结构异常：{}",e.getMessage());
            }
        });
        return agileDataStandardShow;
    }


    /**
     * 根据入口查询模板
     * @param executeContext
     * @return
     */
    private AgileReportPageTemplate getPageDefine(ExecuteContext executeContext){
        Map<String,Object> agileData = executeContext.getAgileData();
        String entry = MapUtils.getString(agileData,"entry");
        if(StringUtils.equals("1",entry)){
            entry = "SCREEN";
        } else {
            entry = "GENERA";
        }
        AgileReportPageTemplate agileReportPageTemplate = null;
        for(AgileReportPageTemplate template : pageDefines){
            String supportKey = "AGILE_DATA_TEMPLATE_" + entry;
            if(StringUtils.equals(template.supportKey(), supportKey)){
                agileReportPageTemplate = template;
                break;
            }
        }
        return agileReportPageTemplate;
    }
}
