package com.digiwin.athena.show.service.impl;

import cn.hutool.json.JSONUtil;
import com.digiwin.athena.agiledataecho.constant.ErrorCodeEnum;
import com.digiwin.athena.agiledataecho.domain.AgileDataUserDefine;
import com.digiwin.athena.agiledataecho.dto.aiBoard.*;
import com.digiwin.athena.agiledataecho.proxy.adt.AdtService;
import com.digiwin.athena.agiledataecho.proxy.scrumbi.ScrumBIService;
import com.digiwin.athena.agiledataecho.proxy.scrumbi.dto.ScrumBILayoutReq;
import com.digiwin.athena.agiledataecho.service.aiBoard.EchoAiBoardService;
import com.digiwin.athena.appcore.auth.domain.AuthoredUser;
import com.digiwin.athena.appcore.exception.BusinessException;
import com.digiwin.athena.appcore.util.MessageUtils;
import com.digiwin.athena.show.assistant.AgileDataConstants;
import com.digiwin.athena.show.assistant.EchoShowConstants;
import com.digiwin.athena.show.assistant.ExecuteContext;
import com.digiwin.athena.show.component.AbstractComponent;
import com.digiwin.athena.show.component.gridster.GridGroupDTO;
import com.digiwin.athena.show.component.index.AgileDataIndexComponent;
import com.digiwin.athena.show.domain.agileDataDTO.ConvertAgileDataRelation;
import com.digiwin.athena.show.domain.agileDataDTO.ConvertGetAgileData;
import com.digiwin.athena.show.domain.board.*;
import com.digiwin.athena.show.domain.dynamicLayout.BordDynamicDTO;
import com.digiwin.athena.show.domain.queryDefine.SnapDataDTO;
import com.digiwin.athena.show.domain.showDefine.DynamicAgileData;
import com.digiwin.athena.show.infrastructure.meta.ECHOConstants;
import com.digiwin.athena.show.manager.themeMap.ThemeAgileDataService;
import com.digiwin.athena.show.manager.themeMap.domain.ThemeMapBoardDTO;
import com.digiwin.athena.show.metadata.ApiMetadata;
import com.digiwin.athena.show.service.AgileDataBestShowRuleService;
import com.digiwin.athena.show.service.AgileReportService;
import com.digiwin.athena.show.service.DynamicLayoutBoardService;
import com.digiwin.athena.show.util.layout.AgileDataBoardComponentBuild;
import com.digiwin.athena.show.util.layout.AgileDataCheck;
import com.digiwin.athena.show.util.layout.BoardLayoutPlanner;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.data.mongodb.core.MongoTemplate;
import org.springframework.data.mongodb.core.query.Criteria;
import org.springframework.data.mongodb.core.query.Query;
import org.springframework.data.mongodb.core.query.Update;
import org.springframework.stereotype.Service;

import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.atomic.AtomicReference;
import java.util.stream.Collectors;

@Slf4j
@Service
public class DynamicLayoutBoardServiceImpl implements DynamicLayoutBoardService {

    @Autowired
    private AgileDataBestShowRuleService agileDataBestShowRuleService;

    @Autowired
    private ThemeAgileDataService themeAgileDataService;

    @Autowired
    private AdtService adtService;

    @Autowired
    private AgileReportService agileReportService;

    @Autowired
    private AgileDataPageDefineThemeMapService agileDataPageDefineThemeMapService;

    @Autowired
    private MessageUtils messageUtils;

    @Autowired
    private EchoAiBoardService echoAiBoardService;

    @Autowired
    private ScrumBIService scrumBIService;

    @Autowired
    @Qualifier("mongoAgileDataPageViewTemplate")
    private MongoTemplate mongoAgileDataPageViewTemplate;

    private static final String BOARD_LAYOUT_NAME = "agileData_board_layout";
    /** 默认按照 12 列布局 */
    private static final int DEFAULT_LAYOUT_COLS = 12;


    /**
     * 获取看板动态布局
     * @param board
     * @return
     */
    @Override
    public List<GridGroupDTO> calcDynamicLayoutBoard(AuthoredUser user, BordDynamicDTO board, String local) {
        //获取KM看板结构
        ThemeMapBoardDTO themeMapBoardDTO = themeAgileDataService.getThemeMapBoard(board.getBoardId());
        if (themeMapBoardDTO == null) {
            throw BusinessException.create(ErrorCodeEnum.BOARD_NO_DATA.getErrCode(), messageUtils.getMessage("board.no.data"));
        }
        if (CollectionUtils.isEmpty(themeMapBoardDTO.getDataBoardQuestions())) {
            throw BusinessException.create(ErrorCodeEnum.BOARD_NO_DATA.getErrCode(), messageUtils.getMessage("board.no.data"));
        }
        //获取用户自定义布局
        BoardLayoutMongoData userDefineLayout = this.getDynamicLayoutBoard(user, board.getBoardId());
        //待重构，对话看板入口，走AI布局设定
        String entry = themeMapBoardDTO.getDataEntry();
        if (StringUtils.equals("aiBoard", entry)) {
            log.info("开始构建看板布局信息，{}", board.getBoardId());
            return this.getAILayout(user, local, themeMapBoardDTO, userDefineLayout);
        }
        //重新按自定义排序
        if (userDefineLayout != null) {
            Map<String, Integer> orderMap = Maps.newHashMap();
            for (int i = 0; i < userDefineLayout.getBoardLayout().size(); i++) {
                orderMap.put(userDefineLayout.getBoardLayout().get(i).getId(), i);
            }
            Map<ThemeMapBoardDTO.DataBoardQuestion, Integer> originalIndexMap = Maps.newHashMap();
            for (int i = 0; i < themeMapBoardDTO.getDataBoardQuestions().size(); i++) {
                originalIndexMap.put(themeMapBoardDTO.getDataBoardQuestions().get(i), i);
            }
            List<ThemeMapBoardDTO.DataBoardQuestion> dataBoardQuestions = themeMapBoardDTO.getDataBoardQuestions()
                    .stream()
                    .sorted(Comparator
                            .comparingInt((ThemeMapBoardDTO.DataBoardQuestion item) -> orderMap.containsKey(item.getQuestionId()) ? 0 : 1)
                            .thenComparing(Comparator
                                    .comparingInt((ThemeMapBoardDTO.DataBoardQuestion item) -> orderMap.getOrDefault(item.getQuestionId(), Integer.MAX_VALUE))
                                    .thenComparing(item -> originalIndexMap.get(item)))
                    )
                    .collect(Collectors.toList());
            themeMapBoardDTO.setDataBoardQuestions(dataBoardQuestions);
        }

        //设置呈现元素
        List<AbstractComponent> abstractComponents = Lists.newArrayList();

        themeMapBoardDTO.getDataBoardQuestions().stream().forEach(question -> {
            //呈现类型为指标卡，重新通过ADE获取数据结构
            if (question.getShowType() == null) {
                question.setShowType(0);
            }
            switch (question.getShowType()) {
                case 2:
                    SnapDataDTO snapDataDTO = adtService.transAgileData(String.valueOf(question.getShowType()), question.getSnapshotId(), local);
                    AgileDataIndexComponent agileDataIndexComponent = getIndexComponent(question.getQuestionId(), snapDataDTO, local);
                    if (agileDataIndexComponent != null) {
                        abstractComponents.add(agileDataIndexComponent);
                    } else {
                        abstractComponents.add(getEmptyIndexComponent(question.getQuestionId()));
                    }
                    break;
                case 3:
                case 4:
                case 5:
                case 6:
                    abstractComponents.add(AgileDataBoardComponentBuild.buildBoardEcharts(question.getQuestionId()));
                    break;
                case 0:
                default:
                    abstractComponents.add(AgileDataBoardComponentBuild.buildBoardGrid(question.getQuestionId()));
                    break;
            }
        });
        //动态结构计算
        List<GridGroupDTO> items;
        if (userDefineLayout != null) {
            items = ConvertAgileDataRelation.createUserDefineBoardItems(abstractComponents, userDefineLayout);
        } else {
            items = ConvertAgileDataRelation.createBoardItems(abstractComponents, userDefineLayout);
        }
        return items;
    }

    /**
     * 获取AI布局
     * @param authoredUser
     * @param local
     * @param themeMapBoardDTO
     * @param userDefineLayout
     * @return
     */
    private List<GridGroupDTO> getAILayout(AuthoredUser authoredUser, String local,
                                           ThemeMapBoardDTO themeMapBoardDTO, BoardLayoutMongoData userDefineLayout) {
        List<GridGroupDTO> items = Lists.newArrayList();
        //查询AI看板布局
        if (themeMapBoardDTO.getMessageId() == null) {
            return items;
        }
        AIBoardQueryReq aiBoardQueryReq = new AIBoardQueryReq();
        aiBoardQueryReq.setMessageId(themeMapBoardDTO.getMessageId());
        EchoAIBoardDTO echoAIBoardDTO = echoAiBoardService.queryAgileDataAIBoard(authoredUser, aiBoardQueryReq);
        if (echoAIBoardDTO == null) {
            return items;
        }
        //获取解析成功的看板问句
        getAnalyzeResult(echoAIBoardDTO);
        AIBoardLayout aiBoardLayout = echoAIBoardDTO.getBoardLayout();

        if (aiBoardLayout == null) {
            //获取语义布局
            List<EchoAIBoardLayoutDTO> layoutDTOList = scrumBIService.getAIBoardLayout(authoredUser, ScrumBILayoutReq.buildScrumBILayoutReq(echoAIBoardDTO), local);
            log.info("layoutDTOList:{}", JSONUtil.toJsonStr(layoutDTOList));
            if (CollectionUtils.isEmpty(layoutDTOList)) {
                return items;
            }
            aiBoardLayout = BoardLayoutPlanner.getAiBoardLayout(layoutDTOList, echoAIBoardDTO, aiBoardLayout);
            boolean hasLightFields = aiBoardLayout.getCharts().stream().anyMatch(c -> c.getImportance() != null || c.getRow() != null);
            if (hasLightFields) {
                int layoutCols = aiBoardLayout.getOverallLayout() != null && aiBoardLayout.getOverallLayout().getLayoutCols() != null
                        ? aiBoardLayout.getOverallLayout().getLayoutCols() : 12;
                int layoutRows = aiBoardLayout.getOverallLayout() != null && aiBoardLayout.getOverallLayout().getLayoutRows() != null
                        ? aiBoardLayout.getOverallLayout().getLayoutRows() : 18;

                List<BoardLayoutPlanner.Seed> seeds = BoardLayoutPlanner.getSeedList(aiBoardLayout, echoAIBoardDTO);
                if (!seeds.isEmpty()) {
                    AIBoardLayout planned = BoardLayoutPlanner.plan(seeds, layoutCols, layoutRows);
                    log.info("seeds:{}", JSONUtil.toJsonStr(seeds));
                    log.info("planned:{}", JSONUtil.toJsonStr(planned));
                    EchoAIBoardDTO updateAIBoardDTO = new EchoAIBoardDTO();
                    updateAIBoardDTO.setMessageId(themeMapBoardDTO.getMessageId());
                    updateAIBoardDTO.setBoardLayout(planned);
                    echoAiBoardService.updateAgileDataAIBoard(authoredUser, updateAIBoardDTO);
                    echoAIBoardDTO.setBoardLayout(planned);
                    aiBoardLayout = planned;
                }
            }
        } else {
            List<ThemeMapBoardDTO.DataBoardQuestion> newQuestions = resolveNewQuestions(themeMapBoardDTO, userDefineLayout, aiBoardLayout);
            if (CollectionUtils.isNotEmpty(newQuestions)) {
                boolean userLayoutUpdated = updateUserDefineLayout(authoredUser, themeMapBoardDTO, userDefineLayout, aiBoardLayout, newQuestions, echoAIBoardDTO);
                if (userLayoutUpdated) {
                    persistUserLayout(authoredUser, themeMapBoardDTO.getBoardId(), userDefineLayout);
                } else {
                    boolean boardLayoutUpdated = updateBoardLayout(themeMapBoardDTO, echoAIBoardDTO, aiBoardLayout, newQuestions);
                    if (boardLayoutUpdated) {
                        EchoAIBoardDTO updateAIBoardDTO = new EchoAIBoardDTO();
                        updateAIBoardDTO.setMessageId(themeMapBoardDTO.getMessageId());
                        updateAIBoardDTO.setBoardLayout(aiBoardLayout);
                        echoAiBoardService.updateAgileDataAIBoard(authoredUser, updateAIBoardDTO);
                    }
                }
            }
        }
        items = ConvertAgileDataRelation.createAIUserDefineLayout(themeMapBoardDTO, aiBoardLayout, userDefineLayout);
        return items;
    }

    /**
     * 计算看板中比 AI 布局多出的问句
     */
    private List<ThemeMapBoardDTO.DataBoardQuestion> resolveNewQuestions(ThemeMapBoardDTO themeMapBoardDTO,
                                                                         BoardLayoutMongoData userDefineLayout,
                                                                         AIBoardLayout aiBoardLayout) {
        if (themeMapBoardDTO == null || CollectionUtils.isEmpty(themeMapBoardDTO.getDataBoardQuestions())) {
            return Lists.newArrayList();
        }
        Set<String> existedIds = Sets.newHashSet();
        if (userDefineLayout != null && CollectionUtils.isNotEmpty(userDefineLayout.getBoardLayout())) {
            userDefineLayout.getBoardLayout().forEach(item -> {
                if (item == null) {
                    return;
                }
                String id = item.getId();
                if (StringUtils.isNotBlank(id)) {
                    existedIds.add(id);
                }
            });
        }
        if (CollectionUtils.isEmpty(existedIds) && aiBoardLayout != null && CollectionUtils.isNotEmpty(aiBoardLayout.getCharts())) {
            aiBoardLayout.getCharts().forEach(chart -> {
                if (chart == null) {
                    return;
                }
                String chartId = chart.getChartId();
                if (StringUtils.isNotBlank(chartId)) {
                    existedIds.add(chartId);
                }
            });
        }
        if (CollectionUtils.isEmpty(existedIds)) {
            return Lists.newArrayList(themeMapBoardDTO.getDataBoardQuestions());
        }
        return themeMapBoardDTO.getDataBoardQuestions().stream()
                .filter(question -> {
                    String questionId = question.getQuestionId();
                    if (StringUtils.isNotBlank(questionId) && existedIds.contains(questionId)) {
                        return false;
                    }
                    String resolvedId = resolveBoardChartId(question);
                    return !(StringUtils.isNotBlank(resolvedId) && existedIds.contains(resolvedId));
                })
                .collect(Collectors.toList());
    }

    /**
     * 将新增问句补充到用户自定义布局，按最后一行顺延
     */
    private boolean updateUserDefineLayout(AuthoredUser authoredUser, ThemeMapBoardDTO themeMapBoardDTO,
                                           BoardLayoutMongoData userDefineLayout,
                                           AIBoardLayout aiBoardLayout,
                                           List<ThemeMapBoardDTO.DataBoardQuestion> newQuestions,
                                           EchoAIBoardDTO echoAIBoardDTO) {
        if (userDefineLayout == null) {
            return false;
        }
        List<GridGroupDTO> layout = userDefineLayout.getBoardLayout();
        if (layout == null) {
            layout = Lists.newArrayList();
            userDefineLayout.setBoardLayout(layout);
        }
        boolean updated = removeInvalidUserLayoutItems(layout, themeMapBoardDTO);
        int nextRowStart = calculateNextRowStart(layout);
        for (ThemeMapBoardDTO.DataBoardQuestion question : newQuestions) {
            if (layout.stream().anyMatch(item -> StringUtils.equals(item.getId(), question.getQuestionId()))) {
                continue;
            }
            String showType = resolveChartShowType(question, echoAIBoardDTO);
            LayoutSize layoutSize = resolveLayoutSize(showType, question.getSnapshotId());
            GridGroupDTO gridGroupDTO = buildUserLayoutGroup(question, showType, nextRowStart, layoutSize);
            layout.add(gridGroupDTO);
            nextRowStart += layoutSize.getRows();
            updated = true;
        }
        return updated;
    }

    /**
     * 将新增问句补充到 AI 布局，保持末行宽度占满
     */
    private boolean updateBoardLayout(ThemeMapBoardDTO themeMapBoardDTO, EchoAIBoardDTO echoAIBoardDTO,
                                      AIBoardLayout aiBoardLayout,
                                      List<ThemeMapBoardDTO.DataBoardQuestion> newQuestions) {
        if (aiBoardLayout == null) {
            return false;
        }
        List<AIBoardLayout.layoutArea> charts = aiBoardLayout.getCharts();
        if (charts == null) {
            charts = Lists.newArrayList();
            aiBoardLayout.setCharts(charts);
        }
        boolean updated = removeInvalidBoardCharts(aiBoardLayout, themeMapBoardDTO);
        int nextRowStart = calculateNextRowStart(aiBoardLayout);
        for (ThemeMapBoardDTO.DataBoardQuestion question : newQuestions) {
            if(question == null){
                continue;
            }
            String matchedChartId = resolveBoardChartId(question);
            if (charts.stream().anyMatch(chart -> StringUtils.equals(chart.getChartId(), matchedChartId))) {
                continue;
            }
            String showType = resolveChartShowType(question, echoAIBoardDTO);
            LayoutSize layoutSize = resolveLayoutSize(showType, question.getSnapshotId());
            AIBoardLayout.layoutArea layoutArea = buildLayoutArea(themeMapBoardDTO, echoAIBoardDTO, question, matchedChartId, showType, nextRowStart, layoutSize);
            charts.add(layoutArea);
            nextRowStart += layoutSize.getRows();
            updated = true;
        }
        if (updated) {
            aiBoardLayout.getOverallLayout().setLayoutRows(nextRowStart);
        }
        return updated;
    }

    private boolean removeInvalidUserLayoutItems(List<GridGroupDTO> layout, ThemeMapBoardDTO themeMapBoardDTO) {
        if (CollectionUtils.isEmpty(layout)) {
            return false;
        }
        if (themeMapBoardDTO == null || CollectionUtils.isEmpty(themeMapBoardDTO.getDataBoardQuestions())) {
            return false;
        }
        Set<String> validQuestionIds = themeMapBoardDTO.getDataBoardQuestions().stream()
                .map(ThemeMapBoardDTO.DataBoardQuestion::getQuestionId)
                .filter(StringUtils::isNotBlank)
                .collect(Collectors.toSet());
        if (validQuestionIds.isEmpty()) {
            return false;
        }
        return layout.removeIf(item -> item == null
                || StringUtils.isBlank(item.getId())
                || !validQuestionIds.contains(item.getId()));
    }

    private boolean removeInvalidBoardCharts(AIBoardLayout aiBoardLayout, ThemeMapBoardDTO themeMapBoardDTO) {
        if (aiBoardLayout == null || CollectionUtils.isEmpty(aiBoardLayout.getCharts())) {
            return false;
        }
        if (themeMapBoardDTO == null || CollectionUtils.isEmpty(themeMapBoardDTO.getDataBoardQuestions())) {
            return false;
        }
        Set<String> validChartIds = themeMapBoardDTO.getDataBoardQuestions().stream()
                .map(this::resolveBoardChartId)
                .filter(StringUtils::isNotBlank)
                .collect(Collectors.toSet());
        if (validChartIds.isEmpty()) {
            return false;
        }
        return aiBoardLayout.getCharts().removeIf(chart -> chart == null
                || StringUtils.isBlank(chart.getChartId())
                || !validChartIds.contains(chart.getChartId()));
    }

    /**
     * 持久化用户自定义布局
     */
    private void persistUserLayout(AuthoredUser authoredUser, String boardId, BoardLayoutMongoData userDefineLayout) {
        if (userDefineLayout == null || CollectionUtils.isEmpty(userDefineLayout.getBoardLayout())) {
            return;
        }
        Query query = Query.query(Criteria.where("userId").is(authoredUser.getUserId())
                .and("tenantId").is(authoredUser.getTenantId())
                .and("boardId").is(boardId));
        Update update = new Update();
        update.set("boardLayout", userDefineLayout.getBoardLayout());
        mongoAgileDataPageViewTemplate.upsert(query, update, BoardLayoutMongoData.class, BOARD_LAYOUT_NAME);
    }

    /**
     * 构造用户自定义布局项
     */
    private GridGroupDTO buildUserLayoutGroup(ThemeMapBoardDTO.DataBoardQuestion question, String showType,
                                              int rowStart, LayoutSize layoutSize) {
        GridGroupDTO gridGroupDTO = new GridGroupDTO();
        gridGroupDTO.setId(question.getQuestionId());
        gridGroupDTO.setType(showType);
        gridGroupDTO.setX(0);
        gridGroupDTO.setY(rowStart);
        gridGroupDTO.setCols(layoutSize.getCols());
        gridGroupDTO.setRows(layoutSize.getRows());
        return gridGroupDTO;
    }

    /**
     * 构造 AI 布局区域，统一列宽并顺延行坐标
     */
    private AIBoardLayout.layoutArea buildLayoutArea(ThemeMapBoardDTO themeMapBoardDTO, EchoAIBoardDTO echoAIBoardDTO,
                                                     ThemeMapBoardDTO.DataBoardQuestion question, String chartId, String showType,
                                                     int rowStart, LayoutSize layoutSize) {
        AIBoardLayout.layoutArea layoutArea = new AIBoardLayout.layoutArea();
        layoutArea.setChartId(chartId);
        layoutArea.setShowType(showType);
        layoutArea.setTitle(resolveChartTitle(themeMapBoardDTO, question, echoAIBoardDTO));

        AIBoardLayout.layoutDimensions dimensions = new AIBoardLayout.layoutDimensions();
        dimensions.setAreaCols(layoutSize.getCols());
        dimensions.setAreaRow(layoutSize.getRows());
        layoutArea.setDimensions(dimensions);

        AIBoardLayout.layoutPosition position = new AIBoardLayout.layoutPosition();
        AIBoardLayout.layoutIndex colIndex = new AIBoardLayout.layoutIndex();
        colIndex.setStart(0);
        colIndex.setEnd(layoutSize.getCols());
        position.setLayoutCol(colIndex);

        AIBoardLayout.layoutIndex rowIndex = new AIBoardLayout.layoutIndex();
        rowIndex.setStart(rowStart);
        rowIndex.setEnd(rowStart + layoutSize.getRows());
        position.setLayoutRow(rowIndex);
        layoutArea.setPosition(position);
        return layoutArea;
    }

    /**
     * 优先从 AI 问句结果获取呈现类型
     */
    private String resolveChartShowType(ThemeMapBoardDTO.DataBoardQuestion question, EchoAIBoardDTO echoAIBoardDTO) {
        if (echoAIBoardDTO != null && CollectionUtils.isNotEmpty(echoAIBoardDTO.getQuestionInfo())) {
            for (AIBoardQuestionDTO questionInfo : echoAIBoardDTO.getQuestionInfo()) {
                if (StringUtils.equals(String.valueOf(questionInfo.getQuestionId()), question.getQuestionId())
                        && StringUtils.isNotBlank(questionInfo.getShowType())) {
                    return questionInfo.getShowType();
                }
            }
        }
        return question.getShowType() == null ? null : String.valueOf(question.getShowType());
    }

    /**
     * 兜底补齐图表标题
     */
    private String resolveChartTitle(ThemeMapBoardDTO themeMapBoardDTO, ThemeMapBoardDTO.DataBoardQuestion question,
                                     EchoAIBoardDTO echoAIBoardDTO) {
        if (echoAIBoardDTO != null && CollectionUtils.isNotEmpty(echoAIBoardDTO.getQuestionInfo())) {
            for (AIBoardQuestionDTO questionInfo : echoAIBoardDTO.getQuestionInfo()) {
                if (StringUtils.equals(String.valueOf(questionInfo.getQuestionId()), question.getQuestionId())
                        && StringUtils.isNotBlank(questionInfo.getChartTitle())) {
                    return questionInfo.getChartTitle();
                }
            }
        }
        if (themeMapBoardDTO != null && CollectionUtils.isNotEmpty(themeMapBoardDTO.getDataBoardQuestions())) {
            return question.getQuestion();
        }
        return null;
    }

    /**
     * 计算自定义布局下一行的起点
     */
    private int calculateNextRowStart(List<GridGroupDTO> layout) {
        if (CollectionUtils.isEmpty(layout)) {
            return 0;
        }
        return layout.stream()
                .mapToInt(item -> (item.getY() == null ? 0 : item.getY()) + (item.getRows() == null ? 0 : item.getRows()))
                .max()
                .orElse(0);
    }

    /**
     * 计算 AI 布局下一行的起点
     */
    private int calculateNextRowStart(AIBoardLayout aiBoardLayout) {
        if (aiBoardLayout == null || CollectionUtils.isEmpty(aiBoardLayout.getCharts())) {
            return 0;
        }
        return aiBoardLayout.getCharts().stream()
                .mapToInt(chart -> {
                    AIBoardLayout.layoutPosition position = chart.getPosition();
                    Integer rowStart = null;
                    Integer rowEnd = null;
                    if (position != null && position.getLayoutRow() != null) {
                        rowStart = position.getLayoutRow().getStart();
                        rowEnd = position.getLayoutRow().getEnd();
                    }
                    if (rowEnd != null) {
                        return rowStart + chart.getDimensions().getAreaRow();
                    }
                    int start = rowStart == null ? 0 : rowStart;
                    return start + extractRowHeight(chart, null);
                })
                .max()
                .orElse(0);
    }

    /**
     * 提取布局项高度
     */
    private int extractRowHeight(AIBoardLayout.layoutArea chart, String snapshotId) {
        if (chart == null) {
            return 0;
        }
        Integer areaRow = chart.getDimensions() == null ? null : chart.getDimensions().getAreaRow();
        if (areaRow != null && areaRow > 0) {
            return areaRow;
        }
        if (chart.getPosition() != null && chart.getPosition().getLayoutRow() != null) {
            AIBoardLayout.layoutIndex row = chart.getPosition().getLayoutRow();
            Integer start = row.getStart();
            Integer end = row.getEnd();
            if (end != null) {
                if (start != null) {
                    return Math.max(0, end - start);
                }
                return end;
            }
        }
        LayoutSize layoutSize = resolveLayoutSize(chart.getShowType(), null);
        return layoutSize.getRows();
    }

    /**
     * 依据呈现类型返回固定尺寸
     */
    private LayoutSize resolveLayoutSize(String showType, String snapshotId) {
        if (StringUtils.equals("2", showType)) {
            if (StringUtils.isEmpty(snapshotId)) {
                return LayoutSize.of(2, 2);
            }
            SnapDataDTO snapDataDTO = adtService.transAgileData(String.valueOf(showType), snapshotId, "zh_CN");
            AgileDataIndexComponent agileDataIndexComponent = getIndexComponent(snapshotId, snapDataDTO, "zh_CN");
            if (agileDataIndexComponent != null && CollectionUtils.isNotEmpty(agileDataIndexComponent.getGroup())) {
                int size = agileDataIndexComponent.getGroup().size() * 2;
                int rows = (size + 12 - 1) / 12;
                if (rows > 1) {
                    size = 12;
                }
                return LayoutSize.of(size, rows * 2);
            } else {
                return LayoutSize.of(2, 2);
            }
        }
        if (StringUtils.equals("0", showType)) {
            return LayoutSize.of(DEFAULT_LAYOUT_COLS, 6);
        }
        return LayoutSize.of(6, 6);
    }

    /**
     * 保证布局列数满足默认宽度
     */
    private int ensureLayoutCols(AIBoardLayout aiBoardLayout) {
        if (aiBoardLayout.getOverallLayout() == null) {
            AIBoardLayout.layoutAll layoutAll = new AIBoardLayout.layoutAll();
            layoutAll.setLayoutCols(DEFAULT_LAYOUT_COLS);
            layoutAll.setLayoutRows(0);
            aiBoardLayout.setOverallLayout(layoutAll);
        }
        Integer layoutCols = aiBoardLayout.getOverallLayout().getLayoutCols();
        if (layoutCols == null || layoutCols < DEFAULT_LAYOUT_COLS) {
            aiBoardLayout.getOverallLayout().setLayoutCols(DEFAULT_LAYOUT_COLS);
            return DEFAULT_LAYOUT_COLS;
        }
        return layoutCols;
    }

    /**
     * 解析看板问句与布局 chartId 的真实映射
     */
    private String resolveBoardChartId(ThemeMapBoardDTO.DataBoardQuestion question) {
        if (question == null) {
            return null;
        }
        Map<String, Object> dataParam = question.getDataParam();
        if (MapUtils.isNotEmpty(dataParam)) {
            Object chartId = dataParam.get("questionId");
            if (chartId != null) {
                return String.valueOf(chartId);
            }
        }
        return question.getQuestionId();
    }


    private static final class LayoutSize {
        private final int cols;
        private final int rows;

        private LayoutSize(int cols, int rows) {
            this.cols = cols;
            this.rows = rows;
        }

        static LayoutSize of(int cols, int rows) {
            return new LayoutSize(cols, rows);
        }

        int getCols() {
            return cols;
        }

        int getRows() {
            return rows;
        }
    }

    private void getAnalyzeResult(EchoAIBoardDTO echoAIBoardDTO) {
        List<AIBoardQuestionDTO> filteredQuestions = echoAIBoardDTO.getQuestionInfo().stream()
                .filter(question -> question.getAnalyzeStatus() != null && question.getAnalyzeStatus() == 0)
                .collect(Collectors.toList());
        echoAIBoardDTO.setQuestionInfo(filteredQuestions);
    }


    /**
     * 获取指标卡组件
     * @param questionId
     * @param snapDataDTO
     * @return
     */
    private AgileDataIndexComponent getIndexComponent(String questionId, SnapDataDTO snapDataDTO, String local) {
        if (snapDataDTO == null) {
            return null;
        }
        String actionId = snapDataDTO.getPageData().keySet().iterator().next();
        ExecuteContext executeContext = new ExecuteContext();
        executeContext.setLocale(local);
        ApiMetadata apiMetadata = agileDataPageDefineThemeMapService.getApiMetadata(snapDataDTO, executeContext, actionId);
        return AgileDataBoardComponentBuild.buildBoardIndex(questionId, ConvertGetAgileData.getRootMetadataField(apiMetadata));
    }

    private AgileDataIndexComponent getEmptyIndexComponent(String questionId) {
        AgileDataIndexComponent agileDataIndexComponent = new AgileDataIndexComponent();
        agileDataIndexComponent.setType(ECHOConstants.ComponentType.STATISTIC);
        List<AgileDataIndexComponent.AgileDataIndex> group = Lists.newArrayList();
        group.add(new AgileDataIndexComponent.AgileDataIndex());
        agileDataIndexComponent.setGroup(group);
        agileDataIndexComponent.setId(questionId);
        return agileDataIndexComponent;
    }

    /**
     * 生成看板呈现结构
     * @param user
     * @param boardLayoutParamsDTO
     * @param local
     * @return
     */
    @Override
    public Map<String, Object> generateBoardLayout(AuthoredUser user, BoardLayoutParamsDTO boardLayoutParamsDTO, String local) {
        Map<String, Object> result = Maps.newHashMap();
        //获取看板数据
        BoardSnapResultDTO boardData = adtService.getBoardData(boardLayoutParamsDTO, local);
        if (boardData.getSnapShotDTO() == null) {
            result.put("result", false);
            result.put("msg", boardData.getMsg());
            result.put("title", boardData.getTitle());
            return result;
        }
        String snapshotId = boardData.getSnapShotDTO().getSnapshotId();
        //生成DSL结构
        ExecuteContext executeContext = ExecuteContext.createByHttpRequest();
        //获取敏捷报表信息
        SnapDataDTO agileReportDTO = adtService.getAdtReportBySnapshotId(snapshotId, executeContext.getLocale());
        //构建标题
        this.getDefineTitle(boardData.getSnapShotDTO(), executeContext);
        if (AgileDataCheck.sceneBasicCheck(agileReportDTO)) {
            result.put("result", false);
            result.put("title", executeContext.getDefineTitle());
            result.put("msg", MessageUtils.getMessageByCurrentLanguage(EchoShowConstants.getLanMap("查询结果为空，请确认查询条件是否正确。", "查詢結果為空，請確認查詢條件是否正確。", "no data")));
            return result;
        }
        //解析报表信息
        executeContext.appendAgileDataPageInfo(AgileDataConstants.REPORT_PAGE, agileReportDTO, null, "0");
        //封装呈现内容
        DynamicAgileData dynamicAgileData = agileReportService.createAgileReportTemplateTable(executeContext, agileReportDTO);
        result.put("dsl", dynamicAgileData);
        if (dynamicAgileData != null) {
            result.put("result", true);
        } else {
            result.put("result", false);
        }

        Object dataTipMessage = boardData.getSnapShotDTO().getContext().getBizParams().get("dataTipMessage");
        if (dataTipMessage != null) {
            result.put("dataTipMessage", dynamicAgileData);
        }
        return result;
    }

    /**
     * 看板标题修改
     * @param boardInfo
     * @param executeContext
     */
    private void getDefineTitle(SnapShotInfo boardInfo, ExecuteContext executeContext) {
        if (boardInfo == null) {
            return;
        }
        SnapShotInfo.SnapShotInfoContext context = boardInfo.getContext();
        if (context == null) {
            return;
        }
        Map<String, Object> bizParams = context.getBizParams();
        if (MapUtils.isEmpty(bizParams)) {
            return;
        }
        Map<String, Object> showDefine = MapUtils.getMap(bizParams, "showDefine");
        if (MapUtils.isEmpty(showDefine)) {
            return;
        }
        List<Map<String, Object>> showType = (List<Map<String, Object>>) MapUtils.getObject(showDefine, "showType");
        if (CollectionUtils.isEmpty(showType)) {
            return;
        }
        AtomicReference<String> title = new AtomicReference<>(StringUtils.EMPTY);
        showType.stream().forEach(showTypeDTO -> {
            List<Map<String, Object>> typeList = (List<Map<String, Object>>) MapUtils.getObject(showTypeDTO, "type");
            if (CollectionUtils.isNotEmpty(typeList)) {
                typeList.stream().forEach(type -> title.set(MapUtils.getString(type, "title")));
            }
        });
        executeContext.setDefineTitle(title.get());
    }

    /**
     * 保存看板布局
     * @param authoredUser
     * @param boardLayoutDTO
     * @return
     */
    @Override
    public Boolean saveDynamicLayoutBoard(AuthoredUser authoredUser, BoardLayoutDTO boardLayoutDTO) {
        if (CollectionUtils.isNotEmpty(boardLayoutDTO.getBoardLayout())) {
            Query query = Query.query(Criteria.where("userId").is(authoredUser.getUserId())
                    .and("tenantId").is(authoredUser.getTenantId())
                    .and("boardId").is(boardLayoutDTO.getBoardId()));
            BoardLayoutMongoData boardLayoutMongoData = mongoAgileDataPageViewTemplate.findOne(query, BoardLayoutMongoData.class, BOARD_LAYOUT_NAME);
            if (boardLayoutMongoData == null) {
                boardLayoutMongoData = new BoardLayoutMongoData();
                boardLayoutMongoData.setBoardLayout(boardLayoutDTO.getBoardLayout());
                boardLayoutMongoData.setBoardId(boardLayoutDTO.getBoardId());
                boardLayoutMongoData.setUserId(authoredUser.getUserId());
                boardLayoutMongoData.setTenantId(authoredUser.getTenantId());
                mongoAgileDataPageViewTemplate.insert(boardLayoutMongoData, BOARD_LAYOUT_NAME);
            } else {
                Update update = new Update();
                update.set("boardLayout", boardLayoutDTO.getBoardLayout());
                mongoAgileDataPageViewTemplate.upsert(query, update, AgileDataUserDefine.class, BOARD_LAYOUT_NAME);
            }
        }
        return true;
    }

    /**
     * 获取看板布局
     * @param authoredUser
     * @param boardId
     * @return
     */
    @Override
    public BoardLayoutMongoData getDynamicLayoutBoard(AuthoredUser authoredUser, String boardId) {
        Query query = Query.query(Criteria.where("userId").is(authoredUser.getUserId())
                .and("tenantId").is(authoredUser.getTenantId())
                .and("boardId").is(boardId));
        return mongoAgileDataPageViewTemplate.findOne(query, BoardLayoutMongoData.class, BOARD_LAYOUT_NAME);
    }
}