package com.digiwin.athena.show.service.ppt.impl;

import com.digiwin.athena.appcore.auth.domain.AuthoredUser;
import com.digiwin.athena.appcore.util.JsonUtils;
import com.digiwin.athena.show.domain.ppt.AgileDataFileDTO;
import com.digiwin.athena.show.domain.ppt.AgileDataFileQueryReq;
import com.digiwin.athena.show.service.ppt.AgileDataFileManagerService;
import com.google.common.collect.Maps;
import com.mongodb.BasicDBObject;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.data.mongodb.core.MongoTemplate;
import org.springframework.data.mongodb.core.query.Criteria;
import org.springframework.data.mongodb.core.query.Query;
import org.springframework.data.mongodb.core.query.Update;
import org.springframework.stereotype.Service;

import java.time.LocalTime;
import java.util.Map;
import java.util.Objects;

@Slf4j
@Service
public class AgileDataFileManagerServiceImpl implements AgileDataFileManagerService {

    @Autowired
    @Qualifier("mongoAgileDataPageViewTemplate")
    private MongoTemplate mongoAgileDataPageViewTemplate;

    private static final String MONGODB_COLLECTION_NAME = "agile_data_file_analyze";

    /**
     * 查询分析规划问句信息
     * @param agileDataFileQueryReq
     * @return
     */
    @Override
    public AgileDataFileDTO queryAgileDataFile(AuthoredUser user,AgileDataFileQueryReq agileDataFileQueryReq) {
        if(agileDataFileQueryReq == null){
            return null;
        }
        if(agileDataFileQueryReq.getMessageId() == null){
            return null;
        }
        //根据消息ID查询分析规划结果
        Query query = Query.query(Criteria.where("userId").is(user.getUserId())
                .and("tenantId").is(user.getTenantId())
                .and("messageId").is(agileDataFileQueryReq.getMessageId()));
        AgileDataFileDTO agileDataFileDTO = mongoAgileDataPageViewTemplate.findOne(query, AgileDataFileDTO.class, MONGODB_COLLECTION_NAME);
        if(agileDataFileDTO == null){
            return  null;
        }
        //筛选分析规划结果
        if(CollectionUtils.isNotEmpty(agileDataFileQueryReq.getQuestionIds())){
            if(CollectionUtils.isNotEmpty(agileDataFileDTO.getQuestionInfo())){
                agileDataFileDTO.getQuestionInfo().removeIf(questionInfo ->
                        !agileDataFileQueryReq.getQuestionIds().contains(questionInfo.getQuestionId()));
            }
        }
        return agileDataFileDTO;
    }

    /**
     * 新增分析规划结果
     * @param user
     * @param agileDataFileDTO
     * @return
     */
    @Override
    public Boolean saveAgileDataFile(AuthoredUser user, AgileDataFileDTO agileDataFileDTO) {
        if(agileDataFileDTO == null){
            return false;
        }
        agileDataFileDTO.setTenantId(user.getTenantId());
        agileDataFileDTO.setUserId(user.getUserId());
        agileDataFileDTO.setCreateTime(LocalTime.now());
        mongoAgileDataPageViewTemplate.insert(agileDataFileDTO,MONGODB_COLLECTION_NAME);
        return true;
    }

    /**
     * 更新分析规划结果
     * @param user
     * @param agileDataFileDTO
     * @return
     */
    @Override
    public Boolean updateAgileDataFile(AuthoredUser user, AgileDataFileDTO agileDataFileDTO) {
        if(agileDataFileDTO == null){
            return false;
        }
        agileDataFileDTO.setTenantId(user.getTenantId());
        agileDataFileDTO.setUserId(user.getUserId());

        //更新规划问句
        if(CollectionUtils.isNotEmpty(agileDataFileDTO.getQuestionInfo())) {
            //更新规划问句
            agileDataFileDTO.getQuestionInfo().stream().forEach(questionInfo -> {
                Query query = new Query(Criteria.where("messageId").is(agileDataFileDTO.getMessageId())
                        .and("tenantId").is(user.getTenantId())
                        .and("questionInfo.questionId").is(questionInfo.getQuestionId())
                        .and("userId").is(user.getUserId()));
                Update update = new Update();
                if(agileDataFileDTO.getMessageStatus() != null){
                    update.set("messageStatus", agileDataFileDTO.getMessageStatus());
                }

                // 构建更新操作
                if(questionInfo.getLock() != null) {
                    update.set("questionInfo.$.lock", questionInfo.getLock());
                }
                // 更新问句，同时清除数据集
                if(StringUtils.isNotEmpty(questionInfo.getQuestion())) {
                    log.info("更新PPT问句信息，{}", JsonUtils.objectToString(agileDataFileDTO));
                    update.set("questionInfo.$.question", questionInfo.getQuestion());
                    update.set("questionInfo.$.title", null);
                    update.set("questionInfo.$.datasetIds", null);
                }
                if(questionInfo.getAnalyzeStatus() != null) {
                    update.set("questionInfo.$.analyzeStatus", questionInfo.getAnalyzeStatus());
                }
                if(questionInfo.getAnalyzeResult() != null) {
                    update.set("questionInfo.$.analyzeResult", questionInfo.getAnalyzeResult());
                }
                if(StringUtils.isNotEmpty(questionInfo.getSnapshotId())) {
                    update.set("questionInfo.$.snapshotId", questionInfo.getSnapshotId());
                }
                if(StringUtils.isNotEmpty(questionInfo.getSummarizeData())) {
                    update.set("questionInfo.$.summarizeData", questionInfo.getSummarizeData());
                }
                if(Objects.nonNull(questionInfo.getAnalyzeMessageId())) {
                    update.set("questionInfo.$.analyzeMessageId", questionInfo.getAnalyzeMessageId());
                }

                update.set("updateTime", LocalTime.now());

                // 执行更新（更新第一个匹配的文档）
                mongoAgileDataPageViewTemplate.updateFirst(query, update, MONGODB_COLLECTION_NAME);
            });
        } else if(CollectionUtils.isNotEmpty(agileDataFileDTO.getPptInfo())){
            //更新ppt状态
            agileDataFileDTO.getPptInfo().stream().forEach(pptInfo -> {
                Query query = new Query(Criteria.where("messageId").is(agileDataFileDTO.getMessageId())
                        .and("tenantId").is(user.getTenantId())
                        .and("userId").is(user.getUserId())
                        .and("pptInfo.pptId").is(pptInfo.getPptId()));
                AgileDataFileDTO agileDataFile = mongoAgileDataPageViewTemplate.findOne(query, AgileDataFileDTO.class,MONGODB_COLLECTION_NAME);
                if(agileDataFile != null) {
                    Query updateQuery = new Query(Criteria.where("messageId").is(agileDataFileDTO.getMessageId())
                            .and("tenantId").is(user.getTenantId())
                            .and("userId").is(user.getUserId())
                            .and("pptInfo.pptId").is(pptInfo.getPptId()));
                    Update update = new Update();
                    if (StringUtils.isNotEmpty(pptInfo.getGenerateStatus())) {
                        update.set("pptInfo.$.generateStatus", pptInfo.getGenerateStatus());
                    }
                    if (StringUtils.isNotEmpty(pptInfo.getPptUrl())) {
                        update.set("pptInfo.$.pptUrl", pptInfo.getPptUrl());
                    }
                    if (StringUtils.isNotEmpty(pptInfo.getPptPreviewUrl())) {
                        update.set("pptInfo.$.pptPreviewUrl", pptInfo.getPptPreviewUrl());
                    }
                    if (StringUtils.isNotEmpty(pptInfo.getPptDocId())) {
                        update.set("pptInfo.$.pptDocId", pptInfo.getPptDocId());
                    }
                    if (StringUtils.isNotEmpty(pptInfo.getPptName())) {
                        update.set("pptInfo.$.pptName", pptInfo.getPptName());
                    }
                    if (StringUtils.isNotEmpty(pptInfo.getCreateDate())) {
                        update.set("pptInfo.$.createDate", pptInfo.getCreateDate());
                    }
                    if (StringUtils.isNotEmpty(pptInfo.getExpireDate())) {
                        update.set("pptInfo.$.expireDate", pptInfo.getExpireDate());
                    }
                    if (StringUtils.isNotEmpty(pptInfo.getPptPageSize())) {
                        update.set("pptInfo.$.pptPageSize", pptInfo.getPptPageSize());
                    }
                    update.set("updateTime", LocalTime.now());
                    mongoAgileDataPageViewTemplate.updateFirst(updateQuery, update, MONGODB_COLLECTION_NAME);
                } else {
                    Query addQuery = new Query(Criteria.where("messageId").is(agileDataFileDTO.getMessageId())
                            .and("tenantId").is(user.getTenantId())
                            .and("userId").is(user.getUserId()));
                    Map<String, Object> pptMap = Maps.newHashMap();
                    pptMap.put("pptId",pptInfo.getPptId());
                    Update update = new Update()
                            .push("pptInfo", pptMap);
                    update.set("updateTime", LocalTime.now());
                    mongoAgileDataPageViewTemplate.updateFirst(addQuery, update, MONGODB_COLLECTION_NAME);
                }
            });

        } else{
            //更新消息状态
            Query query = new Query(Criteria.where("messageId").is(agileDataFileDTO.getMessageId())
                    .and("tenantId").is(user.getTenantId())
                    .and("userId").is(user.getUserId()));
            Update update = new Update();
            if (agileDataFileDTO.getMessageStatus() != null) {
                update.set("messageStatus", agileDataFileDTO.getMessageStatus());
            }
            update.set("updateTime", LocalTime.now());
            mongoAgileDataPageViewTemplate.updateFirst(query, update, MONGODB_COLLECTION_NAME);
        }
        return true;
    }

    /**
     * 删除分析规划结果
     * @param user
     * @param agileDataFileQueryReq
     * @return
     */
    @Override
    public Boolean deleteAgileDataFile(AuthoredUser user, AgileDataFileQueryReq agileDataFileQueryReq) {
        if(agileDataFileQueryReq == null){
            return false;
        }
        //根据消息ID查询分析规划结果
        Query query = Query.query(Criteria.where("userId").is(user.getUserId())
                .and("tenantId").is(user.getTenantId())
                .and("messageId").is(agileDataFileQueryReq.getMessageId()));
        if(CollectionUtils.isNotEmpty(agileDataFileQueryReq.getQuestionIds())){
            BasicDBObject pullCondition = new BasicDBObject("questionId",
                    new BasicDBObject("$in", agileDataFileQueryReq.getQuestionIds())
            );
            Update update = new Update();
            update.pull("questionInfo", pullCondition);

            mongoAgileDataPageViewTemplate.updateFirst(
                    query,
                    update,
                    MONGODB_COLLECTION_NAME
            );
        } else {
            mongoAgileDataPageViewTemplate.remove(query, MONGODB_COLLECTION_NAME);
        }
        return true;
    }
}
