package com.digiwin.athena.show.util.layout;

import com.digiwin.athena.agiledataecho.domain.AIBoardScrumBIShowTypeEnum;
import com.digiwin.athena.agiledataecho.dto.aiBoard.AIBoardLayout;
import com.digiwin.athena.agiledataecho.dto.aiBoard.AIBoardQuestionDTO;
import com.digiwin.athena.agiledataecho.dto.aiBoard.EchoAIBoardDTO;
import com.digiwin.athena.agiledataecho.dto.aiBoard.EchoAIBoardLayoutDTO;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import org.apache.commons.collections.CollectionUtils;

import java.util.*;
import java.util.stream.Collectors;

/**
 * 看板布局规划贪心算法（实现Gridster布局）
 */
public final class BoardLayoutPlanner {

    private BoardLayoutPlanner() {}

    /**
     * 规划输入
     */
    public static class Seed {
        public String chartId;
        public String showType; // 按项目既有值（如 0=表格, 2=指标卡, 其它=图形枚举值）
        public String title;
        public double importance = 1.0;
        public int row = 1;

        public Seed(String chartId, String showType, String title, Double importance, Integer row) {
            this.chartId = chartId;
            this.showType = showType;
            this.title = title;
            if (importance != null) this.importance = importance;
            if (row != null) this.row = row;
        }
    }

    /**
     * 基于 Skyline 贪心的 12 列网格布局规划
     * - 表格: 12x6 固定
     * - 指标卡: 2x2 固定（同组内统一自新行开始成块）
     * - 图形: 默认 6x4；若同组仅此一个且无指标卡，则直接铺满整行为 12x6（全宽x6）
     * - 仅允许同组内横向填充；不同组不回填上一组行的右侧空位（下一组地板=当前全局最大高度）
     * - 行组顺序由 seed.row 升序决定；总高按实际 skyline 更新，不再对齐到 6 的倍数
     */
    public static AIBoardLayout plan(List<Seed> seeds, int layoutCols, int initialLayoutRows) {
        AIBoardLayout layout = new AIBoardLayout();
        AIBoardLayout.layoutAll overall = new AIBoardLayout.layoutAll();
        overall.setLayoutCols(layoutCols);
        overall.setLayoutRows(initialLayoutRows);
        layout.setOverallLayout(overall);

        if (seeds == null || seeds.isEmpty()) {
            layout.setCharts(new ArrayList<>());
            return layout;
        }

        // 记录入参顺序，后续仅调整返回列表顺序，不影响布局计算
        List<String> originalOrder = seeds.stream()
                .map(s -> Objects.toString(s.chartId, ""))
                .collect(Collectors.toList());

        // 排序：row 升序 -> importance 升序
        seeds.sort(Comparator
                .comparingInt((Seed s) -> s.row)
                .thenComparingDouble((Seed s) -> s.importance)
        );

        int cols = Math.max(1, layoutCols);
        int totalRows = Math.max(6, initialLayoutRows);
        List<AIBoardLayout.layoutArea> areas = new ArrayList<>();
        List<Placed> placed = new ArrayList<>();

        // 全局 Skyline：按组设置地板=当前全局最大高度，仅允许同组内右侧填充，不允许跨组回填
        int[] skyline = new int[cols];
        int i = 0;
        while (i < seeds.size()) {
            int currentRow = seeds.get(i).row;
            List<Seed> group = new ArrayList<>();
            while (i < seeds.size() && seeds.get(i).row == currentRow) {
                group.add(seeds.get(i));
                i++;
            }

            int groupFloor = currentMax(skyline); // 该组从全局最高点开始，禁止跨组回填
            // 先放置非指标(表格+图形)，后放置指标卡，保证指标卡成块且从新的一行开始
            List<Seed> others = new ArrayList<>();
            List<Seed> metrics = new ArrayList<>();
            for (Seed seed : group) {
                if (isMetric(seed.showType)) metrics.add(seed); else others.add(seed);
            }

            // 非指标：表格优先（全宽独占一行），再放置图形；按组地板放置，允许利用上一组右侧空位
            others.sort((a, b) -> {
                boolean at = isTable(a.showType);
                boolean bt = isTable(b.showType);
                if (at && !bt) return -1;
                if (!at && bt) return 1;
                // 同类维持原排序（已由外部按 importance 排过）
                return 0;
            });
            for (Seed seed : others) {
                Size base = decideBaseSize(seed.showType);
                boolean isChart = isChart(seed.showType);
                int w = base.w;
                int h = base.h;

                // 表格始终铺满整行
                if (isTable(seed.showType)) {
                    w = cols;
                }

                // 图形且本组仅此一个且无指标卡：直接铺满整行（cols x 6）
                if (isChart && metrics.isEmpty() && others.size() == 1) {
                    w = cols; h = 6;
                    Pos p = new Pos(0, groupFloor);
                    place(skyline, p.x, p.y, w, h);
                    totalRows = Math.max(totalRows, currentMax(skyline));
                    AIBoardLayout.layoutArea area = buildArea(seed, p.x, p.y, w, h);
                    areas.add(area);
                    placed.add(new Placed(area, seed.showType, p.x, p.y, w, h));
                    continue;
                }

                Pos tryBase = findPositionWithFloor(skyline, cols, w, groupFloor);
                place(skyline, tryBase.x, tryBase.y, w, h);
                totalRows = Math.max(totalRows, currentMax(skyline));
                AIBoardLayout.layoutArea area = buildArea(seed, tryBase.x, tryBase.y, w, h);
                areas.add(area);
                placed.add(new Placed(area, seed.showType, tryBase.x, tryBase.y, w, h));
            }

            // 指标卡：优先尝试与同组行内元素同一“行高”对齐（宽不变），仅当该行剩余连续空间足够容纳所有指标卡时
            if (!metrics.isEmpty()) {
                int rowY = groupFloor;
                int rowHeight = getRowMaxHeightAtY(placed, rowY);
                boolean placedInSameRow = false;
                if (rowHeight > 0) {
                    int needed = metrics.size() * 2; // 每张指标卡宽 2
                    Segment seg = largestFreeSegmentAtY(skyline, cols, rowY);
                    if (seg.length >= needed) {
                        int cursor = seg.start;
                        for (Seed seed : metrics) {
                            int w = 2, h = rowHeight;
                            place(skyline, cursor, rowY, w, h);
                            totalRows = Math.max(totalRows, currentMax(skyline));
                            AIBoardLayout.layoutArea area = buildArea(seed, cursor, rowY, w, h);
                            areas.add(area);
                            placed.add(new Placed(area, seed.showType, cursor, rowY, w, h));
                            cursor += w;
                        }
                        placedInSameRow = true;
                    }
                }
                // 否则：统一从新的一行开始，成块连续摆放（保持 2x2）
                if (!placedInSameRow) {
                    int metricFloor = Math.max(currentMax(skyline), groupFloor);
                    for (Seed seed : metrics) {
                        Size base = decideBaseSize(seed.showType); // 2x2
                        int w = base.w;
                        int h = base.h;
                        Pos p = findPositionWithFloor(skyline, cols, w, metricFloor);
                        place(skyline, p.x, p.y, w, h);
                        totalRows = Math.max(totalRows, currentMax(skyline));
                        AIBoardLayout.layoutArea area = buildArea(seed, p.x, p.y, w, h);
                        areas.add(area);
                        placed.add(new Placed(area, seed.showType, p.x, p.y, w, h));
                    }
                }
            }
            overall.setLayoutRows(Math.max(totalRows, currentMax(skyline)));
        }

        // 行末填充：对非指标元素，若在其垂直区间内右侧无障碍，则将其横向扩展至最近障碍或边界（填满该行）
        stretchRowForNonMetrics(placed, cols);

        // 将输出列表顺序恢复为入参顺序（仅调整列表顺序，不改变已计算的位置与尺寸）
        Map<String, Deque<AIBoardLayout.layoutArea>> bucket = new HashMap<>();
        for (AIBoardLayout.layoutArea a : areas) {
            String id = Objects.toString(a.getChartId(), "");
            bucket.computeIfAbsent(id, k -> new ArrayDeque<>()).addLast(a);
        }
        List<AIBoardLayout.layoutArea> orderedAreas = new ArrayList<>(areas.size());
        for (String id : originalOrder) {
            Deque<AIBoardLayout.layoutArea> q = bucket.get(id);
            if (q != null) {
                AIBoardLayout.layoutArea a = q.pollFirst();
                if (a != null) orderedAreas.add(a);
            }
        }
        // 追加任何未被覆盖到的元素（健壮性处理）
        for (Map.Entry<String, Deque<AIBoardLayout.layoutArea>> e : bucket.entrySet()) {
            Deque<AIBoardLayout.layoutArea> q = e.getValue();
            while (q != null && !q.isEmpty()) orderedAreas.add(q.pollFirst());
        }

        layout.setCharts(orderedAreas);
        return layout;
    }

    private static AIBoardLayout.layoutArea buildArea(Seed seed, int x, int y, int w, int h) {
        AIBoardLayout.layoutArea area = new AIBoardLayout.layoutArea();
        area.setChartId(seed.chartId);
        area.setShowType(seed.showType);
        area.setTitle(seed.title);

        AIBoardLayout.layoutDimensions dim = new AIBoardLayout.layoutDimensions();
        dim.setAreaCols(w);
        dim.setAreaRow(h);
        area.setDimensions(dim);

        AIBoardLayout.layoutIndex col = new AIBoardLayout.layoutIndex();
        col.setStart(x);
        col.setEnd(x + w);
        AIBoardLayout.layoutIndex row = new AIBoardLayout.layoutIndex();
        row.setStart(y);
        row.setEnd(y + h);
        AIBoardLayout.layoutPosition pos = new AIBoardLayout.layoutPosition();
        pos.setLayoutCol(col);
        pos.setLayoutRow(row);
        area.setPosition(pos);
        return area;
    }

    private static boolean isTable(String showType) {
        return Objects.equals(showType, "0");
    }

    private static boolean isMetric(String showType) {
        return Objects.equals(showType, "2");
    }

    private static boolean isChart(String showType) {
        return !isTable(showType) && !isMetric(showType);
    }

    private static Size decideBaseSize(String showType) {
        // 表格固定 12x6；指标卡固定 2x2；图形统一 6x4
        if (isTable(showType)) return new Size(12, 6);
        if (isMetric(showType)) return new Size(2, 2);
        return new Size(6, 4);
    }

    private static void place(int[] skyline, int x, int y, int w, int h) {
        for (int i = x; i < x + w; i++) {
            skyline[i] = y + h;
        }
    }

    private static int currentMax(int[] skyline) {
        int m = 0;
        for (int v : skyline) m = Math.max(m, v);
        return m;
    }

    private static Pos findPositionWithFloor(int[] skyline, int cols, int w, int floor) {
        if (w > cols) return null;
        int bestX = 0;
        int bestY = Integer.MAX_VALUE;
        for (int x = 0; x <= cols - w; x++) {
            int y = 0;
            for (int i = x; i < x + w; i++) y = Math.max(y, skyline[i]);
            y = Math.max(y, floor);
            if (y < bestY) {
                bestY = y;
                bestX = x;
            }
        }
        return new Pos(bestX, bestY);
    }

    private static boolean rowIsEmptyAtFloor(int[] skyline, int cols, int floor) {
        for (int i = 0; i < cols; i++) {
            if (skyline[i] > floor) return false;
        }
        return true;
    }

    private static int getRowMaxHeightAtY(List<Placed> placed, int y) {
        int h = 0;
        for (Placed p : placed) {
            if (p.y == y) h = Math.max(h, p.h);
        }
        return h;
    }

    private static class Segment { int start; int length; Segment(int s, int l){start=s;length=l;} }

    private static Segment largestFreeSegmentAtY(int[] skyline, int cols, int y) {
        int bestStart = -1, bestLen = 0;
        int curStart = -1, curLen = 0;
        for (int i = 0; i < cols; i++) {
            if (skyline[i] <= y) {
                if (curStart == -1) { curStart = i; curLen = 1; }
                else { curLen++; }
            } else {
                if (curLen > bestLen) { bestLen = curLen; bestStart = curStart; }
                curStart = -1; curLen = 0;
            }
        }
        if (curLen > bestLen) { bestLen = curLen; bestStart = curStart; }
        if (bestStart == -1) return new Segment(0, 0);
        return new Segment(bestStart, bestLen);
    }

    private static class Size {
        final int w; final int h;
        Size(int w, int h) { this.w = w; this.h = h; }
    }

    private static class Pos {
        final int x; final int y;
        Pos(int x, int y) { this.x = x; this.y = y; }
    }

    private static class Placed {
        final AIBoardLayout.layoutArea area;
        final String showType;
        int x, y, w, h;
        Placed(AIBoardLayout.layoutArea area, String showType, int x, int y, int w, int h) {
            this.area = area; this.showType = showType; this.x = x; this.y = y; this.w = w; this.h = h;
        }
    }

    private static void stretchRowForNonMetrics(List<Placed> placed, int cols) {
        for (Placed p : placed) {
            if (p == null) continue;
            if (Objects.equals(p.showType, "2") || Objects.equals(p.showType, "0")) continue;
            // 仅当该元素从行首开始
            if (p.x != 0) continue;
            int yStart = p.y;
            int yEnd = p.y + p.h;
            int obstacleX = cols;
            for (Placed q : placed) {
                if (q == p) continue;
                int qY1 = q.y, qY2 = q.y + q.h;
                boolean overlapY = qY1 < yEnd && qY2 > yStart;
                if (!overlapY) continue;
                if (q.x >= p.x + p.w) {
                    obstacleX = Math.min(obstacleX, q.x);
                }
            }
            int newW = Math.max(p.w, obstacleX - p.x);
            if (newW > p.w) {
                p.w = newW;
                // 更新 area（宽高与位置）
                AIBoardLayout.layoutDimensions dim = new AIBoardLayout.layoutDimensions();
                dim.setAreaCols(p.w);
                dim.setAreaRow(p.h);
                p.area.setDimensions(dim);

                AIBoardLayout.layoutIndex col = new AIBoardLayout.layoutIndex();
                col.setStart(p.x);
                col.setEnd(p.x + p.w);
                AIBoardLayout.layoutIndex row = new AIBoardLayout.layoutIndex();
                row.setStart(p.y);
                row.setEnd(p.y + p.h);
                AIBoardLayout.layoutPosition pos = new AIBoardLayout.layoutPosition();
                pos.setLayoutCol(col);
                pos.setLayoutRow(row);
                p.area.setPosition(pos);
            }
        }
    }

    public static List<Seed> getSeedList(AIBoardLayout aiBoardLayout, EchoAIBoardDTO echoAIBoardDTO) {
        List<Seed> seeds = Lists.newArrayList();
        aiBoardLayout.getCharts().forEach(c -> {
            String chartId = c.getChartId();
            Double importance = c.getImportance() == null ? null : c.getImportance();
            Integer row = c.getRow();
            Integer stVal = null;
            String title = null;
            if (CollectionUtils.isNotEmpty(echoAIBoardDTO.getQuestionInfo())) {
                for (AIBoardQuestionDTO q : echoAIBoardDTO.getQuestionInfo()) {
                    if (q.getAnalyzeStatus() != null && q.getAnalyzeStatus() == 0
                            && String.valueOf(q.getQuestionId()).equals(chartId)) {
                        stVal = AIBoardScrumBIShowTypeEnum.getShowTypeValueByBIType(q.getShowType());
                        title = q.getChartTitle();
                        break;
                    }
                }
            }
            String showTypeStr = (stVal != null) ? String.valueOf(stVal) : c.getShowType();
            String resolvedTitle = (title != null) ? title : c.getTitle();
            seeds.add(new Seed(chartId, showTypeStr, resolvedTitle, importance, row));
        });
        return seeds;
    }


    public static AIBoardLayout getAiBoardLayout(List<EchoAIBoardLayoutDTO> layoutDTOList, EchoAIBoardDTO echoAIBoardDTO, AIBoardLayout aiBoardLayout) {
        if (CollectionUtils.isNotEmpty(layoutDTOList)) {
            AIBoardLayout newLayout = new AIBoardLayout();
            AIBoardLayout.layoutAll overall = new AIBoardLayout.layoutAll();
            overall.setLayoutCols(12);
            overall.setLayoutRows(24);
            newLayout.setOverallLayout(overall);

            List<AIBoardLayout.layoutArea> charts = Lists.newArrayList();
            List<AIBoardQuestionDTO> qs = echoAIBoardDTO.getQuestionInfo();
            Map<Long, AIBoardQuestionDTO> qMap = (qs == null) ? Maps.newHashMap() :
                    qs.stream().collect(Collectors.toMap(AIBoardQuestionDTO::getQuestionId, v -> v, (a, b) -> a));
            int groupRow = 1; // 段起始行（summary 所在行）

            for (EchoAIBoardLayoutDTO seg : layoutDTOList) {
                List<Long> summary = seg.getSummary();
                List<Long> details = seg.getDetails();

                // 1) summary：全部落在同一组行 groupRow；importance 按列表顺序（越靠前越大）
                if (CollectionUtils.isNotEmpty(summary)) {
                    int n = summary.size();
                    for (int idx = 0; idx < n; idx++) {
                        Long sid = summary.get(idx);
                        AIBoardQuestionDTO q = qMap.get(sid);
                        if (q == null) continue;
                        AIBoardLayout.layoutArea c = new AIBoardLayout.layoutArea();
                        c.setChartId(String.valueOf(q.getQuestionId()));
                        c.setShowType(String.valueOf(AIBoardScrumBIShowTypeEnum.getShowTypeValueByBIType(q.getShowType())));
                        c.setTitle(q.getChartTitle());
                        c.setRow(groupRow);
                        // summary 的 importance 按输入顺序正序递增，确保与 layoutDTOList 顺序一致
                        c.setImportance(1.0 + (idx + 1) * 0.0001);
                        charts.add(c);
                    }
                }

                // 2) details：放在 groupRow+1，按宽度贪心装箱；不够一行则换到新组行
                int lastDetailGroupRow = groupRow;
                if (CollectionUtils.isNotEmpty(details)) {
                    int detailGroupRow = groupRow + 1;
                    int remaining = 12;
                    int n = details.size();
                    for (int idx = 0; idx < n; idx++) {
                        Long did = details.get(idx);
                        AIBoardQuestionDTO q = qMap.get(did);
                        if (q == null) continue;
                        String st = String.valueOf(AIBoardScrumBIShowTypeEnum.getShowTypeValueByBIType(q.getShowType()));
                        int width = computeBaseWidth(st);
                        if (remaining < width && remaining != 12) { // 放不下，换行
                            detailGroupRow++;
                            remaining = 12;
                        }

                        AIBoardLayout.layoutArea c = new AIBoardLayout.layoutArea();
                        c.setChartId(String.valueOf(q.getQuestionId()));
                        c.setShowType(st);
                        c.setTitle(q.getChartTitle());
                        c.setRow(detailGroupRow);
                        // details 的 importance 按输入顺序正序递增，确保与 layoutDTOList 顺序一致
                        c.setImportance(1.0 + (idx + 1) * 0.0001);
                        charts.add(c);

                        remaining -= Math.min(width, remaining);
                        lastDetailGroupRow = detailGroupRow;
                    }
                    // 下一段从最后一个 details 行的下一行开始
                    groupRow = lastDetailGroupRow + 1;
                } else {
                    // 无 details：下一段从 summary 行+1 开始
                    groupRow = groupRow + 1;
                }
            }
            newLayout.setCharts(charts);
            echoAIBoardDTO.setBoardLayout(newLayout);
            aiBoardLayout = newLayout;
        }
        return aiBoardLayout;
    }

    /**
     * 基础宽度推断（与 BoardLayoutPlanner 保持一致）
     */
    private static int computeBaseWidth(String showType) {
        if ("0".equals(showType)) return 12; // 表格
        if ("2".equals(showType)) return 2;  // 指标卡
        return 6; // 图形统一 6 列
    }

}