package com.digiwin.athena.agiledataecho.app.env;

import lombok.Data;
import lombok.extern.slf4j.Slf4j;
import org.slf4j.Logger;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;

import java.lang.management.ManagementFactory;
import java.lang.management.RuntimeMXBean;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.time.ZoneOffset;

/**
 * ADT配置信息
 */
@Component
@Slf4j
public class EchoEnvProperties {

    @Value("${ade.uri:}")
    private String adeUri;

    @Value("${adt.uri:}")
    private String adtUri;

    @Value("${semc.uri:}")
    private String semcUrl;

    @Value("${chatBI.uri:}")
    private String chatBIUrl;

    @Value("${chatBI.strategy:}")
    private String chatStrategy;

    @Value("${chatBI.gptModel:}")
    private String chatGptModel;

    @Value("${gmc.uri:}")
    private String gmcUrl;

    @Value("${themeMap.uri:}")
    private String knowledgeMapsUrl;


    /**
     * dap中间件
     */
    @Value("${iam.uri:}")
    private String iamUri;

    @Value("${eoc.uri:}")
    private String eocUri;

    @Value("${language:}")
    private String defaultLanguage;

    private LocalDateTime currentTime;
    private ZoneId currentZoneId;
    private long freeMemory;
    private long totalMemory;
    private long maxMemory;
    private String osName;
    private String host;
    private String ipAddress;
    private long runtime;
    private int threadCount;
    private long pid;
    private LocalDateTime startTime;






    public void refresh() {
        this.currentTime = LocalDateTime.now();
        this.currentZoneId = ZoneId.systemDefault();
        RuntimeMXBean runtimeMXBean = ManagementFactory.getRuntimeMXBean();
        Runtime runtime = Runtime.getRuntime();
        long freeMemory = runtime.freeMemory();
        this.setFreeMemory(byteToM(freeMemory));
        long totalMemory = runtime.totalMemory();
        this.setTotalMemory(byteToM(totalMemory));
        long maxMemory = runtime.maxMemory();
        this.setMaxMemory(byteToM(maxMemory));
        this.setOsName(System.getProperty("os.name"));

        try {
            InetAddress localHost = InetAddress.getLocalHost();
            String hostName = localHost.getHostName();
            this.setHost(hostName);
            if (this.ipAddress == null) {
                this.ipAddress = localHost.getHostAddress();
            }
        } catch (UnknownHostException var13) {
            log.warn("UnknownHostException", var13);
            this.setHost("未知");
        }

        this.setIpAddress(this.ipAddress);
        long startTime = runtimeMXBean.getStartTime();
        LocalDateTime startDate = LocalDateTime.ofEpochSecond(startTime / 1000L, 0, ZoneOffset.ofHours(0));
        this.setStartTime(startDate);
        this.setRuntime(runtimeMXBean.getUptime());
        this.setThreadCount(ManagementFactory.getThreadMXBean().getThreadCount());
        this.setPid(tryGetPid());
    }

    static long byteToM(long bytes) {
        return bytes / 1024L / 1024L;
    }

    static long tryGetPid() {
        try {
            String name = ManagementFactory.getRuntimeMXBean().getName();
            String pid = name.split("@")[0];
            return Long.parseLong(pid);
        } catch (NumberFormatException var2) {
            return 0L;
        }
    }


    public static Logger getLog() {
        return log;
    }

    public String getAdeUri() {
        return adeUri;
    }

    public void setAdeUri(String adeUri) {
        this.adeUri = adeUri;
    }

    public String getAdtUri() {
        return adtUri;
    }

    public void setAdtUri(String adtUri) {
        this.adtUri = adtUri;
    }

    public String getSemcUrl() {
        return semcUrl;
    }

    public void setSemcUrl(String semcUrl) {
        this.semcUrl = semcUrl;
    }

    public String getChatBIUrl() {
        return chatBIUrl;
    }

    public void setChatBIUrl(String chatBIUrl) {
        this.chatBIUrl = chatBIUrl;
    }

    public String getChatStrategy() {
        return chatStrategy;
    }

    public void setChatStrategy(String chatStrategy) {
        this.chatStrategy = chatStrategy;
    }

    public String getChatGptModel() {
        return chatGptModel;
    }

    public void setChatGptModel(String chatGptModel) {
        this.chatGptModel = chatGptModel;
    }

    public String getGmcUrl() {
        return gmcUrl;
    }

    public void setGmcUrl(String gmcUrl) {
        this.gmcUrl = gmcUrl;
    }

    public String getKnowledgeMapsUrl() {
        return knowledgeMapsUrl;
    }

    public void setKnowledgeMapsUrl(String knowledgeMapsUrl) {
        this.knowledgeMapsUrl = knowledgeMapsUrl;
    }

    public String getIamUri() {
        return iamUri;
    }

    public void setIamUri(String iamUri) {
        this.iamUri = iamUri;
    }

    public String getEocUri() {
        return eocUri;
    }

    public void setEocUri(String eocUri) {
        this.eocUri = eocUri;
    }

    public String getDefaultLanguage() {
        return defaultLanguage;
    }

    public void setDefaultLanguage(String defaultLanguage) {
        this.defaultLanguage = defaultLanguage;
    }

    public LocalDateTime getCurrentTime() {
        return currentTime;
    }

    public void setCurrentTime(LocalDateTime currentTime) {
        this.currentTime = currentTime;
    }

    public ZoneId getCurrentZoneId() {
        return currentZoneId;
    }

    public void setCurrentZoneId(ZoneId currentZoneId) {
        this.currentZoneId = currentZoneId;
    }

    public long getFreeMemory() {
        return freeMemory;
    }

    public void setFreeMemory(long freeMemory) {
        this.freeMemory = freeMemory;
    }

    public long getTotalMemory() {
        return totalMemory;
    }

    public void setTotalMemory(long totalMemory) {
        this.totalMemory = totalMemory;
    }

    public long getMaxMemory() {
        return maxMemory;
    }

    public void setMaxMemory(long maxMemory) {
        this.maxMemory = maxMemory;
    }

    public String getOsName() {
        return osName;
    }

    public void setOsName(String osName) {
        this.osName = osName;
    }

    public String getHost() {
        return host;
    }

    public void setHost(String host) {
        this.host = host;
    }

    public String getIpAddress() {
        return ipAddress;
    }

    public void setIpAddress(String ipAddress) {
        this.ipAddress = ipAddress;
    }

    public long getRuntime() {
        return runtime;
    }

    public void setRuntime(long runtime) {
        this.runtime = runtime;
    }

    public int getThreadCount() {
        return threadCount;
    }

    public void setThreadCount(int threadCount) {
        this.threadCount = threadCount;
    }

    public long getPid() {
        return pid;
    }

    public void setPid(long pid) {
        this.pid = pid;
    }

    public LocalDateTime getStartTime() {
        return startTime;
    }

    public void setStartTime(LocalDateTime startTime) {
        this.startTime = startTime;
    }
}