package com.digiwin.athena.agiledataecho.proxy.eoc.impl;

import com.digiwin.athena.agiledataecho.app.env.EchoEnvProperties;
import com.digiwin.athena.agiledataecho.constant.ErrorCodeEnum;
import com.digiwin.athena.agiledataecho.domain.*;
import com.digiwin.athena.agiledataecho.proxy.eoc.EocService;
import com.digiwin.athena.agiledataecho.util.BatchConsumer;
import com.digiwin.athena.agiledataecho.util.BatchDataOperation;
import com.digiwin.athena.agiledataecho.util.EocApiConstant;
import com.digiwin.athena.appcore.auth.GlobalConstant;
import com.digiwin.athena.appcore.auth.domain.AuthoredUser;
import com.digiwin.athena.appcore.util.ExceptionUtil;
import com.digiwin.athena.appcore.util.JsonUtils;
import com.digiwin.athena.appcore.util.MessageUtils;
import com.fasterxml.jackson.core.type.TypeReference;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import net.sf.json.JSONArray;
import net.sf.json.JSONObject;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.i18n.LocaleContextHolder;
import org.springframework.core.ParameterizedTypeReference;
import org.springframework.http.*;
import org.springframework.stereotype.Service;
import org.springframework.util.StringUtils;
import org.springframework.web.client.RestTemplate;

import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.*;

@Slf4j
@Service
public class EocServiceImpl implements EocService {
    @Autowired
    EchoEnvProperties envProperties;

    @Autowired
    RestTemplate restTemplate;

    @Autowired
    private MessageUtils messageUtils;

    /**
     * 根据userId获取员工详情
     *
     * @param userId
     * @return
     */
    public EocEmployeeDTO getEmployeeByUserId(String userId) {
        return getEmployee(userId, null);
//        return EocEmployeeDTO.builder()
//                .id(userId)
//                .name(userId)
//                .build();
    }

    /**
     * 根据empId获取员工详情
     *
     * @param empId
     * @return
     */
    public EocEmployeeDTO getEmployeeByEmpId(String empId) {
        return getEmployee(null, empId);
    }

    private EocEmployeeDTO getEmployee(String userId, String empId) {
        String uri = envProperties.getEocUri() + EocApiConstant.EMP_INFO;

        HttpHeaders headers = getHeaders();
        addLang(headers);

        Map<String, Object> param = new HashMap<>();
        param.put("userId", userId);
        param.put("empId", empId);
        param.put("showException",false);
        HttpEntity<?> httpEntity = new HttpEntity<>(param, headers);
        try {
            ResponseEntity<EocResultDTO> responseEntity = restTemplate.exchange(uri, HttpMethod.POST, httpEntity, EocResultDTO.class, new HashMap<>());
            EocResultDTO result = responseEntity.getBody();
            if (result != null) {
                if(result.isSuccess()) {
                return JsonUtils.jsonToObject(JSONObject.fromObject(result.getData()).toString(), EocEmployeeDTO.class);
                }
                else {
                    log.warn(result.getMessage());
                }
            }
        } catch (Exception e) {
            log.warn(e.getMessage(), e);
        }
        return null;
    }

    /**
     * 根据部门id获取所有员工列表
     *
     * @param deptId
     * @return
     */
    public List<EocDeptEmployeeDTO> getDepartEmployees(String deptId) {
        String uri = envProperties.getEocUri() + EocApiConstant.DEPT_EMPS;

        HttpHeaders headers = getHeaders();
        addLang(headers);

        Map<String, String> param = new HashMap<>();
        param.put("deptId", deptId);

        HttpEntity<?> httpEntity = new HttpEntity<>(param, headers);
        ResponseEntity<EocResultDTO> responseEntity = restTemplate.exchange(uri, HttpMethod.POST, httpEntity, EocResultDTO.class, new HashMap<>());
        EocResultDTO result = responseEntity.getBody();
        if (result != null && result.isSuccess()) {
            return JsonUtils.jsonToObject(JSONArray.fromObject(result.getData()).toString(), new TypeReference<List<EocDeptEmployeeDTO>>() {
            });
        }
        return null;
    }


    /**
     * 根据职责ID获取该职责下的所有员工信息
     *
     * @param dutyId     职责id
     * @param deptId     部门id
     * @param empContent 员工编号Id或员工名称
     * @return 查询结果
     */
    public List<EocDutyEmployeeDTO> getDutyEmployees(String dutyId, String deptId, String empContent) {
        //根据职能id获取职能api地址
        //http://172.16.2.141:22694/html/web/controller/share/share.html#5f1f81f741fcbe5bc6508c23
        String uri = envProperties.getEocUri() + EocApiConstant.DUTY_EMPS;

        HttpHeaders headers = getHeaders();
        addLang(headers);

        Map<String, String> param = new HashMap<>();
        param.put("id", dutyId);
        param.put("deptId", deptId);
        param.put("empContent", empContent);

        HttpEntity<?> httpEntity = new HttpEntity<>(param, headers);
        ResponseEntity<EocResultDTO> responseEntity = restTemplate.exchange(uri, HttpMethod.POST, httpEntity, EocResultDTO.class, new HashMap<>());
        EocResultDTO result = responseEntity.getBody();
        if (result != null && result.isSuccess()) {
            return JsonUtils.jsonToObject(JSONArray.fromObject(result.getData()).toString(), new TypeReference<List<EocDutyEmployeeDTO>>() {
            });
        }
        return null;
    }


    /**
     * 根据部门id获取该部门的主管信息
     *
     * @param deptId 部门id
     * @return 查询结果
     */
    public EocDirectEmployeeDTO getDirectorEmployee(String deptId) {
        //根据职能id获取职能api地址
        //http://172.16.2.141:22694/html/web/controller/share/share.html#5f32045241fcbe5bc6508d62
        String uri = envProperties.getEocUri() + EocApiConstant.DEPT_DIRECTOR;

        HttpHeaders headers = getHeaders();
        addLang(headers);

        Map<String, String> param = new HashMap<>();
        param.put("deptId", deptId);

        HttpEntity<?> httpEntity = new HttpEntity<>(param, headers);
        ResponseEntity<EocResultDTO> responseEntity = restTemplate.exchange(uri, HttpMethod.POST, httpEntity, EocResultDTO.class, new HashMap<>());
        EocResultDTO result = responseEntity.getBody();
        if (result != null && result.isSuccess()) {
            return JsonUtils.jsonToObject(JSONObject.fromObject(result.getData()).toString(), EocDirectEmployeeDTO.class);
        }
        return null;


    }

    /**
     * 获取部门主管
     * @param deptId
     * @param companyId
     * @return
     */
    @Override
    public EocDirectEmployeeDTO getDirectorEmployee(String deptId, String companyId) {
        // 根据职能id获取职能api地址
        String uri = envProperties.getEocUri() + EocApiConstant.DEPT_DIRECTOR;

        HttpHeaders headers = getHeaders();
        addLang(headers);

        Map<String, String> param = new HashMap<>();
        param.put("deptId", deptId);
        param.put("companyId", companyId);

        HttpEntity<?> httpEntity = new HttpEntity<>(param, headers);
        ResponseEntity<EocResultDTO> responseEntity = restTemplate.exchange(uri, HttpMethod.POST, httpEntity, EocResultDTO.class, new HashMap<>());
        EocResultDTO result = responseEntity.getBody();
        if (result != null && result.isSuccess()) {
            return JsonUtils.jsonToObject(JSONObject.fromObject(result.getData()).toString(), EocDirectEmployeeDTO.class);
        }
        return null;


    }

    /**
     * 获取当前租户下的所有用户
     *
     * @return
     */
    public List<EocTenantAllUserDTO.EocUserDTO> getAllUserByTenant() {
        String uri = envProperties.getEocUri() + EocApiConstant.PAGE_EMP;

        HttpHeaders headers = getHeaders();
        addLang(headers);

        Map<String, String> param = new HashMap<>();

        HttpEntity<?> httpEntity = new HttpEntity<>(param, headers);
        ResponseEntity<EocTenantAllUserDTO> responseEntity = restTemplate.exchange(uri, HttpMethod.GET, httpEntity, EocTenantAllUserDTO.class, new HashMap<>());
        EocTenantAllUserDTO result = responseEntity.getBody();
        if (result != null && result.isSuccess()) {
            return result.getData().getList();
        }
        return Collections.emptyList();
    }

    @Override
    public List<EocTenantAllUserDTO.EocUserDTO> getAllUserByParams(Map<String,Object> eocParams) {

        String uri = envProperties.getEocUri() + EocApiConstant.PAGE_EMP_PARAMS;

        // 拼接参数
        Map<String, Object> param = new HashMap<>();
        if (MapUtils.isNotEmpty(eocParams)){
            for (String key : eocParams.keySet()) {
                uri = uri + key + "=" + "{" +key + "}&";
                param.put(key,JsonUtils.objectToString(eocParams.get(key)));
            }
            uri = uri.substring(0,uri.length() - 1);
        }

        HttpHeaders headers = getHeaders();
        addLang(headers);


        HttpEntity<?> httpEntity = new HttpEntity<>(headers);
        ResponseEntity<EocTenantAllUserDTO> responseEntity = restTemplate.exchange(uri, HttpMethod.GET, httpEntity, EocTenantAllUserDTO.class, param);
        EocTenantAllUserDTO result = responseEntity.getBody();
        if (result != null && result.isSuccess()) {
            return result.getData().getList();
        }
        return Collections.emptyList();
    }

    /**
     * 根据 userId 获取该人员部门的主管信息
     *
     * @param userId 部门id
     * @return 查询结果
     */
    public EocDirectEmployeeDTO getDirectorEmployeeByUserId(String userId) {
        //http://172.16.2.141:22694/html/web/controller/share/share.html#5f30b37341fcbe5bc6508d32
        String uri = envProperties.getEocUri() + EocApiConstant.EMP_DIRECTOR;

        HttpHeaders headers = getHeaders();
        addLang(headers);

        Map<String, String> param = new HashMap<>();
        param.put("userId", userId);

        HttpEntity<?> httpEntity = new HttpEntity<>(param, headers);
        ResponseEntity<EocResultDTO> responseEntity = restTemplate.exchange(uri, HttpMethod.POST, httpEntity, EocResultDTO.class, new HashMap<>());
        EocResultDTO result = responseEntity.getBody();
        if (result != null && result.isSuccess()) {
            return JsonUtils.jsonToObject(JSONObject.fromObject(result.getData()).toString(), EocDirectEmployeeDTO.class);
        }
        return null;
    }

    /**
     * 判断指定 userId 是否是部门主管
     *
     * @param userId
     * @return
     */
    public Boolean isDeptDirector(String userId) {

        List<EocTenantAllUserDTO.EocUserDTO> eocUserDTOList = getAllUserByDirectorId(userId, 1);
        if (CollectionUtils.isNotEmpty(eocUserDTOList)) {
            return Boolean.TRUE;
        }

        EocEmployeeDTO eocEmployeeDTO = getEmployeeByUserId(userId);
        if (eocEmployeeDTO != null && CollectionUtils.isNotEmpty(eocEmployeeDTO.getDepts())) {
            for (EocEmployeeDTO.EocEmployeeDepartDTO dept : eocEmployeeDTO.getDepts()) {
                if (Objects.equals(dept.getDeptDirectorId(), eocEmployeeDTO.getId())) {
                    return Boolean.TRUE;
                }
            }
        }

        return Boolean.FALSE;
    }

    /**
     * 获取指定人员作为部门主管，其部门的人员列表
     *
     * @param userId
     * @return
     */
    public List<EocDeptEmployeeDTO> getDeptDirectorEmployees(String userId) {

        List<EocDeptEmployeeDTO> result = new ArrayList<>();

        //取所属部门
        EocEmployeeDTO eocEmployeeDTO = getEmployeeByUserId(userId);
        if (eocEmployeeDTO == null || CollectionUtils.isEmpty(eocEmployeeDTO.getDepts())) {
            return result;
        }

        //取部门人员列表,过滤相同人员
        for (EocEmployeeDTO.EocEmployeeDepartDTO dept : eocEmployeeDTO.getDepts()) {
            if (Objects.equals(dept.getDeptDirectorId(), eocEmployeeDTO.getId())) {
                List<EocDeptEmployeeDTO> list = getDepartEmployees(dept.getId());
                if (CollectionUtils.isNotEmpty(list)) {
                    for (EocDeptEmployeeDTO eocDeptEmployeeDTO : list) {
                        if (!StringUtils.isEmpty(eocDeptEmployeeDTO.getUserId())) {
                            if (result.stream().filter(x -> Objects.equals(x.getUserId(), eocDeptEmployeeDTO.getUserId())).count() == 0) {
                                result.add(eocDeptEmployeeDTO);
                            }
                        }
                    }
                }
            }
        }

        return result;
    }

    /**
     * 查找指定直属主管所属的人员列表
     *
     * @param directorId 要查询的直属主管userId
     * @param size       获取几个人
     * @return
     */
    public List<EocTenantAllUserDTO.EocUserDTO> getAllUserByDirectorId(String directorId, Integer size) {

        //http://172.16.2.141:22694/html/web/controller/share/share.html#5f110cf741fcbe5bc65089cd

        //根据userId获取eocId
        EocEmployeeDTO eocEmployeeDTO = getEmployeeByUserId(directorId);
        if(eocEmployeeDTO == null){
            return null;
        }
        String uri = envProperties.getEocUri() + EocApiConstant.PAGE_EMP_SIMPLE;

        Map params = new HashMap();
        params.put("directorId", eocEmployeeDTO.getId());

        Map<String, String> param = new HashMap<>();
        param.put("pageSize", size.toString());
        param.put("params", JsonUtils.objectToString(params));

        HttpHeaders headers = getHeaders();
        addLang(headers);

        HttpEntity<?> httpEntity = new HttpEntity<>(headers);
        ResponseEntity<EocTenantAllUserDTO> responseEntity = restTemplate.exchange(uri, HttpMethod.GET, httpEntity, EocTenantAllUserDTO.class, param);
        EocTenantAllUserDTO result = responseEntity.getBody();
        if (result != null && result.isSuccess()) {
            return result.getData().getList();
        }
        return null;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public List<EocTenantAllUserDTO.EocUserDTO> getEmployeeInfoByUserId(Collection<String> userIds) {
        // eoc接口 http://172.16.2.141:22694/html/web/controller/share/share.html#5f110cf741fcbe5bc65089cd
        final List<EocTenantAllUserDTO.EocUserDTO> employeeInfoList = new ArrayList<>();

        BatchDataOperation batchDataOperation = new BatchDataOperation(Lists.newArrayList(userIds));
        /**
         * userId可能是邮箱地址（邮箱地址可能比较长），因此这里批处理大小设置为50，设置的小一点、保守一点
         * @since BUG-103452
         * @modify by majianfu
         */
        batchDataOperation.batchOperate((BatchConsumer<List<String>>) tmpUserIds -> employeeInfoList.addAll(getBatchEmployeeInfo(tmpUserIds)), 50);

        return employeeInfoList;
    }

    private List<EocTenantAllUserDTO.EocUserDTO> getBatchEmployeeInfo(List<String> userIds) {
        Map<String, Object> params = new HashMap<>();
        params.put("userIds", userIds);

        EocTenantAllUserDTO queryEmpInfo = this.queryEmpInfo(null, null, null, params);
        if (null != queryEmpInfo && null != queryEmpInfo.getData()) {
            return queryEmpInfo.getData().getList();
        } else {
            return Collections.emptyList();
        }
    }

    private EocTenantAllUserDTO queryEmpInfo(Integer pageNum, Integer pageSize, String orderBy, Map<String, Object> params) {
        String url = envProperties.getEocUri() + EocApiConstant.PAGE_EMP_SIMPLE_ORDER_BY;

        Map<String, Object> uriVariables = new HashMap<>();
        // 默认第一页
        uriVariables.put("pageNum", null != pageNum ? pageNum : 1);
        // 默认获取所有数据
        uriVariables.put("pageSize", null != pageSize ? pageSize : Integer.MAX_VALUE);
        uriVariables.put("orderBy", orderBy);
        // 默认传空Map
        uriVariables.put("params", JsonUtils.objectToString(null != params ? params : new HashMap<>()));

        HttpHeaders headers = getHeaders();
        addLang(headers);

        HttpEntity<?> reqEntity = new HttpEntity<>(headers);
        try {
            ResponseEntity<EocTenantAllUserDTO> respEntity = restTemplate.exchange(url, HttpMethod.GET, reqEntity, EocTenantAllUserDTO.class, uriVariables);
            return respEntity.getBody();
        } catch (Exception ex) {
            log.error("查询员工信息失败，参数：{}，错误内容：{}", uriVariables, ex.getMessage());
            throw ExceptionUtil.wrap(ErrorCodeEnum.EOC_SEARCH_STAFF_INFO_FAIL.getErrCode(),ex);
        }
    }

    /**
     * 查找指定人员作为直属主管其所有下属成员列表
     * 包括其作为某部门直属主管其部门内所有成员，以及eoc中设置其为直属主管的
     * 包括 getDeptDirectorEmployees 和 getAllUserByDirectorId 的集合
     *
     * @param userId
     * @return
     */
    public List<EocDeptEmployeeDTO> getUserByDirectorId(String userId) {
        List<EocDeptEmployeeDTO> eocDeptEmployeeDTOList = getDeptDirectorEmployees(userId);
        if (eocDeptEmployeeDTOList == null) {
            eocDeptEmployeeDTOList = new ArrayList<>();
        }

        List<EocTenantAllUserDTO.EocUserDTO> eocUserDTOList = getAllUserByDirectorId(userId, 2000);
        if (CollectionUtils.isEmpty(eocUserDTOList)) {
            return eocDeptEmployeeDTOList;
        }

        for (EocTenantAllUserDTO.EocUserDTO eocUserDTO : eocUserDTOList) {
            if (eocDeptEmployeeDTOList.stream().filter(x -> Objects.equals(x.getUserId(), eocUserDTO.getUserId())).count() == 0) {
                EocDeptEmployeeDTO eocUser = EocDeptEmployeeDTO.builder()
                        .userId(eocUserDTO.getUserId())
                        .userName(eocUserDTO.getUserName())
                        .empId(eocUserDTO.getId())
                        .empName(eocUserDTO.getName())
                        .deptId(eocUserDTO.getDeptId())
                        .deptName(eocUserDTO.getDeptName())
                        .status(eocUserDTO.getStatus())
                        .build();
                eocDeptEmployeeDTOList.add(eocUser);
            }
        }

        return eocDeptEmployeeDTOList;
    }

    /**
     * 查询所有部门信息以及部门下所有员工的信息【4.10】
     *
     * @return
     */
    @Override
    public List<EocDeptDTO.Dept> queryAllDeptAndEmployee() {
        //http://172.16.2.141:22694/html/web/controller/share/share.html#618c8c7a4c557f27fa76f39c

        List<EocDeptDTO.Dept> result = new ArrayList<>();
        String uri = envProperties.getEocUri() + EocApiConstant.PAGE_DEPT_CASCADE;

        Map params = new HashMap();
        params.put("pageSize", 2000);

        HttpHeaders headers = getHeaders();
        addLang(headers);
        try {
            HttpEntity<?> httpEntity = new HttpEntity<>(headers);
            ResponseEntity<EocDeptDTO> responseEntity = restTemplate.exchange(uri, HttpMethod.GET, httpEntity, EocDeptDTO.class, params);
            EocDeptDTO body = responseEntity.getBody();
            if (body != null && body.isSuccess() && body.getData() != null) {
                result = body.getData().getList();
            }
            return result;
        } catch (Exception ex) {
            log.error(ErrorCodeEnum.EOC_SEARCH_EOC_V2_DEPT_CASCADE_FAIL.getErrMsg() + "，参数：{}，错误内容：{}", params, ex.getMessage());
            throw ExceptionUtil.wrap(ErrorCodeEnum.EOC_SEARCH_STAFF_INFO_FAIL.getErrCode(),ex);
        }


    }

    public JSONArray queryMembersOfTheDepartment(String departmentSid) {
        String uri = envProperties.getEocUri() + EocApiConstant.PAGE_DEPT_DEPARTMENT_SID_EMPS;
        Map<String, Object> params = new HashMap<>();
        params.put("departmentSid", departmentSid);
        params.put("pageSize", 2000);

        HttpHeaders headers = getHeaders();
        addLang(headers);

        HttpEntity<?> httpEntity = new HttpEntity<>(headers);
        ResponseEntity<EocResultDTO> responseEntity = restTemplate.exchange(uri, HttpMethod.GET, httpEntity, EocResultDTO.class, params);
        EocResultDTO result = responseEntity.getBody();
        if (result != null && result.isSuccess()) {
            return JSONArray.fromObject(((LinkedHashMap) result.getData()).get("list"));
        }
        return new JSONArray();
    }

    public JSONArray queryCurrentTenantAllDepartment() {
        String uri = envProperties.getEocUri() + EocApiConstant.PAGE_DEPT_LIST;
        Map<String, Object> params = new HashMap<>();
        params.put("pageSize", 2000);

        HttpHeaders headers = getHeaders();
        addLang(headers);

        HttpEntity<?> httpEntity = new HttpEntity<>(headers);
        ResponseEntity<EocResultDTO> responseEntity = restTemplate.exchange(uri, HttpMethod.GET, httpEntity, EocResultDTO.class, params);
        EocResultDTO result = responseEntity.getBody();
        if (result != null && result.isSuccess()) {
            return JSONArray.fromObject(((LinkedHashMap) result.getData()).get("list"));
        }
        return new JSONArray();
    }


    /**
     * 根据用户查询用户所属部门以及子部门的员工列表
     *
     * @param userId 用户id
     * @return JSONArray
     */
    public JSONArray queryUserDepartmentInfo(String userId) {
        String uri = envProperties.getEocUri() + EocApiConstant.EMP_SUBORDINATE;

        HttpHeaders headers = getHeaders();
        addLang(headers);

        Map<String, String> param = new HashMap<>();
        param.put("userId", userId);

        HttpEntity<?> httpEntity = new HttpEntity<>(param, headers);
        ResponseEntity<EocResultDTO> responseEntity = restTemplate.exchange(uri, HttpMethod.POST, httpEntity, EocResultDTO.class, new HashMap<>());
        EocResultDTO result = responseEntity.getBody();
        if (result != null && result.isSuccess()) {
            return JSONArray.fromObject(result.getData());
        }
        return new JSONArray();
    }

    private HttpHeaders getHeaders() {
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        return headers;
    }

    private void addLang(HttpHeaders headers) {
        headers.add("locale", LocaleContextHolder.getLocale().toString());
    }

    @Override
    public String getEmpIdByUserId(String userId, String token) {
        //获取员工ID
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        headers.add(GlobalConstant.IAM_USER_TOKEN, token);
        //post body
        Map<String, String> param = new HashMap<>();
        param.put("userId", userId);
        HttpEntity<?> httpEntity = new HttpEntity<>(param, headers);
        String url = envProperties.getEocUri() + EocApiConstant.EMP_ID;
        ResponseEntity<Map> responseEntity = restTemplate.postForEntity(url, httpEntity, Map.class);
        Map body = responseEntity.getBody();
        if (body == null) {
            return null;
        } else {
            return body.get("data").toString();
        }
    }

    /**
     * 通过员工ID获取用户ID
     *
     * @param empId 用户id
     * @param token
     * @return 用户基本信息
     */
    @Override
    public String getUserIdByEmpId(String empId, String token) {
        //获取员工ID
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        headers.add(GlobalConstant.IAM_USER_TOKEN, token);
        //post body
        Map<String, String> param = new HashMap<>();
        param.put("empId", empId);
        HttpEntity<?> httpEntity = new HttpEntity<>(param, headers);
        String url = envProperties.getEocUri() + EocApiConstant.EMP_USER_ID;
        ResponseEntity<Map> responseEntity = restTemplate.postForEntity(url, httpEntity, Map.class);
        Map body = responseEntity.getBody();
        if (body == null) {
            return null;
        } else {
            return body.get("data").toString();
        }
    }

    @Override
    public ProxyUserResp getProxyUser(String userId, String token) {
        //获取员工ID
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        headers.add(GlobalConstant.IAM_USER_TOKEN, token);
        DateTimeFormatter df = DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss");
        //post body
        Map<String, String> param = new HashMap<>();
        param.put("userId", userId);
        param.put("agentDate", df.format(LocalDateTime.now()));
        HttpEntity<?> httpEntity = new HttpEntity<>(param, headers);
        String url = envProperties.getEocUri() + EocApiConstant.EMP_AGENT;
        ResponseEntity<ProxyUserDTOWithCodeResp> responseEntity = restTemplate.postForEntity(url, httpEntity, ProxyUserDTOWithCodeResp.class);
        ProxyUserDTOWithCodeResp baseResult = responseEntity.getBody();
        if (baseResult.getCode() == 200 && baseResult.getData() != null) {
            return baseResult.getData();
        }
        return null;
    }

    @Override
    public List<ProxyUserResp> getProxyTargetUsers(String userId, String token) {
        //获取员工ID
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        headers.add(GlobalConstant.IAM_USER_TOKEN, token);
        //post body
        Map<String, String> param = new HashMap<>();
        param.put("userId", userId);
        HttpEntity<?> httpEntity = new HttpEntity<>(param, headers);
        String url = envProperties.getEocUri() + EocApiConstant.EMP_AGENT_TARGET;
        ResponseEntity<ProxyUserDTOListWithCodeResp> responseEntity = restTemplate.postForEntity(url, httpEntity, ProxyUserDTOListWithCodeResp.class);
        ProxyUserDTOListWithCodeResp baseResult = responseEntity.getBody();
        if (baseResult != null && baseResult.getCode() == 200 && baseResult.getData() != null) {
            return baseResult.getData();
        }
        return null;
    }

    @Override
    public List<EocUserDTO> getEocUserListByUserId(String userId, String token) {
        //获取员工ID
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        headers.add(GlobalConstant.IAM_USER_TOKEN, token);
        //post body
        Map<String, String> param = new HashMap<>();
        param.put("userId", userId);
        HttpEntity<?> httpEntity = new HttpEntity<>(param, headers);
        String url = envProperties.getEocUri() + EocApiConstant.EMP_MAIN_EMPS;
        ResponseEntity<EocUserListDTOWithCode> responseEntity = restTemplate.postForEntity(url, httpEntity, EocUserListDTOWithCode.class);
        EocUserListDTOWithCode baseResult = responseEntity.getBody();
        if (baseResult.getCode() == 200) {
            return baseResult.getData();
        } else {
            return null;
        }
    }

    /**
     * 获取用户直属领导
     *
     * @param userId
     * @param token
     * @return
     */
    public EocUserDTO getUserLeader(String userId, String token) {
        //获取员工ID
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        headers.add(GlobalConstant.IAM_USER_TOKEN, token);
        //post body
        Map<String, String> param = new HashMap<>();
        param.put("userId", userId);
        HttpEntity<?> httpEntity = new HttpEntity<>(param, headers);
        String url = envProperties.getEocUri() + EocApiConstant.EMP_DIRECTOR;
        ResponseEntity<EocUserDTOWithCode> responseEntity = restTemplate.postForEntity(url, httpEntity, EocUserDTOWithCode.class);
        EocUserDTOWithCode baseResult = responseEntity.getBody();
        if (baseResult.getCode() == 200) {
            return baseResult.getData();
        } else {
            return null;
        }
    }

    /**
     * 获取部门直属领导
     *
     * @param deptId
     * @param token
     * @return
     */
    @Override
    public EocUserDTO getDeptLeader(String deptId, String token) {
        //获取员工ID
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        headers.add(GlobalConstant.IAM_USER_TOKEN, token);
        //post body
        Map<String, String> param = new HashMap<>();
        param.put("deptId", deptId);
        HttpEntity<?> httpEntity = new HttpEntity<>(param, headers);
        String url = envProperties.getEocUri() + EocApiConstant.DEPT_DIRECTOR;
        ResponseEntity<EocUserDTOWithCode> responseEntity = restTemplate.postForEntity(url, httpEntity, EocUserDTOWithCode.class);
        EocUserDTOWithCode baseResult = responseEntity.getBody();
        if (baseResult.getCode() == 200) {
            return baseResult.getData();
        } else {
            return null;
        }
    }

    /**
     * 根据租户Sid和用户id获取EOC员工信息
     *
     * @Author：SYQ
     * @Date：2021/7/5 16:44
     */
    public String getEocByTenantAndUser(AuthoredUser user, String tenantSid, String userId) {
        if (org.apache.commons.lang.StringUtils.isEmpty(tenantSid) || org.apache.commons.lang.StringUtils.isEmpty(userId)) {
            return "";
        }
        String eocUserId = "";
        //获取用户信息
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        headers.add(GlobalConstant.IAM_USER_TOKEN, user.getToken());
        //获取当前用户是否开启代理以及代理人地址
        String url = envProperties.getEocUri() + EocApiConstant.EMP_TENANT_USER_ID_INFO;
        //查询参数
        Map<String, Object> paramMap = new HashMap<>();
        paramMap.put("tenantSid", tenantSid);
        paramMap.put("userId", userId);
        HttpEntity<?> httpEntity = new HttpEntity<>(paramMap, headers);
        //获取当前用户是否开启代理以及代理人
        try {
            ResponseEntity<Map> response = restTemplate.postForEntity(url, httpEntity, Map.class);
            //判断API是否调用成功
            if (response.getStatusCode() == HttpStatus.OK) {
                //获取data数据
                JSONObject jsonObject = JSONObject.fromObject(response.getBody().get("data"));
                //获取eoc员工id
                eocUserId = jsonObject.getString("id");
            }
        } catch (Exception e) {
            log.error(e.getMessage(), e);
        }

        return eocUserId;
    }

    /**
     * 获取EOC用户信息
     *
     * @Author：SYQ
     * @Date：2021/6/15 17:07
     */
    public String getEocEmpInfo(String userId, AuthoredUser user) {
        String eocEmpId = "";

        EocEmployeeDTO employee = getEmployee(userId, null);
        if (Objects.isNull(employee)){
            return eocEmpId;
        }
        return employee.getId();
    }

    /**
     * 获取EOC用户信息
     *
     * @Author：xusijia
     * @Date：2022/5/31 16:40
     */
   /* public Map<String, Object> getEocEmpInfos(String userId, AuthoredUser user) {
        Map<String, Object> objectMap = new HashMap<>();
        EocEmployeeDTO employee = getEmployee(userId, null);
        //获取当前用户是否开启代理以及代理人
        try {
            String json = JsonUtils.objectToString(employee);
            //获取eoc员工id
            objectMap = JsonUtils.jsonToObject(json, Map.class);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
        }
        return objectMap;
    }*/

    @Override
    public BizUnitInfoDTO getBizUnitInfo(JSONObject bizUnitCodeJSONObj) {
        // {"eoc_company_id":"TCm01","eoc_site_id":"TPt01"}
        BizUnitInfoDTO reqBody = new BizUnitInfoDTO();
        if (!bizUnitCodeJSONObj.containsKey("eoc_company_id")) {
            return reqBody;
        }

        String companyId = bizUnitCodeJSONObj.getString("eoc_company_id");
        if (org.apache.commons.lang.StringUtils.isBlank(companyId)) {
            return reqBody;
        }
        reqBody.setCompanyId(companyId);
        if (bizUnitCodeJSONObj.containsKey("eoc_site_id")) {
            reqBody.setFactoryId(bizUnitCodeJSONObj.getString("eoc_site_id"));
        }

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        HttpEntity<?> httpEntity = new HttpEntity<>(reqBody, headers);

        String url = envProperties.getEocUri() + EocApiConstant.QRY_BIZ_UNIT_NAME_URL;
        try {
            ResponseEntity<BizUnitInfoRespDTO> respEntity = restTemplate.postForEntity(url, httpEntity, BizUnitInfoRespDTO.class);
            if (respEntity.getStatusCode() == HttpStatus.OK) {
                BizUnitInfoDTO bizUnitInfoDTO = Optional.ofNullable(respEntity.getBody()).map(BizUnitInfoRespDTO::getData).orElse(null);
                if (null != bizUnitInfoDTO && org.apache.commons.lang.StringUtils.isNotBlank(bizUnitInfoDTO.getCompanyName())) {
                    return bizUnitInfoDTO;
                } else { // 没能正确获取到公司名称，则返回reqBody
                    log.warn("[BizUnitInfoDTO getBizUnitInfo] Can't get valid info for {}", JsonUtils.objectToString(reqBody));
                    return reqBody;
                }
            } else {
                return new BizUnitInfoDTO();
            }
        } catch (Exception ex) {
            return new BizUnitInfoDTO();
        }

    }

    @Override
    public TenantBizUnitDTO getCompanyInfoForTenant() {
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        Map<String, String> param = new HashMap<>();
        HttpEntity<?> httpEntity = new HttpEntity<>(param, headers);

        String url = envProperties.getEocUri() + EocApiConstant.CORP_COMPANY;
        ResponseEntity<TenantBizUnitDTO> respEntity = restTemplate.postForEntity(url, httpEntity, TenantBizUnitDTO.class);
        return respEntity.getBody();
    }

    @Override
    public TenantBizUnitDTO getFactoryInfoForTenant() {
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        Map<String, String> param = new HashMap<>();
        HttpEntity<?> httpEntity = new HttpEntity<>(param, headers);

        String url = envProperties.getEocUri() + EocApiConstant.CORP_FACTORY;
        ResponseEntity<TenantBizUnitDTO> respEntity = restTemplate.postForEntity(url, httpEntity, TenantBizUnitDTO.class);
        return respEntity.getBody();
    }

    public static void main(String[] args) {
        Map respEntity = new HashMap();
        System.out.println(respEntity.get("1"));
    }


    /**
     * 获取用户的部门
     * @param userId
     * @param size
     * @return
     */
    @Override
    public List<EocUserDeptDTO.EocUserDept> getDeptByUserId(String userId, Integer size) {
        String uri = envProperties.getEocUri() + EocApiConstant.PAGE_EMP_SIMPLE;

        Map params = new HashMap();
        params.put("userIds", Arrays.asList(userId));

        Map<String, String> param = new HashMap<>();
        param.put("pageSize", size.toString());
        param.put("params", JsonUtils.objectToString(params));

        HttpHeaders headers = getHeaders();
        addLang(headers);

        HttpEntity<?> httpEntity = new HttpEntity<>(headers);
        ResponseEntity<EocUserDeptDTO> responseEntity = restTemplate.exchange(uri, HttpMethod.GET, httpEntity, EocUserDeptDTO.class, param);
        EocUserDeptDTO result = responseEntity.getBody();
        if (result != null && result.isSuccess()) {
            return result.getData().getList();
        }
        return null;
    }



    @Override
    public List<EocEmployeeDTO> getEmployees(List<String> userIds, List<String> empIds) {
        String uri = envProperties.getEocUri() + EocApiConstant.EMP_INFOS;

        HttpHeaders headers = getHeaders();
        addLang(headers);

        Map<String, Object> param = new HashMap<>();
        param.put("userIds", userIds);
        param.put("empIds", empIds);
        HttpEntity<?> httpEntity = new HttpEntity<>(param, headers);
        try {
            ResponseEntity<EocBaseDTO<List<EocEmployeeDTO>>> responseEntity = restTemplate.exchange(uri, HttpMethod.POST, httpEntity,
                    new ParameterizedTypeReference<EocBaseDTO<List<EocEmployeeDTO>>>() {}, new HashMap<>());
            EocBaseDTO result = responseEntity.getBody();
            if (result != null) {
                if(result.isSuccess()) {
                    return responseEntity.getBody().getData();
                }
                else {
                    log.warn(result.getMessage());
                }
            }
        } catch (Exception e) {
            log.warn(e.getMessage(), e);
        }
        return null;
    }

    @Override
    public List<ProxyUserResp> getProxyUserBatch(List<String> userIds, String token) {
        //获取员工ID
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        headers.add(GlobalConstant.IAM_USER_TOKEN, token);
        DateTimeFormatter df = DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss");
        //post body
        Map<String, Object> param = new HashMap<>();
        param.put("userIds", userIds);
        param.put("agentDate", df.format(LocalDateTime.now()));
        HttpEntity<?> httpEntity = new HttpEntity<>(param, headers);
        String url = envProperties.getEocUri() + EocApiConstant.EMP_AGENT_BATCH;
        ResponseEntity<ProxyUserBatchResp> responseEntity = restTemplate.postForEntity(url, httpEntity, ProxyUserBatchResp.class);
        ProxyUserBatchResp baseResult = responseEntity.getBody();
        if (baseResult != null && baseResult.getCode() == 200 && CollectionUtils.isNotEmpty(baseResult.getData())) {
            return baseResult.getData();
        }
        return Collections.emptyList();
    }
}
