package com.digiwin.athena.agiledataecho.service.imp;


import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.digiwin.athena.agiledataecho.constant.ErrorCodeEnum;
import com.digiwin.athena.agiledataecho.dao.EchoBulletinReadRecordMapper;
import com.digiwin.athena.agiledataecho.domain.EchoBulletinReadRecord;
import com.digiwin.athena.agiledataecho.proxy.gmc.model.GmcBulletinResultDTO;
import com.digiwin.athena.agiledataecho.proxy.gmc.model.GmcBulletinUnReadResultDTO;
import com.digiwin.athena.agiledataecho.proxy.gmc.model.GmcBulletinUnReadSaveDTO;
import com.digiwin.athena.agiledataecho.proxy.gmc.service.GmcService;
import com.digiwin.athena.agiledataecho.proxy.iam.UserService;
import com.digiwin.athena.agiledataecho.proxy.knowledgemaps.KnowledgeMapsService;
import com.digiwin.athena.agiledataecho.proxy.knowledgemaps.model.KnowledgeMapsTenantPurchasedAgileDataApplicationModel;
import com.digiwin.athena.agiledataecho.service.EchoGmcService;
import com.digiwin.athena.agiledataecho.util.SystemLanguageUtils;
import com.digiwin.athena.appcore.auth.domain.AuthoredUser;
import com.digiwin.athena.appcore.util.ExceptionUtil;
import com.digiwin.athena.appcore.util.JsonUtils;
import com.fasterxml.jackson.core.type.TypeReference;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.util.*;
import java.util.stream.Collectors;

/**
 * GMC接口实现
 *
 * @author yanfeng
 * @date 2024/7/23
 */
@Slf4j
@Service
public class EchoGmcServiceImpl implements EchoGmcService {

    @Autowired
    private GmcService gmcService;

    @Autowired
    private KnowledgeMapsService knowledgeMapsService;

    @Autowired
    private EchoBulletinReadRecordMapper bulletinReadRecordMapper;

    @Autowired
    private UserService userService;

    @Override
    public List<GmcBulletinResultDTO> queryReadBulletinList(AuthoredUser user, Integer pageNum, Integer pageSize) {
        String token = user.getToken();
        String tenantId = user.getTenantId();
        String userId = user.getUserId();

        List<KnowledgeMapsTenantPurchasedAgileDataApplicationModel> knowledgeMapsTenantPurchasedAgileDataApplicationModels = knowledgeMapsService.queryTenantPurchasedAgileDataApplicationList(token, tenantId);
        if (CollectionUtils.isEmpty(knowledgeMapsTenantPurchasedAgileDataApplicationModels)) {
            log.warn("已读公告:当前用户未购买敏捷数据应用");
            return Collections.emptyList();
        }
        List<String> codes = knowledgeMapsTenantPurchasedAgileDataApplicationModels.stream()
                .map(KnowledgeMapsTenantPurchasedAgileDataApplicationModel::getCode)
                .collect(Collectors.toList());

        List<GmcBulletinResultDTO> bulletinList = queryBulletinList(user, codes, pageNum, pageSize);
        // 查询当前租户下用户已读公告列表
        List<EchoBulletinReadRecord> agileDataBulletinReadRecords = bulletinReadRecordMapper.selectList(new LambdaQueryWrapper<EchoBulletinReadRecord>()
                .eq(EchoBulletinReadRecord::getTenantId, tenantId).eq(EchoBulletinReadRecord::getUserId, userId));

        // 使用Set来优化查找已读公告的性能
        Set<String> readSids = agileDataBulletinReadRecords.stream()
                .map(EchoBulletinReadRecord::getSid)
                .collect(Collectors.toSet());

        bulletinList.forEach(bulletin -> {
            if (!readSids.contains(bulletin.getSid())) {
                bulletinReadRecordMapper.insert(new EchoBulletinReadRecord(userId, tenantId, bulletin.getSid()));
            }
            bulletin.setIsRead(true);
        });
        return bulletinList;
    }

    @Override
    public GmcBulletinUnReadResultDTO queryUnreadBulletinList(AuthoredUser user, Integer pageNum, Integer pageSize) {
        String token = user.getToken();
        String tenantId = user.getTenantId();
        String userId = user.getUserId();

        List<KnowledgeMapsTenantPurchasedAgileDataApplicationModel> knowledgeMapsTenantPurchasedAgileDataApplicationModels = knowledgeMapsService.queryTenantPurchasedAgileDataApplicationList(token, tenantId);
        log.info("未读公告获取的km购买应用列表:{}", JsonUtils.objectToString(knowledgeMapsTenantPurchasedAgileDataApplicationModels));
        if (CollectionUtils.isEmpty(knowledgeMapsTenantPurchasedAgileDataApplicationModels)) {
            log.warn("未读公告:当前用户未购买敏捷数据应用");
            return null;
        }
        List<String> codes = knowledgeMapsTenantPurchasedAgileDataApplicationModels.stream()
                .map(KnowledgeMapsTenantPurchasedAgileDataApplicationModel::getCode)
                .collect(Collectors.toList());

        List<GmcBulletinResultDTO> bulletinList = queryBulletinList(user, codes, pageNum, pageSize);

        GmcBulletinUnReadResultDTO gmcBulletinUnReadResultDTO = new GmcBulletinUnReadResultDTO();
        // 查询当前租户下用户已读公告列表
        List<EchoBulletinReadRecord> agileDataBulletinReadRecords = bulletinReadRecordMapper.selectList(new LambdaQueryWrapper<EchoBulletinReadRecord>()
                .eq(EchoBulletinReadRecord::getTenantId, tenantId).eq(EchoBulletinReadRecord::getUserId, userId));

        if (!CollectionUtils.isEmpty(agileDataBulletinReadRecords)) {
            bulletinList.removeIf(bulletin -> agileDataBulletinReadRecords.stream().anyMatch(record -> record.getSid().equals(bulletin.getSid())));
        }
        gmcBulletinUnReadResultDTO.setCount(bulletinList.size());
        gmcBulletinUnReadResultDTO.setBulletinList(bulletinList);
        return gmcBulletinUnReadResultDTO;
    }

    @Override
    public Boolean saveReadBulletin(AuthoredUser user, GmcBulletinUnReadSaveDTO saveDTO) {
        try {
            bulletinReadRecordMapper.insertBatch(user.getUserId(), user.getTenantId(), saveDTO.getSidList());
        } catch (Exception e) {
            log.error("保存已读公告失败:{}", e.getMessage());
            return false;
        }
        return true;
    }

    private List<GmcBulletinResultDTO> queryBulletinList(AuthoredUser user, List<String> codes, Integer pageNum, Integer pageSize) {
        String token = user.getToken();
        String tenantId = user.getTenantId();
        String userId = user.getUserId();
        List<GmcBulletinResultDTO> bulletins = gmcService.queryBulletinList(token, tenantId, codes, pageNum, pageSize);
        if (CollectionUtils.isEmpty(bulletins)) {
            log.warn("当前用户未查询到公告列表,codes:{},user:{}", codes, JsonUtils.objectToString(user));
            return Collections.emptyList();
        }
        Comparator<String> tComparator = Comparator.reverseOrder();
        bulletins.sort((o1, o2) -> tComparator.compare(o1.getUpdateTime(), o2.getUpdateTime()));
        String traditional;
        try {
            // 根据用户语言设置公告列表的显示语言
            String locale = userService.getUserLangNameByUserId(userId, tenantId, token);
            if (Locale.TAIWAN.toString().equals(locale)) {
                traditional = SystemLanguageUtils.toTraditional(JsonUtils.objectToString(bulletins));
            } else {
                traditional = SystemLanguageUtils.toSimple(JsonUtils.objectToString(bulletins));
            }
        } catch (Exception e) {
            throw ExceptionUtil.wrap(ErrorCodeEnum.GMC_DAP_TRANSLATORS_FAIL.getErrCode(), e);
        }
        return JsonUtils.jsonToObject(traditional, new TypeReference<List<GmcBulletinResultDTO>>() {
        });
    }

}