package com.digiwin.athena.agiledataecho.service.imp;

import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.digiwin.athena.agiledataecho.dao.EchoChargingMapper;
import com.digiwin.athena.agiledataecho.domain.EchoCharging;
import com.digiwin.athena.agiledataecho.proxy.adt.AdtService;
import com.digiwin.athena.agiledataecho.service.EchoLiteChargingService;
import com.digiwin.athena.agiledataecho.util.LogUtils;
import com.digiwin.athena.appcore.auth.domain.AuthoredUser;
import com.digiwin.athena.appcore.util.JsonUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.redis.core.StringRedisTemplate;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.time.LocalDateTime;
import java.util.concurrent.TimeUnit;

/**
 * 敏数Lite版本体验次数计费
 */
@Slf4j
@Service
public class EchoLiteChargingServiceImpl implements EchoLiteChargingService {

    @Resource
    private EchoChargingMapper echoChargingMapper;

    @Autowired
    private AdtService adtService;

    private static final Integer CHARGING_NUM = 1;

    private static final String CHARGING_TYPE = "lite";

    @Autowired
    private StringRedisTemplate stringRedisTemplate;

    private static final String AGILE_DATA_ECHO_CACHE = "agiledataecho:cache";

    private static final String CHARGING_REDIS_KEY = "charging_redis_key";
    /**
     * 体验次数扣费
     * @param authoredUser
     * @param chargingType
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public Boolean changeCharging(AuthoredUser authoredUser, String chargingType) {
        EchoCharging echoCharging = new EchoCharging();
        try{
            QueryWrapper<EchoCharging> queryWrapper = new QueryWrapper<>();
            queryWrapper.eq("user_id", authoredUser.getUserId()).eq("tenant_id", authoredUser.getTenantId())
                    .eq("charging_type", StringUtils.isEmpty(chargingType) ? CHARGING_TYPE : chargingType);
            echoCharging = echoChargingMapper.selectOne(queryWrapper);
            Integer userCharging = 0;
            //adt 配置租户值
            String redisValue = stringRedisTemplate.opsForValue().get(AGILE_DATA_ECHO_CACHE + CHARGING_REDIS_KEY + authoredUser.getTenantId());
            if(StringUtils.isNotEmpty(redisValue)){
                userCharging =  Integer.parseInt(redisValue);
            }else{
                userCharging = adtService.getChargingUserValue(authoredUser);
                stringRedisTemplate.opsForValue().set(AGILE_DATA_ECHO_CACHE + CHARGING_REDIS_KEY + authoredUser.getTenantId(),
                        String.valueOf(userCharging), 24 * 3600 * 1000, TimeUnit.SECONDS);
            }
            if(echoCharging == null){
                //初始化体验次数
                this.initCharging(authoredUser,chargingType,userCharging,1);
            } else {
                //扣减体验次数
                chargingType = chargingType == null ? CHARGING_TYPE : chargingType;
                echoChargingMapper.updateChargingRemain(authoredUser.getTenantId(),authoredUser.getUserId(),chargingType,LocalDateTime.now());
                //剩余次数不够，提示用户
                boolean result = !(echoCharging.getChargingRemain() + 1 >= userCharging);
                LogUtils.buildAgileLog(LogUtils.MODULE_ECHO, "liteDeduction", LogUtils.SUCCESS,
                        JsonUtils.objectToString(echoCharging), JsonUtils.objectToString(result), "");
                return result;
            }
        }catch (Exception e){
            log.error("敏数体验次数扣减失败:{}",e.getMessage());
            LogUtils.buildAgileLog(LogUtils.MODULE_ECHO, "liteDeduction", LogUtils.ERROR,
                    JsonUtils.objectToString(echoCharging), "ECHO敏数体验次数扣减失败", "");
        }
        return true;
    }

    /**
     * 初始化用户技术数量
     * @param authoredUser
     * @param chargingNum
     */
    @Override
    public void initCharging(AuthoredUser authoredUser,String chargingType,Integer chargingNum,Integer chargingRemain) {
        EchoCharging echoCharging = new EchoCharging();
        echoCharging.setUserId(authoredUser.getUserId());
        echoCharging.setUserName(authoredUser.getUserName());
        echoCharging.setTenantId(authoredUser.getTenantId());
        echoCharging.setTenantName(authoredUser.getTenantName());
        echoCharging.setChargingNum(chargingNum == null ? CHARGING_NUM : chargingNum);
        echoCharging.setChargingRemain(chargingRemain == null ? CHARGING_NUM : chargingRemain);
        echoCharging.setChargingType(StringUtils.isEmpty(chargingType) ? CHARGING_TYPE : chargingType);
        echoCharging.setCreateDate(LocalDateTime.now());
        echoCharging.setModifyDate(LocalDateTime.now());
        echoChargingMapper.insert(echoCharging);
    }

    /**
     * 修改剩余次数
     * @param userId
     * @param tenantId
     * @param chargingRemain
     */
    @Override
    public void initChargingRemain(String userId, String tenantId, Integer chargingRemain,String chargingType) {
        chargingRemain = chargingRemain == null ? CHARGING_NUM : chargingRemain;
        chargingType = chargingType == null ? CHARGING_TYPE : chargingType;
        QueryWrapper<EchoCharging> queryWrapper = new QueryWrapper<>();
        queryWrapper.eq("user_id", userId).eq("tenant_id", tenantId)
                .eq("charging_type", chargingType);
        EchoCharging echoCharging = echoChargingMapper.selectOne(queryWrapper);
        if(echoCharging == null){
            AuthoredUser authoredUser = new AuthoredUser();
            authoredUser.setUserId(userId);
            authoredUser.setTenantId(tenantId);
            //初始化体验次数
            this.initCharging(authoredUser,chargingType,chargingRemain,chargingRemain);
        } else {
            echoChargingMapper.reductionChargingRemain(tenantId,userId,chargingRemain,chargingType,LocalDateTime.now());
        }
    }
}
