package com.digiwin.athena.agiledataecho.service.imp;

import com.alibaba.nacos.common.utils.CollectionUtils;
import com.alibaba.nacos.common.utils.StringUtils;
import com.digiwin.athena.agiledataecho.constant.CommonConstants;
import com.digiwin.athena.agiledataecho.dao.EchoSchemaElLogMapper;
import com.digiwin.athena.agiledataecho.domain.EchoSchemaElementLog;
import com.digiwin.athena.agiledataecho.dto.EchoInformationResDTO;
import com.digiwin.athena.agiledataecho.dto.EchoSchemaEIResDTO;
import com.digiwin.athena.agiledataecho.dto.EchoSchemaElReqDTO;
import com.digiwin.athena.agiledataecho.proxy.iam.UserService;
import com.digiwin.athena.agiledataecho.service.AbsEchoService;
import com.digiwin.athena.agiledataecho.service.EchoInformationService;
import com.digiwin.athena.agiledataecho.service.EchoSchemaElService;
import com.digiwin.athena.appcore.auth.domain.AuthoredUser;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.lang.reflect.Field;
import java.util.*;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;

/**
 * @Author: SunHong
 * @Date: 2024/8/5 15:52
 * @Description: 语义上报实现
 */
@Service
@Slf4j
public class EchoSchemaElServiceImpl extends AbsEchoService implements EchoSchemaElService {

    @Autowired
    private EchoInformationService echoInformationService;
    @Resource
    private EchoSchemaElLogMapper echoSchemaElLogMapper;

    @Autowired
    private UserService userService;

    @Override
    public void submit(EchoSchemaElReqDTO req, AuthoredUser user) {
        EchoSchemaElementLog echoSchemaElementLog = new EchoSchemaElementLog();
        BeanUtils.copyProperties(req, echoSchemaElementLog);
        echoSchemaElementLog.setCreateTime(new Date());

        TranslationService translationService = new TranslationService();
        List<String> translates = new ArrayList<>();
        translates.add(echoSchemaElementLog.getTarget());
        translates.add(echoSchemaElementLog.getPoint());
        translates.add(echoSchemaElementLog.getSchemaGroup());
        translates.add(echoSchemaElementLog.getSchemaCondition());
        translates.add(echoSchemaElementLog.getSchemaSort());
        // 执行翻译
        List<CompletableFuture<String>> translationFutures = translates.stream()
                .map(translationService::translate)
                .collect(Collectors.toList());
        CompletableFuture<Void> allTranslations = CompletableFuture.allOf(translationFutures.toArray(new CompletableFuture[0]));
        allTranslations.thenRun(() -> {
            echoSchemaElementLog.setComplexTarget(translationFutures.get(0).join());
            echoSchemaElementLog.setComplexPoint(translationFutures.get(1).join());
            echoSchemaElementLog.setComplexSchemaGroup(translationFutures.get(2).join());
            echoSchemaElementLog.setComplexSchemaCondition(translationFutures.get(3).join());
            echoSchemaElementLog.setComplexSchemaSort(translationFutures.get(4).join());
            echoSchemaElLogMapper.insert(echoSchemaElementLog);
        }).join();
    }

    @Override
    public List<EchoSchemaEIResDTO> getSchemaElInfoByMessageId(String messageId, AuthoredUser user) {
        List<EchoSchemaEIResDTO> results = new ArrayList<>();
        List<EchoSchemaElementLog> echoSchemaElementLogs = echoSchemaElLogMapper.getSchemaElInfoByMessageId(messageId);
        if(CollectionUtils.isEmpty(echoSchemaElementLogs)){
            return new ArrayList<>();
        }
        String userLangName = userService.getUserLangNameByUserId(
                user.getUserId(), user.getTenantId(), user.getToken());
        for(EchoSchemaElementLog res : echoSchemaElementLogs){
            EchoSchemaEIResDTO echoSchemaElementLog = new EchoSchemaEIResDTO();
            if("zh_TW".equals(userLangName)){
                BeanUtils.copyProperties(res, echoSchemaElementLog);
                echoSchemaElementLog.setTarget(res.getComplexTarget());
                echoSchemaElementLog.setPoint(res.getComplexPoint());
                echoSchemaElementLog.setSchemaGroup(res.getComplexSchemaGroup());
                echoSchemaElementLog.setSchemaCondition(res.getComplexSchemaCondition());
                echoSchemaElementLog.setSchemaSort(res.getComplexSchemaSort());
                results.add(echoSchemaElementLog);
            }else{
                BeanUtils.copyProperties(res, echoSchemaElementLog);
                results.add(echoSchemaElementLog);
            }
        }
        return results;
    }

    @Override
    public LinkedHashMap<String,Object> getInfoAndSchema(String messageId, AuthoredUser user) {
        LinkedHashMap<String,Object> res = new LinkedHashMap<>();
        List<String> tileList = CommonConstants.ECHO_SIMPLE_TITLE;
        List<String> echoList = CommonConstants.ECHO_INFO_SIMPLE_LIST;
        List<String> schemaList = CommonConstants.SCHEMA_SIMPLE_LIST;

        String userLangName = userService.getUserLangNameByUserId(
                user.getUserId(), user.getTenantId(), user.getToken());
        if("zh_TW".equals(userLangName)){
            tileList = CommonConstants.ECHO_COMPLEX_TITLE;
            echoList = CommonConstants.ECHO_INFO_COMPLEX_LIST;
            schemaList = CommonConstants.SCHEMA_COMPLEX_LIST;
        }
        List<EchoInformationResDTO> echoInfos = echoInformationService.getExceptionInfoByMessageId(messageId,user);
        List<EchoSchemaEIResDTO> echoSchemaElementLogs = getSchemaElInfoByMessageId(messageId,user);
        try {
            res.put(tileList.get(0),convertEchoInfoObject(echoInfos,echoList));
            res.put(tileList.get(1),convertEchoSchemaObject(echoSchemaElementLogs,schemaList));
        }catch (Exception e){
            log.info("getInfoAndSchema convertObject is error");
        }
        return res;
    }

    public static List<Map<String,Object>> convertEchoInfoObject(List<EchoInformationResDTO> list,List<String> enums)  {
        if(CollectionUtils.isEmpty(list)){
            return new ArrayList<>();
        }
        EchoInformationResDTO echoInformation = list.get(0);
        List<Map<String,Object>> res = new ArrayList<>();
        Field[] fields = echoInformation.getClass().getDeclaredFields();
        for (Field field : fields) {
            Map<String,Object> map = new HashMap<>();
            switch (field.getName()){
                case "appName":
                    map.put(enums.get(0), StringUtils.isEmpty(echoInformation.getAppName()) ? "" : echoInformation.getAppName());
                    break;
                case "description":
                    map.put(enums.get(1), StringUtils.isEmpty(echoInformation.getDescription()) ? "" : echoInformation.getDescription());
                    break;
                case "createTime":
                    map.put(enums.get(2),Objects.isNull(echoInformation.getCreateTime()) ? "" : echoInformation.getCreateTime());
                    break;
                case "content":
                    map.put(enums.get(3),StringUtils.isEmpty(echoInformation.getContent()) ? "" : echoInformation.getContent());
                    break;
                case "extend":
                    map.put(enums.get(4),StringUtils.isEmpty(echoInformation.getExtend()) ? "" : echoInformation.getExtend());
                    break;
            }
            if(!map.isEmpty()){
                res.add(map);
            }
        }
        return res;
    }

    public static List<Map<String,Object>> convertEchoSchemaObject(List<EchoSchemaEIResDTO> list,List<String> enums )  {
        if(CollectionUtils.isEmpty(list)){
            return new ArrayList<>();
        }
        EchoSchemaEIResDTO echoSchemaElementLog = list.get(0);
        List<Map<String,Object>> res = new ArrayList<>();
        Field[] fields = echoSchemaElementLog.getClass().getDeclaredFields();
        for (Field field : fields) {
            Map<String,Object> map = new HashMap<>();
            switch (field.getName()){
                case "target":
                    map.put(enums.get(0),StringUtils.isEmpty(echoSchemaElementLog.getTarget()) ? "" : echoSchemaElementLog.getTarget());
                    break;
                case "point":
                    map.put(enums.get(1),StringUtils.isEmpty(echoSchemaElementLog.getPoint()) ? "" : echoSchemaElementLog.getPoint());
                    break;
                case "schemaGroup":
                    map.put(enums.get(2),StringUtils.isEmpty(echoSchemaElementLog.getSchemaGroup()) ? "" : echoSchemaElementLog.getSchemaGroup());
                    break;
                case "schemaTime":
                    map.put(enums.get(3),StringUtils.isEmpty(echoSchemaElementLog.getSchemaTime()) ? "" : echoSchemaElementLog.getSchemaTime());
                    break;
                case "schemaCondition":
                    map.put(enums.get(4),StringUtils.isEmpty(echoSchemaElementLog.getSchemaCondition()) ? "" : echoSchemaElementLog.getSchemaCondition());
                    break;
                case "schemaRestrict":
                    map.put(enums.get(5),StringUtils.isEmpty(echoSchemaElementLog.getSchemaRestrict()) ? "" : echoSchemaElementLog.getSchemaRestrict());
                    break;
                case "schemaSort":
                    map.put(enums.get(6),StringUtils.isEmpty(echoSchemaElementLog.getSchemaSort()) ? "" : echoSchemaElementLog.getSchemaSort());
                    break;
            }
            if(!map.isEmpty()){
                res.add(map);
            }
        }
        return res;
    }
}