package com.digiwin.athena.show.metadata;

import com.digiwin.athena.agiledataecho.app.env.EchoEnvProperties;
import com.digiwin.athena.agiledataecho.constant.ErrorCodeEnum;
import com.digiwin.athena.appcore.domain.BaseResultDTO;
import com.digiwin.athena.appcore.exception.BusinessException;
import com.digiwin.athena.appcore.util.HttpUtils;
import com.digiwin.athena.appcore.util.JsonUtils;
import com.digiwin.athena.appcore.util.MessageUtils;
import com.digiwin.athena.show.util.ClassResourceUtil;
import com.fasterxml.jackson.core.type.TypeReference;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import net.sf.json.JSONObject;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.core.ParameterizedTypeReference;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpMethod;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Service;
import org.springframework.util.StringUtils;
import org.springframework.web.client.RestTemplate;

import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * @description: 元数据服务
 **/
@Slf4j
@Service
public class MetadataServiceImpl implements MetadataService {

    @Autowired
    private EchoEnvProperties envProperties;

    @Autowired
    private RestTemplate restTemplate;

    @Autowired
    MessageUtils messageUtils;

    private static final String patternStr = "^\\((?<length>\\d*?),(?<place>\\d*?)\\)$|^(?<size>\\d*?)$";

    private static final Pattern pattern = Pattern.compile(patternStr);
    /**
     * 元数据获取地址
     */
    static String METADATA_URL = "knowledgegraph/Action/Metadata";

    /**
     * 资产中心数据获取地址
     */
    static String MODEL_CENTER_METADATA_URL = "scene/modelCenterMetaData";

    @Override
    public ApiMetadata getMetadata(String locale, String actionId) {
        if (!StringUtils.hasText(actionId)) {
            return null;
        }
        if (StringUtils.isEmpty(locale)) {
            locale = "zh_CN";
        }
        String url = envProperties.getKnowledgeMapsUrl() + METADATA_URL + "?actionId=" + actionId;
        HttpHeaders headers = new HttpHeaders();
        headers.add("locale", locale);
        HttpEntity<String> requestEntity = new HttpEntity<>(null, headers);
        try {
            ResponseEntity<Object> respEntity = restTemplate.exchange(url, HttpMethod.GET, requestEntity,
                    new ParameterizedTypeReference<Object>() {
                    });
            BaseResultDTO<MetadataDataDTO> resultDTO = HttpUtils.descResponseRtnBody(url, null, respEntity, new TypeReference<MetadataDataDTO>() {
            });
            if (resultDTO == null) {
                throw BusinessException.create(ErrorCodeEnum.METADATA_GET_ERROR.getErrCode(), String.format(messageUtils.getMessage("exception.metadata.error"), actionId));

            }
            MetadataDataDTO metadataDataDTO = resultDTO.getResponse();
            if (null == metadataDataDTO) {
                throw BusinessException.create(ErrorCodeEnum.METADATA_GET_ERROR.getErrCode(), String.format(messageUtils.getMessage("exception.metadata.error"), actionId));
            }
            return createApiMetadata(metadataDataDTO);
        } catch (Exception e) {
            log.error("url: {}, error: ", url, e);
            throw e;
        }
    }

    @Override
    public ApiMetadata getModelMetadata(String locale, String modelId) {
        if (!StringUtils.hasText(modelId)) {
            return null;
        }
        if (StringUtils.isEmpty(locale)) {
            locale = "zh_CN";
        }
        String url = envProperties.getKnowledgeMapsUrl() + MODEL_CENTER_METADATA_URL;
        HttpHeaders headers = new HttpHeaders();
        headers.add("locale", locale);
        Map<String,Object> body = Maps.newHashMap();
        body.put("modelIds", Lists.newArrayList(modelId));
        HttpEntity<String> requestEntity = new HttpEntity(body, headers);
        try {
            ResponseEntity<Object> respEntity = restTemplate.exchange(url, HttpMethod.POST, requestEntity,
                    new ParameterizedTypeReference<Object>() {
                    });
            BaseResultDTO<MetadataDataDTO> resultDTO = HttpUtils.descResponseRtnBody(url, null, respEntity, new TypeReference<MetadataDataDTO>() {
            });
            if (resultDTO == null) {
                throw BusinessException.create(ErrorCodeEnum.METADATA_GET_ERROR.getErrCode(), String.format(messageUtils.getMessage("exception.metadata.error"), modelId));

            }
            MetadataDataDTO metadataDataDTO = resultDTO.getResponse();
            if (null == metadataDataDTO) {
                throw BusinessException.create(ErrorCodeEnum.METADATA_GET_ERROR.getErrCode(), String.format(messageUtils.getMessage("exception.metadata.error"), modelId));
            }
            return createApiMetadata(metadataDataDTO);
        } catch (Exception e) {
            log.error("url: {}, error: ", url, e);
            throw e;
        }
    }

    @Override
    public ApiMetadata getMockMetadata(String actionId) {
        log.info("=== getMockMetadata, actionId: {}", actionId);
        if (!StringUtils.hasText(actionId)) {
            return null;
        }
        String resourcePath = MessageFormat.format("mock/apiMetadata/{0}.json", actionId);
        if (StringUtils.isEmpty(resourcePath)) {
            return null;
        }
        try {
            String apiMetadataStr = ClassResourceUtil.readFile(resourcePath);
            JSONObject jsonStr = JsonUtils.jsonToObject(apiMetadataStr, JSONObject.class);
            if (jsonStr.containsKey("response")) {
                MetadataDataDTO metadataDataDTO = JsonUtils.jsonToObject(jsonStr.get("response").toString(), MetadataDataDTO.class);
                return createApiMetadata(metadataDataDTO);
            }
            return null;
        } catch (Exception e) {
            return null;
        }
    }


    /**
     * 创建单个元数据结构
     *
     * @param metadataDataDTO
     * @return
     */
    @Override
    public ApiMetadata createApiMetadata(MetadataDataDTO metadataDataDTO) {
        //通过bpm推送过来的元数据，重新构建元数据

        ApiMetadata apiMetadata = new ApiMetadata();
        String actionId = metadataDataDTO.getActionId();
        String serviceName = metadataDataDTO.getServiceName();
        apiMetadata.setActionId(actionId);
        apiMetadata.setServiceName(serviceName);
        if (metadataDataDTO.getRequest() != null && metadataDataDTO.getRequest().getParameters() != null) {
            apiMetadata.setRequestFields(createMetadataField(new ArrayList<>(), metadataDataDTO.getRequest().getParameters()));
        }
        if (metadataDataDTO.getResponse() != null && metadataDataDTO.getResponse().getData() != null) {
            apiMetadata.setResponseFields(createMetadataField(new ArrayList<>(), metadataDataDTO.getResponse().getData()));
        }
        apiMetadata.setSceneNodeType(metadataDataDTO.getSceneNodeType());
        apiMetadata.setShowType(metadataDataDTO.getShowType());
        apiMetadata.setName(metadataDataDTO.getName());

        return apiMetadata;
    }

    /**
     * 创建单个元数据结构
     *
     * @param apiMetadataObj
     * @return
     */
    @Override
    public ApiMetadata createApiMetadata(String actionId, String serviceName, Object apiMetadataObj) {
        //通过bpm推送过来的元数据，重新构建元数据
        MetadataDataDTO metadataDataDTO = new MetadataDataDTO();
        metadataDataDTO.setActionId(actionId);
        metadataDataDTO.setServiceName(serviceName);
        MetadataFieldDTO data = JsonUtils.jsonToObject(JsonUtils.objectToString(apiMetadataObj), MetadataFieldDTO.class);
        MetadataResponseDTO responseDTO = new MetadataResponseDTO();
        responseDTO.setData(data);
        metadataDataDTO.setResponse(responseDTO);
        ApiMetadata apiMetadata = new ApiMetadata();
        apiMetadata.setActionId(actionId);
        apiMetadata.setServiceName(serviceName);
        if (metadataDataDTO.getRequest() != null && metadataDataDTO.getRequest().getParameters() != null) {
            apiMetadata.setRequestFields(createMetadataField(new ArrayList<>(), metadataDataDTO.getRequest().getParameters()));
        }
        if (metadataDataDTO.getResponse() != null && metadataDataDTO.getResponse().getData() != null) {
            apiMetadata.setResponseFields(createMetadataField(new ArrayList<>(), metadataDataDTO.getResponse().getData()));
        }

        return apiMetadata;
    }

    /**
     * 创建对象
     *
     * @param metadataFieldDTOs
     * @return
     */
    private static List<MetadataField> createMetadataField(List<MetadataField> requestFields, List<MetadataFieldDTO> metadataFieldDTOs) {
        for (MetadataFieldDTO metadataFieldDTO : metadataFieldDTOs) {
            MetadataField metadataField = new MetadataField();
            metadataField.setName(metadataFieldDTO.getData_name());
            if (metadataFieldDTO.getRequired() == null) {
                metadataField.setRequired(false);
            } else {
                metadataField.setRequired(metadataFieldDTO.getRequired());
            }
            metadataField.setDescription(metadataFieldDTO.getDescription());
            metadataField.setRemark(metadataFieldDTO.getRemark());
            metadataField.setBusinessKey(metadataFieldDTO.is_businesskey());
            metadataField.setDataKey(metadataFieldDTO.getIs_datakey());
            metadataField.setDataType(metadataFieldDTO.getData_type());
            metadataField.setEnumKey(metadataFieldDTO.getEnum_key());
            metadataField.setCanFilter(metadataFieldDTO.getCan_filter());
            metadataField.setCanSort(metadataFieldDTO.getCan_sort());
            metadataField.setPercent(metadataFieldDTO.getPercent());
            metadataField.setDecimal(metadataFieldDTO.getDecimal());

            //解析数据精度或者长度
            if (metadataFieldDTO.getPrecision() != null) {
                Matcher m = pattern.matcher(metadataFieldDTO.getPrecision());
                if (m.matches()) {
                    Precision precision = new Precision();
                    //如果只有长度
                    if (m.group("size") != null) {
                        precision.setLength(Integer.parseInt(m.group("size")));
                        metadataField.setPrecision(precision);
                    } else if (m.group("length") != null && m.group("place") != null) {
                        precision.setLength(Integer.parseInt(m.group("length")));
                        precision.setPlace(Integer.parseInt(m.group("place")));
                        metadataField.setPrecision(precision);
                    }
                }
            }
            metadataField.setArray(metadataFieldDTO.is_array());
            if (CollectionUtils.isNotEmpty(metadataFieldDTO.getField())) {
                List<MetadataField> subRequestFields = new ArrayList<>();
                metadataField.setSubFields(createMetadataField(subRequestFields, metadataFieldDTO.getField()));
            }
            requestFields.add(metadataField);
        }
        return requestFields;
    }


    /**
     * 创建对象
     *
     * @param metadataFieldDTO
     * @return
     */
    private static List<MetadataField> createMetadataField(List<MetadataField> requestFields, MetadataFieldDTO metadataFieldDTO) {
        MetadataField metadataField = new MetadataField();
        metadataField.setName(metadataFieldDTO.getData_name());
        if (metadataFieldDTO.getRequired() == null) {
            metadataField.setRequired(false);
        } else {
            metadataField.setRequired(metadataFieldDTO.getRequired());
        }
        metadataField.setDescription(metadataFieldDTO.getDescription());
        metadataField.setRemark(metadataFieldDTO.getRemark());
        metadataField.setDataKey(metadataFieldDTO.getIs_datakey());
        metadataField.setBusinessKey(metadataFieldDTO.is_businesskey());
        metadataField.setDataType(metadataFieldDTO.getData_type());
        metadataField.setArray(metadataFieldDTO.is_array());
        metadataField.setEnumKey(metadataFieldDTO.getEnum_key());
        metadataField.setCanFilter(metadataFieldDTO.getCan_filter());
        metadataField.setCanSort(metadataFieldDTO.getCan_sort());
        if (CollectionUtils.isNotEmpty(metadataFieldDTO.getField())) {
            List<MetadataField> subRequestFields = new ArrayList<>();
            metadataField.setSubFields(createMetadataField(subRequestFields, metadataFieldDTO.getField()));
        }
        requestFields.add(metadataField);
        return requestFields;
    }

}
