package com.digiwin.athena.show.service.impl;

import com.digiwin.athena.agiledataecho.constant.ErrorCodeEnum;
import com.digiwin.athena.agiledataecho.domain.AgileDataUserDefine;
import com.digiwin.athena.agiledataecho.proxy.adt.AdtService;
import com.digiwin.athena.appcore.auth.domain.AuthoredUser;
import com.digiwin.athena.appcore.exception.BusinessException;
import com.digiwin.athena.appcore.util.MessageUtils;
import com.digiwin.athena.show.assistant.AgileDataConstants;
import com.digiwin.athena.show.assistant.ExecuteContext;
import com.digiwin.athena.show.component.AbstractComponent;
import com.digiwin.athena.show.component.gridster.GridGroupDTO;
import com.digiwin.athena.show.component.index.AgileDataIndexComponent;
import com.digiwin.athena.show.domain.agileDataDTO.ConvertGetAgileData;
import com.digiwin.athena.show.domain.board.BoardLayoutDTO;
import com.digiwin.athena.show.domain.agileDataDTO.AgileDataBestShowRule;
import com.digiwin.athena.show.domain.agileDataDTO.ConvertAgileDataRelation;
import com.digiwin.athena.show.domain.board.BoardLayoutMongoData;
import com.digiwin.athena.show.domain.board.BoardLayoutParamsDTO;
import com.digiwin.athena.show.domain.board.SnapShotInfo;
import com.digiwin.athena.show.domain.dynamicLayout.BordDynamicDTO;
import com.digiwin.athena.show.domain.queryDefine.SnapDataDTO;
import com.digiwin.athena.show.domain.showDefine.DynamicAgileData;
import com.digiwin.athena.show.infrastructure.meta.ECHOConstants;
import com.digiwin.athena.show.manager.themeMap.ThemeAgileDataService;
import com.digiwin.athena.show.manager.themeMap.domain.ThemeMapBoardDTO;
import com.digiwin.athena.show.metadata.ApiMetadata;
import com.digiwin.athena.show.metadata.MetadataField;
import com.digiwin.athena.show.service.AgileDataBestShowRuleService;
import com.digiwin.athena.show.service.AgileReportService;
import com.digiwin.athena.show.service.DynamicLayoutBoardService;
import com.digiwin.athena.show.util.layout.AgileDataBoardComponentBuild;
import com.digiwin.athena.show.util.layout.AgileDataCheck;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.data.mongodb.core.MongoTemplate;
import org.springframework.data.mongodb.core.query.Criteria;
import org.springframework.data.mongodb.core.query.Query;
import org.springframework.data.mongodb.core.query.Update;
import org.springframework.stereotype.Service;

import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

@Service
public class DynamicLayoutBoardServiceImpl implements DynamicLayoutBoardService {

    @Autowired
    private AgileDataBestShowRuleService agileDataBestShowRuleService;

    @Autowired
    private ThemeAgileDataService themeAgileDataService;

    @Autowired
    private AdtService adtService;

    @Autowired
    private AgileReportService agileReportService;

    @Autowired
    private AgileDataPageDefineThemeMapService agileDataPageDefineThemeMapService;

    @Autowired
    private MessageUtils messageUtils;

    @Autowired
    @Qualifier("mongoAgileDataPageViewTemplate")
    private MongoTemplate mongoAgileDataPageViewTemplate;

    private static final String BOARD_LAYOUT_NAME = "agileData_board_layout";


    /**
     * 获取看板动态布局
     * @param board
     * @return
     */
    @Override
    public List<GridGroupDTO> calcDynamicLayoutBoard(AuthoredUser user,BordDynamicDTO board,String local) {
        //获取KM看板结构
        ThemeMapBoardDTO themeMapBoardDTO = themeAgileDataService.getThemeMapBoard(board.getBoardId());
        if(themeMapBoardDTO == null){
            throw BusinessException.create(ErrorCodeEnum.BOARD_NO_DATA.getErrCode(), messageUtils.getMessage("board.no.data"));
        }
        if(CollectionUtils.isEmpty(themeMapBoardDTO.getDataBoardQuestions())){
            throw BusinessException.create(ErrorCodeEnum.BOARD_NO_DATA.getErrCode(), messageUtils.getMessage("board.no.data"));
        }
        //获取用户自定义布局
        BoardLayoutMongoData userDefineLayout = this.getDynamicLayoutBoard(user,board.getBoardId());
        //自定义布局数量与看板区域数据一致，使用用户自定义布局
        if(userDefineLayout != null && CollectionUtils.isNotEmpty(userDefineLayout.getBoardLayout())){
            //用户自定义看板区域内容与收藏内容一致，使用用户自定义布局
            List<ThemeMapBoardDTO.DataBoardQuestion> userBoard = themeMapBoardDTO.getDataBoardQuestions().stream()
                    .filter(themeBoard -> userDefineLayout.getBoardLayout().stream()
                            .noneMatch(userDefineBoard -> StringUtils.equals(userDefineBoard.getId(),themeBoard.getQuestionId())))
                    .collect(Collectors.toList());
            //看板收藏内容与用户自定义内容一致，输出自定义布局
            if(themeMapBoardDTO.getDataBoardQuestions().size() == userDefineLayout.getBoardLayout().size() && CollectionUtils.isEmpty(userBoard)){
                return userDefineLayout.getBoardLayout();
            }
        }

        //设置呈现元素
        List<AbstractComponent> abstractComponents = Lists.newArrayList();

        themeMapBoardDTO.getDataBoardQuestions().stream().forEach(question->{
            //呈现类型为指标卡，重新通过ADE获取数据结构
            if(question.getShowType() == null){
                question.setShowType(0);
            }
            switch (question.getShowType()){
                case 2:
                    SnapDataDTO snapDataDTO = adtService.transAgileData(String.valueOf(question.getShowType()),question.getSnapshotId(),local);
                    AgileDataIndexComponent agileDataIndexComponent = getIndexComponent(question.getQuestionId(),snapDataDTO,local);
                    if(agileDataIndexComponent != null) {
                        abstractComponents.add(agileDataIndexComponent);
                    } else {
                        abstractComponents.add(getEmptyIndexComponent(question.getQuestionId()));
                    }
                    break;
                case 3:
                case 4:
                case 5:
                case 6:
                    abstractComponents.add(AgileDataBoardComponentBuild.buildBoardEcharts(question.getQuestionId()));
                    break;
                case 0:
                default:
                    abstractComponents.add(AgileDataBoardComponentBuild.buildBoardGrid(question.getQuestionId()));
                    break;
            }
        });



        //封装规则
        List<AgileDataBestShowRule> bestShowRules = agileDataBestShowRuleService.getBestShowRules(null);


        //动态结构计算
        List<GridGroupDTO> items = ConvertAgileDataRelation.createItems(null,  bestShowRules, abstractComponents,userDefineLayout);
        return items;
    }

    /**
     * 获取指标卡组件
     * @param questionId
     * @param snapDataDTO
     * @return
     */
    private AgileDataIndexComponent getIndexComponent(String questionId,SnapDataDTO snapDataDTO,String local) {
        if(snapDataDTO == null){
            return null;
        }
        String actionId = snapDataDTO.getPageData().keySet().iterator().next();
        ExecuteContext executeContext  = new ExecuteContext();
        executeContext.setLocale(local);
        ApiMetadata apiMetadata = agileDataPageDefineThemeMapService.getApiMetadata(snapDataDTO,executeContext,actionId);
        return AgileDataBoardComponentBuild.buildBoardIndex(questionId, ConvertGetAgileData.getRootMetadataField(apiMetadata));
    }

    private AgileDataIndexComponent getEmptyIndexComponent(String questionId) {
        AgileDataIndexComponent agileDataIndexComponent = new AgileDataIndexComponent();
        agileDataIndexComponent.setType(ECHOConstants.ComponentType.STATISTIC);
        List<AgileDataIndexComponent.AgileDataIndex> group = Lists.newArrayList();
        group.add(new AgileDataIndexComponent.AgileDataIndex());
        agileDataIndexComponent.setGroup(group);
        agileDataIndexComponent.setId(questionId);
        return agileDataIndexComponent;
    }

    /**
     * 生成看板呈现结构
     * @param user
     * @param boardLayoutParamsDTO
     * @param local
     * @return
     */
    @Override
    public Map<String, Object> generateBoardLayout(AuthoredUser user, BoardLayoutParamsDTO boardLayoutParamsDTO, String local) {
        Map<String, Object> result = Maps.newHashMap();
        //获取看板数据
        SnapShotInfo snapShotInfo = adtService.getBoardData(boardLayoutParamsDTO,local);
        if(snapShotInfo == null){
            return result;
        }
        String snapshotId = snapShotInfo.getSnapshotId();
        //生成DSL结构
        ExecuteContext executeContext = ExecuteContext.createByHttpRequest();
        //获取敏捷报表信息
        SnapDataDTO agileReportDTO = adtService.getAdtReportBySnapshotId(snapshotId, executeContext.getLocale());
        if (AgileDataCheck.sceneBasicCheck(agileReportDTO)) {
            return result;
        }
        //解析报表信息
        executeContext.appendAgileDataPageInfo(AgileDataConstants.REPORT_PAGE, agileReportDTO,null,"0");
        //封装呈现内容
        DynamicAgileData dynamicAgileData = agileReportService.createAgileReportTemplateTable(executeContext, agileReportDTO);
        result.put("dsl",dynamicAgileData);
        Object dataTipMessage = snapShotInfo.getContext().getBizParams().get("dataTipMessage");
        if(dataTipMessage != null){
            result.put("dataTipMessage",dynamicAgileData);
        }
        return result;
    }

    /**
     * 保存看板布局
     * @param authoredUser
     * @param boardLayoutDTO
     * @return
     */
    @Override
    public Boolean saveDynamicLayoutBoard(AuthoredUser authoredUser, BoardLayoutDTO boardLayoutDTO) {
        if(CollectionUtils.isNotEmpty(boardLayoutDTO.getBoardLayout())){
            Query query = Query.query(Criteria.where("userId").is(authoredUser.getUserId())
                    .and("tenantId").is(authoredUser.getTenantId())
                    .and("boardId").is(boardLayoutDTO.getBoardId()));
            BoardLayoutMongoData boardLayoutMongoData = mongoAgileDataPageViewTemplate.findOne(query, BoardLayoutMongoData.class, BOARD_LAYOUT_NAME);
            if(boardLayoutMongoData == null) {
                boardLayoutMongoData = new BoardLayoutMongoData();
                boardLayoutMongoData.setBoardLayout(boardLayoutDTO.getBoardLayout());
                boardLayoutMongoData.setBoardId(boardLayoutDTO.getBoardId());
                boardLayoutMongoData.setUserId(authoredUser.getUserId());
                boardLayoutMongoData.setTenantId(authoredUser.getTenantId());
                mongoAgileDataPageViewTemplate.insert(boardLayoutMongoData, BOARD_LAYOUT_NAME);
            } else {
                Update update = new Update();
                update.set("boardLayout",boardLayoutDTO.getBoardLayout());
                mongoAgileDataPageViewTemplate.upsert(query,update, AgileDataUserDefine.class,BOARD_LAYOUT_NAME);
            }
        }
        return true;
    }

    /**
     * 获取看板布局
     * @param authoredUser
     * @param boardId
     * @return
     */
    @Override
    public BoardLayoutMongoData getDynamicLayoutBoard(AuthoredUser authoredUser, String boardId) {
        Query query = Query.query(Criteria.where("userId").is(authoredUser.getUserId())
                .and("tenantId").is(authoredUser.getTenantId())
                .and("boardId").is(boardId));
        return mongoAgileDataPageViewTemplate.findOne(query, BoardLayoutMongoData.class, BOARD_LAYOUT_NAME);
    }
}
