package com.digiwin.athena.show.util.layout;

import com.digiwin.athena.show.assistant.ExecuteContext;
import com.digiwin.athena.show.component.AbstractComponent;
import com.digiwin.athena.show.component.gridster.GridGroupDTO;
import com.digiwin.athena.show.domain.agileDataDTO.*;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang.StringUtils;

import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * 构建标准布局
 */
public class AgileDataGridsterBestShow {

    private AgileDataGridsterBestShow() {
        throw new IllegalStateException("Utility class");
    }


    /**
     * 创建
     *
     * @param relations
     */
    public static List<GridGroupDTO> createItems(List<AgileDataRelationDTO> relations, List<AbstractComponent> components, ExecuteContext executeContext) {
        //获取用户级别看板规则
        executeContext.getAgileData();
        if(ConvertAgileDataUtils.isNeedDefaultModel(executeContext)){
            return buildUserItems(components,executeContext);
        }
        //关系未配置，走默认规则
        if (CollectionUtils.isEmpty(relations) || relations.size() < 2) {
            return createDefaultItems(relations, components);
        }
        //关系未绑定，走默认规则
        List<AgileDataRelationDTO> componentsRelation = relations.stream().filter(relation -> CollectionUtils.isEmpty(relation.getAbstractComponents())).collect(Collectors.toList());
        if (CollectionUtils.isNotEmpty(componentsRelation)) {
            return createDefaultItems(relations, components);
        }
        //设置默认数据流关系
        setEmptyRelation(relations, components);
        //设置层级
        setRelationLevel(relations, 0);
        //按关系初始化结构
        List<AgileDataGridsterArrange> arranges = initArrange(relations);
        //按区域调整宽高
        ConvertAgileDataRelation.reSize(arranges);
        //按区域宽高二次调整
//        ConvertAgileDataRelation.reSize(arranges);
        //组装为前端结构
        return ConvertAgileDataRelation.assembleGridster(arranges);
    }

    /**
     * 构建用户级大屏看板
     * @param components
     * @param executeContext
     */
    private static List<GridGroupDTO> buildUserItems(List<AbstractComponent> components, ExecuteContext executeContext){
        List<GridGroupDTO> items = Lists.newArrayList();
        Map<String,Object> agileDataMap = executeContext.getAgileData();
        Object screenModelObj = MapUtils.getObject(agileDataMap,"screenModel");
        String actionIds = MapUtils.getString(agileDataMap,"screenActionIds");
        if(StringUtils.isEmpty(actionIds)){
            return null;
        }
        if(screenModelObj == null){
            return null;
        }
        List<String> actionIdList = Lists.newArrayList(actionIds.split(","));
        AgileDataScreenModel screenModel = (AgileDataScreenModel) screenModelObj;
        List<AgileDataScreenModel.ScreenModel> layoutModels = screenModel.getModels();
        for (int i = 0; i < layoutModels.size(); i++) {
            String actionId = null;
            if(i < actionIdList.size()) {
                actionId = actionIdList.get(i);
            }
            //对应的模板信息
            AgileDataScreenModel.ScreenModel layoutModel = layoutModels.get(i);

            // 获取组件
            String finalActionId = actionId;
            AbstractComponent abstractComponent = null;
            if(finalActionId != null) {
                 abstractComponent = components.stream()
                        .filter(component -> StringUtils.equals(finalActionId, component.getId()) || StringUtils.equals(finalActionId, component.getPath()))
                        .findFirst().orElse(null);
            }

            // 处理布局模型和组件的关系
            if (layoutModel != null) {
                GridGroupDTO item = new GridGroupDTO();
                if(abstractComponent == null) {
                    item.setId("tempArea" + i);
                } else {
                    item.setId(abstractComponent.getId());
                    item.setType(abstractComponent.getType());
                }
                item.setCols(layoutModel.getCols());
                item.setRows(layoutModel.getRows());
                item.setX(layoutModel.getX());
                item.setY(layoutModel.getY());
                items.add(item);
            }
        }
        return items;
    }

    /**
     * 设置无关系数据流
     *
     * @param relations
     * @param components
     */
    private static void setEmptyRelation(List<AgileDataRelationDTO> relations, List<AbstractComponent> components) {
        Map<String, AgileDataRelationDTO> relationDTOMap = getRelationMap(relations);
        components.stream().forEach(component -> {
            if (!relationDTOMap.containsKey(component.getId())) {
                AgileDataRelationDTO agileDataRelationDTO = new AgileDataRelationDTO();
                agileDataRelationDTO.setActionId(component.getId());
                agileDataRelationDTO.setSort(99999);
                agileDataRelationDTO.setAbstractComponents(Lists.newArrayList(component));
                relations.add(agileDataRelationDTO);
            }
        });
    }

    /**
     * 平铺关系结构
     *
     * @param relations
     * @return
     */
    private static Map<String, AgileDataRelationDTO> getRelationMap(List<AgileDataRelationDTO> relations) {
        Map<String, AgileDataRelationDTO> relationDTOMap = Maps.newHashMap();
        relations.stream().forEach(relation -> {
            relationDTOMap.put(relation.getActionId(), relation);
            if (CollectionUtils.isNotEmpty(relation.getSubAction())) {
                relationDTOMap.putAll(getRelationMap(relation.getSubAction()));
            }
        });
        return relationDTOMap;
    }


    /**
     * 初始化整体结构
     *
     * @param relations
     * @return
     */
    private static List<AgileDataGridsterArrange> initArrange(List<AgileDataRelationDTO> relations) {
        List<AgileDataGridsterArrange> arranges = Lists.newArrayList();
        //行排序,级别为1的切分行，按顺序排序
        relations = relations.stream().filter(relation -> 1 == relation.getLevel()).sorted(Comparator.comparing(AgileDataRelationDTO::getSort)).collect(Collectors.toList());
        //构建结构
        relations.stream().forEach(relation -> {
            //每一层级作为一行
            AgileDataGridsterArrange agileDataGridsterArrange = new AgileDataGridsterArrange();
            //当前按根节点关系按层级再次分组
            Map<Integer, List<AgileDataRelationDTO>> relationMap = groupByLevel(relation);
            //构建区域
            relationMap.forEach((level, subRelations) -> {
                AgileDataGridsterGroup group = new AgileDataGridsterGroup();
                agileDataGridsterArrange.getGroups().add(group);
                subRelations.stream().forEach(subRelation -> subRelation.getAbstractComponents().stream().forEach(component -> {
                    AgileDataGridsterArea area = new AgileDataGridsterArea(component,subRelation.getType());
                    //设置宽高
                    area.setSize();
                    //设置后续节点，用于拉齐宽度
                    Set<Integer> nextLevel = Sets.newHashSet();
                    getNextLevel(subRelation, nextLevel);
                    area.setNextLevel(nextLevel);
                    //构建组
                    group.getAreas().add(area);
                    group.setSize();
                    group.setDirection("row");
                }));
            });
            agileDataGridsterArrange.setSize();
            arranges.add(agileDataGridsterArrange);
        });


        return arranges;
    }

    /**
     * 获取后续节点
     *
     * @param relationDTO
     * @param xIndex
     */
    private static void getNextLevel(AgileDataRelationDTO relationDTO, Set<Integer> xIndex) {
        if (CollectionUtils.isNotEmpty(relationDTO.getSubAction())) {
            relationDTO.getSubAction().stream().forEach(subRelation -> {
                xIndex.add(subRelation.getLevel());
                if (CollectionUtils.isNotEmpty(subRelation.getSubAction())) {
                    getNextLevel(subRelation, xIndex);
                }
            });
        }
    }

    /**
     * 按根节点分组，组内排序
     *
     * @param relation
     * @return
     */
    private static Map<Integer, List<AgileDataRelationDTO>> groupByLevel(AgileDataRelationDTO relation) {
        //平铺节点
        List<AgileDataRelationDTO> relations = Lists.newArrayList();
        getAllRelation(relation, relations);
        //分组，排序
        return relations.stream()
                .collect(Collectors.groupingBy(AgileDataRelationDTO::getLevel,
                        Collectors.collectingAndThen(
                                Collectors.toList(),
                                list -> {
                                    list.sort(Comparator.comparing(AgileDataRelationDTO::getSort));
                                    return list;
                                })));
    }

    /**
     * 平铺节点
     *
     * @param relation
     * @param relations
     */
    public static void getAllRelation(AgileDataRelationDTO relation, List<AgileDataRelationDTO> relations) {
        if (CollectionUtils.isNotEmpty(relation.getAbstractComponents())) {
            relations.add(relation);
        }
        //关系未配置，走默认规则
        if (CollectionUtils.isNotEmpty(relation.getSubAction())) {
            relation.getSubAction().stream().forEach(subRelation -> getAllRelation(subRelation, relations));
        }
    }

    /**
     * 默认按组件类型规则布局
     *
     * @param relations
     * @param components
     * @return
     */
    private static List<GridGroupDTO> createDefaultItems(List<AgileDataRelationDTO> relations, List<AbstractComponent> components) {
        List<AgileDataBestShowRule> bestShowRules = Lists.newArrayList();
        Map<String, List<AbstractComponent>> abstractComponentMap = components.stream().collect(Collectors.groupingBy(AbstractComponent::getType));
        //图形规则
        if (abstractComponentMap.containsKey("ECHARTS")) {
            int size = abstractComponentMap.get("ECHARTS").size();
            AgileDataBestShowRule bestShowRule = new AgileDataBestShowRule();
            bestShowRule.setRuleType("ECHARTS_NUM");
            switch (size) {
                case 1:
                case 2:
                case 3:
                case 5:
                case 6:
                    //一行3个
                    bestShowRule.setRuleValue("3");
                    break;
                case 4:
                    //一行2个
                    bestShowRule.setRuleValue("2");
                    break;
                case 7:
                    //一行4个
                    bestShowRule.setRuleValue("4");
                    break;
                default:
                    //一行2个
                    bestShowRule.setRuleValue("2");
                    break;
            }
            bestShowRules.add(bestShowRule);
        }
        return ConvertAgileDataRelation.createItems(relations, bestShowRules, components,null);
    }


    /**
     * 设置层级
     */
    private static void setRelationLevel(List<AgileDataRelationDTO> relations, int index) {
        if (CollectionUtils.isEmpty(relations)) {
            return;
        }
        relations.stream().forEach(relation -> {
            relation.setLevel(index + 1);
            if (relation.getSort() == null) {
                relation.setSort(99999);
            }
            if (CollectionUtils.isNotEmpty(relation.getSubAction())) {
                setRelationLevel(relation.getSubAction(), index + 1);
            }
        });
    }


}
